//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Sample/RoughnessItems.cpp
//! @brief     Implements classes RoughnessItems
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/Model/Sample/RoughnessItems.h"
#include "GUI/Support/XML/UtilXML.h"

namespace {
namespace Tag {

const QString Sigma("Sigma");
const QString Hurst("Hurst");
const QString LateralCorrelationLength("LateralCorrelationLength");

} // namespace Tag
} // namespace

BasicRoughnessItem::BasicRoughnessItem()
{
    m_sigma.init("Sigma", "rms of the roughness", 1.0, Unit::nanometer, "sigma");
    m_hurst.init("Hurst",
                 "Hurst parameter which describes how jagged the interface,\n "
                 "dimensionless [0.0, 1.0], where 0.0 gives more spikes, \n1.0 more smoothness.",
                 0.3, Unit::unitless, 3 /* decimals */, RealLimits::limited(0.0, 1.0), "hurst");
    m_lateralCorrelationLength.init("Correlation length",
                                    "Lateral correlation length of the roughness", 5.0,
                                    Unit::nanometer, "corrLen");
}

void BasicRoughnessItem::writeTo(QXmlStreamWriter* w) const
{
    XML::writeAttribute(w, XML::Attrib::version, uint(1));

    m_sigma.writeTo(w, Tag::Sigma);
    m_hurst.writeTo(w, Tag::Hurst);
    m_lateralCorrelationLength.writeTo(w, Tag::LateralCorrelationLength);
}

void BasicRoughnessItem::readFrom(QXmlStreamReader* r)
{
    const uint version = XML::readUIntAttribute(r, XML::Attrib::version);
    Q_UNUSED(version)

    while (r->readNextStartElement()) {
        QString tag = r->name().toString();

        if (tag == Tag::Sigma)
            m_sigma.readFrom(r, tag);
        else if (tag == Tag::Hurst)
            m_hurst.readFrom(r, tag);
        else if (tag == Tag::LateralCorrelationLength)
            m_lateralCorrelationLength.readFrom(r, tag);
        else
            r->skipCurrentElement();
    }
}
