package kingpin

import (
	"os"
	"path/filepath"
)

var (
	// CommandLine is the default Kingpin parser.
	CommandLine = New(filepath.Base(os.Args[0]), "")
)

// Command adds a new command to the default parser.
func Command(name, help string) *CmdClause {
	return CommandLine.Command(name, help)
}

// Flag adds a new flag to the default parser.
func Flag(name, help string) *Clause {
	return CommandLine.Flag(name, help)
}

// Arg adds a new argument to the top-level of the default parser.
func Arg(name, help string) *Clause {
	return CommandLine.Arg(name, help)
}

// Struct creates a command-line from a struct.
func Struct(v interface{}) *Application {
	err := CommandLine.Struct(v)
	FatalIfError(err, "")
	return CommandLine
}

// Parse and return the selected command. Will call the termination handler if
// an error is encountered.
func Parse() string {
	selected := MustParse(CommandLine.Parse(os.Args[1:]))
	if selected == "" && CommandLine.cmdGroup.have() {
		Usage()
		CommandLine.terminate(0)
	}
	return selected
}

// Errorf prints an error message to stderr.
func Errorf(format string, args ...interface{}) {
	CommandLine.Errorf(format, args...)
}

// Fatalf prints an error message to stderr and exits.
func Fatalf(format string, args ...interface{}) {
	CommandLine.Fatalf(format, args...)
}

// FatalIfError prints an error and exits if err is not nil. The error is printed
// with the given prefix.
func FatalIfError(err error, format string, args ...interface{}) {
	CommandLine.FatalIfError(err, format, args...)
}

// FatalUsage prints an error message followed by usage information, then
// exits with a non-zero status.
func FatalUsage(format string, args ...interface{}) {
	CommandLine.FatalUsage(format, args...)
}

// FatalUsageContext writes a printf formatted error message to stderr, then
// usage information for the given ParseContext, before exiting.
func FatalUsageContext(context *ParseContext, format string, args ...interface{}) {
	CommandLine.FatalUsageContext(context, format, args...)
}

// Usage prints usage to stderr.
func Usage() {
	CommandLine.Usage(os.Args[1:])
}

// UsageTemplate associates a template with a flag. The flag must be a Bool() and must
// already be defined.
func UsageTemplate(template string) *Application {
	return CommandLine.UsageTemplate(template)
}

// MustParse can be used with app.Parse(args) to exit with an error if parsing fails.
func MustParse(command string, err error) string {
	if err != nil {
		Fatalf(T("{{.Arg0}}, try --help", V{"Arg0": err}))
	}
	return command
}

// Version adds a flag for displaying the application version number.
func Version(version string) *Application {
	return CommandLine.Version(version)
}
