//*****************************************************************************
//                                 PnlValue.cpp                               *
//                                --------------                              *
//  Started     : 14/09/2004                                                  *
//  Last Update : 11/04/2011                                                  *
//  Copyright   : (C) 2004 by MSWaters                                        *
//  Email       : M.Waters@bom.gov.au                                         *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "utility/PnlValue.hpp"

//*****************************************************************************
// Implement an event table.

BEGIN_EVENT_TABLE( PnlValue, wxPanel )

  EVT_SPIN_UP  ( SpinCtrl::ID_SPINBTN, PnlValue::OnSpnScroll )
  EVT_SPIN_DOWN( SpinCtrl::ID_SPINBTN, PnlValue::OnSpnScroll )

  EVT_CHOICE   ( ID_CHO_UNITS,         PnlValue::OnChoUnits  )

END_EVENT_TABLE( )

//*****************************************************************************
// Constructor.

PnlValue::PnlValue( void ) : wxPanel( )
{
}

//*****************************************************************************
// Destructor.

PnlValue::~PnlValue( )
{
}

//*****************************************************************************
// Convert a temperature value to Celcius from the units specified in the units
// choice control.
// Note: This conversion will only be done if the units are of temperature.
//
// Argument List :
//   pfValue - The value to be converted
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PnlValue::bTempToDegC( float * pfValue )
{
  if( ! bIsCreated( ) )                             return( FALSE );
  if( ! m_bUseUnits || ! m_oChoUnits.IsShown( ) )   return( FALSE );
  if( m_oChoUnits.eGetUnitsType( ) != eUNITS_TEMP ) return( FALSE );

  switch( m_oChoUnits.GetSelection( ) )
  {
    case 0 :                                       break; // Celcius
    case 1 : *pfValue = (*pfValue + 32.0) / 2.44 ; break; // Fahrenheit
    case 2 : *pfValue -= 273.15;                   break; // Kelvin
    default: return( FALSE );
  }

  return( TRUE );
}

//*****************************************************************************
// Convert a phase value to Degree from the units specified in the units
// choice control.
// Note: This conversion will only be done if the units are of phase.
//
// Argument List :
//   pfValue - The value to be converted
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PnlValue::bPhaseToDeg( float * pfValue )
{
  if( ! bIsCreated( ) )                              return( FALSE );
  if( ! m_bUseUnits || ! m_oChoUnits.IsShown( ) )    return( FALSE );
  if( m_oChoUnits.eGetUnitsType( ) != eUNITS_PHASE ) return( FALSE );

  switch( m_oChoUnits.GetSelection( ) )
  {
    case 0 :                     break; // Degree
    case 1 : *pfValue /= 6.2832; break; // Radian
    case 2 : *pfValue *= 0.9;    break; // Grad
    default: return( FALSE );
  }

  return( TRUE );
}

//*****************************************************************************
// Layout the display objects.

void  PnlValue::DoLayout( void )
{
  wxBoxSizer  * poSzr;
  wxSizerFlags  oFlags;

  // Create the sizer for the panel
  poSzr = new wxBoxSizer( wxHORIZONTAL );

  oFlags.Expand( );
  oFlags.Border( wxTOP | wxBOTTOM, 1 );

  // Add the value label
  if( m_oLblName.GetParent( ) != NULL )
  {
    oFlags.Proportion( 1 );
// The lines marked with ??? are a fix for a bug in wxWidgets ie. the following line has no effect.
    oFlags.Align( wxALIGN_LEFT | wxALIGN_CENTER_VERTICAL );
oFlags.Border( wxTOP, 7 );            // ??? 26/05/2007
    poSzr->Add( &m_oLblName, oFlags );
oFlags.Border( wxTOP | wxBOTTOM, 1 ); // ??? 26/05/2007
    poSzr->AddSpacer( 5 );
  }

  // Add the value itself
  oFlags.Proportion( 0 );
  oFlags.Center( );
  poSzr->Add( &m_oSpnValue, oFlags );

  // Add units for the value
  if( m_bUseUnits )
  {
    poSzr->AddSpacer( 2 );
    poSzr->Add( &m_oChoUnits, oFlags );
oFlags.Border( wxTOP, 7 );            // ??? 26/05/2007
    poSzr->Add( &m_oLblUnits, oFlags );
  }

  // Set the panel sizer and the min. & init. sizes as calculated by the sizer
  SetSizer( poSzr );
  poSzr->SetSizeHints( this );
}

//*****************************************************************************
// Create an instance of this object.
//
// Argument List :
//   poWin  - The parent window
//   oWinID - The window identifier
//   iWidth - The width of the name label in pixels
//   roPosn - The position
//   bUnits - Flag indicating if units should be accommodated
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PnlValue::bCreate( wxWindow * poWin, wxWindowID oWinID, int iWidth,
                         const wxPoint & roPosn, bool bUnits )
{
  bool  bRtn=TRUE;
  int   iWd, iHt;

  if( bIsCreated( ) )                                     return( TRUE );

  // Create the base class (wxPanel)
  if( ! Create( poWin, oWinID, roPosn ) )                 return( FALSE );

  // Create the variable name label
  if( iWidth > 0 )
    if( ! m_oLblName.Create( this, ID_UNUSED, wxT(""), wxDefaultPosition,
                             wxSize( iWidth, PNLVALUE_HT ), wxALIGN_LEFT ) )
                                                          bRtn = FALSE;

  // Create the value spin control
  if( ! m_oSpnValue.bCreate( this, ID_SPN_VALUE, 80 ) )   bRtn = FALSE;

  // Create the units controls
  m_bUseUnits = bUnits;
  if( m_bUseUnits )
  {
    // Create the units choice box
    if( ! m_oChoUnits.bCreate( this, ID_CHO_UNITS, 85 ) ) bRtn = FALSE;

    // Create the units label
    m_oChoUnits.GetClientSize( &iWd, &iHt );
    if( ! m_oLblUnits.Create( this, ID_LBL_UNITS, wxT(""), wxDefaultPosition,
                              wxSize( iWd, PNLVALUE_HT ), wxST_NO_AUTORESIZE ) )
                                                          bRtn = FALSE;
    m_oLblUnits.Show( FALSE );
  }

  // Layout the display objects
  DoLayout( );

  return( bRtn );
}

//*****************************************************************************
// Clear the spin and choice values.
//
// Return Values :
//   Success - TRUE
//   Failure - FALSE

bool  PnlValue::bClear( void )
{
  bool  bRtn=TRUE;

  if( ! m_oSpnValue.bClear( ) ) bRtn = FALSE;
  if( ! m_oChoUnits.bClear( ) ) bRtn = FALSE;

  return( bRtn );
}

//*****************************************************************************
// Set default object attributes.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PnlValue::bSetDefaults( void )
{
  bool  bRtn=TRUE;

  if( ! m_oSpnValue.bSetDefaults( ) ) bRtn = FALSE;
  if( ! m_oChoUnits.bSetDefaults( ) ) bRtn = FALSE;

  return( bRtn );
}

//*****************************************************************************
// Set the name label.
//
// Argument List :
//   rosName - The name of the variable
//
// Return Values :
//   Success - TRUE
//   Failure - FALSE

bool  PnlValue::bSetName( const wxString & rosName )
{
  if( ! bIsCreated( ) )                 return( FALSE );
  if( m_oLblName.GetParent( ) == NULL ) return( FALSE );
  if( rosName.IsEmpty( ) )              return( FALSE );

  m_oLblName.SetLabel( rosName );

  return( TRUE );
}

//*****************************************************************************
// Set the variable type for the spin control to display.
//
// Argument List :
//   eVType - The variable type
//
// Return Values :
//   Success - TRUE
//   Failure - FALSE

bool  PnlValue::bSetVarType( eVarType eVType )
{
  if( ! bIsCreated( ) ) return( FALSE );

  return( m_oSpnValue.bSetVarType( eVType ) );
}

//*****************************************************************************
// Set the spin control range.
//
// Argument List :
//   fMinValue - The spin control minimum value
//   fMaxValue - The spin control maximum value
//
// Return Values :
//   Success - TRUE
//   Failure - FALSE

bool  PnlValue::bSetRange( float fMinValue, float fMaxValue )
{
  if( ! bIsCreated( ) ) return( FALSE );

  return( m_oSpnValue.bSetRange( fMinValue, fMaxValue ) );
}

//*****************************************************************************
// Set the increment sizes of the spin control.
//
// This spin control can be incremented using two different approaches,
// constant or variable step sizes. A constant step size means that the spin
// control is incremented by the same amount throughout it's range. A variable
// step size means that the spin control is incremented by an amount dependent
// on it's current value.
//
// Argument List :
//   fMinIncSz - The minimum spin control increment size
//   fMaxIncSz - The maximum spin control increment size
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PnlValue::bSetIncSz( float fMinIncSz, float fMaxIncSz )
{
  if( ! bIsCreated( ) ) return( FALSE );

  return( m_oSpnValue.bSetIncSz( fMinIncSz, fMaxIncSz ) );
}

//*****************************************************************************
// Set the spin control parameters.
//
// Argument List :
//   fDefValue - The spin control initial value
//   fMinValue - The spin control minimum value
//   fMaxValue - The spin control maximum value
//   fMinIncSz - The spin control minimum increment size
//   fMaxIncSz - The spin control maximum increment size
//
// Return Values :
//   Success - TRUE
//   Failure - FALSE

bool  PnlValue::bSetParms( float fDefValue, float fMinValue, float fMaxValue,
                           float fMinIncSz, float fMaxIncSz )
{
  bool  bRtn;

  if( ! bIsCreated( ) ) return( FALSE );

  bRtn = m_oSpnValue.bSetParms( fDefValue, fMinValue, fMaxValue,
                                                        fMinIncSz, fMaxIncSz );

  return( bRtn );
}

//*****************************************************************************
// Set the spin control value as an integer.
//
// Argument List :
//   liValue  - The long integer value
//
// Return Values :
//   Success - TRUE
//   Failure - FALSE

bool  PnlValue::bSetValue( long liValue )
{
  bool    bRtn=TRUE;
  float   fMan;
  int     iExp;

  if( ! bIsCreated( ) ) return( FALSE );

  // If this is a temperature or phase value convert to the specified units
  fMan = (float) liValue;
  if( bTempToDegC( &fMan ) || bPhaseToDeg( &fMan ) )
  { // This is a temperature or phase value convert it to an integer
    if( fMan >= 0.0 ) liValue = (int) ( fMan + 0.5 );
    else              liValue = (int) ( fMan - 0.5 );
  }

  if( m_bUseUnits && m_oChoUnits.IsShown( ) )
  {
    // Determine the spin control value and the units choice selection
    ConvertType::bParseFlt( (double) liValue, &fMan, &iExp );
    while( iExp % 3 )
    {
      iExp--;
      fMan *= 10.0;
    }

    // Set the spin control and choice control values
    if( ! m_oSpnValue.bSetValue( fMan ) )          bRtn = FALSE;
    if( ! m_oChoUnits.bSetUnits( iExp ) )          bRtn = FALSE;
  }
  else
    if( ! m_oSpnValue.bSetValue( (int) liValue ) ) bRtn = FALSE;

  return( bRtn );
}

//*****************************************************************************
// Set the spin control value as a float.
//
// Argument List :
//   dfValue  - The double float value
//
// Return Values :
//   Success - TRUE
//   Failure - FALSE

bool  PnlValue::bSetValue( double dfValue )
{
  bool   bRtn=TRUE;
  float  fMan=0.0;
  int    iExp=0;

  if( ! bIsCreated( ) )                              return( FALSE );

  fMan = (float) dfValue;
  bTempToDegC( &fMan );  // Convert value to Deg.C  if appropriate
  bPhaseToDeg( &fMan );  // Convert value to Degree if appropriate

  if( m_bUseUnits && m_oChoUnits.IsShown( ) )
  {
    // Determine the spin control value and the units choice selection
    ConvertType::bParseFlt( dfValue, &fMan, &iExp );
    while( iExp % 3 )
    {
      iExp--;
      fMan *= 10.0;
    }

    // Set the spin control and choice control values
    if(      ! m_oChoUnits.bSetUnits( iExp ) )       bRtn = FALSE;
    else if( ! m_oSpnValue.bSetValue( fMan ) )       bRtn = FALSE;
  }
  else
    if( ! m_oSpnValue.bSetValue( (float) dfValue ) ) bRtn = FALSE;

  return( bRtn );
}

//*****************************************************************************
// Set the spin control value as a string.
//
// Argument List :
//   rosValue - The string value
//
// Return Values :
//   Success - TRUE
//   Failure - FALSE

bool  PnlValue::bSetValue( const wxString & rosValue )
{
  wxString  os1;
  double    df1, df2;
  long      li1;

  // Don't continue unless the panel has been created
  if( ! bIsCreated( ) )                                return( FALSE );

  // Set the value
  if( ! ConvertType::bStrToDFlt( rosValue, &df1 ) )    return( FALSE );
  if( ! bSetValue( df1 ) )                             return( FALSE );

  // For zero values the units will have been ignored, attempt to set them here
  if( df1 == 0.0 )
  {
    os1 = rosValue;
    os1.at( 0 ) = '1';
    if( ! ConvertType::bStrToDFlt( os1, &df1 ) )       return( FALSE );
    if( ! ConvertType::bParseDFlt( df1, &df2, &li1 ) ) return( FALSE );
    if( ! m_oChoUnits.bSetUnits( (int) li1 ) )         return( FALSE );
  }

  return( TRUE );
}

//*****************************************************************************
// Set the type of units to be displayed in the choice control.
//
// Argument List :
//   eUType  - The units specifier
//
// Return Values :
//   Success - TRUE
//   Failure - FALSE

bool  PnlValue::bSetUnitsType( eUnitsType eUType )
{
  bool  bRtn;

  if( ! m_bUseUnits || ! bIsCreated( ) )       return( FALSE );
  if( m_oChoUnits.eGetUnitsType( ) == eUType ) return( TRUE );
  if( ! m_oChoUnits.bSetUnitsType( eUType ) )  return( FALSE );
  m_oChoUnits.m_eDefUType = eUType;

  // Automatically set the spin control parameters
  switch( eUType )
  {
    case eUNITS_CAP   :  // Capacitance
      bRtn = m_oSpnValue.bSetParms( 100.0,     0.0, 1000.0, 1.0, 100.0 );
      break;

    case eUNITS_IND   :  // Inductance
      bRtn = m_oSpnValue.bSetParms( 100.0,     0.0, 1000.0, 1.0, 100.0 );
      break;

    case eUNITS_RES   :  // Resistance
      bRtn = m_oSpnValue.bSetParms( 100.0,     0.0, 1000.0, 1.0, 100.0 );
      break;

    case eUNITS_VOLT  :  // Voltage
      bRtn = m_oSpnValue.bSetParms( 100.0, -1000.0, 1000.0, 1.0, 100.0 );
      break;

    case eUNITS_CURR  :  // Current
      bRtn = m_oSpnValue.bSetParms( 100.0, -1000.0, 1000.0, 1.0, 100.0 );
      break;

    case eUNITS_TIME  :  // Time
      bRtn = m_oSpnValue.bSetParms( 100.0,     0.0, 1000.0, 1.0, 100.0 );
      break;

    case eUNITS_FREQ  :  // Frequency
      bRtn = m_oSpnValue.bSetParms( 100.0,     0.0, 1000.0, 1.0, 100.0 );
      break;

    case eUNITS_PHASE :  // Phase
      bRtn = m_oSpnValue.bSetParms(   0.0,     0.0,  360.0, 5.0 );
      break;

    case eUNITS_TEMP  :  // Temperature
      bRtn = m_oSpnValue.bSetParms(  27.0,  -100.0,  200.0, 1.0 );
      break;

    case eUNITS_SCLR  :  // Scalar
      bRtn = m_oSpnValue.bSetParms(   1.0, -1000.0, 1000.0, 1.0, 100.0 );
      break;

    default           :  // Default
      bRtn = m_oSpnValue.bSetParms( 100.0, -1000.0, 1000.0, 1.0, 100.0 );
      break;
  }

  return( bRtn );
}

//*****************************************************************************
// Set the units to be displayed in the choice control or static label.
//
// Argument List :
//   rosUnits - The units to be displayed
//
// Return Values :
//   Success - TRUE
//   Failure - FALSE

bool  PnlValue::bSetUnits( const wxString & rosUnits )
{
  if( ! m_bUseUnits || ! bIsCreated( ) )      return( FALSE );

  if( m_oChoUnits.IsShown( ) )
  {
    if( ! m_oChoUnits.bSetUnits( rosUnits ) ) return( FALSE );
  }
  else m_oLblUnits.SetLabel( wxT("  ") + rosUnits );

  return( TRUE );
}

//*****************************************************************************
// Show or hide the units choice control.
//
// Argument List :
//   bEnable - The new state for the units choice control
//
// Return Values :
//   Success - TRUE
//   Failure - FALSE

bool  PnlValue::bShowUnits( bool bEnable )
{
  if( ! m_bUseUnits || ! bIsCreated( ) ) return( FALSE );

  if( bEnable )
  {
    m_oLblUnits.Show( FALSE );
    m_oChoUnits.Show( TRUE );
  }
  else
  {
    m_oChoUnits.Show( FALSE );
    m_oLblUnits.Show( TRUE );
  }

  return( TRUE );
}

//*****************************************************************************
// Get the variable name.
//
// Return Values :
//   Success - The variable name as a string
//   Failure - An empty string

const wxString & PnlValue::rosGetName( void )
{
  static  wxString  os1;

  os1.Empty( );

  if( ! bIsCreated( ) ) return( os1 );

  os1 = m_oLblName.GetLabel( );

  return( os1 );
}

//*****************************************************************************
// Get the spin control value including units as an integer.
//
// Return Values :
//   Success - The value as a long integer
//   Failure - LONG_MIN

long  PnlValue::liGetValue( void )
{
  float  f1;
  long   li1;
  int    i1;

  // Has the control been created?
  if( ! bIsCreated( ) )  return( LONG_MIN );

  // Get the value combining it with any units
  if( m_bUseUnits && m_oChoUnits.IsShown( ) )
  {
    f1  = m_oSpnValue.fGetValue( );
    if( f1 == -FLT_MAX ) return( LONG_MIN );
    li1 = m_oChoUnits.liGetValue( f1 );
  }
  else
  {
    i1 = m_oSpnValue.iGetValue( );
    if( i1 == INT_MIN )  return( LONG_MIN );
    li1 = (long) i1;
  }

  return( li1 );
}

//*****************************************************************************
// Get the spin control value including units as a float.
//
// Return Values :
//   Success - The value as a double precision float
//   Failure - -DBL_MAX

double  PnlValue::dfGetValue( void )
{
  float   f1;
  double  df1;

  // Has the control been created?
  if( ! bIsCreated( ) ) return( -DBL_MAX );

  // Get the spin control value
  f1 = m_oSpnValue.fGetValue( );
  if( f1 == -FLT_MAX )  return( -DBL_MAX );

  // Combine the spin control value with any units
  if( m_bUseUnits && m_oChoUnits.IsShown( ) )
       df1 = m_oChoUnits.dfGetValue( f1 );
  else df1 = (double) f1;

  return( df1 );
}

//*****************************************************************************
// Get the spin control value including units as a string.
//
// Return Values :
//   Success - The value as a string
//   Failure - An empty string

const wxString & PnlValue::rosGetValue( void )
{
  static  wxString  osValue;
  float   f1;

  osValue.Empty( );

  // Has the control been created?
  if( ! bIsCreated( ) ) return( osValue );

  // Test the spin control value
  f1 = m_oSpnValue.fGetValue( );
  if( f1 == -FLT_MAX )  return( osValue );

  // Combine the spin control value with any units
  if( m_bUseUnits && m_oChoUnits.IsShown( ) )
       osValue = m_oChoUnits.rosGetValue( f1 );
  else osValue = m_oSpnValue.rosGetValue( );

  return( osValue );
}

//*****************************************************************************
// Get the units.
//
// Return Values :
//   Success - The currently selected units
//   Failure - An empty string

const wxString & PnlValue::rosGetUnits( void )
{
  static  wxString  osEmpty;

  if( ! m_bUseUnits || ! bIsCreated( ) ) return( osEmpty );

  return( m_oChoUnits.rosGetUnits( ) );
}

//*****************************************************************************
//                                                                            *
//                             Event Handlers                                 *
//                                                                            *
//*****************************************************************************
// Spin button scroll event handler.
//
// Argument List :
//   roEvtSpn - An object holding information about the event

void  PnlValue::OnSpnScroll( wxSpinEvent & roEvtSpn )
{
  static  bool   bAtLimit=FALSE;
  static  bool   bStartup=TRUE;
          float  f1;
          int    i1;

  if( ! m_bUseUnits || ! m_oChoUnits.IsShown( ) )              return;
  if( m_oChoUnits.eGetUnitsType( ) == eUNITS_TEMP  )           return;
  if( m_oChoUnits.eGetUnitsType( ) == eUNITS_PHASE )           return;

  f1 = m_oSpnValue.fGetValue( );
  i1 = m_oChoUnits.iGetUnits( );

  // Must set the state of bAtLimit when this function first called
  if( bStartup )
  {
    bStartup = FALSE;
    if( f1 == 0.0 )                         bAtLimit = TRUE;
    if( f1 == m_oSpnValue.fGetMaxValue( ) ) bAtLimit = TRUE;
    if( f1 == m_oSpnValue.fGetMinValue( ) ) bAtLimit = TRUE;
  }

  // Determine the event type and act accordingly
  if( roEvtSpn.GetEventType( ) == wxEVT_SCROLL_LINEUP )
  { // Increment the value
    if( f1 > 0.0 )
    { // Positive numbers
      if( f1 < m_oSpnValue.fGetMaxValue( ) ) { bAtLimit = FALSE; return; }
      if( ! bAtLimit )                       { bAtLimit = TRUE;  return; }
      f1 /= 1000.0;
      i1 += 3;
    }
    else
    { // Negative numbers
      if( f1 < -1.0 )                        { bAtLimit = FALSE; return; }
      if( ! bAtLimit )                       { bAtLimit = TRUE;  return; }
      f1 = -1000.0;
      i1 -= 3;
    }
  }
  else if( roEvtSpn.GetEventType( ) == wxEVT_SCROLL_LINEDOWN )
  { // Decrement the value
    if( f1 < 0.0 )
    { // Negative numbers
      if( f1 > m_oSpnValue.fGetMinValue( ) ) { bAtLimit = FALSE; return; }
      if( ! bAtLimit )                       { bAtLimit = TRUE;  return; }
      f1 /= 1000.0;
      i1 += 3;
    }
    else
    { // Positive numbers
      if( f1 > 1.0 )                         { bAtLimit = FALSE; return; }
      if( ! bAtLimit )                       { bAtLimit = TRUE;  return; }
      f1 = 1000.0;
      i1 -= 3;
    }
  }
  else return;

  if( m_oChoUnits.bSetUnits( i1 ) ) m_oSpnValue.bSetValue( f1 );
}

//*****************************************************************************
// Units choice box event handler.
//
// Argument List :
//   roEvtCmd - An object holding information about the event

void  PnlValue::OnChoUnits( wxCommandEvent & roEvtCmd )
{
  if(      m_oChoUnits.eGetUnitsType( ) == eUNITS_TEMP )
  {
    if(      m_oChoUnits.rosGetUnits( ).IsSameAs( wxT("Deg C") ) )
      bSetParms( 27.0, -100.0, 200.0, 1.0 );
    else if( m_oChoUnits.rosGetUnits( ).IsSameAs( wxT("Deg F") ) )
      bSetParms( 70.0, -100.0, 400.0, 1.0 );
    else if( m_oChoUnits.rosGetUnits( ).IsSameAs( wxT("Deg K") ) )
      bSetParms( 300.0, 200.0, 500.0, 1.0 );
  }
  else if( m_oChoUnits.eGetUnitsType( ) == eUNITS_PHASE )
  {
    if(      m_oChoUnits.rosGetUnits( ).IsSameAs( wxT("Degree") ) )
      bSetParms( 0.0, 0.0, 360.0, 1.0 );
    else if( m_oChoUnits.rosGetUnits( ).IsSameAs( wxT("Radian") ) )
      bSetParms( 0.0, 0.0, 6.3, 0.1 );
    else if( m_oChoUnits.rosGetUnits( ).IsSameAs( wxT("Grad") ) )
      bSetParms( 0.0, 0.0, 400.0, 1.0 );
  }
}

//*****************************************************************************
