#!/bin/sh

# Copyright (C) 2015 Canonical Ltd
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License version 3 as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Authored by: Michi Henning <michi.henning@canonical.com>
#              Thomas Voß <thomas.voss@canonical.com>

#
# Script to generate debian files for dual landing in Vivid (gcc 4.9 ABI)
# and Wily and later (gcc 5 ABI).
#
# This script is called from debian/rules and generates:
#
# - control
# - libubuntu-location-service${soversion}.install.${target_arch}
#
# For all but control, this is a straight substition and/or renaming exercise for each file.
# For control, if building on Wily or later, we also fix the "Replaces:" and "Conflicts:"
# entries, so we don't end up with two packages claiming ownership of the same places
# in the file system.
#
# Because the debian files for the different distributions are generated on the fly,
# this allows us to keep a single source tree for both distributions. See ../HACKING
# for more explanations.
#

set -e  # Fail if any command fails.

progname=$(basename $0)

[ $# -ne 1 ] && {
    echo "usage: $progname path-to-debian-dir" >&2
    exit 1
}
dir=$1
version_dir=$(mktemp -d)

# Dump version numbers into files and initialize vars from those files.

sh ${dir}/get-versions.sh ${dir} ${version_dir}

full_version=$(cat "${version_dir}"/libubuntu-location-service.full-version)
major_minor=$(cat "${version_dir}"/libubuntu-location-service.major-minor-version)
soversion=$(cat "${version_dir}"/libubuntu-location-service.soversion)
vivid_soversion=$(cat "${version_dir}"/libubuntu-location-service.vivid-soversion)

warning=$(mktemp -t gen-debian-files-msg.XXX)

trap "rm -fr $warning $version_dir" 0 INT TERM QUIT

warning_msg()
{
    cat >$warning <<EOF
# This file is autogenerated. DO NOT EDIT!
#
# Modifications should be made to $(basename "$1") instead.
# This file is regenerated automatically in the clean target.
#
EOF
}

# Generate debian/control from debian/control.in, substituting the soversion for both libs.
# For wily onwards, we also add an entry for the vivid versions to "Conflicts:" and "Replaces:".

infile="${dir}"/control.in
outfile="${dir}"/control
warning_msg $infile
cat $warning $infile \
    | sed -e "s/@UBUNTU_LOCATION_SERVICE_SOVERSION@/${soversion}/" > "$outfile"

[ "$distro" != "vivid" ] && {
    sed -i -e "/Replaces: libubuntu-location-service2,/a\
\          libubuntu-location-service${vivid_soversion}," \
        "$outfile"
}

# Generate the install files, naming them according to the soversion.

# Install file for binary package
infile="${dir}"/libubuntu-location-service.install.in
outfile="${dir}"/libubuntu-location-service${soversion}.install
warning_msg "$infile"
cat $warning "$infile" >"$outfile"

infile="${dir}"/libubuntu-location-service.install.with-gps.in
outfile="${dir}"/libubuntu-location-service${soversion}.install.with-gps
warning_msg "$infile"
cat $warning "$infile" >"$outfile"

infile="${dir}"/libubuntu-location-service-dev.install.in
outfile="${dir}"/libubuntu-location-service-dev.install
warning_msg "$infile"
cat $warning $infile | sed -e "s/@UBUNTU_LOCATION_SERVICE_SOVERSION@/${soversion}/" > "$outfile"

ln -s "${dir}"/libubuntu-location-service${soversion}.install.with-gps "${dir}"/libubuntu-location-service${soversion}.install.amd64
ln -s "${dir}"/libubuntu-location-service${soversion}.install.with-gps "${dir}"/libubuntu-location-service${soversion}.install.armhf
ln -s "${dir}"/libubuntu-location-service${soversion}.install.with-gps "${dir}"/libubuntu-location-service${soversion}.install.i386

exit 0
