/*
Copyright (c) 2007, Dennis M. Sosnoski.
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice, this
   list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.
 * Neither the name of JiBX nor the names of its contributors may be used
   to endorse or promote products derived from this software without specific
   prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

package org.jibx.runtime.impl;

/**
 * Support class providing methods used by generated code.
 *
 * @author Dennis M. Sosnoski
 */
public abstract class RuntimeSupport
{
    /**
     * Split concatenated class names string into an array of individual class
     * names. This is used by the generated binding factory code, to reduce the
     * size of the generated classes and methods. It takes as input a string
     * consisting of compacted fully-qualified class names separated by '|'
     * delimitor characters. If the compacted name starts with one or more '.'
     * characters the corresponding number of package name levels are taken from
     * the last class name. Empty names are left <code>null</code> in the array.
     *
     * @param count number of names
     * @param blob compacted class names separated by '|' delimitors
     * @return expanded class names
     */
    public static String[] splitClassNames(int count, String blob) {
        String[] names = new String[count];
        String last = "";
        int split = -1;
        int index = 0;
        StringBuffer buff = new StringBuffer();
        while (split < blob.length()) {
            int base = split + 1;
            split = blob.indexOf('|', base);
            if (split < 0) {
                split = blob.length();
            }
            if (split > base) {
                int mark = 0;
                while (blob.charAt(base) == '.') {
                    mark = last.indexOf('.', mark) + 1;
                    base++;
                }
                buff.setLength(0);
                if (mark > 0) {
                    buff.append(last.substring(0, mark));
                }
                buff.append(blob.substring(base, split));
                names[index] = last = buff.toString().intern();
            }
            index++;
        }
        return names;
    }
    
    /**
     * Expand names URI indexes into an array of individual names URIs. This is
     * used by the generated binding factory code, to reduce the size of the
     * generated classes and methods. It takes as input a string where each
     * character is a namespace index, biased by +2 in order to avoid the use of
     * null characters, along with a table of corresponding namespace URIs. The
     * character value 1 is used as a marker for the case where no namespace is
     * associated with an index.
     *
     * @param blob string of characters representing namespace indexes
     * @param uris namespace URIs defined in binding
     * @return expanded class names
     */
    public static String[] expandNamespaces(String blob, String[] uris) {
        String[] nameuris = new String[blob.length()];
        for (int i = 0; i < blob.length(); i++) {
            int index = blob.charAt(i) - 2;
            if (index >= 0) {
                nameuris[i] = uris[index];
            }
        }
        return nameuris;
    }
    
    /**
     * Split concatenated names string into an array of individual names. This
     * is used by the generated binding factory code, to reduce the size of the
     * generated classes and methods. It takes as input a string consisting of
     * names separated by '|' delimitor characters.
     *
     * @param count number of names
     * @param blob element names separated by '|' delimitors
     * @return expanded class names
     */
    public static String[] splitNames(int count, String blob) {
        String[] names = new String[count];
        int split = -1;
        int index = 0;
        while (split < blob.length()) {
            int base = split + 1;
            split = blob.indexOf('|', base);
            if (split < 0) {
                split = blob.length();
            }
            if (split > base) {
                names[index] = blob.substring(base, split);
            }
            index++;
        }
        return names;
    }
}