/***********************************************************************
 *
 * This MobilityDB code is provided under The PostgreSQL License.
 * Copyright (c) 2016-2024, Université libre de Bruxelles and MobilityDB
 * contributors
 *
 * MobilityDB includes portions of PostGIS version 3 source code released
 * under the GNU General Public License (GPLv2 or later).
 * Copyright (c) 2001-2024, PostGIS contributors
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose, without fee, and without a written
 * agreement is hereby granted, provided that the above copyright notice and
 * this paragraph and the following two paragraphs appear in all copies.
 *
 * IN NO EVENT SHALL UNIVERSITE LIBRE DE BRUXELLES BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES, INCLUDING
 * LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION,
 * EVEN IF UNIVERSITE LIBRE DE BRUXELLES HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * UNIVERSITE LIBRE DE BRUXELLES SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS ON
 * AN "AS IS" BASIS, AND UNIVERSITE LIBRE DE BRUXELLES HAS NO OBLIGATIONS TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 *****************************************************************************/

/**
 * @file
 * @brief Spatial restriction functions for temporal points
 */

/* C */
#include <assert.h>
/* PostgreSQL */
#include <postgres.h>
#include <utils/float.h>
#include <utils/timestamp.h>
/* PostGIS */
#include <liblwgeom.h>
#include <liblwgeom_internal.h>
#include <lwgeodetic.h>
/* MEOS */
#include <meos.h>
#include <meos_internal.h>
#include "general/lifting.h"
#include "general/span.h"
#include "general/temporal_restrict.h"
#include "general/tsequence.h"
#include "general/type_util.h"
#include "point/tpoint_spatialfuncs.h"
#include "point/tpoint_spatialrels.h"

/*****************************************************************************
 * Force a temporal point to be 2D
 *****************************************************************************/

/**
 * @brief Force a point to be in 2D
 */
static Datum
point_force2d(Datum point, Datum srid)
{
  const POINT2D *p = DATUM_POINT2D_P(point);
  GSERIALIZED *result = geopoint_make(p->x, p->y, 0.0, false, false,
    DatumGetInt32(srid));
  return PointerGetDatum(result);
}

/**
 * @brief Force a temporal point to be in 2D
 * @param[in] temp Temporal point
 * @pre The temporal point has Z dimension
 */
static Temporal *
tpoint_force2d(const Temporal *temp)
{
  /* Ensure validity of the arguments */
  if (! ensure_not_null((void *) temp) || ! ensure_tgeo_type(temp->temptype) ||
      ! ensure_has_Z(temp->flags))
    return NULL;

  /* We only need to fill these parameters for tfunc_temporal */
  LiftedFunctionInfo lfinfo;
  memset(&lfinfo, 0, sizeof(LiftedFunctionInfo));
  lfinfo.func = (varfunc) &point_force2d;
  lfinfo.numparam = 1;
  int32 srid = tpoint_srid(temp);
  lfinfo.param[0] = Int32GetDatum(srid);
  lfinfo.argtype[0] = temp->temptype;
  lfinfo.restype = T_TGEOMPOINT;
  lfinfo.tpfunc_base = NULL;
  lfinfo.tpfunc = NULL;
  return tfunc_temporal(temp, &lfinfo);
}

/*****************************************************************************/

/**
 * @brief Return the timestamp at which a segment of a temporal point takes a
 * base value (iterator function)
 * @details To take into account roundoff errors, the function considers that
 * two values are equal even if their coordinates may differ by @p MEOS_EPSILON.
 * @param[in] inst1,inst2 Temporal values
 * @param[in] value Base value
 * @param[out] t Timestamp
 * @return Return true if the point is found in the temporal point
 * @pre The segment is not constant and has linear interpolation
 * @note The resulting timestamptz may be at an exclusive bound
 */
static bool
tpointsegm_timestamp_at_value1_iter(const TInstant *inst1,
  const TInstant *inst2, Datum value, TimestampTz *t)
{
  Datum value1 = tinstant_val(inst1);
  Datum value2 = tinstant_val(inst2);
  /* Is the lower bound the answer? */
  bool result = true;
  if (datum_point_eq(value1, value))
    *t = inst1->t;
  /* Is the upper bound the answer? */
  else if (datum_point_eq(value2, value))
    *t = inst2->t;
  else
  {
    double dist;
    double fraction = (double) geosegm_locate_point(value1, value2, value, &dist);
    if (fabs(dist) >= MEOS_EPSILON)
      result = false;
    else
    {
      double duration = (double) (inst2->t - inst1->t);
      *t = inst1->t + (TimestampTz) (duration * fraction);
    }
  }
  return result;
}

/**
 * @brief Return the timestamp at which a temporal point sequence is equal to a
 * point
 * @details This function is called by the #tpointseq_interperiods function
 * while computing atGeometry to find the timestamp at which an intersection
 * point found by PostGIS is located. This function differs from function
 * #tpointsegm_intersection_value in particular since the latter is used for
 * finding crossings during synchronization and thus it is required that the
 * timestamp in strictly between the timestamps of a segment.
 * @param[in] seq Temporal point sequence
 * @param[in] value Base value
 * @param[out] t Timestamp
 * @return Return true if the point is found in the temporal point
 * @pre The point is known to belong to the temporal sequence (taking into
 * account roundoff errors), the temporal sequence has linear interpolation,
 * and is simple
 * @note The resulting timestamp may be at an exclusive bound
 */
static bool
tpointseq_timestamp_at_value(const TSequence *seq, Datum value,
  TimestampTz *t)
{
  const TInstant *inst1 = TSEQUENCE_INST_N(seq, 0);
  for (int i = 1; i < seq->count; i++)
  {
    const TInstant *inst2 = TSEQUENCE_INST_N(seq, i);
    /* We are sure that the segment is not constant since the
     * sequence is simple */
    if (tpointsegm_timestamp_at_value1_iter(inst1, inst2, value, t))
      return true;
    inst1 = inst2;
  }
  /* We should never arrive here */
  meos_error(ERROR, MEOS_ERR_TEXT_INPUT,
    "The value has not been found due to roundoff errors");
  return false;
}

/*****************************************************************************
 * Restriction functions for a spatiotemporal box
 *****************************************************************************/

/*
 * Region codes for the Cohen-Sutherland algorithm for 3D line clipping
 */

const int INSIDE  = 0;  /* 000000 */
const int LEFT    = 1;  /* 000001 */
const int RIGHT   = 2;  /* 000010 */
const int BOTTOM  = 4;  /* 000100 */
const int TOP     = 8;  /* 001000 */
const int FRONT   = 16; /* 010000 */
const int BACK    = 32; /* 100000 */

/*
 * Border codes for the excluding the top border of a spatiotemporal box
 */
const int XMAX    = 1;  /* 001 */
const int YMAX    = 2;  /* 010 */
const int ZMAX    = 4;  /* 100 */

/**
 * @brief Compute region code for a point(x, y, z)
 */
static int
computeRegionCode(double x, double y, double z, bool hasz, const STBox *box)
{
  /* Initialized as being inside */
  int code = INSIDE;
  if (x < box->xmin)      /* to the left of the box */
    code |= LEFT;
  else if (x > box->xmax) /* to the right of the box */
    code |= RIGHT;
  if (y < box->ymin)      /* below the box */
    code |= BOTTOM;
  else if (y > box->ymax) /* above the box */
    code |= TOP;
  if (hasz)
  {
    if (z < box->zmin)      /* in front of the box */
      code |= FRONT;
    else if (z > box->zmax) /* behind the the box */
      code |= BACK;
  }
  return code;
}

/**
 * @brief Compute max border code for a point(x, y, z)
 */
static int
computeMaxBorderCode(double x, double y, double z, bool hasz, const STBox *box)
{
  /* Initialized as being inside */
  int code = INSIDE;
  /* Check if we are on a max border
   * Note: After clipping, we don't need to apply fabs() */
  if (box->xmax - x < MEOS_EPSILON) /* on xmax border */
    code |= XMAX;
  if (box->ymax - y < MEOS_EPSILON) /* on ymax border */
    code |= YMAX;
  if (hasz && box->zmax - z < MEOS_EPSILON) /* on zmax border */
    code |= ZMAX;
  return code;
}

/**
 * @brief Clip the paramaters t0 and t1 for the Liang-Barsky clipping algorithm
 */
bool
clipt(double p, double q, double *t0, double *t1)
{
  double r;
  if (p < 0)
  {
    /* entering visible region, so compute t0 */
    if (q < 0)
    {
      /* t0 will be nonnegative, so continue */
      r = q / p;
      if (r > *t1)
        /* t0 will exceed t1, so reject */
        return false;
      if (r > *t0)
        *t0 = r; /* t0 is max of r's */
    }
  }
  else if (p > 0)
  {
    /* exiting visible region, so compute t1 */
    if (q < p)
    {
      /* t1 will be <= 1, so continue */
      r = q / p;
      if (r < *t0)
        /* t1 will be <= t0, so reject */
        return false;
      if (r < *t1)
        *t1 = r; /* t1 is min of r's */
    }
  }
  else /* p == 0 */
  {
    if (q < 0)
      /* line parallel and outside */
      return false;
  }
  return true;
}

/**
 * @brief Clip a line segment using the Liang-Barsky algorithm
 * @param[in] point1,point2 Input points
 * @param[in] box Bounding box
 * @param[in] hasz Has Z dimension?
 * @param[in] border_inc True when the box contains the upper border, otherwise
 * the upper border is assumed as outside of the box.
 * @param[out] point3,point4 Output points
 * @param[out] p3_inc,p4_inc Are the points included or not in the box?
 * These are only written/returned when @p border_inc is false
 * @return True if the line segment defined by p1,p2 intersects the bounding
 * box, false otherwise.
 * @note It is possible to mix 2D/3D geometries, the Z dimension is only
 * considered if both the temporal point and the box have Z dimension
 * @see The Liang-Barsky algorithm is explained in
 * https://www.researchgate.net/publication/255657434_Some_Improvements_to_a_Parametric_Line_Clipping_Algorithm
 */
static bool
liangBarskyClip(GSERIALIZED *point1, GSERIALIZED *point2, const STBox *box,
  bool hasz, bool border_inc, GSERIALIZED **point3, GSERIALIZED **point4,
  bool *p3_inc, bool *p4_inc)
{
  assert(MEOS_FLAGS_GET_X(box->flags));
  assert(! geopoint_eq(point1, point2));
  assert(! hasz || (MEOS_FLAGS_GET_Z(box->flags) &&
    (bool) FLAGS_GET_Z(point1->gflags) && (bool) FLAGS_GET_Z(point2->gflags)));

  int srid = box->srid;
  assert(srid == gserialized_get_srid(point1) &&
    srid == gserialized_get_srid(point2));

  /* Get the input points */
  double x1, y1, z1 = 0.0, x2, y2, z2 = 0.0;
  if (hasz)
  {
    const POINT3DZ *pt1 = GSERIALIZED_POINT3DZ_P(point1);
    const POINT3DZ *pt2 = GSERIALIZED_POINT3DZ_P(point2);
    x1 = pt1->x; x2 = pt2->x;
    y1 = pt1->y; y2 = pt2->y;
    z1 = pt1->z; z2 = pt2->z;
  }
  else
  {
    const POINT2D *pt1 = GSERIALIZED_POINT2D_P(point1);
    const POINT2D *pt2 = GSERIALIZED_POINT2D_P(point2);
    x1 = pt1->x; x2 = pt2->x;
    y1 = pt1->y; y2 = pt2->y;
  }

  if ((x1 < box->xmin && x2 < box->xmin) || (x1 > box->xmax && x2 > box->xmax) ||
      (y1 < box->ymin && y2 < box->ymin) || (y1 > box->ymax && y2 > box->ymax) ||
      (hasz && z1 < box->zmin && z2 < box->zmin) ||
      (hasz && z1 > box->zmax && z2 > box->zmax))
    /* trivial reject */
    return false;

  /* not trivial reject */
  double t0 = 0, t1 = 1;
  double dx = x2 - x1;
  if (clipt(-dx, x1 - box->xmin, &t0, &t1) && /* left */
      clipt(dx, box->xmax - x1, &t0, &t1)) /* right */
  {
    double dy = y2 - y1;
    if (clipt(-dy, y1 - box->ymin, &t0, &t1) && /* bottom */
        clipt(dy, box->ymax - y1, &t0, &t1)) /* top */
    {
      bool found = true;
      double dz = 0;
      if (hasz)
      {
        dz = z2 - z1;
        if (! clipt(-dz, z1 - box->zmin, &t0, &t1) || /* front */
            ! clipt(dz, box->zmax - z1, &t0, &t1)) /* back */
          found = false;
      }
      if (found)
      {
        /* compute coordinates */
        if (t1 < 1)
        {
          /* compute V1' */
          x2 = x1 + t1 * dx;
          y2 = y1 + t1 * dy;
          if (hasz)
            z2 = z1 + t1 * dz;
        }
        if (t0 > 0)
        {
          /* compute V0' */
          x1 = x1 + t0 * dx;
          y1 = y1 + t0 * dy;
          if (hasz)
            z1 = z1 + t0 * dz;
        }
        /* Remove the max border */
        if (! border_inc)
        {
          /* Compute max border codes for the input points */
          int max_code1 = computeMaxBorderCode(x1, y1, z1, hasz, box);
          int max_code2 = computeMaxBorderCode(x2, y2, z2, hasz, box);
          /* If the whole segment is on a max border, remove it */
          if (max_code1 & max_code2)
            return false;
          /* Point is included if its max_code is 0 */
          if (p3_inc && p4_inc)
          {
            *p3_inc = (max_code1 == 0);
            *p4_inc = (max_code2 == 0);
          }
        }
        if (point3 && point4)
        {
          *point3 = geopoint_make(x1, y1, z1, hasz, false, srid);
          *point4 = geopoint_make(x2, y2, z2, hasz, false, srid);
        }
        return true;
      }
    }
  }
  return false;
}

/*****************************************************************************/

/**
 * @brief Return a temporal point instant restricted to (the complement of) a
 * spatiotemporal box (iterator function)
 * @pre The arguments have the same SRID. This is verified in
 * #tpoint_restrict_stbox
 */
static bool
tpointinst_restrict_stbox_iter(const TInstant *inst, const STBox *box,
  bool border_inc, bool atfunc)
{
  bool hasz = MEOS_FLAGS_GET_Z(inst->flags) && MEOS_FLAGS_GET_Z(box->flags);
  bool hast = MEOS_FLAGS_GET_T(box->flags);

  /* Restrict to the T dimension */
  if (hast && ! contains_span_value(&box->period, DatumGetTimestampTz(inst->t)))
    return ! atfunc;

  /* Restrict to the XY(Z) dimension */
  Datum value = tinstant_val(inst);
  /* Get the input point */
  double x, y, z = 0.0;
  if (hasz)
  {
    const POINT3DZ *pt = DATUM_POINT3DZ_P(value);
    x = pt->x;
    y = pt->y;
    z = pt->z;
  }
  else
  {
    const POINT2D *pt = DATUM_POINT2D_P(value);
    x = pt->x;
    y = pt->y;
  }
  /* Compute region code for the input point */
  int reg_code = computeRegionCode(x, y, z, hasz, box);
  int max_code = 0;
  if (! border_inc)
    max_code = computeMaxBorderCode(x, y, z, hasz, box);
  if ((reg_code | max_code) != 0)
    return ! atfunc;

  /* Point is inside the region */
  return atfunc;
}

/**
 * @ingroup meos_internal_temporal_restrict
 * @brief Return a temporal point instant restricted to (the complement of) a
 * spatiotemporal box
 * @param[in] inst Temporal instant point
 * @param[in] box Spatiotemporal box
 * @param[in] border_inc True when the box contains the upper border
 * @param[in] atfunc True if the restriction is at, false for minus
 * @pre The box has X dimension and the arguments have the same SRID.
 * This is verified in #tpoint_restrict_stbox
 * @csqlfn #Tpoint_at_stbox(), minus_stbox()
 */
TInstant *
tpointinst_restrict_stbox(const TInstant *inst, const STBox *box,
  bool border_inc, bool atfunc)
{
  assert(inst); assert(box); assert(tgeo_type(inst->temptype));
  if (tpointinst_restrict_stbox_iter(inst, box, border_inc, atfunc))
    return tinstant_copy(inst);
  return NULL;
}

/**
 * @brief Return a temporal point discrete sequence restricted to (the
 * complement of) a spatiotemporal box
 * @param[in] seq Temporal discrete sequence point
 * @param[in] box Spatiotemporal box
 * @param[in] border_inc True when the box contains the upper border
 * @param[in] atfunc True if the restriction is at, false for minus
 * @pre Instantaneous sequences have been managed in the calling function
 */
TSequence *
tpointseq_disc_restrict_stbox(const TSequence *seq, const STBox *box,
  bool border_inc, bool atfunc)
{
  assert(seq); assert(box); assert(tgeo_type(seq->temptype));
  assert(MEOS_FLAGS_GET_INTERP(seq->flags) == DISCRETE);
  assert (seq->count > 1);

  const TInstant **instants = palloc(sizeof(TInstant *) * seq->count);
  int ninsts = 0;
  for (int i = 0; i < seq->count; i++)
  {
    const TInstant *inst = TSEQUENCE_INST_N(seq, i);
    if (tpointinst_restrict_stbox_iter(inst, box, border_inc, atfunc))
      instants[ninsts++] = inst;
  }
  TSequence *result = NULL;
  if (ninsts > 0)
    result = tsequence_make(instants, ninsts, true, true, DISCRETE,
      NORMALIZE_NO);
  pfree(instants);
  return result;
}

/**
 * @brief Return a temporal sequence point with step interpolation restricted
 * to a
 * spatiotemporal box
 * @param[in] seq Temporal point
 * @param[in] box Spatiotemporal box
 * @param[in] border_inc True when the box contains the upper border
 * @param[in] atfunc True if the restriction is at, false for minus
 * @pre Instantaneous sequences have been managed in the calling function
 * @note The function computes the "at" restriction on all dimensions. Then,
 * for the "minus" restriction, it computes the complement of the "at"
 * restriction with respect to the time dimension.
 */
TSequenceSet *
tpointseq_step_restrict_stbox(const TSequence *seq, const STBox *box,
  bool border_inc, bool atfunc)
{
  assert(seq); assert(box);
  assert(tgeo_type(seq->temptype));
  assert(MEOS_FLAGS_GET_INTERP(seq->flags) == STEP);
  assert(seq->count > 1);

  /* Compute the time span of the result if the box has T dimension */
  bool hast = MEOS_FLAGS_GET_T(box->flags);
  Span timespan;
  if (hast)
  {
    if (! inter_span_span(&seq->period, &box->period, &timespan))
      return atfunc ? NULL : tsequence_to_tsequenceset(seq);
  }

  /* Compute the sequences for "at" restriction */
  bool lower_inc;
  TSequence **sequences = palloc(sizeof(TSequence *) * seq->count);
  TInstant **instants = palloc(sizeof(TInstant *) * seq->count);
  TimestampTz start = DatumGetTimestampTz(seq->period.lower);
  int ninsts = 0, nseqs = 0;
  for (int i = 0; i < seq->count; i++)
  {
    const TInstant *inst = TSEQUENCE_INST_N(seq, i);
    if (tpointinst_restrict_stbox_iter(inst, box, border_inc, REST_AT))
      instants[ninsts++] = (TInstant *) inst;
    else
    {
      if (ninsts > 0)
      {
        /* Continue the last instant of the sequence until the time of inst2
         * projected to the time dimension (if any) */
        Datum value = tinstant_val(instants[ninsts - 1]);
        bool tofree = false;
        bool upper_inc = false;
        if (hast)
        {
          Span extend, inter;
          span_set(TimestampTzGetDatum(instants[ninsts - 1]->t),
            TimestampTzGetDatum(inst->t), true, false, T_TIMESTAMPTZ,
            T_TSTZSPAN, &extend);
          if (inter_span_span(&timespan, &extend, &inter))
          {
            if (TimestampTzGetDatum(inter.lower) !=
                TimestampTzGetDatum(inter.upper))
            {
              instants[ninsts++] = tinstant_make(value, seq->temptype,
                DatumGetTimestampTz(inter.upper));
              tofree = true;
            }
            else
              upper_inc = true;
          }
        }
        else
        {
          /* Continue the last instant of the sequence until the time of inst2 */
          instants[ninsts++] = tinstant_make(value, seq->temptype, inst->t);
          tofree = true;
        }
        lower_inc = (instants[0]->t == start) ? seq->period.lower_inc : true;
        sequences[nseqs++] = tsequence_make((const TInstant **) instants,
          ninsts, lower_inc, upper_inc, STEP, NORMALIZE_NO);
        if (tofree)
          pfree(instants[ninsts - 1]);
        ninsts = 0;
      }
    }
  }
  /* Add a last sequence with the remaining instants */
  if (ninsts > 0)
  {
    lower_inc = (instants[0]->t == start) ? seq->period.lower_inc : true;
    TimestampTz end = DatumGetTimestampTz(seq->period.upper);
    bool upper_inc = (instants[ninsts - 1]->t == end) ?
      seq->period.upper_inc : false;
    sequences[nseqs++] = tsequence_make((const TInstant **) instants, ninsts,
      lower_inc, upper_inc, STEP, NORMALIZE_NO);
  }
  pfree(instants);

  /* Clean up and return if no sequences have been found with "at" */
  if (nseqs == 0)
  {
    pfree(sequences);
    return atfunc ? NULL : tsequence_to_tsequenceset(seq);
  }

  /* Construct the result for "at" restriction */
  TSequenceSet *result_at = tsequenceset_make_free(sequences, nseqs,
    NORMALIZE_NO);
  /* If "at" restriction, return */
  if (atfunc)
    return result_at;

  /* If "minus" restriction, compute the complement wrt time */
  SpanSet *ss = tsequenceset_time(result_at);
  TSequenceSet *result = tcontseq_restrict_tstzspanset(seq, ss, atfunc);
  pfree(ss); pfree(result_at);
  return result;
}

/*****************************************************************************/

/**
 * @brief Restrict the temporal point to the spatial dimensions of a
 * spatiotemporal box
 * @param[in] seq Temporal point sequence
 * @param[in] box Bounding box
 * @param[in] border_inc True when the box contains the upper border
 * @note Instantaneous sequences must be managed since this function is called
 * after restricting to the time dimension
 */
TSequenceSet *
tpointseq_linear_at_stbox_xyz(const TSequence *seq, const STBox *box,
  bool border_inc)
{
  assert(MEOS_FLAGS_GET_INTERP(seq->flags) == LINEAR);

  /* Instantaneous sequence */
  if (seq->count == 1)
  {
    if (tpointinst_restrict_stbox_iter(TSEQUENCE_INST_N(seq, 0), box,
        border_inc, REST_AT))
      return tsequence_to_tsequenceset(seq);
    return NULL;
  }

  /* General case */
  bool hasz_seq = MEOS_FLAGS_GET_Z(seq->flags);
  bool hasz_box = MEOS_FLAGS_GET_Z(box->flags);
  bool hasz = hasz_seq && hasz_box;
  TSequence **sequences = palloc(sizeof(TSequence *) * seq->count);
  TInstant **instants = palloc(sizeof(TInstant *) * seq->count);
  TInstant **tofree = palloc(sizeof(TInstant *) * seq->count * 2);
  const TInstant *inst1 = TSEQUENCE_INST_N(seq, 0);
  GSERIALIZED *p1 = DatumGetGserializedP(tinstant_val(inst1));
  bool lower_inc = seq->period.lower_inc;
  bool upper_inc;
  int ninsts = 0, nseqs = 0, nfree = 0;
  for (int i = 1; i < seq->count; i++)
  {
    const TInstant *inst2 = TSEQUENCE_INST_N(seq, i);
    upper_inc = (i == seq->count - 1) ? seq->period.upper_inc : false;
    GSERIALIZED *p2 = DatumGetGserializedP(tinstant_val(inst2));
    GSERIALIZED *p3, *p4;
    TInstant *inst1_2d, *inst2_2d;
    bool makeseq = false;
    if (geopoint_eq(p1, p2))
    {
      /* Constant segment */
      if (tpointinst_restrict_stbox_iter(inst1, box, border_inc, REST_AT))
      {
        /* If ninsts > 0 the instant was added in the previous iteration */
        if (ninsts == 0)
          instants[ninsts++] = (TInstant *) inst1;
        instants[ninsts++] = (TInstant *) inst2;
      }
      else
        makeseq = true;
    }
    else
    {
      /* Clip the segment */
      bool p3_inc = true, p4_inc = true;
      bool found = liangBarskyClip(p1, p2, box, hasz, border_inc, &p3, &p4,
        &p3_inc, &p4_inc);
      if (found)
      {
        /* If p2 != p4 or p4_inc is false, we exit the box,
         * so end the previous sequence and start a new one */
        if (! geopoint_eq(p2, p4) || ! p4_inc)
          makeseq = true;
        /* To reduce roundoff errors, (1) find the timestamps at which the
         * segment take the points returned by the clipping function and
         * (2) project the temporal points to the timestamps instead  */
        TimestampTz t1, t2;
        Datum d3 = PointerGetDatum(p3);
        Datum d4 = PointerGetDatum(p4);
        if (hasz_seq && ! hasz)
        {
          /* Force the computation at 2D */
          inst1_2d = (TInstant *) tpoint_force2d((Temporal *) inst1);
          inst2_2d = (TInstant *) tpoint_force2d((Temporal *) inst2);
          p1 = DatumGetGserializedP(tinstant_val(inst1_2d));
          p2 = DatumGetGserializedP(tinstant_val(inst2_2d));
        }
        /* Compute timestamp t1 of point p3 */
        if (geopoint_eq(p1, p3))
          t1 = inst1->t;
        else if (geopoint_eq(p2, p3))
          t1 = inst2->t;
        else /* inst1->t < t1(p3) < inst2->t */
        {
          if (hasz_seq && ! hasz)
            tpointsegm_timestamp_at_value1_iter(inst1_2d, inst2_2d, d3, &t1);
          else
            tpointsegm_timestamp_at_value1_iter(inst1, inst2, d3, &t1);
        }
        /* Compute timestamp t2 of point p4  */
        if (geopoint_eq(p2, p4))
          t2 = inst2->t;
        else if (geopoint_eq(p3, p4))
          t2 = t1;
        else /* inst1->t < t2(p4) < inst2->t */
        {
          if (hasz_seq && ! hasz)
            tpointsegm_timestamp_at_value1_iter(inst1_2d, inst2_2d, d4, &t2);
          else
            tpointsegm_timestamp_at_value1_iter(inst1, inst2, d4, &t2);
        }
        if (hasz_seq && ! hasz)
        {
          pfree(inst1_2d); pfree(inst2_2d);
        }
        pfree(p3); pfree(p4);
        /* Project the segment to the timestamps if necessary and add the
         * instants */
        Datum inter1 = 0, inter2; /* make compiler quiet */
        bool free1 = false, free2 = false;
        /* If ninsts > 0 the instant was added in the previous iteration */
        if (ninsts == 0)
        {
          if (t1 == inst1->t)
          {
            instants[ninsts++] = (TInstant *) inst1;
            lower_inc &= p3_inc;
          }
          else if (t1 != inst2->t)
          {
            inter1 = tsegment_value_at_timestamptz(inst1, inst2, LINEAR, t1);
            free1 = true;
            instants[ninsts] = tinstant_make(inter1, inst1->temptype, t1);
            tofree[nfree++] = instants[ninsts++];
            lower_inc = p3_inc;
          }
          else if (t1 == inst2->t)
          {
            /* We have t1 == t2 == inst2->t and since found = true, we know
             * that we are on an inclusive border (p3_inc == p4_inc == true).
             * Thus, lower_inc is only false if we are at the last segment
             * and seq->period.upper_inc is false. */
            instants[ninsts++] = (TInstant *) inst2;
            lower_inc = (i == seq->count - 1) ? seq->period.upper_inc : true;
          }
        }
        if (t1 == t2)
        {
          /* If we are here: p3_inc == p4_inc == true, otherwise found would be false.
           * And we also know that p1 != p2, so segment has nonzero length.
           * Thus, we are in 1 of 3 cases: */
          if (t1 == inst1->t)
            /* We are exiting the box at the start of a segment
             * Start of segments are assumed inclusive except for the first one */
            upper_inc = (i == 1) ? seq->period.lower_inc : true;
          else if (t1 == inst2->t)
            /* We are entering the box at the end of a segment
             * End of segments are assumed exclusive except for the last one */
            upper_inc = (i == seq->count - 1) ? seq->period.upper_inc : false;
          else
            /* We clip the box at one of its inclusive corners */
            upper_inc = true;
        }
        else
        {
          if (t2 != inst2->t)
          {
            inter2 = tsegment_value_at_timestamptz(inst1, inst2, LINEAR, t2);
            free2 = true;
            /* Add the instant only if it is different from the previous one
             * Otherwise, assume that t1 == t2 and skip t2 */
            if (! free1 || ! geopoint_eq(DatumGetGserializedP(inter1),
                  DatumGetGserializedP(inter2)))
            {
              instants[ninsts] = tinstant_make(inter2, inst1->temptype, t2);
              tofree[nfree++] = instants[ninsts++];
              upper_inc = p4_inc;
            }
            else
              upper_inc = lower_inc;
          }
          else
          {
            instants[ninsts++] = (TInstant *) inst2;
            upper_inc &= p4_inc;
          }
        }
        if (free1)
          pfree(DatumGetPointer(inter1));
        if (free2)
          pfree(DatumGetPointer(inter2));
      }
      else
        makeseq = true;
    }
    if (makeseq && ninsts > 0)
    {
      /* We can occasionally have ninsts == 1 and lower_inc == upper_inc == false
       * These are cases where the sequence starts / ends on an inclusive border
       * and we have seq->lower_inc / seq->upper_inc being false.
       * Don't create a sequence, but still reset ninsts and lower_inc */
      if (ninsts > 1 || lower_inc || upper_inc)
        sequences[nseqs++] = tsequence_make((const TInstant **) instants, ninsts,
          lower_inc, upper_inc, LINEAR, NORMALIZE_NO);
      ninsts = 0;
      lower_inc = true;
    }
    inst1 = inst2;
    p1 = DatumGetGserializedP(tinstant_val(inst2));
  }
  /* See above for explanation of condition */
  if (ninsts > 0 && (ninsts > 1 || lower_inc || upper_inc))
    sequences[nseqs++] = tsequence_make((const TInstant **) instants, ninsts,
      lower_inc, upper_inc, LINEAR, NORMALIZE_NO);
  pfree_array((void **) tofree, nfree);
  pfree(instants);
  if (nseqs == 0)
  {
    pfree(sequences);
    return NULL;
  }
  return tsequenceset_make_free(sequences, nseqs, NORMALIZE);
}

/**
 * @brief Return a temporal point sequence restricted to (the complement of) a
 * spatiotemporal box
 * @details The function computes the "at" restriction on all dimensions. Then,
 * for the "minus" restriction, it computes the complement of the "at"
 * restriction with respect to the time dimension.
 * The function first filters the temporal point wrt the time dimension
 * to reduce the number of instants before computing the restriction to the
 * spatial dimension.
 * @param[in] seq Temporal sequence point
 * @param[in] box Spatiotemporal box
 * @param[in] border_inc True when the box contains the upper border
 * @param[in] atfunc True if the restriction is at, false for minus
 * @pre The box has X dimension and the arguments have the same SRID.
 * This is verified in #tpoint_restrict_stbox
 * @note Instantaneous sequences have been managed in the calling function
 */
TSequenceSet *
tpointseq_linear_restrict_stbox(const TSequence *seq, const STBox *box,
  bool border_inc, bool atfunc)
{
  assert(seq); assert(box);
  assert(tgeo_type(seq->temptype));
  assert(MEOS_FLAGS_LINEAR_INTERP(seq->flags));
  assert(seq->count > 1);

  /* Restrict to the temporal dimension */
  bool hast = MEOS_FLAGS_GET_T(box->flags);
  TSequence *at_t = hast ?
    tcontseq_at_tstzspan(seq, &box->period) : (TSequence *) seq;

  /* Restrict to the spatial dimension */
  TSequenceSet *result_at = NULL;
  if (at_t)
  {
    result_at = tpointseq_linear_at_stbox_xyz(at_t, box, border_inc);
    if (hast)
      pfree(at_t);
  }

  /* If "at" restriction, return */
  if (atfunc)
    return result_at;

  /* If "minus" restriction, compute the complement wrt time */
  if (! result_at)
    return tsequence_to_tsequenceset(seq);

  SpanSet *ss = tsequenceset_time(result_at);
  TSequenceSet *result = tcontseq_restrict_tstzspanset(seq, ss, atfunc);
  pfree(ss); pfree(result_at);
  return result;
}

/**
 * @ingroup meos_internal_temporal_restrict
 * @brief Return a temporal point sequence restricted to (the complement of) a
 * spatiotemporal box
 * @param[in] seq Temporal sequence point
 * @param[in] box Spatiotemporal box
 * @param[in] border_inc True when the box contains the upper border
 * @param[in] atfunc True if the restriction is at, false for minus
 * @pre The box has X dimension and the arguments have the same SRID.
 * This is verified in #tpoint_restrict_stbox
 * @csqlfn #Tpoint_at_stbox(), minus_stbox()
 */
Temporal *
tpointseq_restrict_stbox(const TSequence *seq, const STBox *box, bool border_inc,
  bool atfunc)
{
  assert(seq); assert(box); assert(tgeo_type(seq->temptype));
  interpType interp = MEOS_FLAGS_GET_INTERP(seq->flags);

  /* Instantaneous sequence */
  if (seq->count == 1)
  {
    if (tpointinst_restrict_stbox_iter(TSEQUENCE_INST_N(seq, 0), box,
        border_inc, atfunc))
      return (interp == DISCRETE) ? (Temporal *) tsequence_copy(seq) :
        (Temporal *) tsequence_to_tsequenceset(seq);
    return NULL;
  }

  /* General case */
  if (interp == DISCRETE)
    return (Temporal *) tpointseq_disc_restrict_stbox((TSequence *) seq, box,
      border_inc, atfunc);
  else if (interp == STEP)
    return (Temporal *) tpointseq_step_restrict_stbox((TSequence *) seq, box,
      border_inc, atfunc);
  else /* interp == LINEAR */
    return (Temporal *) tpointseq_linear_restrict_stbox((TSequence *) seq, box,
      border_inc, atfunc);
}

/**
 * @ingroup meos_internal_temporal_restrict
 * @brief Return a temporal point sequence restricted set to (the complement of) a
 * spatiotemporal box
 * @param[in] ss Temporal sequence set point
 * @param[in] box Spatiotemporal box
 * @param[in] border_inc True when the box contains the upper border
 * @param[in] atfunc True if the restriction is at, false for minus
 * @pre The box has X dimension and the arguments have the same SRID.
 * This is verified in #tpoint_restrict_stbox
 * @csqlfn #Tpoint_at_stbox(), minus_stbox()
 */
TSequenceSet *
tpointseqset_restrict_stbox(const TSequenceSet *ss, const STBox *box,
  bool border_inc, bool atfunc)
{
  assert(ss); assert(box); assert(tgeo_type(ss->temptype));
  const TSequence *seq;
  TSequenceSet *result = NULL;

  /* Singleton sequence set */
  if (ss->count == 1)
  {
    seq = TSEQUENCESET_SEQ_N(ss, 0);
    /* We can safely cast since the composing sequences are continuous */
    return (TSequenceSet *) tpointseq_restrict_stbox(seq, box, border_inc,
      atfunc);
  }

  /* General case */

  /* Initialize to 0 due to the bounding box test below */
  TSequenceSet **seqsets = palloc0(sizeof(TSequenceSet *) * ss->count);
  int totalseqs = 0;
  for (int i = 0; i < ss->count; i++)
  {
    /* Bounding box test */
    seq = TSEQUENCESET_SEQ_N(ss, i);
    STBox box1;
    tspatialseq_set_stbox(seq, &box1);
    if (atfunc && ! overlaps_stbox_stbox(&box1, box))
      continue;
    else
    {
      /* We can safely cast since the composing sequences are continuous */
      seqsets[i] = (TSequenceSet *) tpointseq_restrict_stbox(seq, box,
        border_inc, atfunc);
      if (seqsets[i])
        totalseqs += seqsets[i]->count;
    }
  }
  /* Assemble the sequences from all the sequence sets */
  if (totalseqs > 0)
    result = tseqsetarr_to_tseqset(seqsets, ss->count, totalseqs);
  pfree_array((void **) seqsets, ss->count);
  return result;
}

/**
 * @ingroup meos_internal_temporal_restrict
 * @brief Return a temporal point restricted to (the complement of) a
 * spatiotemporal box
 * @param[in] temp Temporal point
 * @param[in] box Spatiotemporal box
 * @param[in] border_inc True when the box contains the upper border
 * @param[in] atfunc True if the restriction is at, false for minus
 * @note It is possible to mix 2D/3D geometries, the Z dimension is only
 * considered if both the temporal point and the box have Z dimension
 */
Temporal *
tpoint_restrict_stbox(const Temporal *temp, const STBox *box, bool border_inc,
  bool atfunc)
{
  assert(temp); assert(box); assert(tgeo_type(temp->temptype));
  /* At least one of MEOS_FLAGS_GET_X and MEOS_FLAGS_GET_T is true */
  bool hasx = MEOS_FLAGS_GET_X(box->flags);
  bool hast = MEOS_FLAGS_GET_T(box->flags);
  assert(hasx || hast);
  /* Ensure validity of the arguments */
  if (! ensure_same_geodetic(temp->flags, box->flags) ||
      (MEOS_FLAGS_GET_X(box->flags) &&
        ! ensure_same_srid(tpoint_srid(temp), stbox_srid(box))))
    return NULL;

  /* Short-circuit restriction to only T dimension */
  if (hast && ! hasx)
    return temporal_restrict_tstzspan(temp, &box->period, atfunc);

  /* Parameter test */
  assert(tpoint_srid(temp) == stbox_srid(box));
  assert(MEOS_FLAGS_GET_GEODETIC(temp->flags) ==
    MEOS_FLAGS_GET_GEODETIC(box->flags));

  /* Bounding box test */
  STBox box1;
  tspatial_set_stbox(temp, &box1);
  bool overlaps = overlaps_stbox_stbox(&box1, box);
  if (! overlaps)
    return atfunc ? NULL : temporal_cp(temp);

  assert(temptype_subtype(temp->subtype));
  switch (temp->subtype)
  {
    case TINSTANT:
      return (Temporal *) tpointinst_restrict_stbox((TInstant *) temp,
        box, border_inc, atfunc);
    case TSEQUENCE:
      return (Temporal *) tpointseq_restrict_stbox((TSequence *) temp,
        box, border_inc, atfunc);
    default: /* TSEQUENCESET */
      return (Temporal *) tpointseqset_restrict_stbox((TSequenceSet *) temp,
        box, border_inc, atfunc);
  }
}

#if MEOS
/**
 * @ingroup meos_temporal_restrict
 * @brief Return a temporal point restricted to a spatiotemporal box
 * @param[in] temp Temporal point
 * @param[in] box Spatiotemporal box
 * @param[in] border_inc True when the box contains the upper border
 * @csqlfn #Tpoint_at_stbox()
 */
Temporal *
tpoint_at_stbox(const Temporal *temp, const STBox *box, bool border_inc)
{
  /* Ensure validity of the arguments */
  if (! ensure_valid_tpoint_box(temp, box))
    return NULL;
  return tpoint_restrict_stbox(temp, box, border_inc, REST_AT);
}

/**
 * @ingroup meos_temporal_restrict
 * @brief Return a temporal point restricted to the complement of a
 * spatiotemporal box
 * @param[in] temp Temporal point
 * @param[in] box Spatiotemporal box
 * @param[in] border_inc True when the box contains the upper border
 * @csqlfn #Tpoint_minus_stbox()
 */
Temporal *
tpoint_minus_stbox(const Temporal *temp, const STBox *box, bool border_inc)
{
  /* Ensure validity of the arguments */
  if (! ensure_valid_tpoint_box(temp, box))
    return NULL;
  return tpoint_restrict_stbox(temp, box, border_inc, REST_MINUS);
}
#endif /* MEOS */

/*****************************************************************************
 * Restriction functions for a spatiotemporal box keeping the original
 * segments WITHOUT clipping
 *****************************************************************************/

/**
 * @ingroup meos_internal_temporal_restrict
 * @brief Return a temporal point sequence with the segments that intersect
 * a spatiotemporal box in BOTH the spatial and the temporal dimension (if any)
 * @param[in] seq Temporal sequence point
 * @param[in] box Spatiotemporal box
 * @param[in] border_inc True when the box contains the upper border
 * @pre The box has X dimension and the arguments have the same SRID.
 * This is verified in #tpoint_restrict_stbox
 * @csqlfn #Tpoint_at_stbox(), minus_stbox()
 */
TSequenceSet *
tpointseq_at_stbox_segm(const TSequence *seq, const STBox *box,
  bool border_inc)
{
  assert(MEOS_FLAGS_GET_INTERP(seq->flags) == LINEAR);

  /* Instantaneous sequence */
  if (seq->count == 1)
  {
    if (tpointinst_restrict_stbox_iter(TSEQUENCE_INST_N(seq, 0), box,
        border_inc, REST_AT))
      return tsequence_to_tsequenceset(seq);
    return NULL;
  }

  /* General case */
  bool hasz_seq = MEOS_FLAGS_GET_Z(seq->flags);
  bool hasz_box = MEOS_FLAGS_GET_Z(box->flags);
  bool hasz = hasz_seq && hasz_box;
  bool hast = MEOS_FLAGS_GET_T(box->flags);
  TSequence **sequences = palloc(sizeof(TSequence *) * (seq->count - 1));
  const TInstant **instants = palloc(sizeof(TInstant *) * seq->count);
  const TInstant *inst1 = TSEQUENCE_INST_N(seq, 0);
  GSERIALIZED *p1 = DatumGetGserializedP(tinstant_val(inst1));
  bool lower_inc = seq->period.lower_inc;
  bool lower_inc_seq = lower_inc, upper_inc;
  int nseqs = 0, ninsts = 0;
  for (int i = 1; i < seq->count; i++)
  {
    const TInstant *inst2 = TSEQUENCE_INST_N(seq, i);
    upper_inc = (i == seq->count - 1) ? seq->period.upper_inc : false;
    GSERIALIZED *p2 = DatumGetGserializedP(tinstant_val(inst2));
    /* Keep the segment if intersects the bounding box */
    bool inter = false;
    if (geopoint_eq(p1, p2))
    {
      /* Constant segment */
      if (tpointinst_restrict_stbox_iter(inst1, box, border_inc, REST_AT))
        inter = true;
    }
    else
    {
      /* Keep the segment if intersects the bounding box in BOTH the spatial
       * dimension and the temporal dimension (if any) */
      if (liangBarskyClip(p1, p2, box, hasz, border_inc, NULL, NULL, NULL,
          NULL))
      {
        if (hast)
        {
          Span s;
          span_set(TimestampTzGetDatum(inst1->t),
            TimestampTzGetDatum(inst2->t), lower_inc, upper_inc, T_TIMESTAMPTZ,
              T_TSTZSPAN, &s);
          if (overlaps_span_span(&s, &box->period))
            inter = true;
        }
        else
          inter = true;
      }
    }
    if (inter)
    {
      if (ninsts == 0)
        instants[ninsts++] = inst1;
      instants[ninsts++] = inst2;
    }
    else if (ninsts > 0)
    {
      sequences[nseqs++] = tsequence_make(instants, ninsts, lower_inc_seq,
        upper_inc, LINEAR, NORMALIZE_NO);
      ninsts = 0;
      lower_inc_seq = lower_inc;
    }
    lower_inc = true;
    inst1 = inst2;
    p1 = p2;
  }
  if (ninsts > 0)
    sequences[nseqs++] = tsequence_make(instants, ninsts, lower_inc_seq,
      upper_inc, LINEAR, NORMALIZE_NO);
  pfree(instants);
  return tsequenceset_make_free(sequences, nseqs, NORMALIZE);
}

/**
 * @ingroup meos_internal_temporal_restrict
 * @brief Return a temporal point sequence set with the segments that intersect
 * a spatiotemporal box in BOTH the spatial and the temporal dimension (if any)
 * @param[in] ss Temporal sequence set point
 * @param[in] box Spatiotemporal box
 * @param[in] border_inc True when the box contains the upper border
 * @pre The box has X dimension and the arguments have the same SRID.
 * This is verified in #tpoint_restrict_stbox
 * @csqlfn #Tpoint_at_stbox(), minus_stbox()
 */
TSequenceSet *
tpointseqset_at_stbox_segm(const TSequenceSet *ss, const STBox *box,
  bool border_inc)
{
  assert(ss); assert(box); assert(tgeo_type(ss->temptype));
  const TSequence *seq;
  TSequenceSet *result = NULL;

  /* Singleton sequence set */
  if (ss->count == 1)
    /* We can safely cast since the composing sequences are continuous */
    return (TSequenceSet *) tpointseq_at_stbox_segm(TSEQUENCESET_SEQ_N(ss, 0),
      box, border_inc);

  /* General case */

  /* Initialize to 0 due to the bounding box test below */
  TSequenceSet **seqsets = palloc0(sizeof(TSequenceSet *) * ss->count);
  int totalseqs = 0;
  for (int i = 0; i < ss->count; i++)
  {
    /* Bounding box test */
    seq = TSEQUENCESET_SEQ_N(ss, i);
    STBox box1;
    tspatialseq_set_stbox(seq, &box1);
    if (! overlaps_stbox_stbox(&box1, box))
      continue;
    else
    {
      /* We can safely cast since the composing sequences are continuous */
      seqsets[i] = (TSequenceSet *) tpointseq_at_stbox_segm(seq, box,
        border_inc);
      if (seqsets[i])
        totalseqs += seqsets[i]->count;
    }
  }
  /* Assemble the sequences from all the sequence sets */
  if (totalseqs > 0)
    result = tseqsetarr_to_tseqset(seqsets, ss->count, totalseqs);
  pfree_array((void **) seqsets, ss->count);
  return result;
}

/**
 * @ingroup meos_internal_temporal_restrict
 * @brief Return a temporal point with the segments that intersect
 * a spatiotemporal box in BOTH the spatial and the temporal dimension (if any)
 * @param[in] temp Temporal point
 * @param[in] box Spatiotemporal box
 * @param[in] border_inc True when the box contains the upper border
 * @note It is possible to mix 2D/3D geometries, the Z dimension is only
 * considered if both the temporal point and the box have Z dimension
 * @pre This function supposes all the checks have been done in the calling
 * function
 */
Temporal *
tpoint_at_stbox_segm(const Temporal *temp, const STBox *box, bool border_inc)
{
  assert(temp); assert(box); assert(tgeo_type(temp->temptype));
  /* The following implies that temp->subtype != TINSTANT */
  assert(MEOS_FLAGS_GET_INTERP(temp->flags) == LINEAR);
  /* At least one of MEOS_FLAGS_GET_X and MEOS_FLAGS_GET_T is true */
  bool hasx = MEOS_FLAGS_GET_X(box->flags);
  bool hast = MEOS_FLAGS_GET_T(box->flags);
  assert(hasx || hast);
  /* Ensure validity of the arguments */
  if (! ensure_same_geodetic(temp->flags, box->flags) ||
      (MEOS_FLAGS_GET_X(box->flags) &&
        ! ensure_same_srid(tpoint_srid(temp), stbox_srid(box))))
    return NULL;

  /* Short-circuit restriction to only T dimension */
  if (hast && ! hasx)
    return temporal_restrict_tstzspan(temp, &box->period, REST_AT);

  /* Parameter test */
  assert(tpoint_srid(temp) == stbox_srid(box));
  assert(MEOS_FLAGS_GET_GEODETIC(temp->flags) ==
    MEOS_FLAGS_GET_GEODETIC(box->flags));

  /* Bounding box test */
  STBox box1;
  tspatial_set_stbox(temp, &box1);
  if (! overlaps_stbox_stbox(&box1, box))
    return NULL;

  assert(temptype_subtype(temp->subtype));
  if (temp->subtype == TSEQUENCE)
    return (Temporal *) tpointseq_at_stbox_segm((TSequence *) temp,
        box, border_inc);
  else /* temp->subtype == TSEQUENCESET */
    return (Temporal *) tpointseqset_at_stbox_segm((TSequenceSet *) temp,
      box, border_inc);
}

/*****************************************************************************
 * Restriction functions for geometry and possible a Z span and a time period
 * N.B. In the current PostGIS version there is no true ST_Intersection
 * function for geography, it is implemented as ST_DWithin with tolerance 0
 *****************************************************************************/

/**
 * @brief Return a temporal point instant restricted to (the complement of) a
 * spatiotemporal box (iterator function)
 * @pre The arguments have the same SRID, the geometry is 2D and is not empty.
 * This is verified in #tpoint_restrict_geom_time
 */
static bool
tpointinst_restrict_geom_time_iter(const TInstant *inst, const GSERIALIZED *gs,
  const Span *zspan, const Span *period, bool atfunc)
{
  /* Restrict to the T dimension */
  if (period && ! contains_span_value(period, TimestampTzGetDatum(inst->t)))
    return ! atfunc;

  /* Restrict to the Z dimension */
  Datum value = tinstant_val(inst);
  if (zspan)
  {
    const POINT3DZ *p = DATUM_POINT3DZ_P(value);
    if (! contains_span_value(zspan, Float8GetDatum(p->z)))
      return ! atfunc;
  }

  /* Restrict to the XY dimension */
  if (! DatumGetBool(geom_intersects2d(value, PointerGetDatum(gs))))
    return ! atfunc;

  /* Point intersects the geometry and the Z/T spans */
  return atfunc;
}

/**
 * @ingroup meos_internal_temporal_restrict
 * @brief Return a temporal point instant restricted to (the complement of) a
 * geometry
 * and possibly a Z span and a timestamptz span
 * @param[in] inst Temporal point
 * @param[in] gs Geometry
 * @param[in] zspan Span of values to restrict the Z dimension
 * @param[in] period Period to restrict the T dimension
 * @param[in] atfunc True if the restriction is at, false for minus
 */
TInstant *
tpointinst_restrict_geom_time(const TInstant *inst, const GSERIALIZED *gs,
  const Span *zspan, const Span *period, bool atfunc)
{
  assert(inst); assert(gs); assert(tgeo_type(inst->temptype));
  if (tpointinst_restrict_geom_time_iter(inst, gs, zspan, period, atfunc))
    return tinstant_copy(inst);
  return NULL;
}

/**
 * @brief Return a temporal point discrete sequence restricted to
 * (the complement of) a geometry and possibly a Z span and a timestamptz span
 * @param[in] seq Temporal point
 * @param[in] gs Geometry
 * @param[in] zspan Span of values to restrict the Z dimension
 * @param[in] period Period to restrict the T dimension
 * @param[in] atfunc True if the restriction is at, false for minus
 * @pre Instantaneous sequences have been managed in the calling function
 */
TSequence *
tpointseq_disc_restrict_geom_time(const TSequence *seq, const GSERIALIZED *gs,
  const Span *zspan, const Span *period, bool atfunc)
{
  assert(seq); assert(gs); assert(tgeo_type(seq->temptype));
  assert(MEOS_FLAGS_GET_INTERP(seq->flags) == DISCRETE);
  assert(seq->count > 1);

  const TInstant **instants = palloc(sizeof(TInstant *) * seq->count);
  int ninsts = 0;
  for (int i = 0; i < seq->count; i++)
  {
    const TInstant *inst = TSEQUENCE_INST_N(seq, i);
    if (tpointinst_restrict_geom_time_iter(inst, gs, zspan, period, atfunc))
      instants[ninsts++] = inst;
  }
  TSequence *result = NULL;
  if (ninsts > 0)
    result = tsequence_make(instants, ninsts, true, true, DISCRETE,
      NORMALIZE_NO);
  pfree(instants);
  return result;
}

/**
 * @brief Return a temporal sequence point with step interpolation
 * restricted to a geometry and possibly a Z span and a timestamptz span
 * @param[in] seq Temporal point
 * @param[in] gs Geometry
 * @param[in] zspan Span of values to restrict the Z dimension
 * @param[in] period Period to restrict the T dimension
 * @param[in] atfunc True if the restriction is at, false for minus
 * @pre Instantaneous sequences have been managed in the calling function
 * @note The function computes the "at" restriction on all dimensions. Then,
 * for the "minus" restriction, it computes the complement of the "at"
 * restriction with respect to the time dimension.
 */
TSequenceSet *
tpointseq_step_restrict_geom_time(const TSequence *seq, const GSERIALIZED *gs,
   const Span *zspan, const Span *period, bool atfunc)
{
  assert(seq); ensure_not_null((void *) gs);
  assert(tgeo_type(seq->temptype));
  assert(MEOS_FLAGS_GET_INTERP(seq->flags) == STEP);
  assert(seq->count > 1);

  /* Compute the time span of the result if period is given */
  Span timespan;
  if (period)
  {
    if (! inter_span_span(&seq->period, period, &timespan))
      return atfunc ? NULL : tsequence_to_tsequenceset(seq);
  }

  /* Compute the sequences for "at" restriction */
  bool lower_inc;
  TSequence **sequences = palloc(sizeof(TSequence *) * seq->count);
  TInstant **instants = palloc(sizeof(TInstant *) * seq->count);
  TimestampTz start = DatumGetTimestampTz(seq->period.lower);
  int ninsts = 0, nseqs = 0;
  for (int i = 0; i < seq->count; i++)
  {
    const TInstant *inst = TSEQUENCE_INST_N(seq, i);
    if (tpointinst_restrict_geom_time_iter(inst, gs, zspan, period, REST_AT))
      instants[ninsts++] = (TInstant *) inst;
    else
    {
      if (ninsts > 0)
      {
        /* Continue the last instant of the sequence until the time of inst2
         * projected to the period (if any) */
        Datum value = tinstant_val(instants[ninsts - 1]);
        bool tofree = false;
        bool upper_inc = false;
        if (period)
        {
          Span extend, inter;
          span_set(TimestampTzGetDatum(instants[ninsts - 1]->t),
            TimestampTzGetDatum(inst->t), true, false, T_TIMESTAMPTZ,
              T_TSTZSPAN, &extend);
          if (inter_span_span(&timespan, &extend, &inter))
          {
            if (TimestampTzGetDatum(inter.lower) !=
                TimestampTzGetDatum(inter.upper))
            {
              instants[ninsts++] = tinstant_make(value, seq->temptype,
                DatumGetTimestampTz(inter.upper));
              tofree = true;
            }
            else
              upper_inc = true;
          }
        }
        else
        {
          /* Continue the last instant of the sequence until the time of inst2 */
          instants[ninsts++] = tinstant_make(value, seq->temptype, inst->t);
          tofree = true;
        }
        lower_inc = (instants[0]->t == start) ? seq->period.lower_inc : true;
        sequences[nseqs++] = tsequence_make((const TInstant **) instants,
          ninsts, lower_inc, upper_inc, STEP, NORMALIZE_NO);
        if (tofree)
          pfree(instants[ninsts - 1]);
        ninsts = 0;
      }
    }
  }
  /* Add a last sequence with the remaining instants */
  if (ninsts > 0)
  {
    lower_inc = (instants[0]->t == start) ? seq->period.lower_inc : true;
    TimestampTz end = DatumGetTimestampTz(seq->period.upper);
    bool upper_inc = (instants[ninsts - 1]->t == end) ?
      seq->period.upper_inc : false;
    sequences[nseqs++] = tsequence_make((const TInstant **) instants, ninsts,
      lower_inc, upper_inc, STEP, NORMALIZE_NO);
  }
  pfree(instants);

  /* Clean up and return if no sequences have been found with "at" */
  if (nseqs == 0)
  {
    pfree(sequences);
    return atfunc ? NULL : tsequence_to_tsequenceset(seq);
  }

  /* Construct the result for "at" restriction */
  TSequenceSet *result_at = tsequenceset_make_free(sequences, nseqs,
    NORMALIZE_NO);
  /* If "at" restriction, return */
  if (atfunc)
    return result_at;

  /* If "minus" restriction, compute the complement wrt time */
  SpanSet *ss = tsequenceset_time(result_at);
  TSequenceSet *result = tcontseq_restrict_tstzspanset(seq, ss, atfunc);
  pfree(ss); pfree(result_at);
  return result;
}

/*****************************************************************************/

/**
 * @brief Get the periods at which a temporal sequence point with linear
 * interpolation intersects a geometry
 * @param[in] seq Temporal point
 * @param[in] gsinter Intersection of the temporal point and the geometry
 * @param[out] count Number of elements in the resulting array
 * @pre The temporal sequence is simple, that is, non self-intersecting and
 * the intersecting geometry is non empty
 */
Span *
tpointseq_interperiods(const TSequence *seq, GSERIALIZED *gsinter, int *count)
{
  /* The temporal sequence has at least 2 instants since
   * (1) the test for instantaneous full sequence is done in the calling function
   * (2) the simple components of a non self-intersecting sequence have at least
   *     two instants */
  assert(seq->count > 1);
  const TInstant *start = TSEQUENCE_INST_N(seq, 0);
  const TInstant *end = TSEQUENCE_INST_N(seq, seq->count - 1);
  Span *result;

  /* If the sequence is stationary the whole sequence intersects with the
   * geometry since gsinter is not empty */
  if (seq->count == 2 &&
    datum_point_eq(tinstant_val(start), tinstant_val(end)))
  {
    result = palloc(sizeof(Span));
    result[0] = seq->period;
    *count = 1;
    return result;
  }

  /* General case */
  LWGEOM *geom_inter = lwgeom_from_gserialized(gsinter);
  int type = geom_inter->type;
  int ninter;
  LWPOINT *point_inter = NULL; /* make compiler quiet */
  LWLINE *line_inter = NULL; /* make compiler quiet */
  LWCOLLECTION *coll = NULL; /* make compiler quiet */
  if (type == POINTTYPE)
  {
    ninter = 1;
    point_inter = lwgeom_as_lwpoint(geom_inter);
  }
  else if (type == LINETYPE)
  {
    ninter = 1;
    line_inter = lwgeom_as_lwline(geom_inter);
  }
  else
  /* It is a collection of type MULTIPOINTTYPE, MULTILINETYPE, or
   * COLLECTIONTYPE */
  {
    coll = lwgeom_as_lwcollection(geom_inter);
    ninter = coll->ngeoms;
  }
  Span *periods = palloc(sizeof(Span) * ninter);
  int npers = 0;
  for (int i = 0; i < ninter; i++)
  {
    if (ninter > 1)
    {
      /* Find the i-th intersection */
      LWGEOM *subgeom = coll->geoms[i];
      if (subgeom->type == POINTTYPE)
        point_inter = lwgeom_as_lwpoint(subgeom);
      else /* type == LINETYPE */
        line_inter = lwgeom_as_lwline(subgeom);
      type = subgeom->type;
    }
    TimestampTz t1, t2;
    GSERIALIZED *gspoint;
    /* Each intersection is either a point or a linestring */
    if (type == POINTTYPE)
    {
      gspoint = geo_serialize((LWGEOM *) point_inter);
      tpointseq_timestamp_at_value(seq, PointerGetDatum(gspoint), &t1);
      pfree(gspoint);
      /* If the intersection is not at an exclusive bound */
      if ((seq->period.lower_inc || t1 > start->t) &&
          (seq->period.upper_inc || t1 < end->t))
        span_set(t1, t1, true, true, T_TIMESTAMPTZ, T_TSTZSPAN,
          &periods[npers++]);
    }
    else
    {
      /* Get the fraction of the start point of the intersecting line */
      LWPOINT *point = lwline_get_lwpoint(line_inter, 0);
      gspoint = geo_serialize((LWGEOM *) point);
      lwpoint_free(point);
      tpointseq_timestamp_at_value(seq, PointerGetDatum(gspoint), &t1);
      pfree(gspoint);
      /* Get the fraction of the end point of the intersecting line */
      point = lwline_get_lwpoint(line_inter, line_inter->points->npoints - 1);
      gspoint = geo_serialize((LWGEOM *) point);
      lwpoint_free(point);
      tpointseq_timestamp_at_value(seq, PointerGetDatum(gspoint), &t2);
      pfree(gspoint);
      /* If t1 == t2 and the intersection is not at an exclusive bound */
      if (t1 == t2)
      {
        if ((seq->period.lower_inc || t1 > start->t) &&
            (seq->period.upper_inc || t1 < end->t))
          span_set(t1, t1, true, true, T_TIMESTAMPTZ, T_TSTZSPAN,
            &periods[npers++]);
      }
      else
      {
        TimestampTz lower1 = Min(t1, t2);
        TimestampTz upper1 = Max(t1, t2);
        bool lower_inc1 = (lower1 == start->t) ? seq->period.lower_inc : true;
        bool upper_inc1 = (upper1 == end->t) ? seq->period.upper_inc : true;
        span_set(lower1, upper1, lower_inc1, upper_inc1, T_TIMESTAMPTZ,
          T_TSTZSPAN, &periods[npers++]);
      }
    }
  }
  lwgeom_free(geom_inter);

  if (npers == 0)
  {
    *count = npers;
    pfree(periods);
    return NULL;
  }
  if (npers == 1)
  {
    *count = npers;
    return periods;
  }

  int newcount;
  result = spanarr_normalize(periods, npers, ORDER, &newcount);
  *count = newcount;
  pfree(periods);
  return result;
}

/**
 * @brief Return a temporal sequence point with linear interpolation
 * restricted to a geometry
 * @details The computation is based on the PostGIS function @p ST_Intersection
 * which delegates the computation to GEOS. The geometry must be in 2D.
 * When computing the intersection the Z values of the temporal point must
 * be dropped since the Z values "are copied, averaged or interpolated"
 * as stated in https://postgis.net/docs/ST_Intersection.html
 * After this computation, the Z values are recovered by restricting the
 * original sequence to the time span of the 2D result.
 * @note Instantaneous sequences must be managed since this function is called
 * after restricting to the time dimension
 * @pre The arguments have the same SRID, the geometry is 2D and is not empty.
 * This is verified in #tpoint_restrict_geom_time
 */
static TSequenceSet *
tpointseq_linear_at_geom(const TSequence *seq, const GSERIALIZED *gs)
{
  assert(MEOS_FLAGS_LINEAR_INTERP(seq->flags));
  TSequenceSet *result;

  /* Instantaneous sequence */
  if (seq->count == 1)
  {
    if (tpointinst_restrict_geom_time_iter(TSEQUENCE_INST_N(seq, 0), gs, NULL,
        NULL, REST_AT))
      return tsequence_to_tsequenceset(seq);
    return NULL;
  }

  /* Bounding box test */
  STBox box1, box2;
  tspatialseq_set_stbox(seq, &box1);
  /* Non-empty geometries have a bounding box */
  geo_set_stbox(gs, &box2);
  if (! overlaps_stbox_stbox(&box1, &box2))
    return NULL;

  /* Convert the point to 2D before computing the restriction to geometry */
  bool hasz = MEOS_FLAGS_GET_Z(seq->flags);
  TSequence *seq2d = hasz ?
    (TSequence *) tpoint_force2d((Temporal *) seq) : (TSequence *) seq;

  /* Split the temporal point in an array of non self-intersecting fragments
   * to be able to recover the time dimension after obtaining the spatial
   * intersection */
  int nsimple;
  TSequence **simpleseqs = tpointseq_make_simple(seq2d, &nsimple);
  Span *allperiods = NULL; /* make compiler quiet */
  int totalpers = 0;
  GSERIALIZED *traj, *gsinter;
  Datum inter;

  if (nsimple == 1)
  {
    /* Particular case when the input sequence is simple */
    pfree_array((void **) simpleseqs, nsimple);
    traj = tpointseq_trajectory(seq2d);
    inter = geom_intersection2d(PointerGetDatum(traj), PointerGetDatum(gs));
    gsinter = DatumGetGserializedP(inter);
    if (! gserialized_is_empty(gsinter))
      allperiods = tpointseq_interperiods(seq2d, gsinter, &totalpers);
    PG_FREE_IF_COPY_P(gsinter, DatumGetPointer(inter));
    pfree(DatumGetPointer(inter)); pfree(traj);
    if (totalpers == 0)
    {
      if (hasz)
        pfree(seq2d);
      return NULL;
    }
  }
  else
  {
    /* General case */
    if (hasz)
      pfree(seq2d);
    Span **periods = palloc(sizeof(Span *) * nsimple);
    int *npers = palloc0(sizeof(int) * nsimple);
    /* Loop for every simple fragment of the sequence */
    for (int i = 0; i < nsimple; i++)
    {
      traj = tpointseq_trajectory(simpleseqs[i]);
      inter = geom_intersection2d(PointerGetDatum(traj), PointerGetDatum(gs));
      gsinter = DatumGetGserializedP(inter);
      if (! gserialized_is_empty(gsinter))
      {
        periods[i] = tpointseq_interperiods(simpleseqs[i], gsinter,
          &npers[i]);
        totalpers += npers[i];
      }
      PG_FREE_IF_COPY_P(gsinter, DatumGetPointer(inter));
      pfree(DatumGetPointer(inter)); pfree(traj);
    }
    pfree_array((void **) simpleseqs, nsimple);
    if (totalpers == 0)
    {
      pfree(periods); pfree(npers);
      return NULL;
    }

    /* Assemble the periods into a single array */
    allperiods = palloc(sizeof(Span) * totalpers);
    int k = 0;
    for (int i = 0; i < nsimple; i++)
    {
      for (int j = 0; j < npers[i]; j++)
        allperiods[k++] = periods[i][j];
      if (npers[i] != 0)
        pfree(periods[i]);
    }
    pfree(periods); pfree(npers);
    /* It is necessary to sort the periods */
    spanarr_sort(allperiods, totalpers);
  }
  /* Compute the periodset */
  assert(totalpers > 0);
  SpanSet *ss = spanset_make_free(allperiods, totalpers, NORMALIZE, ORDER);
  /* Recover the Z values from the original sequence */
  result = tcontseq_restrict_tstzspanset(seq, ss, REST_AT);
  pfree(ss);
  return result;
}

/**
 * @brief Return a temporal sequence point with linear interpolation
 * restricted to (the complement of) a geometry and possibly a Z span and a
 * timestamptz span
 * @details The function first filters the temporal point wrt the time
 * dimension to reduce the number of instants before computing the restriction
 * to the geometry, which is an expensive operation. Notice that we need to
 * filter wrt the Z dimension after that since while doing this, the subtype of
 * the temporal point may change from a sequence to a sequence set.
 * @param[in] seq Temporal point
 * @param[in] gs Geometry
 * @param[in] zspan Span of values to restrict the Z dimension
 * @param[in] period Period to restrict the T dimension
 * @param[in] atfunc True if the restriction is at, false for minus
 * @note The function computes the "at" restriction on all dimensions. Then,
 * for the "minus" restriction, it computes the complement of the "at"
 * restriction with respect to the time dimension.
 * @pre Instantaneous sequences have been managed in the calling function
 */
TSequenceSet *
tpointseq_linear_restrict_geom_time(const TSequence *seq,
  const GSERIALIZED *gs, const Span *zspan, const Span *period, bool atfunc)
{
  assert(seq); ensure_not_null((void *) gs);
  assert(tgeo_type(seq->temptype));
  assert(MEOS_FLAGS_LINEAR_INTERP(seq->flags));
  assert(seq->count > 1);

  /* Restrict to the temporal dimension */
  TSequence *at_t = period ?
    tcontseq_at_tstzspan(seq, period) : (TSequence *) seq;

  /* Compute atGeometry for the sequence restricted to the time dimension */
  TSequenceSet *at_xyt = NULL;
  if (at_t)
  {
    at_xyt = tpointseq_linear_at_geom(at_t, gs);
    if (period)
      pfree(at_t);
  }

  /* Restrict to the Z dimension */
  TSequenceSet *result_at = NULL;
  if (at_xyt)
  {
    if (zspan)
    {
      /* Bounding box test for the Z dimension */
      STBox box1;
      tspatialseqset_set_stbox(at_xyt, &box1);
      Span zspan1;
      span_set(Float8GetDatum(box1.zmin), Float8GetDatum(box1.zmax), true, true,
        T_FLOAT8, T_FLOATSPAN, &zspan1);
      if (over_span_span(&zspan1, zspan))
      {
        /* Get the Z coordinate values as a temporal float */
        Temporal *tfloat_z = tpoint_get_coord((Temporal *) at_xyt, 2);
        /* Restrict to the zspan */
        Temporal *tfloat_zspan = tnumber_restrict_span(tfloat_z, zspan, REST_AT);
        pfree(tfloat_z);
        if (tfloat_zspan)
        {
          SpanSet *ss = temporal_time(tfloat_zspan);
          result_at = tsequenceset_restrict_tstzspanset(at_xyt, ss, REST_AT);
          pfree(tfloat_zspan);
          pfree(ss);
        }
      }
      pfree(at_xyt);
    }
    else
      result_at = at_xyt;
  }

  /* If "at" restriction, return */
  if (atfunc)
    return result_at;

  /* If "minus" restriction, compute the complement wrt time */
  if (! result_at)
    return tsequence_to_tsequenceset(seq);

  SpanSet *ss = tsequenceset_time(result_at);
  TSequenceSet *result = tcontseq_restrict_tstzspanset(seq, ss, atfunc);
  pfree(ss); pfree(result_at);
  return result;
}

/**
 * @ingroup meos_internal_temporal_restrict
 * @brief Return a temporal point sequence restricted to (the complement of) a
 * geometry and possibly a Z span and a timestamptz span
 * @param[in] seq Temporal point
 * @param[in] gs Geometry
 * @param[in] zspan Span of values to restrict the Z dimension
 * @param[in] period Period to restrict the T dimension
 * @param[in] atfunc True if the restriction is at, false for minus
 */
Temporal *
tpointseq_restrict_geom_time(const TSequence *seq, const GSERIALIZED *gs,
  const Span *zspan, const Span *period, bool atfunc)
{
  assert(seq); assert(gs); assert(tgeo_type(seq->temptype));
  interpType interp = MEOS_FLAGS_GET_INTERP(seq->flags);

  /* Instantaneous sequence */
  if (seq->count == 1)
  {
    if (tpointinst_restrict_geom_time_iter(TSEQUENCE_INST_N(seq, 0), gs, zspan,
        period, atfunc))
      return (interp == DISCRETE) ? (Temporal *) tsequence_copy(seq) :
        (Temporal *) tsequence_to_tsequenceset(seq);
    return NULL;
  }

  /* General case */
  if (interp == DISCRETE)
    return (Temporal *) tpointseq_disc_restrict_geom_time((TSequence *) seq,
      gs, zspan, period, atfunc);
  else if (interp == STEP)
    return (Temporal *) tpointseq_step_restrict_geom_time((TSequence *) seq,
      gs, zspan, period, atfunc);
  else /* interp == LINEAR */
    return (Temporal *) tpointseq_linear_restrict_geom_time((TSequence *) seq,
      gs, zspan, period, atfunc);
}

/**
 * @ingroup meos_internal_temporal_restrict
 * @brief Return a temporal point sequence set restricted to (the complement
 * of) a geometry and possibly a Z span and a timestamptz span
 * @param[in] ss Temporal point
 * @param[in] gs Geometry
 * @param[in] zspan Span of values to restrict the Z dimension
 * @param[in] period Period to restrict the T dimension
 * @param[in] atfunc True if the restriction is at, false for minus
 */
TSequenceSet *
tpointseqset_restrict_geom_time(const TSequenceSet *ss, const GSERIALIZED *gs,
  const Span *zspan, const Span *period, bool atfunc)
{
  assert(ss); assert(gs); assert(tgeo_type(ss->temptype));
  const TSequence *seq;
  TSequenceSet *result = NULL;

  /* Singleton sequence set */
  if (ss->count == 1)
  {
    seq = TSEQUENCESET_SEQ_N(ss, 0);
    /* We can safely cast since the composing sequences are continuous */
    return (TSequenceSet *) tpointseq_restrict_geom_time(seq, gs, zspan,
      period, atfunc);
  }

  /* General case */
  STBox box2;
  /* Non-empty geometries have a bounding box */
  geo_set_stbox(gs, &box2);

  /* Initialize to 0 due to the bounding box test below */
  TSequenceSet **seqsets = palloc0(sizeof(TSequenceSet *) * ss->count);
  int totalseqs = 0;
  for (int i = 0; i < ss->count; i++)
  {
    /* Bounding box test */
    seq = TSEQUENCESET_SEQ_N(ss, i);
    STBox box1;
    tspatialseq_set_stbox(seq, &box1);
    if (atfunc && ! overlaps_stbox_stbox(&box1, &box2))
      continue;
    else
    {
      /* We can safely cast since the composing sequences are continuous */
      seqsets[i] = (TSequenceSet *) tpointseq_restrict_geom_time(seq, gs,
        zspan, period, atfunc);
      if (seqsets[i])
        totalseqs += seqsets[i]->count;
    }
  }
  /* Assemble the sequences from all the sequence sets */
  if (totalseqs > 0)
    result = tseqsetarr_to_tseqset(seqsets, ss->count, totalseqs);
  pfree_array((void **) seqsets, ss->count);
  return result;
}

/**
 * @ingroup meos_internal_temporal_restrict
 * @brief Return a temporal point restricted to (the complement of) a geometry
 * and possibly a Z span and a timestamptz span
 * @param[in] temp Temporal point
 * @param[in] gs Geometry
 * @param[in] zspan Span of values to restrict the Z dimension
 * @param[in] period Period to restrict the T dimension
 * @param[in] atfunc True if the restriction is at, false for minus
 */
Temporal *
tpoint_restrict_geom_time(const Temporal *temp, const GSERIALIZED *gs,
  const Span *zspan, const Span *period, bool atfunc)
{
  assert(temp); assert(gs); assert(tgeo_type(temp->temptype));
  if (gserialized_is_empty(gs))
    return atfunc ? NULL : temporal_cp(temp);
  /* Ensure validity of the arguments */
  ensure_same_srid(tpoint_srid(temp), gserialized_get_srid(gs));
  ensure_has_not_Z_gs(gs);
  if (zspan)
    ensure_has_Z(temp->flags);

  /* Bounding box test */
  STBox box1, box2;
  tspatial_set_stbox(temp, &box1);
  /* Non-empty geometries have a bounding box */
  geo_set_stbox(gs, &box2);
  if (zspan)
  {
    box2.zmin = DatumGetFloat8(zspan->lower);
    box2.zmax = DatumGetFloat8(zspan->upper);
    MEOS_FLAGS_SET_Z(box2.flags, true);
  }
  if (period)
  {
    memcpy(&box2.period, period, sizeof(Span));
    MEOS_FLAGS_SET_T(box2.flags, true);
  }
  bool overlaps = overlaps_stbox_stbox(&box1, &box2);
  if (! overlaps)
    return atfunc ? NULL : temporal_cp(temp);

  /* Restrict to atStbox prior to do atGeom to improve efficiency */
  interpType interp = MEOS_FLAGS_GET_INTERP(temp->flags);
  Temporal *temp1;
  if (interp == LINEAR && atfunc)
  {
    temp1 = tpoint_at_stbox_segm(temp, &box2, BORDER_INC);
    if (! temp1)
      /* This is not redundant with the bounding box check above */
      return atfunc ? NULL : temporal_cp(temp);
  }
  else
    temp1 = (Temporal *) temp;

  Temporal *result;
  assert(temptype_subtype(temp1->subtype));
  switch (temp1->subtype)
  {
    case TINSTANT:
      return (Temporal *) tpointinst_restrict_geom_time((TInstant *) temp1,
        gs, zspan, period, atfunc);
      break;
    case TSEQUENCE:
      result = tpointseq_restrict_geom_time((TSequence *) temp1,
        gs, zspan, period, atfunc);
      break;
    default: /* TSEQUENCESET */
      result = (Temporal *) tpointseqset_restrict_geom_time((TSequenceSet *)
          temp1, gs, zspan, period, atfunc);
  }
  if (interp == LINEAR && atfunc)
    pfree(temp1);
  return result;
}

#if MEOS
/**
 * @ingroup meos_temporal_restrict
 * @brief Return a temporal point restricted to a geometry
 * @param[in] temp Temporal point
 * @param[in] gs Geometry
 * @param[in] zspan Span of values to restrict the Z dimension
 * @param[in] period Period to restrict the T dimension
 * @csqlfn #Tpoint_at_geom_time()
 */
Temporal *
tpoint_at_geom_time(const Temporal *temp, const GSERIALIZED *gs,
  const Span *zspan, const Span *period)
{
  /* Ensure validity of the arguments */
  if (! ensure_valid_tpoint_geo(temp, gs))
    return NULL;
  return tpoint_restrict_geom_time(temp, gs, zspan, period, REST_AT);
}

/**
 * @ingroup meos_temporal_restrict
 * @brief Return a temporal point restricted to (the complement of) a geometry
 * @param[in] temp Temporal point
 * @param[in] gs Geometry
 * @param[in] zspan Span of values to restrict the Z dimension
 * @param[in] period Period to restrict the T dimension
 * @csqlfn #Tpoint_minus_geom_time()
 */
Temporal *
tpoint_minus_geom_time(const Temporal *temp, const GSERIALIZED *gs,
  const Span *zspan, const Span *period)
{
  /* Ensure validity of the arguments */
  if (! ensure_valid_tpoint_geo(temp, gs))
    return NULL;
  return tpoint_restrict_geom_time(temp, gs, zspan, period, REST_MINUS);
}
#endif /* MEOS */

/*****************************************************************************/
