///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include "ChannelColumnMappingEditor.h"

namespace AtomViz {

/******************************************************************************
 * Initializes the widget.
 *****************************************************************************/
ChannelColumnMappingEditor::ChannelColumnMappingEditor(QWidget* parent)
	: QWidget(parent)
{
	// Create the table sub-widget.
	QVBoxLayout* layout = new QVBoxLayout(this);
	layout->setSpacing(0);
	layout->setContentsMargins(0,0,0,0);
	tableWidget = new QTableWidget(this);
	layout->addWidget(tableWidget);
	
	tableWidget->setColumnCount(2);
	QStringList horizontalHeaders;
	horizontalHeaders << tr("Data Channel");
	horizontalHeaders << tr("Component");
	tableWidget->setHorizontalHeaderLabels(horizontalHeaders);
	tableWidget->resizeColumnToContents(1);
	tableWidget->setEditTriggers(QAbstractItemView::AllEditTriggers);

	// Calculate the optimum with of the 1. column.
	QComboBox* box = new QComboBox();
	box->setSizeAdjustPolicy(QComboBox::AdjustToContents);
	QMapIterator<QString, DataChannel::DataChannelIdentifier> i(DataChannel::standardChannelList());
	while(i.hasNext()) {
		i.next();
		box->addItem(i.key(), i.value());
	}
	tableWidget->setColumnWidth(0, box->sizeHint().width());

	nameItemDelegate.owner = this;
	vectorComponentItemDelegate.owner = this;	
	tableWidget->setItemDelegateForColumn(0, &nameItemDelegate);
	tableWidget->setItemDelegateForColumn(1, &vectorComponentItemDelegate);
	
	QHBoxLayout* layout2 = new QHBoxLayout();
	layout->addSpacing(6);
	layout->addLayout(layout2);
	
	QPushButton* presetMenuButton = new QPushButton(tr("Preset Menu"), this);
	presetMenuButton->setMenu(&presetMenu);
	connect(&presetMenu, SIGNAL(aboutToShow()), this, SLOT(updatePresetMenu()));
	layout2->addWidget(presetMenuButton);
	layout2->addStretch(1);

	QPushButton* useAllChannelsButton = new QPushButton(tr("Output All Channels"), this);
	connect(useAllChannelsButton, SIGNAL(clicked(bool)), this, SLOT(onOutputAllChannels()));
	layout2->addWidget(useAllChannelsButton);
}

/******************************************************************************
 * Fills the editor with the given mapping.
 *****************************************************************************/
void ChannelColumnMappingEditor::setMapping(const ChannelColumnMapping& mapping, AtomsObject* atomsObj)
{
	CHECK_OBJECT_POINTER(atomsObj);
	atoms = atomsObj;
	
	// Reset table contents.
	tableWidget->clearContents();
	QStringList verticalHeaders;

	// Fill in the table cells.
	bool warningMessageShown = false;
	int row = 0;
	for(int i=0; i<mapping.columnCount(); i++) {

		DataChannel::DataChannelIdentifier channelId = mapping.getChannelId(i);
		QString channelName = mapping.getChannelName(i);
		DataChannel* channel;
		if(channelId != DataChannel::UserDataChannel) {
			channel = atomsObj->getStandardDataChannel(channelId);		
			channelName = DataChannel::standardChannelName(channelId);				
		}		
		else {
			channel = atomsObj->findDataChannelByName(channelName);
		}
		
		if(channel == NULL && channelId != DataChannel::AtomIndexChannel) {
			if(!warningMessageShown) {
				QMessageBox::warning(this, tr("Data Channel Mapping"), 
					tr("The current mapping between data channels and file columns is not valid. It contains "
					   "at least one reference to a data channel that does not exist in the atomic dataset. "
					   "All columns containing such a dangling reference will be removed."));					   
				warningMessageShown = true;
			}
			continue;
		}
		tableWidget->setRowCount(row + 1);
				
		QTableWidgetItem* nameItem = new QTableWidgetItem(channelName);
		nameItem->setData(Qt::UserRole, channelId);
		nameItem->setFlags(Qt::ItemIsSelectable | Qt::ItemIsEditable | Qt::ItemIsEnabled);
		tableWidget->setItem(row, 0, nameItem);

		QString componentName;
		if(channel && channel->componentNames().size() > (int)mapping.getVectorComponent(i))
			componentName = channel->componentNames()[mapping.getVectorComponent(i)];
		QTableWidgetItem* vectorComponentItem = new QTableWidgetItem(componentName);
		vectorComponentItem->setData(Qt::UserRole, (int)mapping.getVectorComponent(i));
		vectorComponentItem->setFlags(Qt::ItemIsSelectable | Qt::ItemIsEditable | Qt::ItemIsEnabled);
		tableWidget->setItem(row, 1, vectorComponentItem);

		row++;
	}
	ensureEmptyRowAtEnd();
}

/******************************************************************************
 * Makes sure that the table contains exactly one empty row at the end.
 *****************************************************************************/
void ChannelColumnMappingEditor::ensureEmptyRowAtEnd()
{
	int lastNonEmptyRow = tableWidget->rowCount() - 1;
	for(; lastNonEmptyRow >= 0; --lastNonEmptyRow) {
		if(tableWidget->item(lastNonEmptyRow, 0)->text().isEmpty() == false ||
			tableWidget->item(lastNonEmptyRow, 1)->text().isEmpty() == false)
			break;		
	} 
	
	if(lastNonEmptyRow + 2 != tableWidget->rowCount()) {
		tableWidget->setRowCount(lastNonEmptyRow + 2);

		QTableWidgetItem* nameItem = new QTableWidgetItem("");
		nameItem->setFlags(Qt::ItemIsSelectable | Qt::ItemIsEditable | Qt::ItemIsEnabled);
		tableWidget->setItem(lastNonEmptyRow+1, 0, nameItem);

		QTableWidgetItem* vectorComponentItem = new QTableWidgetItem("");
		vectorComponentItem->setData(Qt::UserRole, 0);
		vectorComponentItem->setFlags(Qt::ItemIsSelectable | Qt::ItemIsEditable | Qt::ItemIsEnabled);
		tableWidget->setItem(lastNonEmptyRow+1, 1, vectorComponentItem);

		updateHeaderLabels();
	}
}

/******************************************************************************
 * Updates the labels of the vertical header.
 *****************************************************************************/
void ChannelColumnMappingEditor::updateHeaderLabels()
{
	QStringList verticalHeaders;
	for(int row = 1; row <= tableWidget->rowCount(); ++row)
		verticalHeaders << tr("Col. %1").arg(row);
	tableWidget->setVerticalHeaderLabels(verticalHeaders);
	tableWidget->resizeRowsToContents();	
}

/******************************************************************************
 * Returns the current contents of the editor.
 * Throws an Exception if the current mapping in the editor is not valid.
 *****************************************************************************/
ChannelColumnMapping ChannelColumnMappingEditor::mapping() const
{
	OVITO_ASSERT(atoms.get());
	ChannelColumnMapping mapping;
		
	for(int row = 0; row < tableWidget->rowCount(); row++) {		
		DataChannel::DataChannelIdentifier channelId = (DataChannel::DataChannelIdentifier)tableWidget->item(row, 0)->data(Qt::UserRole).toInt();
		QString channelName = tableWidget->item(row, 0)->data(Qt::DisplayRole).toString().trimmed();
		int vectorComponent = tableWidget->item(row, 1)->data(Qt::UserRole).toInt();
		
		if(channelName.isEmpty()) continue;

		OVITO_ASSERT(channelId == DataChannel::UserDataChannel || channelName == DataChannel::standardChannelName(channelId));

		DataChannel* channel;
		if(channelId != DataChannel::UserDataChannel)
			channel = atoms->getStandardDataChannel(channelId);
		else
			channel = atoms->findDataChannelByName(channelName);
		
		if(channel == NULL && channelId != DataChannel::AtomIndexChannel) continue;
		
		if(channel && (int)channel->componentCount() <= vectorComponent)
			throw Exception(tr("The vector component specified for column %1 is out of range. The data channel '%2' contains only %3 values per atom.").arg(row+1).arg(channelName).arg(channel->componentCount()));
		mapping.insertColumn(mapping.columnCount(), channelId, channelName, vectorComponent);
	}
		
	return mapping;	
}

/******************************************************************************
 * Updates the entries in the "Load Preset" menu.
 *****************************************************************************/
void ChannelColumnMappingEditor::updatePresetMenu()
{
	presetMenu.clear();

	QMenu* loadMenu = presetMenu.addMenu(QIcon(":/atomviz/mapping_preset_load.png"), tr("&Load Mapping"));
	Q_FOREACH(QString name, ChannelColumnMapping::listPresets()) {
		loadMenu->addAction(name, this, SLOT(onLoadPreset()));
	}
	loadMenu->setEnabled(!loadMenu->isEmpty());

	QMenu* saveMenu = presetMenu.addMenu(QIcon(":/atomviz/mapping_preset_save.png"), tr("&Save Mapping"));
	saveMenu->addAction(tr("&New Preset..."), this, SLOT(onSavePresetAs()));
	saveMenu->addSeparator();
	Q_FOREACH(QString name, ChannelColumnMapping::listPresets()) {
		saveMenu->addAction(name, this, SLOT(onSavePreset()));
	}

	QMenu* deleteMenu = presetMenu.addMenu(QIcon(":/atomviz/mapping_preset_delete.png"), tr("&Delete Mapping"));
	Q_FOREACH(QString name, ChannelColumnMapping::listPresets()) {
		deleteMenu->addAction(name, this, SLOT(onDeletePreset()));
	}
	deleteMenu->setEnabled(!deleteMenu->isEmpty());
}

/******************************************************************************
 * Loads a preset.
 *****************************************************************************/
void ChannelColumnMappingEditor::onLoadPreset()
{
	QAction* action = (QAction*)sender();
	CHECK_POINTER(action);

	try {
		QString name = action->text();
		ChannelColumnMapping m;
		m.loadPreset(name);
	
		setMapping(m, atoms.get());
	}
	catch(const Exception& ex) {
		ex.showError();
	}
}

/******************************************************************************
 * Saves a preset under a new name.
 *****************************************************************************/
void ChannelColumnMappingEditor::onSavePresetAs()
{
	try {
		ChannelColumnMapping m = mapping();
		
		QString name = QInputDialog::getText(this, tr("Save Mapping"), 
			tr("Please enter a name for the column mapping:"));
		if(name.isEmpty()) return;
		
		if(ChannelColumnMapping::listPresets().contains(name)) {
			if(QMessageBox::question(this, tr("Save Mapping"), 
					tr("There already a stored mapping with the same name. Do you want to overwrite it?"), 
					QMessageBox::Yes|QMessageBox::No, QMessageBox::Yes) != QMessageBox::Yes)
				return;
		}			
		
		m.savePreset(name);
	}
	catch(const Exception& ex) {
		ex.showError();
	}
}

/******************************************************************************
 * Saves a preset under an existing name.
 *****************************************************************************/
void ChannelColumnMappingEditor::onSavePreset()
{
	QAction* action = (QAction*)sender();
	CHECK_POINTER(action);

	try {
		ChannelColumnMapping m = mapping();
		
		QString name = action->text();
		if(name.isEmpty()) return;

		if(ChannelColumnMapping::listPresets().contains(name)) {
			if(QMessageBox::question(this, tr("Save Mapping"), 
					tr("Do you want to overwrite the existing mapping '%1'?").arg(name), 
					QMessageBox::Yes|QMessageBox::No, QMessageBox::Yes) != QMessageBox::Yes)
				return;
		}
		
		m.savePreset(name);
	}
	catch(const Exception& ex) {
		ex.showError();
	}
}

/******************************************************************************
 * Deletes a preset.
 *****************************************************************************/
void ChannelColumnMappingEditor::onDeletePreset()
{
	QAction* action = (QAction*)sender();
	CHECK_POINTER(action);

	try {
		QString name = action->text();

		if(QMessageBox::question(this, tr("Delete Mapping"), 
				tr("Do you really want to delete the column mapping '%1'?").arg(name), 
				QMessageBox::Yes|QMessageBox::No, QMessageBox::Yes) != QMessageBox::Yes)
			return;

		ChannelColumnMapping::deletePreset(name);
	}
	catch(const Exception& ex) {
		ex.showError();
	}
}

/******************************************************************************
 * Is called when the user pressed the "Output All Channels" button.
 *****************************************************************************/
void ChannelColumnMappingEditor::onOutputAllChannels()
{
	// Create a data channel -> column mapping that contains all data channels.
	ChannelColumnMapping m;
	Q_FOREACH(DataChannel* channel, atoms->dataChannels()) {
		if(channel->type() != QMetaType::Void) {
			for(size_t k=0; k<channel->componentCount(); k++) {
				m.insertColumn(m.columnCount(), channel->id(), channel->name(), k);
			}
		}
	}
	setMapping(m, atoms.get());
}


};	// End of namespace AtomViz
