<?xml version="1.0"?>
<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                xmlns:exslt="http://exslt.org/common" 
                xmlns:func="http://exslt.org/functions"
                xmlns:merge="http://wohlberg.net/xml/merge"
                xmlns:debug="http://wohlberg.net/xml/debug"
                xmlns:dflts="http://wohlberg.net/xml/defaults"
                extension-element-prefixes="exslt func"
                version="1.0">


<!--
     This file contains templates expanding the child elements of
     defaults elements out into explicit occurences of those
     elements. See the notes at the end of this file for further
     details.

     Copyright © 2005-2010 Brendt Wohlberg <photoml@wohlberg.net>

     This is free software; you can redistribute it and/or modify it 
     under the terms of version 2 of the GNU General Public License 
     at http://www.gnu.org/licenses/gpl-2.0.txt.

     This software is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the 
     GNU General Public License for more details.
 -->



  <!-- Include subtree merge functions and templates -->
  <xsl:include href="merge.xsl"/>
 


  <!-- Stylesheet parameter specifying the set of nodes upon which
       defaults expansion is desired. -->
  <xsl:param name="relevant-elements" select="/.."/>

  <!-- Stylesheet parameter controlling handling of defaults
       elements. Valid values are "remove" for removal of the defaults
       element after expansion, "comment" to comment it out, and
       "indicate" to insert a comment marking its position.  -->
  <xsl:param name="replacement-policy" select="'remove'"/>

  <!-- Stylesheet parameter controlling debug trace messages -->
  <xsl:param name="debug-dflts">0</xsl:param>



<!-- *****************************************************************
     Template matching the document root, which is passed to the 
     expand-defaults template in the $nodes parameter.
     ***************************************************************** -->
  <xsl:template match="/">
    <xsl:copy-of select="/node()[not(self::* or preceding-sibling::*)]"/>
    <xsl:call-template name="expand-defaults">
      <xsl:with-param name="nodes" select="/*"/>
    </xsl:call-template>
    <xsl:copy-of select="/node()[not(self::* or following-sibling::*)]"/>
  </xsl:template>



<!-- *****************************************************************
     Named template calling extension function dflts:expand-defaults. 
     ***************************************************************** -->
  <xsl:template name="expand-defaults">
    <xsl:param name="nodes" select="/.."/>
    <xsl:param name="dflts" select="/.."/>

    <xsl:copy-of select="dflts:expand-defaults($nodes, $dflts)"/>
  </xsl:template>



<!-- *****************************************************************
     Expand child elements of the defaults element into subsequent
     occurences of elements of the same name.
     ***************************************************************** -->
  <func:function name="dflts:expand-defaults">
    <xsl:param name="nodes" select="/.."/>
    <xsl:param name="dflts" select="/.."/>

    <!-- ******** BEGIN DEBUGGING TRACE ******** -->
    <xsl:if test="$debug-dflts&gt;0">
      <xsl:message>
        <xsl:text>---dflts-&gt; </xsl:text>
        <xsl:text>Called dflts:expand-defaults:&#10;</xsl:text>
        <xsl:value-of select="debug:debug-node-set($nodes,'nodes:',
                              $debug-dflts&gt;2)"/>
        <xsl:value-of select="debug:debug-node-set($dflts,'defaults:',
                              $debug-dflts&gt;2)"/>
        <xsl:text>&lt;--dflts--&#10;</xsl:text>
      </xsl:message>
    </xsl:if>
    <!-- ********* END DEBUGGING TRACE ********* -->

    <xsl:variable name="result">
      <!-- Construct enclosing element (with attributes) and call
           dflts:defaults-scope-recursion to handle all children 
           at this level. -->
      <xsl:element name="{name($nodes)}">
        <xsl:copy-of select="$nodes/@*"/>
        <xsl:copy-of select="dflts:defaults-scope-recursion(
                               $nodes/node(), 0, $dflts
                             )"/>
      </xsl:element>
    </xsl:variable>

    <func:result select="exslt:node-set($result)/node()"/>
  </func:function>



<!-- *****************************************************************
     Recursive part of implementation of defaults expansion, handling
     all children at a particular level, moving from one set of nodes
     constituting a new defaults scope to the next by recursive call
     to the same function, and down each child subtree by recursive
     calls back to dflts:expand-defaults.
     ***************************************************************** -->
  <func:function name="dflts:defaults-scope-recursion">
    <xsl:param name="nodes" select="/.."/>
    <xsl:param name="dposn" select="0"/>
    <xsl:param name="dflts" select="/.."/>

    <!-- ******** BEGIN DEBUGGING TRACE ******** -->
    <xsl:if test="$debug-dflts&gt;0">
      <xsl:message>
        <xsl:text>---dflts-&gt; </xsl:text>
        <xsl:text>Called dflts:defaults-scope-recursion:&#10;</xsl:text>
        <xsl:value-of select="debug:debug-node-set($nodes,'nodes:',
                              $debug-dflts&gt;2)"/>
        <xsl:text>dposn: </xsl:text>
        <xsl:value-of select="$dposn"/>
        <xsl:text>&#10;</xsl:text>
        <xsl:value-of select="debug:debug-node-set($dflts,'defaults:',
                              $debug-dflts&gt;2)"/>
        <xsl:text>&lt;--dflts--&#10;</xsl:text>
      </xsl:message>
    </xsl:if>
    <!-- ********* END DEBUGGING TRACE ********* -->


    <xsl:variable name="result">
      <!-- Iterate over all nodes preceding currently selected (via
           $dposn) defaults element. -->
      <xsl:for-each select="$nodes/self::node()[
                            not(self::defaults) and
                            count(preceding-sibling::defaults) = $dposn
                            ]">
        <xsl:choose>
          <!-- #### Begin branch: A #### -->
          <xsl:when test="not(self::*)">
            <!-- Simply emit a copy of non-element nodes. -->
            <!-- ******** BEGIN DEBUGGING TRACE ******** -->
            <xsl:if test="$debug-dflts&gt;0">
              <xsl:message>---dflts-- ### Entered branch A</xsl:message>
            </xsl:if>
          <!-- ********* END DEBUGGING TRACE ********* -->
            <xsl:copy-of select="."/>
          </xsl:when>
          <!-- #### End branch: A #### -->
          <!-- #### Begin branch: B #### -->
          <xsl:when test="$dflts[self::*][1]/*[name(.) = name(current())]">
            <!-- The $dflts subtree has a child with the same name as the 
                 current element, indicating that expansion of the relevant 
                 defaults element into that element is required. Merge the 
                 relevant defaults child with the current element and emit 
                 the result. -->
            <!-- ******** BEGIN DEBUGGING TRACE ******** -->
            <xsl:if test="$debug-dflts&gt;0">
              <xsl:message>---dflts-- ### Entered branch B</xsl:message>
            </xsl:if>
            <!-- ********* END DEBUGGING TRACE ********* -->
            <xsl:copy-of select="merge:merge-subtrees(.,
                           $dflts[self::*][1]/*[name(.) = name(current())])"/>
          </xsl:when>
          <!-- #### End branch: B #### -->
          <!-- #### Begin branch: C #### -->
          <xsl:when test="$relevant-elements and not(
                            $relevant-elements/descendant-or-self::*[name(.) =
                           name(current())])">
            <!-- When the $relevant-elements stylesheet parameter is not
                 an empty node set, it is used to specify which
                 elements are subject to defaults expansion. If the
                 current element does not have a correspondingly named
                 element in $relevant-element, it is not subject to
                 defaults expansion. In this case, no further
                 processing is required for this element, and a copy
                 is simply emitted. -->
            <!-- ******** BEGIN DEBUGGING TRACE ******** -->
            <xsl:if test="$debug-dflts&gt;0">
              <xsl:message>---dflts-- ### Entered branch C</xsl:message>
            </xsl:if>
            <!-- ********* END DEBUGGING TRACE ********* -->
            <xsl:copy-of select="."/>
          </xsl:when>
          <!-- #### End branch: C #### -->
          <!-- #### Begin branch: D #### -->
          <xsl:when test="$relevant-elements and ($relevant-elements/
                            descendant-or-self::*[name(.) =
                            name(current())]/*[@insert = 'true'])">
            <!--  When the $relevant-elements stylesheet parameter is
                  not an empty node set, it is used to specify which
                  elements are subject to defaults expansion. Some
                  elements (indicated in $relevant-elements by the
                  present of an attribute 'insert' with value 'true')
                  require special handling during defaults expansion.
                  The standard defaults expansion approach is to 
                  expand defaults into any element that has the same
                  name as a direct defaults child element. Without
                  special handling therefore, a direct defaults child
                  element is never expanded into the main node set if
                  the node set does not contain a node with the same
                  name. The special handling performed here inserts any
                  direct defaults child element that is a valid child of
                  the first element in $nodes if it is tagged for such
                  insertion in $relevant-elements.  -->
            <!-- ******** BEGIN DEBUGGING TRACE ******** -->
            <xsl:if test="$debug-dflts&gt;0">
              <xsl:message>---dflts-- ### Entered branch D</xsl:message>
            </xsl:if>
            <!-- ********* END DEBUGGING TRACE ********* -->
            <!-- Nodes from $relevant-elements that have a parent with
                 the same name as the current element, and have
                 attribute 'insert' with a value of 'true'. -->
            <xsl:variable name="insert-nodes"
                        select="$relevant-elements/
                                descendant-or-self::*[name(.) =
                                name(current())]/*[@insert = 'true']"/>
            <!-- Call function dflts:insert-defaults for special
                 handling of the result of a recursive call of
                 dflts:expand-defaults on the current element. -->
            <xsl:copy-of select="dflts:insert-defaults(
                                   dflts:expand-defaults(., $dflts),
                                   $dflts, $insert-nodes)"/>
          </xsl:when>
          <!-- #### End branch: D #### -->
          <!-- #### Begin branch: E #### -->
          <xsl:otherwise>
            <!-- No special cases matched, so the only action required
                 is recursion down the current element tree, entailing a
                 recursive call to dflts:expand-default. -->
            <!-- ******** BEGIN DEBUGGING TRACE ******** -->
            <xsl:if test="$debug-dflts&gt;0">
              <xsl:message>---dflts-- ### Entered branch E</xsl:message>
            </xsl:if>
          <!-- ********* END DEBUGGING TRACE ********* -->
            <xsl:copy-of select="dflts:expand-defaults(., $dflts)"/>
          </xsl:otherwise>
          <!-- #### End branch: E #### -->
        </xsl:choose>
      </xsl:for-each>

      <!-- If there is a defaults element following the set of
           elements processed above, call dflts:replace-defaults to
           remove or replace the defaults element with a comment, and
           then do a recursive call to dflts:defaults-scope-recursion
           with $dposn incremented to indicate the nodes follwing this
           defaults element, and the current defaults modified via a
           call to dflts:update-defaults. (This recursive call
           strategy is necessary because XSLT variables cannot be
           modified, preventing the simplest approach of just
           iterating over all nodes, updating the $dflts variable when
           a defaults element is encountered. -->
      <xsl:if test="$dposn &lt; count($nodes/self::defaults)">
        <!-- Remove/replace next defaults element. -->
        <xsl:copy-of select="dflts:replace-defaults(
                               $nodes[self::defaults][$dposn+1]
                             )"/>
        <!-- Recursive call to dflts:defaults-scope-recursion for
             handling set of nodes following next defaults element. -->
        <xsl:copy-of select="dflts:defaults-scope-recursion(
                             $nodes, $dposn+1, 
                               dflts:update-defaults($dflts,
                                 $nodes[self::defaults][$dposn+1]
                               )
                             )"/>
      </xsl:if>
    </xsl:variable>

    <func:result select="exslt:node-set($result)/node()"/>
  </func:function>



<!-- *****************************************************************
     Construct the replacement for a defaults element once it has been
     removed from a subtree. The replacement depends on the stylesheet
     parameter 'replacement-policy': when the value is 'indicate', a
     comment is inserted indicating the position from which the
     defaults element was removed; when the value is 'comment', a
     comment containing a string representation of the defaults
     element is inserted at the same position; and when the value is
     'remove' (or anything else), the replacement is null.
     ***************************************************************** -->
  <func:function name="dflts:replace-defaults">
    <xsl:param name="nodes" select="/.."/>

    <!-- ******** BEGIN DEBUGGING TRACE ******** -->
    <xsl:if test="$debug-dflts&gt;1">
      <xsl:message>
        <xsl:text>---dflts-&gt; </xsl:text>
        <xsl:text>Called dflts:replace-defaults:&#10;</xsl:text>
        <xsl:value-of select="debug:debug-node-set($nodes,'nodes:',
                              $debug-dflts&gt;3)"/>
        <xsl:text>&lt;--dflts--&#10;</xsl:text>
      </xsl:message>
    </xsl:if>
    <!-- ********* END DEBUGGING TRACE ********* -->

    <xsl:variable name="result">
      <xsl:choose>
        <xsl:when test="$replacement-policy = 'indicate'">
          <xsl:comment>
            <xsl:text> A defaults element was removed from here </xsl:text>
            <xsl:text>during expansion </xsl:text>
          </xsl:comment>
        </xsl:when>
        <xsl:when test="$replacement-policy = 'comment'">
          <xsl:comment>
            <xsl:text> </xsl:text>
            <xsl:apply-templates select="$nodes[*]|$nodes[text()]|
                                         $nodes[comment()]|
                                         $nodes[processing-instruction()]"
                                 mode="node-set-as-string"/>
            <xsl:text> </xsl:text>
          </xsl:comment>
        </xsl:when>
        <xsl:otherwise>
          <!-- No result in this case. The xsl:otherwise is just a
               reminder of the third option, and fulfills no direct
               function. -->
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>
    
    <func:result select="exslt:node-set($result)/node()"/>
  </func:function>



<!-- *****************************************************************
     Update the current defaults subtree using a defaults element from
     the document. If the document defaults element has a value of
     'replace' for attribute 'combine', the current defaults subtree
     is deleted and replaced with the document defaults element, while
     if the 'combine' attribute has value 'merge' (or anything else),
     the document defaults element subtree is merged into the current
     defaults subtree.
     ***************************************************************** -->
  <func:function name="dflts:update-defaults">
    <xsl:param name="dflts" select="/.."/>
    <xsl:param name="nodes" select="/.."/>

   <!-- ******** BEGIN DEBUGGING TRACE ******** -->
    <xsl:if test="$debug-dflts&gt;1">
      <xsl:message>
        <xsl:text>---dflts-&gt; </xsl:text>
        <xsl:text>Called dflts:update-defaults:&#10;</xsl:text>
        <xsl:value-of select="debug:debug-node-set($dflts,'dflts:',
                              $debug-dflts&gt;3)"/>
        <xsl:value-of select="debug:debug-node-set($nodes,'nodes:',
                              $debug-dflts&gt;3)"/>
        <xsl:text>&lt;--dflts--&#10;</xsl:text>
      </xsl:message>
    </xsl:if>
    <!-- ********* END DEBUGGING TRACE ********* -->

    <xsl:variable name="result">
      <xsl:choose>
        <xsl:when test="$nodes[1]/@combine='replace'">
          <xsl:copy-of select="merge:prune-trailing-children($nodes)"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:copy-of select="merge:merge-subtrees(
                                 $nodes,
                                 $dflts)"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>

    <!-- ******** BEGIN DEBUGGING TRACE ******** -->
    <xsl:if test="$debug-dflts&gt;2">
      <xsl:message>
        <xsl:text>---dflts-&gt; </xsl:text>
        <xsl:text>Result of dflts:update-defaults:&#10;</xsl:text>
        <xsl:value-of select="debug:debug-node-set(
                        exslt:node-set($result)/node(),'result:',
                              $debug-dflts&gt;3)"/>
        <xsl:text>&lt;--dflts--&#10;</xsl:text>
      </xsl:message>
    </xsl:if>
    <!-- ********* END DEBUGGING TRACE ********* -->

    <func:result select="exslt:node-set($result)/node()"/>
  </func:function>



<!-- *****************************************************************
     Merge the relevant child of the current defaults subtree into the
     first element in the 'nodes' variable.
     ***************************************************************** -->
   <func:function name="dflts:combine-defaults">
    <xsl:param name="nodes" select="/.."/>
    <xsl:param name="dflts" select="/.."/>

   <!-- ******** BEGIN DEBUGGING TRACE ******** -->
    <xsl:if test="$debug-dflts&gt;1">
      <xsl:message>
        <xsl:text>---dflts-&gt; </xsl:text>
        <xsl:text>Called dflts:combine-defaults:&#10;</xsl:text>
        <xsl:value-of select="debug:debug-node-set($nodes,'nodes:',
                              $debug-dflts&gt;3)"/>
        <xsl:value-of select="debug:debug-node-set($dflts,'dflts:',
                              $debug-dflts&gt;3)"/>
        <xsl:text>&lt;--dflts--&#10;</xsl:text>
      </xsl:message>
    </xsl:if>
    <!-- ********* END DEBUGGING TRACE ********* -->

    <xsl:variable name="result">
      <xsl:copy-of select="merge:merge-subtrees(
                             $nodes[self::*][1],
                             $dflts[self::*][1]/*[name(.)=
                                              name($nodes[self::*][1])]
                           )"/>
    </xsl:variable>
    <!-- ******** BEGIN DEBUGGING TRACE ******** -->
    <xsl:if test="$debug-dflts&gt;2">
      <xsl:message>
        <xsl:text>---dflts-&gt; </xsl:text>
        <xsl:text>Result of dflts:combine-defaults:&#10;</xsl:text>
        <xsl:value-of select="debug:debug-node-set(
                                exslt:node-set($result)/node(),'result:',
                              $debug-dflts&gt;3)"/>
        <xsl:text>&lt;--dflts--&#10;</xsl:text>
      </xsl:message>
    </xsl:if>
    <!-- ********* END DEBUGGING TRACE ********* -->

    <func:result select="exslt:node-set($result)/node()"/>
  </func:function>



<!-- *****************************************************************
     Insert children of the current defaults subtree into node sets in
     which the usual defaults merge procedure fails.
     ***************************************************************** -->
   <func:function name="dflts:insert-defaults">
    <xsl:param name="nodes" select="/.."/>
    <xsl:param name="dflts" select="/.."/>
    <xsl:param name="rlvnt" select="/.."/>

   <!-- ******** BEGIN DEBUGGING TRACE ******** -->
    <xsl:if test="$debug-dflts&gt;1">
      <xsl:message>
        <xsl:text>---dflts-&gt; </xsl:text>
        <xsl:text>Called dflts:insert-defaults:&#10;</xsl:text>
        <xsl:value-of select="debug:debug-node-set($nodes,'nodes:',
                              $debug-dflts&gt;3)"/>
        <xsl:value-of select="debug:debug-node-set($dflts,'dflts:',
                              $debug-dflts&gt;3)"/>
        <xsl:value-of select="debug:debug-node-set($rlvnt,'rlvnt:',
                              $debug-dflts&gt;3)"/>
        <xsl:text>&lt;--dflts--&#10;</xsl:text>
      </xsl:message>
    </xsl:if>
    <!-- ********* END DEBUGGING TRACE ********* -->

    <xsl:variable name="result">
      <xsl:choose>
        <xsl:when test="count($rlvnt[self::*])=0">
          <xsl:copy-of select="$nodes"/>
        </xsl:when>
        <xsl:otherwise>
          <!-- The name of the current insertion node. -->
          <xsl:variable name="iname" select="name($rlvnt[self::*][1])"/>
          <!-- The current insertion node from $dflts. -->
          <xsl:variable name="inode"
                        select="$dflts[self::*][1]/*[name(.)=$iname]
                                  [not($nodes[self::*][1]/*[name(.)=$iname])] |
                                $dflts[self::*][1]/*[name(.)=$iname]
                                  /following-sibling::text()"/>
          <!-- Non-element nodes preceding the current insertion node. -->
          <xsl:variable name="pnode"
                        select="$inode[self::*][1]/
                                preceding-sibling::node()[not(self::*)]"/>

          <!-- ******** BEGIN DEBUGGING TRACE ******** -->
          <xsl:if test="$debug-dflts&gt;2">
            <xsl:message>
              <xsl:text>---dflts-&gt; </xsl:text>
              <xsl:text>Variables in dflts:insert-defaults:&#10;</xsl:text>
              <xsl:value-of select="concat('iname: ',$iname,'&#10;')"/>
              <xsl:value-of select="debug:debug-node-set($inode,'inode:',
                                    $debug-dflts&gt;3)"/>
              <xsl:value-of select="debug:debug-node-set($pnode,'pnode:',
                                    $debug-dflts&gt;3)"/>
              <xsl:text>&lt;--dflts--&#10;</xsl:text>
            </xsl:message>
          </xsl:if>
          <!-- ********* END DEBUGGING TRACE ********* -->

          <!-- Do a recursive call. -->
          <xsl:copy-of select="dflts:insert-defaults(
                                 merge:merge-subtrees(
                                   $nodes,
                                   dflts:construct-subtree(
                                     $nodes[self::*][1],
                                     $pnode | $inode
                                   )
                                 ),
                                 $dflts,
                                 dflts:trailing-elements($rlvnt)
                               )"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>

    <!-- ******** BEGIN DEBUGGING TRACE ******** -->
    <xsl:if test="$debug-dflts&gt;2">
      <xsl:message>
        <xsl:text>---dflts-&gt; </xsl:text>
        <xsl:text>Result of dflts:insert-defaults:&#10;</xsl:text>
        <xsl:value-of select="debug:debug-node-set(
                                exslt:node-set($result)/node(),'result:',
                              $debug-dflts&gt;3)"/>
        <xsl:text>&lt;--dflts--&#10;</xsl:text>
      </xsl:message>
    </xsl:if>
    <!-- ********* END DEBUGGING TRACE ********* -->

    <func:result select="exslt:node-set($result)/node()"/>
  </func:function>



<!-- *****************************************************************
     Return all nodes in a node set following the first element node .
     ***************************************************************** -->
  <func:function name="dflts:trailing-elements">
    <xsl:param name="nodes" select="/.."/>
 
    <!-- ******** BEGIN DEBUGGING TRACE ******** -->
    <xsl:if test="$debug-dflts&gt;1">
      <xsl:message>
        <xsl:text>---dflts-&gt; </xsl:text>
        <xsl:text>Called dflts:trailing-elements:&#10;</xsl:text>
        <xsl:value-of select="debug:debug-node-set($nodes,'nodes:',
                              $debug-dflts&gt;3)"/>
        <xsl:text>&lt;--dflts--&#10;</xsl:text>
      </xsl:message>
    </xsl:if>
    <!-- ********* END DEBUGGING TRACE ********* -->

    <xsl:variable name="result">
      <xsl:copy-of select="$nodes[preceding-sibling::*]"/>
    </xsl:variable>
    
    <!-- ******** BEGIN DEBUGGING TRACE ******** -->
    <xsl:if test="$debug-dflts&gt;2">
      <xsl:message>
        <xsl:text>---dflts-&gt; </xsl:text>
        <xsl:text>Result of dflts:trailing-elements:&#10;</xsl:text>
        <xsl:value-of select="debug:debug-node-set(exslt:node-set($result)/
                                    node(),'result:',$debug-dflts&gt;3)"/>
        <xsl:text>&lt;--dflts--&#10;</xsl:text>
      </xsl:message>
    </xsl:if>
    <!-- ********* END DEBUGGING TRACE ********* -->

    <func:result select="exslt:node-set($result)/node()"/>
  </func:function>



<!-- *****************************************************************
     Construct a new subtree using the specified root and children.
     ***************************************************************** -->
  <func:function name="dflts:construct-subtree">
    <xsl:param name="root" select="/.."/>
    <xsl:param name="chld" select="/.."/>

    <!-- ******** BEGIN DEBUGGING TRACE ******** -->
    <xsl:if test="$debug-dflts&gt;1">
      <xsl:message>
        <xsl:text>---dflts-&gt; </xsl:text>
        <xsl:text>Called dflts:construct-subtree:&#10;</xsl:text>
        <xsl:value-of select="debug:debug-node-set($root,'root:',
                              $debug-dflts&gt;3)"/>
        <xsl:value-of select="debug:debug-node-set($chld,'chld:',
                              $debug-dflts&gt;3)"/>
        <xsl:text>&lt;--dflts--&#10;</xsl:text>
      </xsl:message>
    </xsl:if>
    <!-- ********* END DEBUGGING TRACE ********* -->

    <xsl:variable name="result">
      <xsl:element name="{name($root)}">
        <xsl:copy-of select="$root/@*"/>
        <xsl:copy-of select="$chld"/>
      </xsl:element>
    </xsl:variable>
    
    <func:result select="exslt:node-set($result)/node()"/>
  </func:function>



<!-- *****************************************************************
     Construct the "node set as string" representation of an element.
     ***************************************************************** -->
  <xsl:template match="*" mode="node-set-as-string">
    <xsl:text>&lt;</xsl:text>
    <xsl:value-of select="name(.)"/>
    <xsl:apply-templates select="@*" mode="node-set-as-string"/>
    <xsl:text>&gt;</xsl:text>
    <xsl:apply-templates select="*|text()|comment()|processing-instruction()"
                         mode="node-set-as-string"/>
    <xsl:text>&lt;/</xsl:text>
    <xsl:value-of select="name(.)"/>
    <xsl:text>&gt;</xsl:text>
  </xsl:template>



<!-- *****************************************************************
     Construct the "node set as string" representation of an attribute.
     ***************************************************************** -->
  <xsl:template match="@*" mode="node-set-as-string">
    <xsl:text> </xsl:text>
    <xsl:value-of select="name(.)"/>
    <xsl:text>=&quot;</xsl:text>
    <xsl:value-of select="."/>
    <xsl:text>&quot;</xsl:text>
  </xsl:template>



<!-- *****************************************************************
     Construct the "node set as string" representation of a text node.
     ***************************************************************** -->
  <xsl:template match="text()" mode="node-set-as-string">
      <xsl:copy/>
  </xsl:template>



<!-- *****************************************************************
     Construct the "node set as string" representation of a comment.
     ***************************************************************** -->
  <xsl:template match="comment()" mode="node-set-as-string">
    <xsl:text>&lt;COMMENT&gt;</xsl:text>
    <xsl:value-of select="."/>
    <xsl:text>&lt;/COMMENT&gt;</xsl:text>
  </xsl:template>



<!-- *****************************************************************
     Construct the "node set as string" representation of a processing 
     instruction.
     ***************************************************************** -->
  <xsl:template match="processing-instruction()" mode="node-set-as-string">
    <xsl:text>&lt;?</xsl:text>
    <xsl:value-of select="name(.)"/>
    <xsl:text> </xsl:text>
    <xsl:value-of select="."/>
    <xsl:text>?&gt;</xsl:text>
  </xsl:template>



 <!--
     The special-handling insertion mechanism signalled by a child of
     $relevant-elements with attribute 'insert' having the value
     'true' is complex, and its motiviation is best illustrated by use
     of an example. Assume $relevant-elements contains the subtree

       <photo>
         <roll>
           <film insert='true'/>
           <frame/>
         </roll>
       </photo>

     and dflts:expand-defaults is called with $nodes parameter
     containing

       <photo>
         <roll id="2000-01">
           <frame id="01>
           ...
           </frame>
         </roll>
       </photo>


     and $dflts parameter containing

       <defaults>
         <film>
           ...
         </film>
       </defaults>

     Without the special handling mechanism, the film defaults would
     not be expanded into the roll element because the roll element
     does not have a film child element, so the standard insertion
     mechanism which detects elements with the same name as a defaults
     child element is never activated. The special expansion mechanism
     detects these cases and ensures that the appropriate elements are
     inserted. Note that the standard expansion mechanism handles
     merging of a defaults node when the node of the same name is
     itself being processed, while the special expansion mechanism
     operates when the *parent* node is being processed.

-->

</xsl:stylesheet>
