/*
 * Copyright 2008 Klaus Triendl
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free 
 * Software Foundation, 51 Franklin Street, Fifth Floor, 
 * Boston, MA 02110-1301, USA.
*/

#include <iostream>
#include <sigc++/functors/mem_fun.h>
#include <sigc++/adaptors/bind_return.h>
#include <glibmm/main.h>
#include <glibmm/thread.h>
#include <sigx/glib_auto_dispatchable.h>
#include <sigx/glib_threadable.h>
#include <sigx/connection_wrapper.h>
#include <sigx/signal_f.h>
#include <sigx/tunnel_functor.h>
#include <sigx/signal_wrapper.h>

using namespace std;



class MyThread: public sigx::glib_threadable
{
public:
	MyThread();


	// expose the idle signal
	sigx::signal_f<Glib::SignalIdle> signal_idle;
};


class MainThread: public sigx::glib_auto_dispatchable
{
public:
	MainThread();
	void run();

protected:
	void quit();
	void on_self_ready();
	void on_mythread_idle_sync();
	void on_mythread_idle_async();
	
	
protected:
	Glib::RefPtr<Glib::MainLoop> m_loop;
	MyThread m_mythread;
	// store the thread's idle signal
	sigx::glib_signal_idle m_sigThreadIdle;
	sigx::connection_wrapper m_connMyThreadIdle;
};






using namespace std;


MainThread::MainThread(): 
	m_loop(), 
	m_mythread(), 
	m_sigThreadIdle(), 
	m_connMyThreadIdle()
{}

void MainThread::run()
{
	m_loop = Glib::MainLoop::create();

	// one-shot idle handler
	Glib::signal_idle().connect(
		sigc::bind_return(
			sigc::mem_fun(this, &MainThread::on_self_ready), 
			false
		)
	);
	
	m_loop->run();
}

void MainThread::quit()
{
	m_mythread.finish();
	m_loop->quit();
}

void MainThread::on_self_ready()
{
	m_mythread.run();
	m_sigThreadIdle = m_mythread.signal_idle();
	m_connMyThreadIdle = 
		m_sigThreadIdle.connect(
			sigx::open_sync_tunnel(
				sigc::bind_return(
					sigc::mem_fun(this, &MainThread::on_mythread_idle_sync), 
					true
				)
			)
		);
}

void MainThread::on_mythread_idle_sync()
{
	static int count = 0;

	cout << "mythread is idle" << endl;

	++count;
	if (count == 3)
	{
		m_connMyThreadIdle.disconnect();
		// create a one-shot mythread idle handler;
		m_sigThreadIdle.connect(
			// bind_return is used to satisfy the compiler, the bound result
			// can't be used with async tunnels;
			// async tunnels return a type's default value which is false for a 
			// bool and therefore we get disconnected automatically from 
			// mythread's idle signal
			sigc::bind_return(
				sigc::mem_fun(this, &MainThread::on_mythread_idle_async), 
				bool()
			)
		);
	}
}

void MainThread::on_mythread_idle_async()
{
	cout << "mythread is idle the last time" << endl;

	// one-shot idle handler to end the program
	Glib::signal_idle().connect(
		sigc::bind_return(
			sigc::mem_fun(this, &MainThread::quit), 
			false
		)
	);
}


MyThread::MyThread(): 
	signal_idle(make_idle_signal_f())
{}



int main()
{
	Glib::thread_init();

	MainThread app;
	app.run();

	cout << "press <enter>.";
	cin.get();
	return 0;
}
