/*****************************************************************************
    TRAVIS - Trajectory Analyzer and Visualizer
    http://www.travis-analyzer.de/

    Copyright (c) 2009-2014 Martin Brehm
                  2012-2014 Martin Thomas

    This file written by Martin Brehm.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*****************************************************************************/

#include "vorowrapper.h"
#include "voro++.h"
#include "tools.h"
#include "globalvar.h"
#include "maintools.h"


CVoroWrapper::CVoroWrapper()
{
	m_pContainer = NULL;

	m_oaVoroAtoms.SetName("CVoroWrapper::m_oaVoroAtoms");
	m_oaVoroMolecules.SetName("CVoroWrapper::m_oaVoroMolecules");
	m_faPOVFaceColor.SetName("CVoroWrapper::m_faPOVFaceColor");
	m_waPOVFaceColorMol.SetName("CVoroWrapper::m_waPOVFaceColorMol");
	m_waPOVFaceColorElem.SetName("CVoroWrapper::m_waPOVFaceColorElem");
	m_waPOVFaceColorAtom.SetName("CVoroWrapper::m_waPOVFaceColorAtom");
	m_waSurfCoverSM.SetName("CVoroWrapper::m_waSurfCoverSM");
	m_oaSurfCoverData.SetName("CVoroWrapper::m_oaSurfCoverData");
	m_waSurfCoverMol.SetName("CVoroWrapper::m_waSurfCoverMol");
	m_waSurfCoverElem.SetName("CVoroWrapper::m_waSurfCoverElem");
	m_waSurfCoverAtom.SetName("CVoroWrapper::m_waSurfCoverAtom");

	m_fPOVAngle = 90.0f;
	m_sPOVText[0] = 0;
	m_iPOVFrameCounter = 0;
	m_fPOVZoom = 1.0f;
	m_fPOVBoxOuterOpacity = 1.0f;
	m_vPOVBoxClipLow[0] = -1.0f;
	m_vPOVBoxClipLow[1] = -1.0f;
	m_vPOVBoxClipLow[2] = -1.0f;
	m_vPOVBoxClipHigh[0] = 1.0f;
	m_vPOVBoxClipHigh[1] = 1.0f;
	m_vPOVBoxClipHigh[2] = 1.0f;
	m_fPOVAtomCenter = 0.0f;
	m_fPOVScale = 0.95f;
	m_bPOVMolecules = false;
	m_fPOVExplode = 1.0f;
	m_fPOVClip = 1.0f;

	m_fPOVRayTrans = 0.0f;
	m_fPOVNbTrans = 0.0f;
	m_fPOVFaceTrans = 0.0f;
	m_fPOVEdgeTrans = 0.0f;
	m_fPOVNbHighlightFac = 1.3f;
	m_fPOVFaceBleach = 0.5f;
	m_vPOVFaceBleach[0] = 0.5f;
	m_vPOVFaceBleach[1] = 0.5f;
	m_vPOVFaceBleach[2] = 1.0f;

	m_bIncludeWritten = false;

	strcpy(m_sPOVText,"ABCabc");
}


CVoroWrapper::~CVoroWrapper()
{
}


void CVoroWrapper::Build(CTimeStep *ts)
{
	int ijk, q, z, z2, z3, id, i, faces, co;
	CVoroAtom *va;
	CVoroMolecule *vm;
	double /**pp,*/ ta, surf, volume, vg;
	voronoicell_neighbor c;
	vector<int> nb;
	vector<int> fo;
	vector<double> fa;
	int *nbtemp;

	try { nbtemp = new int[m_oaVoroAtoms.GetSize()]; } catch(...) { nbtemp = NULL; }
	if (nbtemp == NULL) NewException((double)m_oaVoroAtoms.GetSize()*sizeof(int),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	try { m_pContainer = new container_periodic_poly(g_fBoxX/1000.0,0,g_fBoxY/1000.0,0,0,g_fBoxZ/1000.0,m_iBlocksX,m_iBlocksY,m_iBlocksZ,g_iVoroMemory); } catch(...) { m_pContainer = NULL; }
	if (m_pContainer == NULL) NewException((double)sizeof(container_periodic_poly),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	for (z=0;z<g_iGesAtomCount;z++)
		m_pContainer->put(z,ts->m_vaCoords[z][0]/1000.0,ts->m_vaCoords[z][1]/1000.0,ts->m_vaCoords[z][2]/1000.0,0.05);

	c_loop_all_periodic vl(*m_pContainer);

	for (z=0;z<m_oaVoroMolecules.GetSize();z++)
	{
		vm = (CVoroMolecule*)m_oaVoroMolecules[z];
		vm->tempfaces = 0;
		vm->tempsurf = 0;
		vm->tempvol = 0;
		for (z2=0;z2<m_oaVoroMolecules.GetSize();z2++)
			vm->m_pNbhTempMol[z2] = 0;
		for (z2=0;z2<g_iGesAtomCount;z2++)
			vm->m_pNbhTempAtoms[z2] = 0;
	}

	for (z=0;z<g_iGesAtomCount;z++)
		m_pAtomTouched[z] = false;

	co = 0;
	vg = 0;
	if (vl.start()) 
	{
		do 
		{
			if (m_pContainer->compute_cell(c,vl))
			{
				co++;

				ijk=vl.ijk;
				q=vl.q;
	//			pp=m_pContainer->p[ijk]+m_pContainer->ps*q;

				id = m_pContainer->id[ijk][q];
				c.face_areas(fa);
				c.face_orders(fo);
				c.neighbors(nb);

				m_pAtomTouched[id] = true;

				va = (CVoroAtom*)m_oaVoroAtoms[m_pAssignAtoms[id]];
				vm = (CVoroMolecule*)m_oaVoroMolecules[m_pAssignMolecules[id]];

				surf = c.surface_area();
				faces = c.number_of_faces();
				volume = c.volume();

				vg += volume;

				va->m_pVolume->AddToBin(volume*1000.0);
				va->m_pSurfaceArea->AddToBin(surf*100.0);
				va->m_pFaces->AddToBin_Int(faces);
				va->m_pMaxRadius->AddToBin(sqrt(c.max_radius_squared())*1000.0);
				va->m_pAVRatio->AddToBin(10.6347231054330961*volume/pow(surf,1.5));

				vm->tempvol += volume*1000.0;

				for (z2=0;z2<m_oaVoroAtoms.GetSize();z2++)
				{
					nbtemp[z2] = 0;
				}

				for (z2=0;z2<m_oaVoroMolecules.GetSize();z2++)
				{
					va->m_pNbhTempMol[z2] = 0;
				}

				ta = 0;
				for (z2=0;z2<faces;z2++)
				{
					va->m_pFaceOrders->AddToBin_Int(fo[z2]);

					if (m_pAssignMolecules[id] != m_pAssignMolecules[nb[z2]])
					{
						nbtemp[m_pAssignAtoms[nb[z2]]]++;
						va->m_pNbhDistAtoms[m_pAssignAtoms[nb[z2]]]->AddToBin(ts->FoldedDistance(id,nb[z2]));

						((CVoroMolecule*)m_oaVoroMolecules[m_pAssignMolecules[nb[z2]]])->m_pNbhDistAtoms[m_pAssignAtoms[id]]->AddToBin(ts->FoldedDistance(id,((CVoroMolecule*)m_oaVoroMolecules[m_pAssignMolecules[nb[z2]]])->m_iCenterOffset));

						if (vm->m_pNbhTempAtoms[nb[z2]] == 0)
						{
							vm->m_pNbhTempAtoms[nb[z2]]++;
							vm->m_pNbhDistAtoms[m_pAssignAtoms[nb[z2]]]->AddToBin(ts->FoldedDistance(id,((CVoroMolecule*)m_oaVoroMolecules[m_pAssignMolecules[nb[z2]]])->m_iCenterOffset));
						}

						if (vm->m_pNbhTempMol[m_pAssignMolecules[nb[z2]]] == 0)
						{
							vm->m_pNbhTempMol[m_pAssignMolecules[nb[z2]]]++;
							vm->m_pNbhDistMolecules[m_pAssignMoleculeTypes[nb[z2]]]->AddToBin(ts->FoldedDistance(((CVoroMolecule*)m_oaVoroMolecules[m_pAssignMolecules[id]])->m_iCenterOffset,((CVoroMolecule*)m_oaVoroMolecules[m_pAssignMolecules[nb[z2]]])->m_iCenterOffset));
						}

						if (va->m_pNbhTempMol[m_pAssignMolecules[nb[z2]]] == 0)
							va->m_pNbhDistMolecules[m_pAssignMoleculeTypes[nb[z2]]]->AddToBin(ts->FoldedDistance(id,((CVoroMolecule*)m_oaVoroMolecules[m_pAssignMolecules[nb[z2]]])->m_iCenterOffset));

						va->m_pNbhTempMol[m_pAssignMolecules[nb[z2]]]++;
						vm->tempsurf += fa[z2]*100.0;
						vm->tempfaces++;
						ta += fa[z2]*100.0;
					}
				}

				va->m_pExposedSurface->AddToBin(ta);
				va->m_pExposedSurfacePerc->AddToBin(ta/surf);

				for (z2=0;z2<m_oaVoroAtoms.GetSize();z2++)
				{
					va->m_pNbhAtoms[z2]->AddToBin_Int(nbtemp[z2]);
				}

				for (z2=0;z2<g_oaMolecules.GetSize();z2++)
				{
					i = 0;
					for (z3=0;z3<m_oaVoroMolecules.GetSize();z3++)
					{
						if (((CVoroMolecule*)m_oaVoroMolecules[z3])->m_iMolecule != z2)
							continue;
						if (va->m_pNbhTempMol[z3] != 0)
							i++;
					}
					va->m_pNbhMolecules[z2]->AddToBin_Int(i);
				}
			} else
			{
				mprintf("\nWarning: Compute_Cell failed.");
			}
		} while (vl.inc());
	}

	for (z=0;z<g_iGesAtomCount;z++)
	{
		if (!m_pAtomTouched[z])
			mprintf("\nWarning: Atom %s(%d) %s%d not touched by voronoi!",((CMolecule*)g_oaMolecules[g_waAtomMolIndex[z]])->m_sName,g_laAtomSMIndex[z]+1,((CAtom*)g_oaAtoms[g_waAtomRealElement[z]])->m_sName,g_waAtomMolNumber[z]+1);
	}

	if ((co != g_iGesAtomCount) || (fabs(vg*1000.0-g_fBoxX*g_fBoxY*g_fBoxZ/1000000.0)/(g_fBoxX*g_fBoxY*g_fBoxZ/1000000.0) > 0.001))
	{
		mprintf("\nVoro++: Problem. %d/%d atoms touched. V=%f/%f A^3.",co,g_iGesAtomCount,vg*1000.0,g_fBoxX*g_fBoxY*g_fBoxZ/1000000.0);
	}

	for (z=0;z<m_oaVoroMolecules.GetSize();z++)
	{
		vm = (CVoroMolecule*)m_oaVoroMolecules[z];
		vm->m_pVolume->AddToBin(vm->tempvol);
		vm->m_pSurfaceArea->AddToBin(vm->tempsurf);
		vm->m_pFaces->AddToBin_Int(vm->tempfaces);
		vm->m_pAVRatio->AddToBin(10.6347231054330961*vm->tempvol/pow(vm->tempsurf,1.5));

		for (z2=0;z2<m_oaVoroAtoms.GetSize();z2++)
		{
			i = 0;
			for (z3=0;z3<g_iGesAtomCount;z3++)
			{
				if (m_pAssignAtoms[z3] != z2)
					continue;
				if (vm->m_pNbhTempAtoms[z3] != 0)
					i++;
			}
			vm->m_pNbhAtoms[z2]->AddToBin_Int(i);
		}

		for (z2=0;z2<g_oaMolecules.GetSize();z2++)
		{
			i = 0;
			for (z3=0;z3<m_oaVoroMolecules.GetSize();z3++)
			{
				if (((CVoroMolecule*)m_oaVoroMolecules[z3])->m_iMolecule != z2)
					continue;
				if (vm->m_pNbhTempMol[z3] != 0)
					i++;
			}
			vm->m_pNbhMolecules[z2]->AddToBin_Int(i);
		}
	}

	delete[] nbtemp;
	delete m_pContainer;
}


void CVoroWrapper::Init()
{
	int z, z2, z3, z4, i;
	CVoroAtom *va;
	CVoroMolecule *vm;
	CMolecule *m;
	CSingleMolecule *sm;

	m_iBlocksX = (int)(pow(g_iGesAtomCount/optimal_particles/g_fBoxX/g_fBoxY/g_fBoxZ,1/3.0)*g_fBoxX)+1;
	m_iBlocksY = (int)(pow(g_iGesAtomCount/optimal_particles/g_fBoxX/g_fBoxY/g_fBoxZ,1/3.0)*g_fBoxY)+1;
	m_iBlocksZ = (int)(pow(g_iGesAtomCount/optimal_particles/g_fBoxX/g_fBoxY/g_fBoxZ,1/3.0)*g_fBoxZ)+1;

	m_fBoxDens = g_iGesAtomCount / g_fBoxX / g_fBoxY / g_fBoxZ * 1000000.0; // Particles / Angstrom^3

	if (m_bVoroStat || m_bSurfCover || m_bWritePOV || g_bVoid)
	{
		try { m_pAssignAtoms = new long[g_iGesAtomCount]; } catch(...) { m_pAssignAtoms = NULL; }
		if (m_pAssignAtoms == NULL) NewException((double)g_iGesAtomCount*sizeof(long),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		
		try { m_pAssignMolecules = new long[g_iGesAtomCount]; } catch(...) { m_pAssignMolecules = NULL; }
		if (m_pAssignMolecules == NULL) NewException((double)g_iGesAtomCount*sizeof(long),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		
		try { m_pAssignMoleculeTypes = new long[g_iGesAtomCount]; } catch(...) { m_pAssignMoleculeTypes = NULL; }
		if (m_pAssignMoleculeTypes == NULL) NewException((double)g_iGesAtomCount*sizeof(long),__FILE__,__LINE__,__PRETTY_FUNCTION__);

		try { m_pAtomTouched = new bool[g_iGesAtomCount]; } catch(...) { m_pAtomTouched = NULL; }
		if (m_pAtomTouched == NULL) NewException((double)g_iGesAtomCount*sizeof(bool),__FILE__,__LINE__,__PRETTY_FUNCTION__);

		i = 0;
		for (z=0;z<g_oaMolecules.GetSize();z++)
		{
			m = (CMolecule*)g_oaMolecules[z];
			for (z2=0;z2<m->m_waAtomCount.GetSize()-1;z2++)
			{
				for (z3=0;z3<m->m_waAtomCount[z2];z3++)
				{
					try { va = new CVoroAtom(); } catch(...) { va = NULL; }
					if (va == NULL) NewException((double)sizeof(CVoroAtom),__FILE__,__LINE__,__PRETTY_FUNCTION__);
					
					va->m_pParent = this;
					va->m_iAtomType = z2;
					va->m_iAtom = z3;
					va->m_iRealAtomType = m->m_baAtomIndex[z2];
					va->m_iMolecule = z;

					m_oaVoroAtoms.Add(va);

					for (z4=0;z4<m->m_laSingleMolIndex.GetSize();z4++)
					{
						sm = (CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[z4]];
	//					mprintf("Mol %d, AT %d, A %d, Sm %d, Offset %f, i=%d.\n",z+1,z2+1,z3+1,z4+1,((CxIntArray*)sm->m_oaAtomOffset[z2])->GetAt(z3),i);
						m_pAssignAtoms[((CxIntArray*)sm->m_oaAtomOffset[z2])->GetAt(z3)] = i;
					}

					i++;
				}
			}
		}


		i = 0;
		for (z=0;z<g_oaMolecules.GetSize();z++)
		{
			m = (CMolecule*)g_oaMolecules[z];
			for (z4=0;z4<m->m_laSingleMolIndex.GetSize();z4++)
			{
				sm = (CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[z4]];

				try { vm = new CVoroMolecule(); } catch(...) { vm = NULL; }
				if (vm == NULL) NewException((double)sizeof(CVoroMolecule),__FILE__,__LINE__,__PRETTY_FUNCTION__);
				
				vm->m_pParent = this;
				vm->m_iMolecule = z;
				vm->m_iSingleMol = z4;
				vm->m_iCenterOffset = ((CxIntArray*)sm->m_oaAtomOffset[sm->m_oaAtomOffset.GetSize()-1])->GetAt(1); // 1 is #2, which is Center of Mass

				m_oaVoroMolecules.Add(vm);

				for (z2=0;z2<m->m_waAtomCount.GetSize()-1;z2++)
				{
					for (z3=0;z3<m->m_waAtomCount[z2];z3++)
					{
						m_pAssignMolecules[((CxIntArray*)sm->m_oaAtomOffset[z2])->GetAt(z3)] = i;
						m_pAssignMoleculeTypes[((CxIntArray*)sm->m_oaAtomOffset[z2])->GetAt(z3)] = z;
					}
				}

				i++;
			}
		}
	}

	g_iVoroMemory = 16;
}


void CVoroWrapper::Init2()
{
	int z;

	if (m_bVoroStat)
	{
		for (z=0;z<m_oaVoroAtoms.GetSize();z++)
		{
			((CVoroAtom*)m_oaVoroAtoms[z])->InitAnalyses();
			try { ((CVoroAtom*)m_oaVoroAtoms[z])->m_pNbhTempMol = new int[m_oaVoroMolecules.GetSize()]; } catch(...) { ((CVoroAtom*)m_oaVoroAtoms[z])->m_pNbhTempMol = NULL; }
			if (((CVoroAtom*)m_oaVoroAtoms[z])->m_pNbhTempMol == NULL) NewException((double)m_oaVoroMolecules.GetSize()*sizeof(int),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		}

		for (z=0;z<m_oaVoroMolecules.GetSize();z++)
		{
			((CVoroMolecule*)m_oaVoroMolecules[z])->InitAnalyses();

			try { ((CVoroMolecule*)m_oaVoroMolecules[z])->m_pNbhTempMol = new int[m_oaVoroMolecules.GetSize()]; } catch(...) { ((CVoroMolecule*)m_oaVoroMolecules[z])->m_pNbhTempMol = NULL; }
			if (((CVoroMolecule*)m_oaVoroMolecules[z])->m_pNbhTempMol == NULL) NewException((double)m_oaVoroMolecules.GetSize()*sizeof(int),__FILE__,__LINE__,__PRETTY_FUNCTION__);
			
			try { ((CVoroMolecule*)m_oaVoroMolecules[z])->m_pNbhTempAtoms = new int[g_iGesAtomCount]; } catch(...) { ((CVoroMolecule*)m_oaVoroMolecules[z])->m_pNbhTempAtoms = NULL; }
			if (((CVoroMolecule*)m_oaVoroMolecules[z])->m_pNbhTempAtoms == NULL) NewException((double)g_iGesAtomCount*sizeof(int),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		}
	}
}


void CVoroWrapper::Parse()
{
	CTimeStep *t;
	CMolecule *m;
	CSingleMolecule *sm;
//	unsigned long col;
	char buf[256];
	int z, z2, z3, z4, ti, ti2, tr, tg, tb;
	float fr, fg, fb;
	CxIntArray tia, tia2;
	CxDoubleArray *fa;
	FILE *a;
	CAtom *at;

	mprintf(WHITE,">>> Voronoi Analysis >>>\n\n");

	m_bVoroStat = AskYesNo("    Compute Voronoi statistics (y/n)? [no] ",false);

	m_bSurfCover = AskYesNo("    Calculate surface coverage of one molecule (y/n)? [no] ",false);

	if (m_bSurfCover)
	{
		if (g_oaMolecules.GetSize() > 1)
		{
			m_iSurfCoverMol = AskRangeInteger_ND("    Use Voronoi cell of which molecule type (1-%d)? ",1,g_oaMolecules.GetSize(),g_oaMolecules.GetSize())-1;
		} else m_iSurfCoverMol = 0;

		m = (CMolecule*)g_oaMolecules[m_iSurfCoverMol];

		if (m->m_laSingleMolIndex.GetSize() > 1)
/*			m_iSurfCoverSM = AskInteger_ND("    Which representants of %s to use (1-%d)? ",m->m_sName,m->m_laSingleMolIndex.GetSize())-1;
				else m_iSurfCoverSM = 0;*/
		{
			AskString("    Which representants of %s to use (allowed 1-%d, eg. 5-7,9)? [all] ",buf,"",m->m_sName,m->m_laSingleMolIndex.GetSize());
			if (strlen(buf) == 0)
			{
				for (z=0;z<m->m_laSingleMolIndex.GetSize();z++)
					m_waSurfCoverSM.Add(z);
			} else ParseIntList(buf,&m_waSurfCoverSM);
		} else m_waSurfCoverSM.Add(0);

		mprintf("\n    Observing voronoi cells of %d molecules.\n\n",m_waSurfCoverSM.GetSize());

		g_iFixMol = m_iSurfCoverMol;

		for (z=0;z<g_oaMolecules.GetSize();z++)
		{
			m = (CMolecule*)g_oaMolecules[z];
			for (z2=0;z2<m->m_waAtomCount.GetSize()-1;z2++)
			{
//				at = (CAtom*)g_oaAtoms[m->m_baAtomIndex[z2]];
				for (z3=0;z3<m->m_waAtomCount[z2];z3++)
				{
					try { fa = new CxDoubleArray("CVoroWrapper::Parse():fa"); } catch(...) { fa = NULL; }
					if (fa == NULL) NewException((double)sizeof(CxDoubleArray),__FILE__,__LINE__,__PRETTY_FUNCTION__);
					
					fa->SetGrow(1000);
					m_oaSurfCoverData.Add(fa);
					m_waSurfCoverMol.Add(z);
					m_waSurfCoverElem.Add(z2);
					m_waSurfCoverAtom.Add(z3);
				}
			}
		}
	}

/***************************************************************************************************************************/

	m_bWritePOV = AskYesNo("    Render image / movie of Voronoi cell of one single molecule (y/n)? [no] ",false);

	if (m_bWritePOV)
	{
		mprintf(GREEN,"\n>>>>>>>>>> Render single Voronoi cell >>>>>>>>>>\n\n");
#ifdef TARGET_WINDOWS
		AskString("    Enter name of POV-Ray executable: [pvengine64.exe] ",m_sPOVExe,"pvengine64.exe");
#else
		AskString("    Enter name of POV-Ray executable: [povray] ",m_sPOVExe,"povray");
#endif

		m_bWritePOVMovie = AskYesNo("    Render movie (y) or image (n)? [image] ",false);

		if (m_bWritePOVMovie)
		{
			m_iPOVResX = AskUnsignedInteger("    Enter image width (in pixels): [800] ",800);
			m_iPOVResY = AskUnsignedInteger("    Enter image height (in pixels): [600] ",600);
		} else
		{
			m_iPOVResX = AskUnsignedInteger("    Enter image width (in pixels): [1024] ",1024);
			m_iPOVResY = AskUnsignedInteger("    Enter image height (in pixels): [768] ",768);
		}

		if (m_bWritePOVMovie)
		{
			system("mkdir POV_Single");

#ifndef TARGET_WINDOWS
			mprintf("\n    Please put the font file \"courbd.ttf\" to the POV-Ray working directory!\n\n");
#endif

			if (g_fTimestepLength == 0)
				g_fTimestepLength = AskFloat("    Enter the length of one trajectory time step in fs: [0.5] ",0.5f);

			m_fPOV_FPS = AskFloat("    Enter frame rate of the animation: [25.0] ",25.0f);

			m_iPOVThreads = AskUnsignedInteger("    Enter number of separate POV-Ray render scripts: [1] ",1);
		} else
		{
			m_iPOVThreads = 1;
		}

		m_iPOVThreadPos = 0;

		m_fPOVCameraDist = AskFloat("    Enter camera distance from molecule: [20.0] ",20.0f);

		if (g_oaMolecules.GetSize() > 1)
		{
			mprintf("\n    Molecule types: ");
			for (z=0;z<g_oaMolecules.GetSize();z++)
			{
				mprintf("%s (%d)",((CMolecule*)g_oaMolecules[z])->m_sName,z+1);
				if (z < g_oaMolecules.GetSize()-1)
					mprintf(", ");
			}
			mprintf("\n\n");

			m_iPOVMol = AskRangeInteger_ND("    Write Voronoi cell of which molecule type (1-%d)? ",1,g_oaMolecules.GetSize(),g_oaMolecules.GetSize())-1;
		} else m_iPOVMol = 0;

		m = (CMolecule*)g_oaMolecules[m_iPOVMol];
		if (m->m_laSingleMolIndex.GetSize() > 1)
		{
			m_iPOVSM = AskRangeInteger_ND("    Which representant of %s to use (1-%d)? ",1,m->m_laSingleMolIndex.GetSize(),m->m_sName,m->m_laSingleMolIndex.GetSize())-1;
		} else m_iPOVSM = 0;

		g_iFixMol = m_iPOVMol;

		m_bPOVRot = AskYesNo("    Fix rotational orientation of the reference molecule (y/n)? [yes] ",true);

		if (m_bPOVRot)
		{
//			g_iRefSystemDim = 3;
_ref3again:
			mprintf("\n    The first atom will be put into the center.\n");
			mprintf("    The second atom will point straight to the right.\n");
			mprintf("    The third atom will point around upwards.\n");
			mprintf("    2nd and 3rd atom may be equal, the up direction is not fixed then.\n\n");
			mprintf("    Please enter three reference atoms (e.g. C1,H2,O1): ");
			inpprintf("! Please enter three reference atoms (e.g. C1,H2,O1):\n");
			myget(buf);
			if (!ParseRefSystem(g_iFixMol,buf,3))
				goto _ref3again;

			if ((g_iFixAtomType[0] == g_iFixAtomType[1]) && (g_iFixAtom[0] == g_iFixAtom[1]))
			{
				eprintf("At least atoms 1 and 2 need to be different.\n\n");
				inpprintf("! At least atoms 1 and 2 need to be different.\n");
				goto _ref3again;
			}

 			mprintf("\n");
			mprintf(WHITE,"Reference plane: Fixing in %s the %d. %s-, the %d. %s- and the %d. %s atom.\n",((CMolecule*)g_oaMolecules[g_iFixMol])->m_sName,g_iFixAtom[0]+1,((CAtom*)g_oaAtoms[g_iFixRealAtomType[0]])->m_sName,g_iFixAtom[1]+1,((CAtom*)g_oaAtoms[g_iFixRealAtomType[1]])->m_sName,g_iFixAtom[2]+1,((CAtom*)g_oaAtoms[g_iFixRealAtomType[2]])->m_sName);
			mprintf("\n");
		} else
		{
//			g_iRefSystemDim = 1;
_ref1again:
			mprintf("    Please enter the atom to put into the box center (e.g. C3): [#2] ");
			inpprintf("! Please enter the atom to put into the box center (e.g. C3): [#2]\n");
			myget(buf);
			if (strlen(buf)==0)
			{
				if (!ParseRefSystem(m_iPOVMol,"#2",1))
				{
					eprintf("Weird error.\n");
					inpprintf("! Weird error.\n");
					abort();
				}
			} else if (!ParseRefSystem(m_iPOVMol,buf,1))
				goto _ref1again;
 			mprintf("\n");
			mprintf(WHITE,"Reference atom: Fixing in %s the %d. %s atom.\n",((CMolecule*)g_oaMolecules[m_iPOVMol])->m_sName,g_iFixAtom[0]+1,((CAtom*)g_oaAtoms[g_iFixRealAtomType[0]])->m_sName);
			mprintf("\n");
		}

		m_bPOVAtoms = AskYesNo("    Draw atoms of reference molecule (y/n)? [yes] ",true);

		m_bPOVDrawNeighbors = AskYesNo("    Draw adjacent molecules (y/n)? [no] ",false);

		if (m_bPOVAtoms)
			m_bPOVAtomGrey = AskYesNo("    Draw the atoms in shades of grey (y) or color (n)? [no] ",false);

		m_bPOVVertices = AskYesNo("    Draw Voronoi cell vertices (y/n)? [no] ",false);

		m_bPOVEdges = AskYesNo("    Draw Voronoi cell edges (y/n)? [yes] ",true);

		m_bPOVFaces = AskYesNo("    Draw Voronoi cell faces (y/n)? [yes] ",true);

		if (m_bPOVFaces)
		{
//			m_fPOVFaceTrans = AskRangeFloat("    Enter face transparency (0.0-1.0): [0.25] ",0.0,1.0f,0.25f);
			m_bPOVFaceColor = AskYesNo("    Colorize faces depending on atom types (y/n)? [no] ",false);
		} else
		{
			m_bPOVFaceColor = false;
		}

		if (m_bPOVFaceColor)
		{
			m_bPOVFaceColorRef = AskYesNo("    Take face colors from ref. mol atom types (y) or from neighbors (n)? [no] ",false);
			ParsePOVFaceColor();
		}

		if (m_bPOVDrawNeighbors && m_bPOVFaceColor && !m_bPOVFaceColorRef)
		{
			m_bPOVNbColorFace = AskYesNo("    Colorize adjacent molecules according to face color (y/n)? [no] ",false);
			m_bPOVHighlightNbAtoms = AskYesNo("    Highlight neighboring atoms from molecules (y/n)? [no] ",false);
		} else
		{
			m_bPOVNbColorFace = false;
			m_bPOVHighlightNbAtoms = false;
		}

		if (m_bWritePOVMovie)
		{
			m_fPOVBoxScript = new FILE*[m_iPOVThreads];

			if (m_iPOVThreads == 1)
			{

#ifdef TARGET_WINDOWS
				sprintf(buf,"POV_Single\\render_single.bat");
#else
				sprintf(buf,"POV_Single/render_single");
#endif

				m_fPOVBoxScript[0] = OpenFileWrite(buf,true);
			} else
			{
				for (z=0;z<m_iPOVThreads;z++)
				{

#ifdef TARGET_WINDOWS
					sprintf(buf,"mkdir POV_Single\\%d",z+1);
					system(buf);
					sprintf(buf,"POV_Single\\%d\\render_single.bat",z+1);
#else
					sprintf(buf,"mkdir POV_Single/%d",z+1);
					system(buf);
					sprintf(buf,"POV_Single/%d/render_single",z+1);
#endif

					m_fPOVBoxScript[z] = OpenFileWrite(buf,true);
				}
			}
		} // END IF Movie

		m_faPOVBoxAtomVisible.SetSize(g_iGesAtomCount);
		for (z=0;z<g_iGesAtomCount;z++)
			m_faPOVBoxAtomVisible[z] = 1.0f;

//		mprintf("    Creating Choreography...\n");
//		CreateChoreography_Single();

		ParseChoreography_Single();

		if (m_bWritePOVMovie)
		{
			mprintf("    Testing Choreography (see choreo_test.csv) ...\n");
			TestChoreography();
			mprintf("    Choreography comprises %d frames (%.2f sec) and %d steps.\n",m_iPOVBoxTotalFrames,(float)m_iPOVBoxTotalFrames/m_fPOV_FPS,m_iPOVBoxTotalSteps);

			m_iPOVCurrentChoreographyPart = 0;
			m_iPOVFrameCounter = 0;
			m_pPOVCurrentValues->CopyFrom((CVoroChoreographyPart*)m_oaBoxChoreography[0]);

#ifdef TARGET_WINDOWS
			a = OpenFileWrite("POV_Single\\video.avs",true);
#else
			a = OpenFileWrite("POV_Single/video.avs",true);
#endif

			mfprintf(a,"video = ImageSource(\"frame_%c05d.png\", start=0, end=%d, fps=%.2f, pixel_type = \"rgb24\" )\n",'%',m_iPOVBoxTotalFrames-1,m_fPOV_FPS);
			mfprintf(a,"return video\n");
			fclose(a);

#ifdef TARGET_LINUX
			a = OpenFileWrite("POV_Single/run_mencoder",true);
			mfprintf(a,"mencoder \"mf://*.png\" -vf scale=400:-11 -oac copy -ovc lavc -lavcopts vcodec=msmpeg4:vbitrate=2000 -ffourcc MP43 -o \"output.avi\"\n",'%');
			fclose(a);
#endif

			m_fPOVCT = OpenFileWrite("choreo_single.csv",true);

			mfprintf(m_fPOVCT,"Frame;Step;Zoom;Angle;Scale;Explode;Clip;OuterOpac;Molecules;AtomCenter;ClipLowX;ClipLowY;ClipLowZ;ClipHighX;ClipHighY;ClipHighZ\n");
		}
		mprintf(GREEN,"\n<<<<<<<<<< End of Render single Voronoi cell <<<<<<<<<<\n\n");
	}

/******************************************************************************************************************************/

	m_bPOVBox = AskYesNo("    Render image / movie of Voronoi decompositon of whole box (y/n)? [no] ",false);

	if (m_bPOVBox)
	{
		mprintf(GREEN,"\n>>>>>>>>>> Render Voronoi Box >>>>>>>>>>\n\n");
#ifdef TARGET_WINDOWS
		AskString("    Enter name of POV-Ray executable: [pvengine64.exe] ",m_sPOVExe,"pvengine64.exe");
#else
		AskString("    Enter name of POV-Ray executable: [povray] ",m_sPOVExe,"povray");
#endif

		m_bWritePOVMovie = AskYesNo("    Render movie (y) or image (n)? [image] ",false);

		if (m_bWritePOVMovie)
		{
			m_iPOVResX = AskUnsignedInteger("    Enter image width (in pixels): [800] ",800);
			m_iPOVResY = AskUnsignedInteger("    Enter image height (in pixels): [600] ",600);
		} else
		{
			m_iPOVResX = AskUnsignedInteger("    Enter image width (in pixels): [1024] ",1024);
			m_iPOVResY = AskUnsignedInteger("    Enter image height (in pixels): [768] ",768);
		}

		if (m_bWritePOVMovie)
		{
			system("mkdir POV_Box");

#ifndef TARGET_WINDOWS
			mprintf("\n    Please put the font file \"courbd.ttf\" to the POV-Ray working directory!\n\n");
#endif

			if (g_fTimestepLength == 0)
				g_fTimestepLength = AskFloat("    Enter the length of one trajectory time step in fs: [0.5] ",0.5f);

			m_fPOV_FPS = AskFloat("    Enter frame rate of the animation: [25.0] ",25.0f);

			m_iPOVThreads = AskUnsignedInteger("    Enter number of separate POV-Ray render scripts: [1] ",1);
		} else
		{
			m_iPOVThreads = 1;
		}

		m_iPOVThreadPos = 0;

		if (m_bWritePOVMovie)
		{
			m_fPOVBoxScript = new FILE*[m_iPOVThreads];

			if (m_iPOVThreads == 1)
			{

#ifdef TARGET_WINDOWS
				sprintf(buf,"POV_Box\\render_box.bat");
#else
				sprintf(buf,"POV_Box/render_box");
#endif

				m_fPOVBoxScript[0] = OpenFileWrite(buf,true);
			} else
			{
				for (z=0;z<m_iPOVThreads;z++)
				{

#ifdef TARGET_WINDOWS
					sprintf(buf,"mkdir POV_Box\\%d",z+1);
					system(buf);
					sprintf(buf,"POV_Box\\%d\\render_box.bat",z+1);
#else
					sprintf(buf,"mkdir POV_Box/%d",z+1);
					system(buf);
					sprintf(buf,"POV_Box/%d/render_box",z+1);
#endif

					m_fPOVBoxScript[z] = OpenFileWrite(buf,true);
				}
			}
		}

		m_iaPOVBoxAtomColors.SetSize(g_iGesAtomCount);

		mprintf(YELLOW,"\n    *** Atom Coloring ***\n\n");
		mprintf("    You can choose one of the following atom coloring methods:\n\n");
		mprintf("      1.) Enter colors by element type\n");
		mprintf("      2.) Enter colors by molecule type\n");
		mprintf("      3.) Enter colors for each atom in each molecule type\n");
		mprintf("      4.) Random colors for each atom\n");
		mprintf("      5.) Random colors for each molecule\n");

		mprintf("\n");
		ti = AskRangeInteger("    Enter your choice: [1] ",1,5,1);

		switch(ti)
		{
			case 1:
_cac1:
				mprintf(WHITE,"\n    Element colors:\n");
				for (z=0;z<g_oaAtoms.GetSize();z++)
				{
					if (z == g_iVirtAtomType)
						continue;
					mprintf("      %d.) %2s:  R=%3d, G=%3d, B=%3d.\n",z+1,((CAtom*)g_oaAtoms[z])->m_sName,((CAtom*)g_oaAtoms[z])->m_pElement->m_iColorR,((CAtom*)g_oaAtoms[z])->m_pElement->m_iColorG,((CAtom*)g_oaAtoms[z])->m_pElement->m_iColorB);
				}
				mprintf("\n");
				ti2 = AskRangeInteger("    Which element color to change (1-%d)? [done] ",0,g_oaAtoms.GetSize()-1,0,g_oaAtoms.GetSize()-1);
				if (ti2 != 0)
				{
					((CAtom*)g_oaAtoms[ti2-1])->m_pElement->m_iColorR = AskRangeInteger("      Enter new red component (0-255): [%d] ",0,255,((CAtom*)g_oaAtoms[ti2-1])->m_pElement->m_iColorR,((CAtom*)g_oaAtoms[ti2-1])->m_pElement->m_iColorR);
					((CAtom*)g_oaAtoms[ti2-1])->m_pElement->m_iColorG = AskRangeInteger("      Enter new green component (0-255): [%d] ",0,255,((CAtom*)g_oaAtoms[ti2-1])->m_pElement->m_iColorG,((CAtom*)g_oaAtoms[ti2-1])->m_pElement->m_iColorG);
					((CAtom*)g_oaAtoms[ti2-1])->m_pElement->m_iColorB = AskRangeInteger("      Enter new blue component (0-255): [%d] ",0,255,((CAtom*)g_oaAtoms[ti2-1])->m_pElement->m_iColorB,((CAtom*)g_oaAtoms[ti2-1])->m_pElement->m_iColorB);
					goto _cac1;
				}
				mprintf("\n    Assigning colors...\n");
				for (z=0;z<g_iGesAtomCount;z++)
				{
					at = (CAtom*)g_oaAtoms[g_waAtomRealElement[z]];
					m_iaPOVBoxAtomColors[z] = (at->m_pElement->m_iColorR << 16) + (at->m_pElement->m_iColorG << 8) + at->m_pElement->m_iColorB;
				}
				break;

			case 2:
				tia.SetSize(g_oaMolecules.GetSize()*3);
				for (z=0;z<g_oaMolecules.GetSize();z++)
				{
					tia[z*3] = ((z+1) & 1) * 255;
					tia[z*3+1] = (((z+1) & 2) / 2) * 255;
					tia[z*3+2] = (((z+1) & 4) / 4) * 255;
				}
_cac2:
				mprintf(WHITE,"\n    Molecule type colors:\n");
				for (z=0;z<g_oaMolecules.GetSize();z++)
					mprintf("      %d.) %s:  R=%3d, G=%3d, B=%3d.\n",z+1,((CMolecule*)g_oaMolecules[z])->m_sName,tia[z*3],tia[z*3+1],tia[z*3+2]);
				mprintf("\n");
				ti2 = AskRangeInteger("    Which molecule type color to change (1-%d)? [done] ",0,g_oaMolecules.GetSize(),0,g_oaMolecules.GetSize());
				if (ti2 != 0)
				{
					tia[(ti2-1)*3] = AskRangeInteger("      Enter new red component (0-255): [%d] ",0,255,tia[(ti2-1)*3],tia[(ti2-1)*3]);
					tia[(ti2-1)*3+1] = AskRangeInteger("      Enter new green component (0-255): [%d] ",0,255,tia[(ti2-1)*3+1],tia[(ti2-1)*3+1]);
					tia[(ti2-1)*3+2] = AskRangeInteger("      Enter new blue component (0-255): [%d] ",0,255,tia[(ti2-1)*3+2],tia[(ti2-1)*3+2]);
					goto _cac2;
				}
				mprintf("\n    Assigning colors...\n");
				for (z=0;z<g_iGesAtomCount;z++)
				{
					ti2 = g_waAtomMolIndex[z];
					m_iaPOVBoxAtomColors[z] = (tia[ti2*3] << 16) + (tia[ti2*3+1] << 8) + tia[ti2*3+2];
				}
				break;

			case 3:
				ti2 = 0;
				for (z=0;z<g_oaMolecules.GetSize();z++)
				{
					m = (CMolecule*)g_oaMolecules[z];
					for (z2=0;z2<m->m_baAtomIndex.GetSize();z2++)
					{
						if (m->m_baAtomIndex[z2] == g_iVirtAtomType)
							continue;
						ti2 += m->m_waAtomCount[z2];
					}
				}
				tia.SetSize(ti2*3);
				ti2 = 0;
				for (z=0;z<g_oaMolecules.GetSize();z++)
				{
					m = (CMolecule*)g_oaMolecules[z];
					for (z2=0;z2<m->m_baAtomIndex.GetSize();z2++)
					{
						if (m->m_baAtomIndex[z2] == g_iVirtAtomType)
							continue;
						for (z3=0;z3<m->m_waAtomCount[z2];z3++)
						{
							tia[ti2*3] = ((CAtom*)g_oaAtoms[m->m_baAtomIndex[z2]])->m_pElement->m_iColorR;
							tia[ti2*3+1] = ((CAtom*)g_oaAtoms[m->m_baAtomIndex[z2]])->m_pElement->m_iColorG;
							tia[ti2*3+2] = ((CAtom*)g_oaAtoms[m->m_baAtomIndex[z2]])->m_pElement->m_iColorB;
							ti2++;
						}
					}
				}
_cac3:
				mprintf(WHITE,"\n    Molecule/atom colors:\n");
				ti2 = 0;
				for (z=0;z<g_oaMolecules.GetSize();z++)
				{
					m = (CMolecule*)g_oaMolecules[z];
					for (z2=0;z2<m->m_baAtomIndex.GetSize();z2++)
					{
						if (m->m_baAtomIndex[z2] == g_iVirtAtomType)
							continue;
						for (z3=0;z3<m->m_waAtomCount[z2];z3++)
						{
							mprintf("      %3d.) %s %s%d:  R=%3d, G=%3d, B=%3d.\n",ti2+1,m->m_sName,((CAtom*)g_oaAtoms[m->m_baAtomIndex[z2]])->m_sName,z3+1,tia[ti2*3],tia[ti2*3+1],tia[ti2*3+2]);
							ti2++;
						}
					}
					mprintf("\n");
				}
				AskString("    Enter a list of atom colors to change (e.g. \"1,3-5,8\"): [done] ",buf,"");
				if (strlen(buf) != 0)
				{
					if (!ParseIntList(buf,&tia2))
					{
						eprintf("    Wrong input.\n");
						goto _cac3;
					}
					for (z=0;z<tia2.GetSize();z++)
					{
						if ((tia2[z] > ti2) || (tia2[z] < 1))
						{
							eprintf("    Wrong input: %d.\n",tia2[z]);
							goto _cac3;
						}
					}
					tr = AskRangeInteger_ND("      Enter new red component (0-255): ",0,255);
					tg = AskRangeInteger_ND("      Enter new green component (0-255): ",0,255);
					tb = AskRangeInteger_ND("      Enter new blue component (0-255): ",0,255);
					for (z=0;z<tia2.GetSize();z++)
					{
						tia[tia2[z]*3] = tr;
						tia[tia2[z]*3+1] = tg;
						tia[tia2[z]*3+2] = tb;
					}
					goto _cac3;
				}
				mprintf("\n    Assigning colors...\n");
				ti2 = 0;
				for (z=0;z<g_oaMolecules.GetSize();z++)
				{
					m = (CMolecule*)g_oaMolecules[z];
					for (z3=0;z3<m->m_baAtomIndex.GetSize();z3++)
					{
						if (m->m_baAtomIndex[z3] == g_iVirtAtomType)
							continue;
						for (z4=0;z4<m->m_waAtomCount[z3];z4++)
						{
							for (z2=0;z2<m->m_laSingleMolIndex.GetSize();z2++)
							{
								sm = (CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[z2]];
								m_iaPOVBoxAtomColors[((CxIntArray*)sm->m_oaAtomOffset[z3])->GetAt(z4)] = (tia[ti2*3] << 16) + (tia[ti2*3+1] << 8) + tia[ti2*3+2];
							}
							ti2++;
						}
					}
				}
				break;

			case 4:
				mprintf("\n    Assigning colors...\n");
				for (z=0;z<g_iGesAtomCount;z++)
				{
					HSV2RGB((rand()%10001)/10000.0f,powf((rand()%10001)/10000.0f,0.5),((rand()%4001)+6000)/10000.0f,fr,fg,fb);

					fr *= 255.0f;
					fg *= 255.0f;
					fb *= 255.0f;

					m_iaPOVBoxAtomColors[z] = (((long)fr) << 16) + (((long)fg) << 8) + ((long)fb);
				}
				break;

			case 5:
				mprintf("\n    Assigning colors...\n");
				for (z=0;z<g_oaSingleMolecules.GetSize();z++)
				{
					HSV2RGB((rand()%10001)/10000.0f,powf((rand()%10001)/10000.0f,0.5),((rand()%4001)+6000)/10000.0f,fr,fg,fb);

					fr *= 255.0f;
					fg *= 255.0f;
					fb *= 255.0f;

					sm = (CSingleMolecule*)g_oaSingleMolecules[z];

					for (z2=0;z2<sm->m_oaAtomOffset.GetSize()-1;z2++) // "-1" to skip virtual atoms
						for (z3=0;z3<((CxIntArray*)sm->m_oaAtomOffset[z2])->GetSize();z3++)
							m_iaPOVBoxAtomColors[((CxIntArray*)sm->m_oaAtomOffset[z2])->GetAt(z3)] = (((long)fr) << 16) + (((long)fg) << 8) + ((long)fb);
				}
				break;

			default:
				eprintf("Should never happen.\n");
				abort();
		}

/*		for (z=0;z<g_iGesAtomCount;z++)
		{
//			CElement *el;
//			el = ((CAtom*)g_oaAtoms[g_waAtomRealElement[z]])->m_pElement;
//			m_iaPOVBoxAtomColors[z] = (((long)el->m_iColorR) << 16) + (((long)el->m_iColorG) << 8) + ((long)el->m_iColorB);

			float fr, fg, fb;

			HSV2RGB((rand()%10001)/10000.0f,powf((rand()%10001)/10000.0f,0.5),((rand()%4001)+6000)/10000.0f,fr,fg,fb);

			fr *= 255.0f;
			fg *= 255.0f;
			fb *= 255.0f;

			m_iaPOVBoxAtomColors[z] = (((long)fr) << 16) + (((long)fg) << 8) + ((long)fb);
		}

		m_iaPOVBoxAtomColors[0] = 0xE06000;
		m_iaPOVBoxAtomColors[1] = 0x00C000;
		m_iaPOVBoxAtomColors[2] = 0x0060E0;*/

/*		// EMIM 
		m = (CMolecule*)g_oaMolecules[0];
		for (z=0;z<m->m_laSingleMolIndex.GetSize();z++)
		{
			sm = (CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[z]];

			// Ring: C1,3,4  N1,2  H1,4,5 
			col = 0xF04000;
			m_iaPOVBoxAtomColors[((CxIntArray*)sm->m_oaAtomOffset[0])->GetAt(0)] = col;
			m_iaPOVBoxAtomColors[((CxIntArray*)sm->m_oaAtomOffset[0])->GetAt(2)] = col;
			m_iaPOVBoxAtomColors[((CxIntArray*)sm->m_oaAtomOffset[0])->GetAt(3)] = col;
			m_iaPOVBoxAtomColors[((CxIntArray*)sm->m_oaAtomOffset[1])->GetAt(0)] = col;
			m_iaPOVBoxAtomColors[((CxIntArray*)sm->m_oaAtomOffset[1])->GetAt(3)] = col;
			m_iaPOVBoxAtomColors[((CxIntArray*)sm->m_oaAtomOffset[1])->GetAt(4)] = col;
			m_iaPOVBoxAtomColors[((CxIntArray*)sm->m_oaAtomOffset[2])->GetAt(0)] = col;
			m_iaPOVBoxAtomColors[((CxIntArray*)sm->m_oaAtomOffset[2])->GetAt(1)] = col;

			// Chain: C2,5,6  H2,3,6,7,8,9,10,11 
			col = 0xFFFF00;
			m_iaPOVBoxAtomColors[((CxIntArray*)sm->m_oaAtomOffset[0])->GetAt(1)] = col;
			m_iaPOVBoxAtomColors[((CxIntArray*)sm->m_oaAtomOffset[0])->GetAt(4)] = col;
			m_iaPOVBoxAtomColors[((CxIntArray*)sm->m_oaAtomOffset[0])->GetAt(5)] = col;
			m_iaPOVBoxAtomColors[((CxIntArray*)sm->m_oaAtomOffset[1])->GetAt(1)] = col;
			m_iaPOVBoxAtomColors[((CxIntArray*)sm->m_oaAtomOffset[1])->GetAt(2)] = col;
			m_iaPOVBoxAtomColors[((CxIntArray*)sm->m_oaAtomOffset[1])->GetAt(5)] = col;
			m_iaPOVBoxAtomColors[((CxIntArray*)sm->m_oaAtomOffset[1])->GetAt(6)] = col;
			m_iaPOVBoxAtomColors[((CxIntArray*)sm->m_oaAtomOffset[1])->GetAt(7)] = col;
			m_iaPOVBoxAtomColors[((CxIntArray*)sm->m_oaAtomOffset[1])->GetAt(8)] = col;
			m_iaPOVBoxAtomColors[((CxIntArray*)sm->m_oaAtomOffset[1])->GetAt(9)] = col;
			m_iaPOVBoxAtomColors[((CxIntArray*)sm->m_oaAtomOffset[1])->GetAt(10)] = col;
		}

		// OAc 
		m = (CMolecule*)g_oaMolecules[1];
		for (z=0;z<m->m_laSingleMolIndex.GetSize();z++)
		{
			sm = (CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[z]];

			// Carbox: C1 O1,2 
			col = 0x0040FF;
			m_iaPOVBoxAtomColors[((CxIntArray*)sm->m_oaAtomOffset[0])->GetAt(0)] = col;
			m_iaPOVBoxAtomColors[((CxIntArray*)sm->m_oaAtomOffset[2])->GetAt(0)] = col;
			m_iaPOVBoxAtomColors[((CxIntArray*)sm->m_oaAtomOffset[2])->GetAt(1)] = col;

			// Chain: C2 H1,2,3 
			col = 0x40FF00;
			m_iaPOVBoxAtomColors[((CxIntArray*)sm->m_oaAtomOffset[0])->GetAt(1)] = col;
			m_iaPOVBoxAtomColors[((CxIntArray*)sm->m_oaAtomOffset[1])->GetAt(0)] = col;
			m_iaPOVBoxAtomColors[((CxIntArray*)sm->m_oaAtomOffset[1])->GetAt(1)] = col;
			m_iaPOVBoxAtomColors[((CxIntArray*)sm->m_oaAtomOffset[1])->GetAt(2)] = col;
		}*/

/*		// Take standard element colors
		for (z=0;z<g_iGesAtomCount;z++)
		{
			at = (CAtom*)g_oaAtoms[g_waAtomRealElement[z]];
			m_iaPOVBoxAtomColors[z] = (at->m_pElement->m_iColorR << 16) + (at->m_pElement->m_iColorG << 8) + at->m_pElement->m_iColorB;
		}*/

		m_faPOVBoxAtomVisible.SetSize(g_iGesAtomCount);
		for (z=0;z<g_iGesAtomCount;z++)
			m_faPOVBoxAtomVisible[z] = 1.0f;

//		mprintf("    Creating Choreography...\n");
//		CreateChoreography();

		ParseChoreography();

		if (m_bWritePOVMovie)
		{
			mprintf("    Testing Choreography...\n");
			TestChoreography();
			mprintf("    Choreography comprises %d frames (%.2f sec) and %d steps.\n",m_iPOVBoxTotalFrames,(float)m_iPOVBoxTotalFrames/m_fPOV_FPS,m_iPOVBoxTotalSteps);

			m_iPOVCurrentChoreographyPart = 0;
			m_iPOVFrameCounter = 0;
			m_pPOVCurrentValues->CopyFrom((CVoroChoreographyPart*)m_oaBoxChoreography[0]);

#ifdef TARGET_WINDOWS
			a = OpenFileWrite("POV_Box\\video.avs",true);
#else
			a = OpenFileWrite("POV_Box/video.avs",true);
#endif

			mfprintf(a,"video = ImageSource(\"frame_%c05d.png\", start=0, end=%d, fps=%.2f, pixel_type = \"rgb24\" )\n",'%',m_iPOVBoxTotalFrames-1,m_fPOV_FPS);
			mfprintf(a,"return video\n");
			fclose(a);

#ifdef TARGET_LINUX
			a = OpenFileWrite("POV_Box/run_mencoder",true);
			mfprintf(a,"mencoder \"mf://*.png\" -vf scale=400:-11 -oac copy -ovc lavc -lavcopts vcodec=msmpeg4:vbitrate=2000 -ffourcc MP43 -o \"output.avi\"\n",'%');
			fclose(a);
#endif

			m_fPOVCT = OpenFileWrite("choreo_box.csv",true);

			mfprintf(m_fPOVCT,"Frame;Step;Zoom;Angle;Scale;Explode;Clip;OuterOpac;Molecules;AtomCenter;ClipLowX;ClipLowY;ClipLowZ;ClipHighX;ClipHighY;ClipHighZ\n");
		}
		mprintf(GREEN,"\n<<<<<<<<<< End of Render Voronoi Box <<<<<<<<<<\n\n");
	}

/***************************************************************************************************************************/

	m_bVoroMetric = AskYesNo("    Compute Voronoi metric between atoms (y/n)? [no] ",false);

	if (m_bVoroMetric)
	{
		m_bVoroMetricCDF = AskYesNo("    Create CDFs for Voronoi metric (y/n)? [yes] ",true);

		if (m_bVoroMetricCDF)
		{
			m_pVoroMetricCDF = new C2DF();
			m_pVoroMetricCDF->m_iRes[0] = 100;
			m_pVoroMetricCDF->m_iRes[1] = 100;
			m_pVoroMetricCDF->m_fMinVal[0] = 0;
			m_pVoroMetricCDF->m_fMaxVal[0] = 700.0;
			m_pVoroMetricCDF->m_fMinVal[1] = 0;
			m_pVoroMetricCDF->m_fMaxVal[1] = 12.0;
			m_pVoroMetricCDF->Create();
			m_pVoroMetricCDF->SetLabelX("Distance [pm]");
			m_pVoroMetricCDF->SetLabelY("Shared Surface [1000 pm^2]");

			m_pVoroMetricCDF2 = new C2DF();
			m_pVoroMetricCDF2->m_iRes[0] = 100;
			m_pVoroMetricCDF2->m_iRes[1] = 100;
			m_pVoroMetricCDF2->m_fMinVal[0] = 0;
			m_pVoroMetricCDF2->m_fMaxVal[0] = 700.0;
			m_pVoroMetricCDF2->m_fMinVal[1] = 0;
			m_pVoroMetricCDF2->m_fMaxVal[1] = 2500.0;
			m_pVoroMetricCDF2->Create();
			m_pVoroMetricCDF2->SetLabelX("Distance [pm]");
			m_pVoroMetricCDF2->SetLabelY("Reciprocal Shared Surface [nm^-2]");
		}

		m_faVoroMetric.SetSize(g_iGesAtomCount*g_iGesAtomCount);
	}

	mprintf("\n");
	mprintf("    Initializing Voronoi data structures...\n");
	g_pVoroWrapper->Init();
	mprintf("\n");

	mprintf("*** Voro: Box density is %f particles / Angstrom^3.\n",m_fBoxDens);
	mprintf("*** Voro: Using %d x %d x %d blocks.\n",m_iBlocksX,m_iBlocksY,m_iBlocksZ);
	mprintf("*** Voro: Using cell memory for %d particles.\n",g_iVoroMemory);

	if (m_bVoroStat || m_bSurfCover || m_bWritePOV)
	{
		mprintf(WHITE,"*** Dumping Voronoi Infos to \"voro.txt\".\n");

		try { t = new CTimeStep(); } catch(...) { t = NULL; }
		if (t == NULL) NewException((double)sizeof(CTimeStep),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		
		t->CopyFrom(&g_TimeStep);

		m = (CMolecule*)g_oaMolecules[0];
		sm = (CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[0]];

		t->CenterPos(t->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[sm->m_baAtomIndex.GetSize()-1])->GetAt(1)]);
		t->CenterPos(CxVector3(-g_fBoxX/2.0,-g_fBoxY/2.0,-g_fBoxZ/2.0));
		t->FoldAtomsPositive();
		g_pVoroWrapper->Dump("voro.txt",t);
	//	g_pVoroWrapper->WritePOV(t,"voro.pov",0,0);
		delete t;
		g_pVoroWrapper->Init2();
	}

//	AskYesNo("    This is hard-coded for Ulrikes Bmim-Br ^^ [accept] ",false);

	mprintf(WHITE,"\n<<< End of Voronoi Analysis <<<\n\n");
}


void CVoroWrapper::Dump(const char *s, CTimeStep *ts)
{
	int ijk, q, z, z2, ci, id;
	double *pp, cv;
	FILE *a;
	voronoicell_neighbor c;
	vector<int> nb;
	vector<int> fo;
	vector<double> fa;
	CVoroAtom *va, *va2;
	CVoroMolecule *vm, *vm2;

	a = OpenFileWrite(s,true);

	mfprintf(a,"*** Voro: Box density is %f particles / Angstrom^3.\n",m_fBoxDens);
	mfprintf(a,"*** Voro: Using %d x %d x %d blocks.\n",m_iBlocksX,m_iBlocksY,m_iBlocksZ);

//	d0 = g_iGesAtomCount / g_fBoxX / g_fBoxY / g_fBoxZ * 1000000.0;
//	d = 2.35 / pow(d0,1/3.0);

	mfprintf(a,"*** Voro: Creating Container...\n");
	fflush(a);

//	m_pContainer = new container_periodic_poly(1.0,0,1.0,0,0,1.0,m_iBlocksX,m_iBlocksY,m_iBlocksZ,g_iVoroMemory);

	try { m_pContainer = new container_periodic_poly(g_fBoxX/1000.0,0,g_fBoxY/1000.0,0,0,g_fBoxZ/1000.0,m_iBlocksX,m_iBlocksY,m_iBlocksZ,g_iVoroMemory); } catch(...) { m_pContainer = NULL; }
	if (m_pContainer == NULL) NewException((double)sizeof(container_periodic_poly),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	mfprintf(a,"*** Voro: Adding %d particles...\n",g_iGesAtomCount);
	fflush(a);

	for (z=0;z<g_iGesAtomCount;z++)
		m_pContainer->put(z,ts->m_vaCoords[z][0]/1000.0,ts->m_vaCoords[z][1]/1000.0,ts->m_vaCoords[z][2]/1000.0,0.05);

	m_fMaxVol = 0;
	m_fMaxSurf = 0;

	c_loop_all_periodic vl(*m_pContainer);

	ci = 0;
	cv = 0;
	if (vl.start()) 
	{
		do 
		{
			if (m_pContainer->compute_cell(c,vl))
			{
				ci++;
				ijk=vl.ijk;
				q=vl.q;
				pp=m_pContainer->p[ijk]+m_pContainer->ps*q;
				id = m_pContainer->id[ijk][q];

				va = (CVoroAtom*)m_oaVoroAtoms[m_pAssignAtoms[id]];
				vm = (CVoroMolecule*)m_oaVoroMolecules[m_pAssignMolecules[id]];

				mfprintf(a,"Cell %5d (%s[%d] %s%d); p=(%.2f | %.2f | %.2f); r=%.2f pm; %d Faces; Vol=%.4f A^3, Surface=%.4f A^2; Max.Radius=%.3f pm\n",id,((CMolecule*)g_oaMolecules[va->m_iMolecule])->m_sName,vm->m_iSingleMol+1,((CAtom*)g_oaAtoms[va->m_iRealAtomType])->m_sName,va->m_iAtom+1,pp[0]*1000.0,pp[1]*1000.0,pp[2]*1000.0,pp[3]*1000.0,c.number_of_faces(),c.volume()*1000.0,c.surface_area()*100.0,sqrt(c.max_radius_squared())*1000.0);

				if (c.volume()*1000.0 > m_fMaxVol)
					m_fMaxVol = c.volume()*1000.0;

				cv += c.volume()*1000.0;

				if (c.surface_area()*100.0 > m_fMaxSurf)
					m_fMaxSurf = c.surface_area()*100.0;

				c.face_areas(fa);
				c.face_orders(fo);
				c.neighbors(nb);

				for (z2=0;z2<c.number_of_faces();z2++)
				{
					va2 = (CVoroAtom*)m_oaVoroAtoms[m_pAssignAtoms[nb[z2]]];
					vm2 = (CVoroMolecule*)m_oaVoroMolecules[m_pAssignMolecules[nb[z2]]];
					mfprintf(a,"  - Face %2d: %2d Vertices, Area=%7.4f A^2, Neighbor is %5d (%s[%d] %s%d)\n",z2+1,fo[z2],fa[z2]*100.0,nb[z2],((CMolecule*)g_oaMolecules[va2->m_iMolecule])->m_sName,vm2->m_iSingleMol+1,((CAtom*)g_oaAtoms[va2->m_iRealAtomType])->m_sName,va2->m_iAtom+1);
				}
			}
		} while (vl.inc());
	}
	mprintf("    %d/%d Particles touched.\n",ci,g_iGesAtomCount);
	mprintf("    Volume %f/%f Angstrom^3.\n",cv,g_fBoxX*g_fBoxY*g_fBoxZ/1000000.0);
	m_fMaxVol *= 3.0;
	m_fMaxSurf *= 3.0;
	delete m_pContainer;
	mfprintf(a,"*** The End ***\n");
	fclose(a);
}


void CVoroWrapper::Finish()
{
	int z, z2;
	char buf[256];
	CVoroAtom *va;
	CVoroMolecule *vm, *vmbase;

	if (m_bPOVBox || m_bWritePOV)
	{
		if (m_bWritePOVMovie)
		{
			for (z=0;z<m_iPOVThreads;z++)
				fclose(m_fPOVBoxScript[z]);
		}
		mprintf(YELLOW,"\n    Note: ");
		mprintf("Please use POV-Ray 3.7 to render the pov files.\n");
	}

#ifdef TARGET_LINUX
	if (m_bWritePOV)
	{
		mprintf("\n");
		if (m_bWritePOVMovie)
		{
			if (m_iPOVThreads == 1)
			{
				sprintf(buf,"chmod 755 POV_Single/render_single");
				mprintf("    Executing \"%s\" ...\n",buf);
				system(buf);
			} else
			{
				for (z=0;z<m_iPOVThreads;z++)
				{
					sprintf(buf,"chmod 755 POV_Single/%d/render_single",z+1);
					mprintf("    Executing \"%s\" ...\n",buf);
					system(buf);
				}
			}
			sprintf(buf,"chmod 755 POV_Single/run_mencoder");
			mprintf("    Executing \"%s\" ...\n",buf);
			system(buf);
		} else
		{
			sprintf(buf,"chmod 755 render_single");
			mprintf("    Executing \"%s\" ...\n",buf);
			system(buf);
		}
	}

	if (m_bPOVBox)
	{
		mprintf("\n");
		if (m_bWritePOVMovie)
		{
			if (m_iPOVThreads == 1)
			{
				sprintf(buf,"chmod 755 POV_Box/render_box");
				mprintf("    Executing \"%s\" ...\n",buf);
				system(buf);
			} else
			{
				for (z=0;z<m_iPOVThreads;z++)
				{
					sprintf(buf,"chmod 755 POV_Box/%d/render_box",z+1);
					mprintf("    Executing \"%s\" ...\n",buf);
					system(buf);
				}
			}
			sprintf(buf,"chmod 755 POV_Box/run_mencoder");
			mprintf("    Executing \"%s\" ...\n",buf);
			system(buf);
		} else
		{
			sprintf(buf,"chmod 755 render_box");
			mprintf("    Executing \"%s\" ...\n",buf);
			system(buf);
		}
	}
#endif

	if (m_bVoroMetric)
	{
		if (m_bVoroMetricCDF)
		{
			m_pVoroMetricCDF->WriteMathematicaNb("","vorometric.nb","",false);
			m_pVoroMetricCDF2->WriteMathematicaNb("","vorometric2.nb","",false);
		}
	}

//	if (m_bWritePOV)
//		fclose(m_fPOVScript);

	if (m_bSurfCover)
	{
		mprintf(WHITE,"*** Voronoi Surface Coverage\n\n");
		mprintf("    Writing Surface data to voro_surfcover.csv\n");
		FinishSurfCover("voro_surfcover.csv");
		mprintf("\n");
	}

	if (m_bVoroStat)
	{
		mprintf(WHITE,"*** Voronoi Functions\n\n");

		mprintf("    Creating the sub-directory \"voronoi\"...\n");
		system("mkdir voronoi");
		mprintf("\n");

		mprintf("    Atom Information:\n");
		for (z=0;z<m_oaVoroAtoms.GetSize();z++)
		{
			va = (CVoroAtom*)m_oaVoroAtoms[z];
			sprintf(buf,"voronoi/voro_atom_%s_%s%d",((CMolecule*)g_oaMolecules[va->m_iMolecule])->m_sName,((CAtom*)g_oaAtoms[va->m_iRealAtomType])->m_sName,va->m_iAtom+1);
			mprintf("      - Writing %s ...\n",buf);
			va->Dump(buf);
		}

		mprintf("\n    Molecule Information:\n");
		for (z=0;z<g_oaMolecules.GetSize();z++)
		{
			vmbase = NULL;
			for (z2=0;z2<m_oaVoroMolecules.GetSize();z2++)
			{
				vm = (CVoroMolecule*)m_oaVoroMolecules[z2];
				if (vm == NULL)
					continue;
				if (vm->m_iMolecule != z)
					continue;
				if (vmbase == NULL)
				{
					vmbase = vm;
				} else
				{
					vmbase->Add(vm);
					delete vm;
					m_oaVoroMolecules[z2] = NULL;
				}
			}
			sprintf(buf,"voronoi/voro_molecule_%s",((CMolecule*)g_oaMolecules[vmbase->m_iMolecule])->m_sName);
			mprintf("      - Writing %s ...\n",buf);
			vmbase->Dump(buf);
		}
		mprintf("\n    Mean Atom Neighbor Count Matrix:\n");
		mprintf("      Writing voro_atom_nbcount.csv ...\n");
		WriteNbAtomCountMatrix("voro_atom_nbcount.csv");

		mprintf("\n    Mean Atom Neighbor Distance Matrix:\n");
		mprintf("      Writing voro_atom_nbdist.csv ...\n");
		WriteNbAtomDistMatrix("voro_atom_nbdist.csv");

		mprintf("\n    Mean Molecule Neighbor Count Matrix:\n");
		mprintf("      Writing voro_molecule_nbcount.csv ...\n");
		WriteNbMoleculeCountMatrix("voro_molecule_nbcount.csv");

		mprintf("\n    Mean Molecule Neighbor Distance Matrix:\n");
		mprintf("      Writing voro_molecule_nbdist.csv ...\n");
		WriteNbMoleculeDistMatrix("voro_molecule_nbdist.csv");

		mprintf("\n    Atom information table:\n");
		mprintf("      Writing voro_atoms.csv ...\n");
		WriteAtomInfo("voro_atoms.csv");

		mprintf("\n    Molecule information table:\n");
		mprintf("      Writing voro_molecules.csv ...\n");
		WriteMoleculeInfo("voro_molecules.csv");
		mprintf("\n");
	}
}


CVoroAtom::CVoroAtom()
{
}


CVoroAtom::~CVoroAtom()
{
}


void CVoroAtom::InitAnalyses()
{
	int z;

	try { m_pSurfaceArea = new CDF(); } catch(...) { m_pSurfaceArea = NULL; }
	if (m_pSurfaceArea == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	m_pSurfaceArea->m_fMinVal = 0;
	m_pSurfaceArea->m_fMaxVal = m_pParent->m_fMaxSurf;
	m_pSurfaceArea->m_iResolution = 10000;
	m_pSurfaceArea->SetLabelX("Surface Area [A^2]");
	m_pSurfaceArea->SetLabelY("Occurrence");
	m_pSurfaceArea->Create();

	try { m_pExposedSurface = new CDF(); } catch(...) { m_pExposedSurface = NULL; }
	if (m_pExposedSurface == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	m_pExposedSurface->m_fMinVal = 0;
	m_pExposedSurface->m_fMaxVal = m_pParent->m_fMaxSurf;
	m_pExposedSurface->m_iResolution = 10000;
	m_pExposedSurface->SetLabelX("Exposed Surface Area [A^2]");
	m_pExposedSurface->SetLabelY("Occurrence");
	m_pExposedSurface->Create();

	try { m_pExposedSurfacePerc = new CDF(); } catch(...) { m_pExposedSurfacePerc = NULL; }
	if (m_pExposedSurfacePerc == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	m_pExposedSurfacePerc->m_fMinVal = 0;
	m_pExposedSurfacePerc->m_fMaxVal = 100.0;
	m_pExposedSurfacePerc->m_iResolution = 10000;
	m_pExposedSurfacePerc->SetLabelX("Exposed Surface Percentage");
	m_pExposedSurfacePerc->SetLabelY("Occurrence");
	m_pExposedSurfacePerc->Create();

	try { m_pVolume = new CDF(); } catch(...) { m_pVolume = NULL; }
	if (m_pVolume == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	m_pVolume->m_fMinVal = 0;
	m_pVolume->m_fMaxVal = m_pParent->m_fMaxVol;
	m_pVolume->m_iResolution = 10000;
	m_pVolume->SetLabelX("Volume [A^3]");
	m_pVolume->SetLabelY("Occurrence");
	m_pVolume->Create();

	try { m_pMaxRadius = new CDF(); } catch(...) { m_pMaxRadius = NULL; }
	if (m_pMaxRadius == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	m_pMaxRadius->m_fMinVal = 0;
	m_pMaxRadius->m_fMaxVal = g_fBoxX/4.0;
	m_pMaxRadius->m_iResolution = int(g_fBoxX*0.2);
	m_pMaxRadius->SetLabelX("Max. Radius [pm]");
	m_pMaxRadius->SetLabelY("Occurrence");
	m_pMaxRadius->Create();

	try { m_pAVRatio = new CDF(); } catch(...) { m_pAVRatio = NULL; }
	if (m_pAVRatio == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	m_pAVRatio->m_fMinVal = 0;
	m_pAVRatio->m_fMaxVal = 1.0;
	m_pAVRatio->m_iResolution = 1000;
	m_pAVRatio->SetLabelX("A/V Ratio");
	m_pAVRatio->SetLabelY("Occurrence");
	m_pAVRatio->Create();

	try { m_pFaces = new CDF(); } catch(...) { m_pFaces = NULL; }
	if (m_pFaces == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	m_pFaces->m_fMinVal = 0;
	m_pFaces->m_fMaxVal = 100;
	m_pFaces->m_iResolution = 101;
	m_pFaces->SetLabelX("Face Count");
	m_pFaces->SetLabelY("Occurrence");
	m_pFaces->Create();

	try { m_pFaceOrders = new CDF(); } catch(...) { m_pFaceOrders = NULL; }
	if (m_pFaceOrders == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	m_pFaceOrders->m_fMinVal = 0;
	m_pFaceOrders->m_fMaxVal = 50;
	m_pFaceOrders->m_iResolution = 51;
	m_pFaceOrders->SetLabelX("Face Order");
	m_pFaceOrders->SetLabelY("Occurrence");
	m_pFaceOrders->Create();

	try { m_pNbhAtoms = new CDF*[m_pParent->m_oaVoroAtoms.GetSize()]; } catch(...) { m_pNbhAtoms = NULL; }
	if (m_pNbhAtoms == NULL) NewException((double)m_pParent->m_oaVoroAtoms.GetSize()*sizeof(CDF*),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	for (z=0;z<m_pParent->m_oaVoroAtoms.GetSize();z++)
	{
		try { m_pNbhAtoms[z] = new CDF(); } catch(...) { m_pNbhAtoms[z] = NULL; }
		if (m_pNbhAtoms[z] == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		
		m_pNbhAtoms[z]->m_fMinVal = 0;
		m_pNbhAtoms[z]->m_fMaxVal = 50;
		m_pNbhAtoms[z]->m_iResolution = 51;
		m_pNbhAtoms[z]->SetLabelX("Neighbor Count");
		m_pNbhAtoms[z]->SetLabelY("Occurrence");
		m_pNbhAtoms[z]->Create();
	}

	try { m_pNbhDistAtoms = new CDF*[m_pParent->m_oaVoroAtoms.GetSize()]; } catch(...) { m_pNbhDistAtoms = NULL; }
	if (m_pNbhDistAtoms == NULL) NewException((double)m_pParent->m_oaVoroAtoms.GetSize()*sizeof(CDF*),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	for (z=0;z<m_pParent->m_oaVoroAtoms.GetSize();z++)
	{
		try { m_pNbhDistAtoms[z] = new CDF(); } catch(...) { m_pNbhDistAtoms[z] = NULL; }
		if (m_pNbhDistAtoms[z] == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		
		m_pNbhDistAtoms[z]->m_fMinVal = 0;
		m_pNbhDistAtoms[z]->m_fMaxVal = g_fBoxX*2.0;
		m_pNbhDistAtoms[z]->m_iResolution = int(g_fBoxX*0.5);
		m_pNbhDistAtoms[z]->SetLabelX("Neighbor Distance [pm]");
		m_pNbhDistAtoms[z]->SetLabelY("Occurrence");
		m_pNbhDistAtoms[z]->Create();
	}

	try { m_pNbhDistMolecules = new CDF*[g_oaMolecules.GetSize()]; } catch(...) { m_pNbhDistMolecules = NULL; }
	if (m_pNbhDistMolecules == NULL) NewException((double)g_oaMolecules.GetSize()*sizeof(CDF*),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	for (z=0;z<g_oaMolecules.GetSize();z++)
	{
		try { m_pNbhDistMolecules[z] = new CDF(); } catch(...) { m_pNbhDistMolecules[z] = NULL; }
		if (m_pNbhDistMolecules[z] == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		
		m_pNbhDistMolecules[z]->m_fMinVal = 0;
		m_pNbhDistMolecules[z]->m_fMaxVal = g_fBoxX*2.0;
		m_pNbhDistMolecules[z]->m_iResolution = int(g_fBoxX*0.5);
		m_pNbhDistMolecules[z]->SetLabelX("Neighbor Distance [pm]");
		m_pNbhDistMolecules[z]->SetLabelY("Occurrence");
		m_pNbhDistMolecules[z]->Create();
	}

	try { m_pNbhMolecules = new CDF*[g_oaMolecules.GetSize()]; } catch(...) { m_pNbhMolecules = NULL; }
	if (m_pNbhMolecules == NULL) NewException((double)g_oaMolecules.GetSize()*sizeof(CDF*),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	for (z=0;z<g_oaMolecules.GetSize();z++)
	{
		try { m_pNbhMolecules[z] = new CDF(); } catch(...) { m_pNbhMolecules[z] = NULL; }
		if (m_pNbhMolecules[z] == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		
		m_pNbhMolecules[z]->m_fMinVal = 0;
		m_pNbhMolecules[z]->m_fMaxVal = 50;
		m_pNbhMolecules[z]->m_iResolution = 51;
		m_pNbhMolecules[z]->SetLabelX("Neighbor Count");
		m_pNbhMolecules[z]->SetLabelY("Occurrence");
		m_pNbhMolecules[z]->Create();
	}
}


CVoroMolecule::CVoroMolecule()
{
}


CVoroMolecule::~CVoroMolecule()
{
}


void CVoroMolecule::InitAnalyses()
{
	int z;

	try { m_pSurfaceArea = new CDF(); } catch(...) { m_pSurfaceArea = NULL; }
	if (m_pSurfaceArea == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	m_pSurfaceArea->m_fMinVal = 0;
	m_pSurfaceArea->m_fMaxVal = m_pParent->m_fMaxSurf*((CMolecule*)g_oaMolecules[m_iMolecule])->m_iAtomGes;
	m_pSurfaceArea->m_iResolution = 1000;
	m_pSurfaceArea->SetLabelX("Surface Area [A^2]");
	m_pSurfaceArea->SetLabelY("Occurrence");
	m_pSurfaceArea->Create();

	try { m_pVolume = new CDF(); } catch(...) { m_pVolume = NULL; }
	if (m_pVolume == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	m_pVolume->m_fMinVal = 0;
	m_pVolume->m_fMaxVal = m_pParent->m_fMaxVol*((CMolecule*)g_oaMolecules[m_iMolecule])->m_iAtomGes;
	m_pVolume->m_iResolution = 1000;
	m_pVolume->SetLabelX("Volume [A^3]");
	m_pVolume->SetLabelY("Occurrence");
	m_pVolume->Create();

	try { m_pAVRatio = new CDF(); } catch(...) { m_pAVRatio = NULL; }
	if (m_pAVRatio == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	m_pAVRatio->m_fMinVal = 0;
	m_pAVRatio->m_fMaxVal = 1.0;
	m_pAVRatio->m_iResolution = 1000;
	m_pAVRatio->SetLabelX("A/V Ratio");
	m_pAVRatio->SetLabelY("Occurrence");
	m_pAVRatio->Create();

	try { m_pFaces = new CDF(); } catch(...) { m_pFaces = NULL; }
	if (m_pFaces == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	m_pFaces->m_fMinVal = 0;
	m_pFaces->m_fMaxVal = 100*((CMolecule*)g_oaMolecules[m_iMolecule])->m_iAtomGes;
	m_pFaces->m_iResolution = int(100*((CMolecule*)g_oaMolecules[m_iMolecule])->m_iAtomGes)+1;
	m_pFaces->SetLabelX("Face count");
	m_pFaces->SetLabelY("Occurrence");
	m_pFaces->Create();

	try { m_pNbhAtoms = new CDF*[m_pParent->m_oaVoroAtoms.GetSize()]; } catch(...) { m_pNbhAtoms = NULL; }
	if (m_pNbhAtoms == NULL) NewException((double)m_pParent->m_oaVoroAtoms.GetSize()*sizeof(CDF*),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	for (z=0;z<m_pParent->m_oaVoroAtoms.GetSize();z++)
	{
		try { m_pNbhAtoms[z] = new CDF(); } catch(...) { m_pNbhAtoms[z] = NULL; }
		if (m_pNbhAtoms[z] == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		
		m_pNbhAtoms[z]->m_fMinVal = 0;
		m_pNbhAtoms[z]->m_fMaxVal = 50;
		m_pNbhAtoms[z]->m_iResolution = 51;
		m_pNbhAtoms[z]->SetLabelX("Neighbor count");
		m_pNbhAtoms[z]->SetLabelY("Occurrence");
		m_pNbhAtoms[z]->Create();
	}

	try { m_pNbhDistAtoms = new CDF*[m_pParent->m_oaVoroAtoms.GetSize()]; } catch(...) { m_pNbhDistAtoms = NULL; }
	if (m_pNbhDistAtoms == NULL) NewException((double)m_pParent->m_oaVoroAtoms.GetSize()*sizeof(CDF*),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	for (z=0;z<m_pParent->m_oaVoroAtoms.GetSize();z++)
	{
		try { m_pNbhDistAtoms[z] = new CDF(); } catch(...) { m_pNbhDistAtoms[z] = NULL; }
		if (m_pNbhDistAtoms[z] == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		
		m_pNbhDistAtoms[z]->m_fMinVal = 0;
		m_pNbhDistAtoms[z]->m_fMaxVal = g_fBoxX*2.0;
		m_pNbhDistAtoms[z]->m_iResolution = int(g_fBoxX*0.5);
		m_pNbhDistAtoms[z]->SetLabelX("Neighbor distance [pm]");
		m_pNbhDistAtoms[z]->SetLabelY("Occurrence");
		m_pNbhDistAtoms[z]->Create();
	}

	try { m_pNbhDistMolecules = new CDF*[g_oaMolecules.GetSize()]; } catch(...) { m_pNbhDistMolecules = NULL; }
	if (m_pNbhDistMolecules == NULL) NewException((double)g_oaMolecules.GetSize()*sizeof(CDF*),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	for (z=0;z<g_oaMolecules.GetSize();z++)
	{
		try { m_pNbhDistMolecules[z] = new CDF(); } catch(...) { m_pNbhDistMolecules[z] = NULL; }
		if (m_pNbhDistMolecules[z] == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		
		m_pNbhDistMolecules[z]->m_fMinVal = 0;
		m_pNbhDistMolecules[z]->m_fMaxVal = g_fBoxX*2.0;
		m_pNbhDistMolecules[z]->m_iResolution = int(g_fBoxX*0.5);
		m_pNbhDistMolecules[z]->SetLabelX("Neighbor distance [pm]");
		m_pNbhDistMolecules[z]->SetLabelY("Occurrence");
		m_pNbhDistMolecules[z]->Create();
	}

	try { m_pNbhMolecules = new CDF*[g_oaMolecules.GetSize()]; } catch(...) { m_pNbhMolecules = NULL; }
	if (m_pNbhMolecules == NULL) NewException((double)g_oaMolecules.GetSize()*sizeof(CDF*),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	for (z=0;z<g_oaMolecules.GetSize();z++)
	{
		try { m_pNbhMolecules[z] = new CDF(); } catch(...) { m_pNbhMolecules[z] = NULL; }
		if (m_pNbhMolecules[z] == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		
		m_pNbhMolecules[z]->m_fMinVal = 0;
		m_pNbhMolecules[z]->m_fMaxVal = 50;
		m_pNbhMolecules[z]->m_iResolution = 51;
		m_pNbhMolecules[z]->SetLabelX("Neighbor count");
		m_pNbhMolecules[z]->SetLabelY("Occurrence");
		m_pNbhMolecules[z]->Create();
	}
}


void CVoroAtom::Dump(const char *s)
{
	FILE *a;
	char buf[256];
	int z, z2;

	sprintf(buf,"%s.txt",s);
	a = OpenFileWrite(buf,true);

	mfprintf(a,"*** Voronoi Analysis for %s %s%d ***\n\n",((CMolecule*)g_oaMolecules[m_iMolecule])->m_sName,((CAtom*)g_oaAtoms[m_iRealAtomType])->m_sName,m_iAtom+1);

	m_pVolume->CalcMeanSD();
	mfprintf(a,"  * Volume:\n");
	mfprintf(a,"      Avg. %10G Angstrom^3\n",m_pVolume->m_fMean);
	mfprintf(a,"      Min. %10G Angstrom^3\n",m_pVolume->m_fMinInput);
	mfprintf(a,"      Max. %10G Angstrom^3\n",m_pVolume->m_fMaxInput);
	mfprintf(a,"      SD   %10G Angstrom^3\n",m_pVolume->m_fSD);
	mfprintf(a,"      Saving Histogram as %s_volume.csv.\n",s);
	mfprintf(a,"        (%.0f bin entries, %.0f skipped)\n\n",m_pVolume->m_fBinEntries,m_pVolume->m_fSkipEntries);
	m_pVolume->NormBinSum(100.0);
	m_pVolume->Write("",s,"_volume.csv",true);

	m_pSurfaceArea->CalcMeanSD();
	mfprintf(a,"  * Surface Area:\n");
	mfprintf(a,"      Avg. %10G Angstrom^2\n",m_pSurfaceArea->m_fMean);
	mfprintf(a,"      Min. %10G Angstrom^2\n",m_pSurfaceArea->m_fMinInput);
	mfprintf(a,"      Max. %10G Angstrom^2\n",m_pSurfaceArea->m_fMaxInput);
	mfprintf(a,"      SD   %10G Angstrom^2\n",m_pSurfaceArea->m_fSD);
	mfprintf(a,"      Saving Histogram as %s_surface.csv.\n",s);
	mfprintf(a,"        (%.0f bin entries, %.0f skipped)\n\n",m_pSurfaceArea->m_fBinEntries,m_pSurfaceArea->m_fSkipEntries);
	m_pSurfaceArea->NormBinSum(100.0);
	m_pSurfaceArea->Write("",s,"_surface.csv",true);

	m_pExposedSurface->CalcMeanSD();
	mfprintf(a,"  * Exposed Surface Area:\n");
	mfprintf(a,"      Avg. %10G Angstrom^2\n",m_pExposedSurface->m_fMean);
	mfprintf(a,"      Min. %10G Angstrom^2\n",m_pExposedSurface->m_fMinInput);
	mfprintf(a,"      Max. %10G Angstrom^2\n",m_pExposedSurface->m_fMaxInput);
	mfprintf(a,"      SD   %10G Angstrom^2\n",m_pExposedSurface->m_fSD);
	mfprintf(a,"      Saving Histogram as %s_exposed_surface.csv.\n",s);
	mfprintf(a,"        (%.0f bin entries, %.0f skipped)\n\n",m_pExposedSurface->m_fBinEntries,m_pExposedSurface->m_fSkipEntries);
	m_pExposedSurface->NormBinSum(100.0);
	m_pExposedSurface->Write("",s,"_exposed_surface.csv",true);

	m_pExposedSurfacePerc->CalcMeanSD();
	mfprintf(a,"  * Exposed Surface Area Percentage:\n");
	mfprintf(a,"      Avg. %10G percent\n",m_pExposedSurfacePerc->m_fMean);
	mfprintf(a,"      Min. %10G percent\n",m_pExposedSurfacePerc->m_fMinInput);
	mfprintf(a,"      Max. %10G percent\n",m_pExposedSurfacePerc->m_fMaxInput);
	mfprintf(a,"      SD   %10G percent\n",m_pExposedSurfacePerc->m_fSD);
	mfprintf(a,"      Saving Histogram as %s_exposed_surface_perc.csv.\n",s);
	mfprintf(a,"        (%.0f bin entries, %.0f skipped)\n\n",m_pExposedSurfacePerc->m_fBinEntries,m_pExposedSurfacePerc->m_fSkipEntries);
	m_pExposedSurfacePerc->NormBinSum(100.0);
	m_pExposedSurfacePerc->Write("",s,"_exposed_surface_perc.csv",true);

	m_pMaxRadius->CalcMeanSD();
	mfprintf(a,"  * Maximum Radius:\n");
	mfprintf(a,"      Avg. %.4f pm\n",m_pMaxRadius->m_fMean);
	mfprintf(a,"      Min. %.4f pm\n",m_pMaxRadius->m_fMinInput);
	mfprintf(a,"      Max. %.4f pm\n",m_pMaxRadius->m_fMaxInput);
	mfprintf(a,"      SD   %.4f pm\n\n",m_pMaxRadius->m_fSD);
	mfprintf(a,"      Saving Histogram as %s_maxradius.csv.\n",s);
	mfprintf(a,"        (%.0f bin entries, %.0f skipped)\n\n",m_pMaxRadius->m_fBinEntries,m_pMaxRadius->m_fSkipEntries);
	m_pMaxRadius->NormBinSum(100.0);
	m_pMaxRadius->Write("",s,"_maxradius.csv",true);

	m_pFaceOrders->CalcMeanSD();
	mfprintf(a,"  * Face Orders:\n");
	mfprintf(a,"      Avg. %.6f Edges\n",m_pFaceOrders->m_fMean);
	mfprintf(a,"      Min. %.0f Edges\n",m_pFaceOrders->m_fMinInput);
	mfprintf(a,"      Max. %.0f Edges\n",m_pFaceOrders->m_fMaxInput);
	mfprintf(a,"      SD   %.6f Edges\n\n",m_pFaceOrders->m_fSD);
	m_pFaceOrders->NormBinSum(100.0);
	for (z=0;z<m_pFaceOrders->m_iResolution;z++)
		if (m_pFaceOrders->m_pBin[z] != 0)
			mfprintf(a,"      - %2d Edges: %10.6f percent\n",z,m_pFaceOrders->m_pBin[z]);
	mfprintf(a,"      Saving Histogram as %s_faceorders.csv.\n",s);
	mfprintf(a,"        (%.0f bin entries, %.0f skipped)\n\n",m_pFaceOrders->m_fBinEntries,m_pFaceOrders->m_fSkipEntries);
	m_pFaceOrders->Write_Int("",s,"_faceorders.csv");

	m_pFaces->CalcMeanSD();
	mfprintf(a,"  * Faces:\n");
	mfprintf(a,"      Avg. %.6f\n",m_pFaces->m_fMean);
	mfprintf(a,"      Min. %.0f\n",m_pFaces->m_fMinInput);
	mfprintf(a,"      Max. %.0f\n",m_pFaces->m_fMaxInput);
	mfprintf(a,"      SD   %.6f\n\n",m_pFaces->m_fSD);
	m_pFaces->NormBinSum(100.0);
	for (z=0;z<m_pFaces->m_iResolution;z++)
		if (m_pFaces->m_pBin[z] != 0)
			mfprintf(a,"      - %2d Faces: %10.6f percent\n",z,m_pFaces->m_pBin[z]);
	mfprintf(a,"      Saving Histogram as %s_faces.csv.\n",s);
	mfprintf(a,"        (%.0f bin entries, %.0f skipped)\n\n",m_pFaces->m_fBinEntries,m_pFaces->m_fSkipEntries);
	m_pFaces->Write_Int("",s,"_faces.csv");

	m_pAVRatio->CalcMeanSD();
	mfprintf(a,"  * A/V Ratio:\n");
	mfprintf(a,"      Avg. %.6f\n",m_pAVRatio->m_fMean);
	mfprintf(a,"      Min. %.6f\n",m_pAVRatio->m_fMinInput);
	mfprintf(a,"      Max. %.6f\n",m_pAVRatio->m_fMaxInput);
	mfprintf(a,"      SD   %.6f\n\n",m_pAVRatio->m_fSD);
	m_pAVRatio->NormBinSum(100.0);
	mfprintf(a,"      Saving Histogram as %s_avratio.csv.\n",s);
	mfprintf(a,"        (%.0f bin entries, %.0f skipped)\n\n",m_pAVRatio->m_fBinEntries,m_pAVRatio->m_fSkipEntries);
	m_pAVRatio->Write("",s,"_avratio.csv",false);

	mfprintf(a,"###### Neighboring Molecules ######\n\n");

	for (z=0;z<g_oaMolecules.GetSize();z++)
	{
		m_pNbhMolecules[z]->CalcMeanSD();
		mfprintf(a,"  * Neighboring %s molecules:\n",((CMolecule*)g_oaMolecules[z])->m_sName);
		mfprintf(a,"      Avg. %.6f\n",m_pNbhMolecules[z]->m_fMean);
		mfprintf(a,"      Min. %.0f\n",m_pNbhMolecules[z]->m_fMinInput);
		mfprintf(a,"      Max. %.0f\n",m_pNbhMolecules[z]->m_fMaxInput);
		mfprintf(a,"      SD   %.6f\n",m_pNbhMolecules[z]->m_fSD);
		m_pNbhMolecules[z]->NormBinSum(100.0);
		for (z2=0;z2<m_pNbhMolecules[z]->m_iResolution;z2++)
			if (m_pNbhMolecules[z]->m_pBin[z2] != 0)
				mfprintf(a,"      - %2d Molecules: %10.6f percent\n",z2,m_pNbhMolecules[z]->m_pBin[z2]);
		sprintf(buf,"_nbcount_mol_%s.csv",((CMolecule*)g_oaMolecules[z])->m_sName);
		mfprintf(a,"      Saving Histogram as %s%s.\n",s,buf);
		mfprintf(a,"        (%.0f bin entries, %.0f skipped)\n\n",m_pNbhMolecules[z]->m_fBinEntries,m_pNbhMolecules[z]->m_fSkipEntries);
		m_pNbhMolecules[z]->Write_Int("",s,buf);
	}

	mfprintf(a,"###### Neighboring Molecule Distances ######\n\n");

	for (z=0;z<g_oaMolecules.GetSize();z++)
	{
		m_pNbhDistMolecules[z]->CalcMeanSD();
		mfprintf(a,"  * Neighboring %s molecule distances:\n",((CMolecule*)g_oaMolecules[z])->m_sName);
		mfprintf(a,"      Avg. %.4f pm\n",m_pNbhDistMolecules[z]->m_fMean);
		mfprintf(a,"      Min. %.4f pm\n",m_pNbhDistMolecules[z]->m_fMinInput);
		mfprintf(a,"      Max. %.4f pm\n",m_pNbhDistMolecules[z]->m_fMaxInput);
		mfprintf(a,"      SD   %.4f pm\n",m_pNbhDistMolecules[z]->m_fSD);
		m_pNbhDistMolecules[z]->NormBinSum(100.0);
		sprintf(buf,"_nbdist_mol_%s.csv",((CMolecule*)g_oaMolecules[z])->m_sName);
		mfprintf(a,"      Saving Histogram as %s%s.\n",s,buf);
		mfprintf(a,"        (%.0f bin entries, %.0f skipped)\n\n",m_pNbhDistMolecules[z]->m_fBinEntries,m_pNbhDistMolecules[z]->m_fSkipEntries);
		m_pNbhDistMolecules[z]->Write("",s,buf,true);
	}

	mfprintf(a,"###### Neighboring Atoms ######\n\n");

	for (z=0;z<m_pParent->m_oaVoroAtoms.GetSize();z++)
	{
		sprintf(buf,"%s %s%d",((CMolecule*)g_oaMolecules[((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iMolecule])->m_sName,((CAtom*)g_oaAtoms[((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iRealAtomType])->m_sName,((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iAtom+1);
		m_pNbhAtoms[z]->CalcMeanSD();
		mfprintf(a,"  * Neighboring %s atoms:\n",buf);
		mfprintf(a,"      Avg. %.6f\n",m_pNbhAtoms[z]->m_fMean);
		mfprintf(a,"      Min. %.0f\n",m_pNbhAtoms[z]->m_fMinInput);
		mfprintf(a,"      Max. %.0f\n",m_pNbhAtoms[z]->m_fMaxInput);
		mfprintf(a,"      SD   %.6f\n",m_pNbhAtoms[z]->m_fSD);
		m_pNbhAtoms[z]->NormBinSum(100.0);
		for (z2=0;z2<m_pNbhAtoms[z]->m_iResolution;z2++)
			if (m_pNbhAtoms[z]->m_pBin[z2] != 0)
				mfprintf(a,"      - %2d Atoms: %10.6f percent\n",z2,m_pNbhAtoms[z]->m_pBin[z2]);
		sprintf(buf,"_nbcount_atom_%s_%s%d.csv",((CMolecule*)g_oaMolecules[((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iMolecule])->m_sName,((CAtom*)g_oaAtoms[((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iRealAtomType])->m_sName,((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iAtom+1);
		mfprintf(a,"      Saving Histogram as %s%s.\n",s,buf);
		mfprintf(a,"        (%.0f bin entries, %.0f skipped)\n\n",m_pNbhAtoms[z]->m_fBinEntries,m_pNbhAtoms[z]->m_fSkipEntries);
		m_pNbhAtoms[z]->Write_Int("",s,buf);
	}

	mfprintf(a,"###### Neighboring Atom Distances ######\n\n");

	for (z=0;z<m_pParent->m_oaVoroAtoms.GetSize();z++)
	{
		sprintf(buf,"%s %s%d",((CMolecule*)g_oaMolecules[((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iMolecule])->m_sName,((CAtom*)g_oaAtoms[((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iRealAtomType])->m_sName,((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iAtom+1);
		m_pNbhDistAtoms[z]->CalcMeanSD();
		mfprintf(a,"  * Neighboring %s atom distances:\n",buf);
		mfprintf(a,"      Avg. %.4f pm\n",m_pNbhDistAtoms[z]->m_fMean);
		mfprintf(a,"      Min. %.4f pm\n",m_pNbhDistAtoms[z]->m_fMinInput);
		mfprintf(a,"      Max. %.4f pm\n",m_pNbhDistAtoms[z]->m_fMaxInput);
		mfprintf(a,"      SD   %.4f pm\n",m_pNbhDistAtoms[z]->m_fSD);
		m_pNbhDistAtoms[z]->NormBinSum(100.0);
		sprintf(buf,"_nbdist_atom_%s_%s%d.csv",((CMolecule*)g_oaMolecules[((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iMolecule])->m_sName,((CAtom*)g_oaAtoms[((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iRealAtomType])->m_sName,((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iAtom+1);
		mfprintf(a,"      Saving Histogram as %s%s.\n",s,buf);
		mfprintf(a,"        (%.0f bin entries, %.0f skipped)\n\n",m_pNbhDistAtoms[z]->m_fBinEntries,m_pNbhDistAtoms[z]->m_fSkipEntries);
		m_pNbhDistAtoms[z]->Write("",s,buf,true);
	}

	fclose(a);
}

void CVoroMolecule::Dump(const char *s)
{
	FILE *a;
	char buf[256];
	int z, z2;

	sprintf(buf,"%s.txt",s);
	a = OpenFileWrite(buf,true);

	mfprintf(a,"*** Voronoi Analysis for Molecule %s ***\n\n",((CMolecule*)g_oaMolecules[m_iMolecule])->m_sName);

	m_pVolume->CalcMeanSD();
	mfprintf(a,"  * Volume:\n");
	mfprintf(a,"      Avg. %10G Angstrom^3\n",m_pVolume->m_fMean);
	mfprintf(a,"      Min. %10G Angstrom^3\n",m_pVolume->m_fMinInput);
	mfprintf(a,"      Max. %10G Angstrom^3\n",m_pVolume->m_fMaxInput);
	mfprintf(a,"      SD   %10G Angstrom^3\n",m_pVolume->m_fSD);
	mfprintf(a,"      Saving Histogram as %s_volume.csv.\n",s);
	mfprintf(a,"        (%.0f bin entries, %.0f skipped)\n\n",m_pVolume->m_fBinEntries,m_pVolume->m_fSkipEntries);
	m_pVolume->NormBinSum(100.0);
	m_pVolume->Write("",s,"_volume.csv",true);

	m_pSurfaceArea->CalcMeanSD();
	mfprintf(a,"  * Surface Area:\n");
	mfprintf(a,"      Avg. %10G Angstrom^2\n",m_pSurfaceArea->m_fMean);
	mfprintf(a,"      Min. %10G Angstrom^2\n",m_pSurfaceArea->m_fMinInput);
	mfprintf(a,"      Max. %10G Angstrom^2\n",m_pSurfaceArea->m_fMaxInput);
	mfprintf(a,"      SD   %10G Angstrom^2\n",m_pSurfaceArea->m_fSD);
	mfprintf(a,"      Saving Histogram as %s_surface.csv.\n",s);
	mfprintf(a,"        (%.0f bin entries, %.0f skipped)\n\n",m_pSurfaceArea->m_fBinEntries,m_pSurfaceArea->m_fSkipEntries);
	m_pSurfaceArea->NormBinSum(100.0);
	m_pSurfaceArea->Write("",s,"_surface.csv",true);

	m_pAVRatio->CalcMeanSD();
	mfprintf(a,"  * A/V Ratio:\n");
	mfprintf(a,"      Avg. %10G Angstrom^2\n",m_pAVRatio->m_fMean);
	mfprintf(a,"      Min. %10G Angstrom^2\n",m_pAVRatio->m_fMinInput);
	mfprintf(a,"      Max. %10G Angstrom^2\n",m_pAVRatio->m_fMaxInput);
	mfprintf(a,"      SD   %10G Angstrom^2\n",m_pAVRatio->m_fSD);
	mfprintf(a,"      Saving Histogram as %s_avratio.csv.\n",s);
	mfprintf(a,"        (%.0f bin entries, %.0f skipped)\n\n",m_pAVRatio->m_fBinEntries,m_pAVRatio->m_fSkipEntries);
	m_pAVRatio->NormBinSum(100.0);
	m_pAVRatio->Write("",s,"_avratio.csv",true);

	m_pFaces->CalcMeanSD();
	mfprintf(a,"  * Faces:\n");
	mfprintf(a,"      Avg. %.6f\n",m_pFaces->m_fMean);
	mfprintf(a,"      Min. %.0f\n",m_pFaces->m_fMinInput);
	mfprintf(a,"      Max. %.0f\n",m_pFaces->m_fMaxInput);
	mfprintf(a,"      SD   %.6f\n\n",m_pFaces->m_fSD);
	m_pFaces->NormBinSum(100.0);
	for (z=0;z<m_pFaces->m_iResolution;z++)
		if (m_pFaces->m_pBin[z] != 0)
			mfprintf(a,"      - %2d Faces: %10.6f percent\n",z,m_pFaces->m_pBin[z]);
	mfprintf(a,"      Saving Histogram as %s_faces.csv.\n",s);
	mfprintf(a,"        (%.0f bin entries, %.0f skipped)\n\n",m_pFaces->m_fBinEntries,m_pFaces->m_fSkipEntries);
	m_pFaces->Write_Int("",s,"_faces.csv");

	mfprintf(a,"###### Neighboring Molecules ######\n\n");

	for (z=0;z<g_oaMolecules.GetSize();z++)
	{
		m_pNbhMolecules[z]->CalcMeanSD();
		mfprintf(a,"  * Neighboring %s molecules:\n",((CMolecule*)g_oaMolecules[z])->m_sName);
		mfprintf(a,"      Avg. %.6f\n",m_pNbhMolecules[z]->m_fMean);
		mfprintf(a,"      Min. %.0f\n",m_pNbhMolecules[z]->m_fMinInput);
		mfprintf(a,"      Max. %.0f\n",m_pNbhMolecules[z]->m_fMaxInput);
		mfprintf(a,"      SD   %.6f\n",m_pNbhMolecules[z]->m_fSD);
		m_pNbhMolecules[z]->NormBinSum(100.0);
		for (z2=0;z2<m_pNbhMolecules[z]->m_iResolution;z2++)
			if (m_pNbhMolecules[z]->m_pBin[z2] != 0)
				mfprintf(a,"      - %2d Molecules: %10.6f percent\n",z2,m_pNbhMolecules[z]->m_pBin[z2]);
		sprintf(buf,"_nbcount_mol_%s.csv",((CMolecule*)g_oaMolecules[z])->m_sName);
		mfprintf(a,"      Saving Histogram as %s%s.\n",s,buf);
		mfprintf(a,"        (%.0f bin entries, %.0f skipped)\n\n",m_pNbhMolecules[z]->m_fBinEntries,m_pNbhMolecules[z]->m_fSkipEntries);
		m_pNbhMolecules[z]->Write_Int("",s,buf);
	}

	mfprintf(a,"###### Neighboring Molecule Distances ######\n\n");

	for (z=0;z<g_oaMolecules.GetSize();z++)
	{
		m_pNbhDistMolecules[z]->CalcMeanSD();
		mfprintf(a,"  * Neighboring %s molecule distances:\n",((CMolecule*)g_oaMolecules[z])->m_sName);
		mfprintf(a,"      Avg. %.4f pm\n",m_pNbhDistMolecules[z]->m_fMean);
		mfprintf(a,"      Min. %.4f pm\n",m_pNbhDistMolecules[z]->m_fMinInput);
		mfprintf(a,"      Max. %.4f pm\n",m_pNbhDistMolecules[z]->m_fMaxInput);
		mfprintf(a,"      SD   %.4f pm\n",m_pNbhDistMolecules[z]->m_fSD);
		m_pNbhDistMolecules[z]->NormBinSum(100.0);
		sprintf(buf,"_nbdist_mol_%s.csv",((CMolecule*)g_oaMolecules[z])->m_sName);
		mfprintf(a,"      Saving Histogram as %s%s.\n",s,buf);
		mfprintf(a,"        (%.0f bin entries, %.0f skipped)\n\n",m_pNbhDistMolecules[z]->m_fBinEntries,m_pNbhDistMolecules[z]->m_fSkipEntries);
		m_pNbhDistMolecules[z]->Write("",s,buf,true);
	}

	mfprintf(a,"###### Neighboring Atoms ######\n\n");

	for (z=0;z<m_pParent->m_oaVoroAtoms.GetSize();z++)
	{
		sprintf(buf,"%s %s%d",((CMolecule*)g_oaMolecules[((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iMolecule])->m_sName,((CAtom*)g_oaAtoms[((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iRealAtomType])->m_sName,((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iAtom+1);
		m_pNbhAtoms[z]->CalcMeanSD();
		mfprintf(a,"  * Neighboring %s atoms:\n",buf);
		mfprintf(a,"      Avg. %.6f\n",m_pNbhAtoms[z]->m_fMean);
		mfprintf(a,"      Min. %.0f\n",m_pNbhAtoms[z]->m_fMinInput);
		mfprintf(a,"      Max. %.0f\n",m_pNbhAtoms[z]->m_fMaxInput);
		mfprintf(a,"      SD   %.6f\n",m_pNbhAtoms[z]->m_fSD);
		m_pNbhAtoms[z]->NormBinSum(100.0);
		for (z2=0;z2<m_pNbhAtoms[z]->m_iResolution;z2++)
			if (m_pNbhAtoms[z]->m_pBin[z2] != 0)
				mfprintf(a,"      - %2d Atoms: %10.6f percent\n",z2,m_pNbhAtoms[z]->m_pBin[z2]);
		sprintf(buf,"_nbcount_atom_%s_%s%d.csv",((CMolecule*)g_oaMolecules[((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iMolecule])->m_sName,((CAtom*)g_oaAtoms[((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iRealAtomType])->m_sName,((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iAtom+1);
		mfprintf(a,"      Saving Histogram as %s%s.\n",s,buf);
		mfprintf(a,"        (%.0f bin entries, %.0f skipped)\n\n",m_pNbhAtoms[z]->m_fBinEntries,m_pNbhAtoms[z]->m_fSkipEntries);
		m_pNbhAtoms[z]->Write_Int("",s,buf);
	}

	mfprintf(a,"###### Neighboring Atom Distances ######\n\n");

	for (z=0;z<m_pParent->m_oaVoroAtoms.GetSize();z++)
	{
		sprintf(buf,"%s %s%d",((CMolecule*)g_oaMolecules[((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iMolecule])->m_sName,((CAtom*)g_oaAtoms[((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iRealAtomType])->m_sName,((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iAtom+1);
		m_pNbhDistAtoms[z]->CalcMeanSD();
		mfprintf(a,"  * Neighboring %s atom distances:\n",buf);
		mfprintf(a,"      Avg. %.4f pm\n",m_pNbhDistAtoms[z]->m_fMean);
		mfprintf(a,"      Min. %.4f pm\n",m_pNbhDistAtoms[z]->m_fMinInput);
		mfprintf(a,"      Max. %.4f pm\n",m_pNbhDistAtoms[z]->m_fMaxInput);
		mfprintf(a,"      SD   %.4f pm\n",m_pNbhDistAtoms[z]->m_fSD);
		m_pNbhDistAtoms[z]->NormBinSum(100.0);
		sprintf(buf,"_nbdist_atom_%s_%s%d.csv",((CMolecule*)g_oaMolecules[((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iMolecule])->m_sName,((CAtom*)g_oaAtoms[((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iRealAtomType])->m_sName,((CVoroAtom*)m_pParent->m_oaVoroAtoms[z])->m_iAtom+1);
		mfprintf(a,"      Saving Histogram as %s%s.\n",s,buf);
		mfprintf(a,"        (%.0f bin entries, %.0f skipped)\n\n",m_pNbhDistAtoms[z]->m_fBinEntries,m_pNbhDistAtoms[z]->m_fSkipEntries);
		m_pNbhDistAtoms[z]->Write("",s,buf,true);
	}

	fclose(a);
}

void CVoroMolecule::Add(CVoroMolecule *m)
{
	int z;

	for (z=0;z<m_pParent->m_oaVoroAtoms.GetSize();z++)
	{
		m_pNbhAtoms[z]->AddFrom(m->m_pNbhAtoms[z]);
		m_pNbhDistAtoms[z]->AddFrom(m->m_pNbhDistAtoms[z]);
	}
	for (z=0;z<g_oaMolecules.GetSize();z++)
	{
		m_pNbhMolecules[z]->AddFrom(m->m_pNbhMolecules[z]);
		m_pNbhDistMolecules[z]->AddFrom(m->m_pNbhDistMolecules[z]);
	}
	m_pVolume->AddFrom(m->m_pVolume);
	m_pSurfaceArea->AddFrom(m->m_pSurfaceArea);
	m_pAVRatio->AddFrom(m->m_pAVRatio);
}

void CVoroWrapper::WriteNbAtomCountMatrix(const char *s)
{
	FILE *a;
	int z, z2;
	CVoroAtom *va;

	a = OpenFileWrite(s,true);

	mfprintf(a,"Row=From, Column=To\n");
	mfprintf(a,"*");
	for (z=0;z<m_oaVoroAtoms.GetSize();z++)
	{
		va = (CVoroAtom*)m_oaVoroAtoms[z];
		mfprintf(a,";  %s %s%d",((CMolecule*)g_oaMolecules[va->m_iMolecule])->m_sName,((CAtom*)g_oaAtoms[va->m_iRealAtomType])->m_sName,va->m_iAtom+1);
	}
	mfprintf(a,"\n");
	
	for (z=0;z<m_oaVoroAtoms.GetSize();z++)
	{
		va = (CVoroAtom*)m_oaVoroAtoms[z];
		mfprintf(a,"%s %s%d",((CMolecule*)g_oaMolecules[va->m_iMolecule])->m_sName,((CAtom*)g_oaAtoms[va->m_iRealAtomType])->m_sName,va->m_iAtom+1);
		for (z2=0;z2<m_oaVoroAtoms.GetSize();z2++)
			mfprintf(a,";  %f",va->m_pNbhAtoms[z2]->m_fMean);
		mfprintf(a,"\n");
	}
	
	fclose(a);
}

void CVoroWrapper::WriteNbAtomDistMatrix(const char *s)
{
	FILE *a;
	int z, z2;
	CVoroAtom *va;

	a = OpenFileWrite(s,true);

	mfprintf(a,"Row=From, Column=To, all numbers in pm\n");
	mfprintf(a,"*");
	for (z=0;z<m_oaVoroAtoms.GetSize();z++)
	{
		va = (CVoroAtom*)m_oaVoroAtoms[z];
		mfprintf(a,";  %s %s%d",((CMolecule*)g_oaMolecules[va->m_iMolecule])->m_sName,((CAtom*)g_oaAtoms[va->m_iRealAtomType])->m_sName,va->m_iAtom+1);
	}
	mfprintf(a,"\n");
	
	for (z=0;z<m_oaVoroAtoms.GetSize();z++)
	{
		va = (CVoroAtom*)m_oaVoroAtoms[z];
		mfprintf(a,"%s %s%d",((CMolecule*)g_oaMolecules[va->m_iMolecule])->m_sName,((CAtom*)g_oaAtoms[va->m_iRealAtomType])->m_sName,va->m_iAtom+1);
		for (z2=0;z2<m_oaVoroAtoms.GetSize();z2++)
			mfprintf(a,";  %f",va->m_pNbhDistAtoms[z2]->m_fMean);
		mfprintf(a,"\n");
	}
	
	fclose(a);
}

void CVoroWrapper::WriteNbMoleculeCountMatrix(const char *s)
{
	FILE *a;
	int z, z2;
	CVoroMolecule *vm;

	a = OpenFileWrite(s,true);

	mfprintf(a,"Row=From, Column=To\n");
	mfprintf(a,"*");
	for (z=0;z<m_oaVoroMolecules.GetSize();z++)
	{
		vm = (CVoroMolecule*)m_oaVoroMolecules[z];
		if (vm == NULL)
			continue;
		mfprintf(a,";  %s",((CMolecule*)g_oaMolecules[vm->m_iMolecule])->m_sName);
	}
	mfprintf(a,"\n");
	
	for (z=0;z<m_oaVoroMolecules.GetSize();z++)
	{
		vm = (CVoroMolecule*)m_oaVoroMolecules[z];
		if (vm == NULL)
			continue;
		mfprintf(a,"%s",((CMolecule*)g_oaMolecules[vm->m_iMolecule])->m_sName);
		for (z2=0;z2<m_oaVoroMolecules.GetSize();z2++)
		{
			if (m_oaVoroMolecules[z2] == NULL)
				continue;
			mfprintf(a,";  %f",vm->m_pNbhMolecules[((CVoroMolecule*)m_oaVoroMolecules[z2])->m_iMolecule]->m_fMean);
		}
		mfprintf(a,"\n");
	}
	
	fclose(a);
}

void CVoroWrapper::WriteNbMoleculeDistMatrix(const char *s)
{
	FILE *a;
	int z, z2;
	CVoroMolecule *vm;

	a = OpenFileWrite(s,true);

	mfprintf(a,"Row=From, Column=To, all numbers in pm\n");
	mfprintf(a,"*");
	for (z=0;z<m_oaVoroMolecules.GetSize();z++)
	{
		vm = (CVoroMolecule*)m_oaVoroMolecules[z];
		if (vm == NULL)
			continue;
		mfprintf(a,";  %s",((CMolecule*)g_oaMolecules[vm->m_iMolecule])->m_sName);
	}
	mfprintf(a,"\n");
	
	for (z=0;z<m_oaVoroMolecules.GetSize();z++)
	{
		vm = (CVoroMolecule*)m_oaVoroMolecules[z];
		if (vm == NULL)
			continue;
		mfprintf(a,"%s",((CMolecule*)g_oaMolecules[vm->m_iMolecule])->m_sName);
		for (z2=0;z2<m_oaVoroMolecules.GetSize();z2++)
		{
			if (m_oaVoroMolecules[z2] == NULL)
				continue;
			mfprintf(a,";  %f",vm->m_pNbhDistMolecules[((CVoroMolecule*)m_oaVoroMolecules[z2])->m_iMolecule]->m_fMean);
		}
		mfprintf(a,"\n");
	}
	
	fclose(a);
}

void CVoroWrapper::WriteAtomInfo(const char *s)
{
	FILE *a;
	int z;
	CVoroAtom *va;

	a = OpenFileWrite(s,true);

	mfprintf(a,"Molecule;  Atom;  Mean Volume [A^3];  Mean Surface [A^2];  Mean Exposed Surface [A^2];  Mean Exposed Surface Percentage;  Mean Faces;  Mean Face Orders;  Mean Max Radius [pm];  Mean A/V Ratio\n");

	for (z=0;z<m_oaVoroAtoms.GetSize();z++)
	{
		va = (CVoroAtom*)m_oaVoroAtoms[z];
		mfprintf(a,"%s;  %s%d;  ",((CMolecule*)g_oaMolecules[va->m_iMolecule])->m_sName,((CAtom*)g_oaAtoms[va->m_iRealAtomType])->m_sName,va->m_iAtom+1);
		mfprintf(a,"%f;  ",va->m_pVolume->m_fMean);
		mfprintf(a,"%f;  ",va->m_pSurfaceArea->m_fMean);
		mfprintf(a,"%f;  ",va->m_pExposedSurface->m_fMean);
		mfprintf(a,"%f;  ",va->m_pExposedSurfacePerc->m_fMean);
		mfprintf(a,"%f;  ",va->m_pFaces->m_fMean);
		mfprintf(a,"%f;  ",va->m_pFaceOrders->m_fMean);
		mfprintf(a,"%f;  ",va->m_pMaxRadius->m_fMean);
		mfprintf(a,"%f\n",va->m_pAVRatio->m_fMean);
	}
	
	fclose(a);
}

void CVoroWrapper::WriteMoleculeInfo(const char *s)
{
	FILE *a;
	int z;
	CVoroMolecule *vm;

	a = OpenFileWrite(s,true);

	mfprintf(a,"Molecule;  Mean Volume [A^3];  Mean Surface [A^2];  Mean Faces;  Mean A/V Ratio\n");

	for (z=0;z<m_oaVoroMolecules.GetSize();z++)
	{
		vm = (CVoroMolecule*)m_oaVoroMolecules[z];
		if (vm == NULL)
			continue;
		mfprintf(a,"%s;  ",((CMolecule*)g_oaMolecules[vm->m_iMolecule])->m_sName);
		mfprintf(a,"%f;  ",vm->m_pVolume->m_fMean);
		mfprintf(a,"%f;  ",vm->m_pSurfaceArea->m_fMean);
		mfprintf(a,"%f;  ",vm->m_pFaces->m_fMean);
		mfprintf(a,"%f\n",vm->m_pAVRatio->m_fMean);
	}
	
	fclose(a);
}


/*void CVoroWrapper::WritePOV(CTimeStep *ts)
{
	char buf[256];
	CxVector3 vec, vec2, vec3;
	CMolecule *m;
	CSingleMolecule *sm;

	m = (CMolecule*)g_oaMolecules[m_iPOVMol];
	sm = (CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[m_iPOVSM]];

	sprintf(buf,"voro_%s_%d_%05d.pov",m->m_sName,m_iPOVSM+1,(int)(g_iSteps/g_iStride));

	if (m_bPOVRot)
	{
		vec = ts->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[g_iFixAtomType[0]])->GetAt(g_iFixAtom[0])];
		vec2 = ts->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[g_iFixAtomType[1]])->GetAt(g_iFixAtom[1])];
		vec3 = ts->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[g_iFixAtomType[2]])->GetAt(g_iFixAtom[2])];
		m_mPOVMat.MatUltra(vec2-vec, vec3-vec); 
//		g_pTempTimestep->Transform(mat);
	}

	vec = ts->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[g_iFixAtomType[0]])->GetAt(g_iFixAtom[0])] - CxVector3(g_fBoxX/2.0,g_fBoxY/2.0,g_fBoxZ/2.0);

	ts->CenterPos(vec);

	ts->FoldAtomsPositive();

	WritePOV(ts,buf,m_iPOVMol,m_iPOVSM);

	m_vPOVRotPos += m_vPOVRotInc;
}*/


void CVoroWrapper::WritePOVHeader(FILE *b)
{
	int z;
	CElement *el;
	
	mfprintf(b,"/*****************************************************************************\n");
	mfprintf(b,"********************* General Parameter Values *******************************\n");
	mfprintf(b,"*****************************************************************************/\n\n");


	if (m_bPOVFaces)
	{
		mfprintf(b,"/**** Faces ****/\n");

		mfprintf(b,"#declare face_draw          = true;\n");
		if (m_bPOVFaceColor)
			mfprintf(b,"//#declare face_color         = < 0.3, 0.3, 1.0 >;\n");
				else mfprintf(b,"#declare face_color         = < 0.3, 0.3, 1.0 >;\n");
//		mfprintf(b,"#declare face_trans         = %.6f;\n",m_fPOVFaceTrans);
		mfprintf(b,"#declare face_specular      = 0.5*(1.0-face_trans);\n");
		mfprintf(b,"#declare face_reflect       = 0;\n");
		mfprintf(b,"#declare face_ambient       = 0.2;\n");
		mfprintf(b,"#declare face_diffuse       = 0.9;\n");

		mfprintf(b,"\n");
	}

	if (m_bPOVEdges)
	{
		mfprintf(b,"/**** Edges ****/\n");

		mfprintf(b,"#declare edge_draw           = true;\n");
/*		if (m_bPOVFaces)
		{
			r_edge = 0.005;
			mfprintf(b,"#declare edge_color          = < 0.3, 0.3, 0.3, 0, %.6f >;\n",m_fPOVEdgeTrans);
		} else
		{
			r_edge = 0.005;
			mfprintf(b,"#declare edge_color          = < 0.4, 0.4, 0.6, 0, %.6f >;\n",m_fPOVEdgeTrans);
		}*/

		if (m_bPOVFaces)
			mfprintf(b,"#declare edge_color          = < 0.3, 0.3, 0.3, 0, edge_trans >;\n");
		else
			mfprintf(b,"#declare edge_color          = < 0.4, 0.4, 0.6, 0, edge_trans >;\n");

		mfprintf(b,"#declare edge_r              = 0.005;\n");
		mfprintf(b,"#declare edge_specular       = 1.0*(1.0-edge_trans);\n");
		mfprintf(b,"#declare edge_reflect        = 0;\n");
		mfprintf(b,"#declare edge_ambient        = 0.1;\n");
		mfprintf(b,"#declare edge_diffuse        = 0.9;\n");

		mfprintf(b,"\n");

		mfprintf(b,"#declare edge_draw_halo1     = false;\n");
		mfprintf(b,"#declare edge_r_halo1        = 0.008;\n");
		mfprintf(b,"#declare edge_l_halo1        = 0.006;\n");
		mfprintf(b,"#declare edge_d_halo1        = 0.2;\n");
		mfprintf(b,"#declare edge_color_halo1    = < 0, 0, 0, 0, edge_trans >;\n");
		mfprintf(b,"#declare edge_specular_halo1 = 0;\n");
		mfprintf(b,"#declare edge_reflect_halo1  = 0;\n");
		mfprintf(b,"#declare edge_ambient_halo1  = 1.0;\n");
		mfprintf(b,"#declare edge_diffuse_halo1  = 0;\n");

		mfprintf(b,"\n");

		mfprintf(b,"#declare edge_draw_halo2     = false;\n");
		mfprintf(b,"#declare edge_r_halo2        = 0.006;\n");
		mfprintf(b,"#declare edge_l_halo2        = 0.008;\n");
		mfprintf(b,"#declare edge_d_halo2        = 0.4;\n");
		mfprintf(b,"#declare edge_color_halo2    = < 1, 1, 1, 0, edge_trans >;\n");
		mfprintf(b,"#declare edge_specular_halo2 = 0;\n");
		mfprintf(b,"#declare edge_reflect_halo2  = 0;\n");
		mfprintf(b,"#declare edge_ambient_halo2  = 1.0;\n");
		mfprintf(b,"#declare edge_diffuse_halo2  = 0;\n");

		mfprintf(b,"\n");

		mfprintf(b,"#declare edge_draw_halo3     = false;\n");
		mfprintf(b,"#declare edge_r_halo3        = 0.004;\n");
		mfprintf(b,"#declare edge_l_halo3        = 0.012;\n");
		mfprintf(b,"#declare edge_d_halo3        = 0.401;\n");
		mfprintf(b,"#declare edge_color_halo3    = < 1, 1, 1, 0, 1.0-((1.0-0.5)*(1.0-edge_trans)) >;\n");
		mfprintf(b,"#declare edge_specular_halo3 = 0;\n");
		mfprintf(b,"#declare edge_reflect_halo3  = 0;\n");
		mfprintf(b,"#declare edge_ambient_halo3  = 1.0;\n");
		mfprintf(b,"#declare edge_diffuse_halo3  = 0;\n");

		mfprintf(b,"\n");
	}

	if (m_bPOVVertices)
	{
		mfprintf(b,"/**** Vertices ****/\n");
		mfprintf(b,"#declare vertex_draw         = true;\n");
		mfprintf(b,"#declare vertex_r            = r_edge+0.0015;\n");
		mfprintf(b,"#declare vertex_color        = < 0.2, 0.2, 0.2, 0, 0 >;\n");
		mfprintf(b,"#declare vertex_specular     = 0.6;\n");
		mfprintf(b,"#declare vertex_reflect      = 0;\n");
		mfprintf(b,"#declare vertex_ambient      = 0.2;\n");
		mfprintf(b,"#declare vertex_diffuse      = 0.8;\n");
		mfprintf(b,"\n");
	}

	if (m_bPOVAtoms)
	{
		mfprintf(b,"/**** Atoms ****/\n");

		mfprintf(b,"#declare atom_draw           = true;\n");
		mfprintf(b,"#declare atom_r              = 0.65;\n");
		if (m_bPOVFaces)
			mfprintf(b,"#declare atom_specular      = 0.4;\n");
				else mfprintf(b,"#declare atom_specular      = 0.7;\n");
		mfprintf(b,"#declare atom_reflect        = 0;\n");
		mfprintf(b,"#declare atom_ambient        = 0.2;\n");
		mfprintf(b,"#declare atom_diffuse        = 0.8;\n");

		mfprintf(b,"\n");

		mfprintf(b,"#declare atom_draw_halo1      = true;\n");
		mfprintf(b,"#declare atom_r_halo1         = 0.006;\n");
		mfprintf(b,"#declare atom_d_halo1         = 0.0125;\n");
		mfprintf(b,"#declare atom_color_halo1     = < 0, 0, 0, 0, 0 >;\n");
		mfprintf(b,"#declare atom_specular_halo1  = 0;\n");
		mfprintf(b,"#declare atom_reflect_halo1   = 0;\n");
		mfprintf(b,"#declare atom_ambient_halo1   = 1.0;\n");
		mfprintf(b,"#declare atom_diffuse_halo1   = 0;\n");

		mfprintf(b,"\n");

		mfprintf(b,"#declare atom_draw_halo2      = true;\n");
		mfprintf(b,"#declare atom_r_halo2         = 0.002;\n");
		mfprintf(b,"#declare atom_d_halo2         = 0.01875;\n");
		if (m_bPOVFaces)
			mfprintf(b,"#declare atom_color_halo2     = < 1.7, 1.7, 1.7, 0, 0 >;\n");
				else mfprintf(b,"#declare atom_color_halo2     = < 1, 1, 1, 0, 0 >;\n");
		mfprintf(b,"#declare atom_specular_halo2  = 0;\n");
		mfprintf(b,"#declare atom_reflect_halo2   = 0;\n");
		mfprintf(b,"#declare atom_ambient_halo2   = 1.0;\n");
		mfprintf(b,"#declare atom_diffuse_halo2   = 0;\n");

		mfprintf(b,"\n");

		mfprintf(b,"#declare atom_draw_halo3      = true;\n");
		mfprintf(b,"#declare atom_r_halo3         = 0.002;\n");
		mfprintf(b,"#declare atom_d_halo3         = 0.025;\n");
		if (m_bPOVFaces)
			mfprintf(b,"#declare atom_color_halo3     = < 1.7, 1.7, 1.7, 0, 0.5 >;\n");
				else mfprintf(b,"#declare atom_color_halo3     = < 1, 1, 1, 0, 0.5 >;\n");
		mfprintf(b,"#declare atom_specular_halo3  = 0;\n");
		mfprintf(b,"#declare atom_reflect_halo3   = 0;\n");
		mfprintf(b,"#declare atom_ambient_halo3   = 1.0;\n");
		mfprintf(b,"#declare atom_diffuse_halo3   = 0;\n");

		mfprintf(b,"\n");

		mfprintf(b,"/**** Bonds ****/\n");

		mfprintf(b,"#declare bond_draw        = true;\n");
		mfprintf(b,"#declare bond_r           = 0.015;\n");
		mfprintf(b,"#declare bond_specular    = 0.7;\n");
		mfprintf(b,"#declare bond_reflect     = 0;\n");
		mfprintf(b,"#declare bond_ambient     = 0.2;\n");
		mfprintf(b,"#declare bond_diffuse     = 0.7;\n");

		mfprintf(b,"\n");

		mfprintf(b,"#declare bond_draw_halo1      = true;\n");
		mfprintf(b,"#declare bond_r_halo1         = 0.006;\n");
		mfprintf(b,"#declare bond_d_halo1         = 0;\n");
		mfprintf(b,"#declare bond_color_halo1     = < 0, 0, 0, 0, 0 >;\n");
		mfprintf(b,"#declare bond_specular_halo1  = 0;\n");
		mfprintf(b,"#declare bond_reflect_halo1   = 0;\n");
		mfprintf(b,"#declare bond_ambient_halo1   = 1.0;\n");
		mfprintf(b,"#declare bond_diffuse_halo1   = 0;\n");

		mfprintf(b,"\n");

		mfprintf(b,"#declare bond_draw_halo2      = true;\n");
		mfprintf(b,"#declare bond_r_halo2         = 0.002;\n");
		mfprintf(b,"#declare bond_d_halo2         = 0.0286;\n");
		if (m_bPOVFaces)
			mfprintf(b,"#declare bond_color_halo2     = < 1.7, 1.7, 1.7, 0, 0 >;\n");
				else mfprintf(b,"#declare bond_color_halo2     = < 1, 1, 1, 0, 0 >;\n");
		mfprintf(b,"#declare bond_specular_halo2  = 0;\n");
		mfprintf(b,"#declare bond_reflect_halo2   = 0;\n");
		mfprintf(b,"#declare bond_ambient_halo2   = 1.0;\n");
		mfprintf(b,"#declare bond_diffuse_halo2   = 0;\n");

		mfprintf(b,"\n");

		mfprintf(b,"#declare bond_draw_halo3      = true;\n");
		mfprintf(b,"#declare bond_r_halo3         = 0.002;\n");
		mfprintf(b,"#declare bond_d_halo3         = 0.0333;\n");
		if (m_bPOVFaces)
			mfprintf(b,"#declare bond_color_halo3     = < 1.7, 1.7, 1.7, 0, 0.5 >;\n");
				else mfprintf(b,"#declare bond_color_halo3     = < 1, 1, 1, 0, 0.5 >;\n");
		mfprintf(b,"#declare bond_specular_halo3  = 0;\n");
		mfprintf(b,"#declare bond_reflect_halo3   = 0;\n");
		mfprintf(b,"#declare bond_ambient_halo3   = 1.0;\n");
		mfprintf(b,"#declare bond_diffuse_halo3   = 0;\n");

		mfprintf(b,"\n");

		mfprintf(b,"#declare bond_draw_stub      = true;\n");
		mfprintf(b,"#declare bond_r_stub         = 0.004;\n");
		mfprintf(b,"#declare bond_l_stub         = 0.004;\n");
		mfprintf(b,"#declare bond_color_stub     = < 0, 0, 0, 0, 0 >;\n");
		mfprintf(b,"#declare bond_specular_stub  = 0;\n");
		mfprintf(b,"#declare bond_reflect_stub   = 0;\n");
		mfprintf(b,"#declare bond_ambient_stub   = 1.0;\n");
		mfprintf(b,"#declare bond_diffuse_stub   = 0;\n");

		mfprintf(b,"\n");
	}

	if (m_bPOVDrawNeighbors)
	{
		mfprintf(b,"/**** Neighboring Atoms ****/\n");

		mfprintf(b,"#declare nbatom_draw        = true;\n");
		mfprintf(b,"#declare nbatom_r           = 0.35;\n");
		mfprintf(b,"#declare nbatom_specular    = 0.7*(1.0-nb_trans);\n");
		mfprintf(b,"#declare nbatom_reflect     = 0;\n");
		mfprintf(b,"#declare nbatom_ambient     = 0.2;\n");
		mfprintf(b,"#declare nbatom_diffuse     = 0.7;\n");
		mfprintf(b,"#declare nbhighlight_fac    = nb_highlight;\n");

		mfprintf(b,"\n");

		mfprintf(b,"#declare nbatom_draw_halo1      = true;\n");
		mfprintf(b,"#declare nbatom_r_halo1         = 0.006;\n");
		mfprintf(b,"#declare nbatom_d_halo1         = 0.0125;\n");
		mfprintf(b,"#declare nbatom_color_halo1     = < 0, 0, 0, 0, 1.0-((1.0-nb_trans)*(1.0-nb_trans)) >;\n");
		mfprintf(b,"#declare nbatom_specular_halo1  = 0;\n");
		mfprintf(b,"#declare nbatom_reflect_halo1   = 0;\n");
		mfprintf(b,"#declare nbatom_ambient_halo1   = 1.0;\n");
		mfprintf(b,"#declare nbatom_diffuse_halo1   = 0;\n");

		mfprintf(b,"\n");

		mfprintf(b,"#declare nbatom_draw_halo2      = true;\n");
		mfprintf(b,"#declare nbatom_r_halo2         = 0.0025;\n");
		mfprintf(b,"#declare nbatom_d_halo2         = 0.01875;\n");
		mfprintf(b,"#declare nbatom_color_halo2     = < 1, 1, 1, 0, 1.0-((1.0-nb_trans)*(1.0-nb_trans)) >;\n");
		mfprintf(b,"#declare nbatom_specular_halo2  = 0;\n");
		mfprintf(b,"#declare nbatom_reflect_halo2   = 0;\n");
		mfprintf(b,"#declare nbatom_ambient_halo2   = 1.0;\n");
		mfprintf(b,"#declare nbatom_diffuse_halo2   = 0;\n");

		mfprintf(b,"\n");

		mfprintf(b,"#declare nbatom_draw_halo3      = true;\n");
		mfprintf(b,"#declare nbatom_r_halo3         = 0.0025;\n");
		mfprintf(b,"#declare nbatom_d_halo3         = 0.025;\n");
		mfprintf(b,"#declare nbatom_color_halo3     = < 1, 1, 1, 0, 1.0-((1.0-0.5)*(1.0-(1.0-((1.0-nb_trans)*(1.0-nb_trans))))) >;\n");
		mfprintf(b,"#declare nbatom_specular_halo3  = 0;\n");
		mfprintf(b,"#declare nbatom_reflect_halo3   = 0;\n");
		mfprintf(b,"#declare nbatom_ambient_halo3   = 1.0;\n");
		mfprintf(b,"#declare nbatom_diffuse_halo3   = 0;\n");

		mfprintf(b,"\n");

		mfprintf(b,"/**** Neighboring Bonds ****/\n");

		mfprintf(b,"#declare nbbond_draw        = true;\n");
		mfprintf(b,"#declare nbbond_r           = 0.008;\n");
		mfprintf(b,"#declare nbbond_specular    = 0.7*(1.0-nb_trans);\n");
		mfprintf(b,"#declare nbbond_reflect     = 0;\n");
		mfprintf(b,"#declare nbbond_ambient     = 0.2;\n");
		mfprintf(b,"#declare nbbond_diffuse     = 0.7;\n");

		mfprintf(b,"\n");

		mfprintf(b,"#declare nbbond_draw_halo1      = true;\n");
		mfprintf(b,"#declare nbbond_r_halo1         = 0.006;\n");
		mfprintf(b,"#declare nbbond_d_halo1         = 0;\n");
		mfprintf(b,"#declare nbbond_color_halo1     = < 0, 0, 0, 0, 1.0-((1.0-nb_trans)*(1.0-nb_trans)) >;\n");
		mfprintf(b,"#declare nbbond_specular_halo1  = 0;\n");
		mfprintf(b,"#declare nbbond_reflect_halo1   = 0;\n");
		mfprintf(b,"#declare nbbond_ambient_halo1   = 1.0;\n");
		mfprintf(b,"#declare nbbond_diffuse_halo1   = 0;\n");

		mfprintf(b,"\n");

		mfprintf(b,"#declare nbbond_draw_halo2      = true;\n");
		mfprintf(b,"#declare nbbond_r_halo2         = 0.0025;\n");
		mfprintf(b,"#declare nbbond_d_halo2         = 0.0286;\n");
		mfprintf(b,"#declare nbbond_color_halo2     = < 1, 1, 1, 0, 1.0-((1.0-nb_trans)*(1.0-nb_trans)) >;\n");
		mfprintf(b,"#declare nbbond_specular_halo2  = 0;\n");
		mfprintf(b,"#declare nbbond_reflect_halo2   = 0;\n");
		mfprintf(b,"#declare nbbond_ambient_halo2   = 1.0;\n");
		mfprintf(b,"#declare nbbond_diffuse_halo2   = 0;\n");

		mfprintf(b,"\n");

		mfprintf(b,"#declare nbbond_draw_halo3      = true;\n");
		mfprintf(b,"#declare nbbond_r_halo3         = 0.0025;\n");
		mfprintf(b,"#declare nbbond_d_halo3         = 0.0333;\n");
		mfprintf(b,"#declare nbbond_color_halo3     = < 1, 1, 1, 0, 1.0-((1.0-0.5)*(1.0-(1.0-((1.0-nb_trans)*(1.0-nb_trans))))) >;\n");
		mfprintf(b,"#declare nbbond_specular_halo3  = 0;\n");
		mfprintf(b,"#declare nbbond_reflect_halo3   = 0;\n");
		mfprintf(b,"#declare nbbond_ambient_halo3   = 1.0;\n");
		mfprintf(b,"#declare nbbond_diffuse_halo3   = 0;\n");

		mfprintf(b,"\n");

		mfprintf(b,"#declare nbbond_draw_stub      = true;\n");
		mfprintf(b,"#declare nbbond_r_stub         = 0.001;\n");
		mfprintf(b,"#declare nbbond_l_stub         = 0.005;\n");
		mfprintf(b,"#declare nbbond_color_stub     = < 0, 0, 0, 0, 1.0-((1.0-nb_trans)*(1.0-nb_trans)) >;\n");
		mfprintf(b,"#declare nbbond_specular_stub  = 0;\n");
		mfprintf(b,"#declare nbbond_reflect_stub   = 0;\n");
		mfprintf(b,"#declare nbbond_ambient_stub   = 1.0;\n");
		mfprintf(b,"#declare nbbond_diffuse_stub   = 0;\n");

		mfprintf(b,"\n");

		mfprintf(b,"/**** Rays ****/\n");

		mfprintf(b,"#declare ray_draw            = true;\n");
		mfprintf(b,"#declare ray_color           = < 1.0, 0.2, 1.0, 0, ray_trans >;\n");
		mfprintf(b,"#declare ray_r               = 0.002;\n");
		mfprintf(b,"#declare ray_specular        = 0.4*(1.0-ray_trans);\n");
		mfprintf(b,"#declare ray_reflect         = 0;\n");
		mfprintf(b,"#declare ray_ambient         = 0.2;\n");
		mfprintf(b,"#declare ray_diffuse         = 0.9;\n");

		mfprintf(b,"\n");
		
		mfprintf(b,"#declare ray_draw_halo1      = true;\n");
		mfprintf(b,"#declare ray_r_halo1         = 0.004;\n");
		mfprintf(b,"#declare ray_d_halo1         = 0;\n");
		mfprintf(b,"#declare ray_color_halo1     = < 0, 0, 0, 0, 1.0-((1.0-ray_trans)*(1.0-ray_trans)) >;\n");
		mfprintf(b,"#declare ray_specular_halo1  = 0;\n");
		mfprintf(b,"#declare ray_reflect_halo1   = 0;\n");
		mfprintf(b,"#declare ray_ambient_halo1   = 1.0;\n");
		mfprintf(b,"#declare ray_diffuse_halo1   = 0;\n");

		mfprintf(b,"\n");
		
		mfprintf(b,"#declare ray_draw_halo2      = true;\n");
		mfprintf(b,"#declare ray_r_halo2         = 0.001;\n");
		mfprintf(b,"#declare ray_d_halo2         = 0.001;\n");
		mfprintf(b,"#declare ray_color_halo2     = < 1, 1, 1, 0, 1.0-((1.0-ray_trans)*(1.0-ray_trans)) >;\n");
		mfprintf(b,"#declare ray_specular_halo2  = 0;\n");
		mfprintf(b,"#declare ray_reflect_halo2   = 0;\n");
		mfprintf(b,"#declare ray_ambient_halo2   = 1.0;\n");
		mfprintf(b,"#declare ray_diffuse_halo2   = 0;\n");

		mfprintf(b,"\n");
		
		mfprintf(b,"#declare ray_draw_halo3      = true;\n");
		mfprintf(b,"#declare ray_r_halo3         = 0.001;\n");
		mfprintf(b,"#declare ray_d_halo3         = 0.0011;\n");
		mfprintf(b,"#declare ray_color_halo3     = < 1, 1, 1, 0, 1.0-((1.0-0.5)*(1.0-(1.0-((1.0-ray_trans)*(1.0-ray_trans))))) >;\n");
		mfprintf(b,"#declare ray_specular_halo3  = 0;\n");
		mfprintf(b,"#declare ray_reflect_halo3   = 0;\n");
		mfprintf(b,"#declare ray_ambient_halo3   = 1.0;\n");
		mfprintf(b,"#declare ray_diffuse_halo3   = 0;\n");

		mfprintf(b,"\n");
		
		mfprintf(b,"#declare ray_draw_stub       = true;\n");
		mfprintf(b,"#declare ray_r_stub          = 0.003;\n");
		mfprintf(b,"#declare ray_l_stub          = 0.004;\n");
		mfprintf(b,"#declare ray_color_stub      = < 0, 0, 0, 0, 1.0-((1.0-ray_trans)*(1.0-ray_trans)) >;\n");
		mfprintf(b,"#declare ray_specular_stub   = 0;\n");
		mfprintf(b,"#declare ray_reflect_stub    = 0;\n");
		mfprintf(b,"#declare ray_ambient_stub    = 1.0;\n");
		mfprintf(b,"#declare ray_diffuse_stub    = 0;\n");

		mfprintf(b,"\n");
	}

	mfprintf(b,"/**** Element Colors ****/\n");
	for (z=0;z<g_oaAtoms.GetSize();z++)
	{
		if (z == g_iVirtAtomType)
			continue;
		el = ((CAtom*)g_oaAtoms[z])->m_pElement;
		mfprintf(b,"#declare element_%s_color   = < %f, %f, %f, 0, 0 >;\n",el->m_sLabel,el->m_iColorR/255.0,el->m_iColorG/255.0,el->m_iColorB/255.0);
		mfprintf(b,"#declare nb_%s_color   = < %f, %f, %f, 0, nb_trans >;\n",el->m_sLabel,el->m_iColorR/255.0,el->m_iColorG/255.0,el->m_iColorB/255.0);
	}

	mfprintf(b,"\n/**** Element Radii ****/\n");
	for (z=0;z<g_oaAtoms.GetSize();z++)
	{
		if (z == g_iVirtAtomType)
			continue;
		el = ((CAtom*)g_oaAtoms[z])->m_pElement;
		mfprintf(b,"#declare element_%s_radius  = %f;\n",el->m_sLabel,el->m_fRadius/1000.0);
	}
	mfprintf(b,"\n");

	mfprintf(b,"#declare camera_dist = %f;\n",m_fPOVCameraDist);
	mfprintf(b,"#declare camera_x = 1.0 * camera_dist * sin(rot_angle) / zoom;\n");
	mfprintf(b,"#declare camera_y = 0.0 * camera_dist / zoom;\n");
	mfprintf(b,"#declare camera_z = 1.0 * camera_dist * cos(rot_angle) / zoom;\n");

	mfprintf(b,"\ncamera {\n");
	mfprintf(b,"	location < camera_x, camera_y, camera_z >\n");
	mfprintf(b,"	sky y\n");
	mfprintf(b,"	right -0.06*x*image_width/image_height\n");
	mfprintf(b,"	up 0.06*y\n");
	mfprintf(b,"	look_at < 0, 0, 0 >\n");
	mfprintf(b,"}\n");
	mfprintf(b,"\n");
	mfprintf(b,"// Solid colored Background\n");
	mfprintf(b,"background { rgb < 1.0, 1.0, 1.0 > }\n");
	mfprintf(b,"\n");

	mfprintf(b,"/***** Color Gradient Background ****/\n");
	mfprintf(b,"sky_sphere {\n");
	mfprintf(b,"  pigment {\n");
	mfprintf(b,"    gradient y\n");
	mfprintf(b,"    color_map {\n");
	mfprintf(b,"      [ 0 color rgb < 0.9, 0.9, 0.9 > ]\n");
	mfprintf(b,"      [ 1 color rgb < 0.85, 0.75, 1.0 > ]\n");
	mfprintf(b,"    }\n");
	mfprintf(b,"    scale 0.1\n");
	mfprintf(b,"    translate -0.05\n");
	mfprintf(b,"  }\n");
	mfprintf(b,"}\n\n");

	mfprintf(b,"// Two light sources\n");
//	mfprintf(b,"light_source { < -8, 20, 20 > color rgb 0.8 }\n");
//	mfprintf(b,"light_source { < 25, 12, 20 > color rgb 0.5 }\n\n");
	mfprintf(b,"light_source { < 21.54*sin(rot_angle-0.7297), 20, 21.54*cos(rot_angle-0.7297) > color rgb 0.8 }\n");
	mfprintf(b,"light_source { < 32.02*sin(rot_angle+0.547), 12, 32.02*cos(rot_angle+0.547) > color rgb 0.5 }\n\n");

	if (m_bWritePOVMovie)
	{
		if (strlen(m_sPOVText) != 0)
		{
			mfprintf(b,"text {\n");
			mfprintf(b,"  ttf \"courbd.ttf\" title_text 0.01, 0\n");
			mfprintf(b,"  pigment { rgb < 0.2, 0.2, 0.2 > } \n");
			mfprintf(b,"  no_shadow\n");
			mfprintf(b,"  scale 0.2*camera_dist/90.0/zoom*1.6\n");
			mfprintf(b,"  translate < -2.65*camera_dist/90.0/zoom*1.27, 1.90*camera_dist/90.0/zoom*1.27, 0 >\n");
			mfprintf(b,"  rotate < 0, rot_angle, 0 >\n");
			mfprintf(b,"}\n\n");
		}
	}

	mfprintf(b,"#macro m_face_color(col)\n");
	mfprintf(b,"  #ifdef(face_color)\n");
	mfprintf(b,"    < face_color.x, face_color.y, face_color.z, 0, face_trans >\n");
	mfprintf(b,"  #else\n");
	mfprintf(b,"    < (1.0-facebleach_val)*col.x+facebleach_val*facebleach_r, (1.0-facebleach_val)*col.y+facebleach_val*facebleach_g, (1.0-facebleach_val)*col.z+facebleach_val*facebleach_b, 0, face_trans >\n");
	mfprintf(b,"  #end\n");
	mfprintf(b,"#end\n\n");

	mfprintf(b,"/*****************************************************************************\n");
	mfprintf(b,"*****************************************************************************/\n");
}


void CVoroWrapper::WritePOV(CTimeStep *ts, const char *s)
{
	int mol, smol;
	FILE *a, *b;
	int ijk, q, z, z2, z3, z4, id, faces, /*co,*/ fc, ti, o, o2;
	CVoroAtom *va;
	CVoroMolecule *vm;
	CMolecule *m;
	CElement *el, *el2;
	CSingleMolecule *sm;
	bool n=false, n2=false;
	double *pp, tx, ty, tz, lb, angle;
	voronoicell_neighbor c;
	vector<int> nb;
	vector<int> fo;
	vector<double> fa;
	vector<int> fv;
	CxVector3 vec, vec1, vec2, vec3, vec1b, vec2b, vecA, vecB, vecC, vecD, cam;
	CxIntArray nbmol, nbsm;
	CxIntArray nbatoms;
	CMolBond *mb;
	float cr, cg, cb;
//	double r_edge = 0.005;
	char buf[256];

	mol = m_iPOVMol;
	smol = m_iPOVSM;

	m = (CMolecule*)g_oaMolecules[mol];
	sm = (CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[smol]];

	if (m_bPOVRot)
	{
		vec = ts->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[g_iFixAtomType[0]])->GetAt(g_iFixAtom[0])];
		vec2 = ts->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[g_iFixAtomType[1]])->GetAt(g_iFixAtom[1])];
		vec3 = ts->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[g_iFixAtomType[2]])->GetAt(g_iFixAtom[2])];
		m_mPOVMat.MatUltra(vec2-vec, vec3-vec); 
//		g_pTempTimestep->Transform(mat);
	}

	vec = ts->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[g_iFixAtomType[0]])->GetAt(g_iFixAtom[0])] - CxVector3(g_fBoxX/2.0,g_fBoxY/2.0,g_fBoxZ/2.0);

	ts->CenterPos(vec);

	ts->FoldAtomsPositive();

	if (m_bWritePOVMovie)
		sprintf(m_sPOVText,"t=%7.3fps",((float)g_iSteps)/g_iStride*g_fTimestepLength/1000.0);

	if (m_bWritePOVMovie)
	{
#ifdef TARGET_WINDOWS
		if (s == NULL)
		{
			if (m_iPOVThreads == 1)
			{
				sprintf(buf,"POV_Single\\voro_box_%05d.pov",g_pVoroWrapper->m_iPOVFrameCounter);
				mfprintf(m_fPOVBoxScript[m_iPOVThreadPos],"%s /NR /EXIT /RENDER voro_box_%05d.pov +w%d +h%d +fn +a0.3 -J0.0 +Oframe_%05d.png\n",m_sPOVExe,g_pVoroWrapper->m_iPOVFrameCounter,m_iPOVResX,m_iPOVResY,g_pVoroWrapper->m_iPOVFrameCounter);
			} else
			{
				sprintf(buf,"POV_Single\\%d\\voro_box_%05d.pov",m_iPOVThreadPos+1,g_pVoroWrapper->m_iPOVFrameCounter);
				mfprintf(m_fPOVBoxScript[m_iPOVThreadPos],"%s /NR /EXIT /RENDER voro_box_%05d.pov +w%d +h%d +fn +a0.3 -J0.0 +Oframe_%05d.png\n",m_sPOVExe,g_pVoroWrapper->m_iPOVFrameCounter,m_iPOVResX,m_iPOVResY,g_pVoroWrapper->m_iPOVFrameCounter);
			}
		} else
		{
			strcpy(buf,s);
			mfprintf(m_fPOVBoxScript[m_iPOVThreadPos],"%s /NR /EXIT /RENDER %s +w%d +h%d +fn +a0.3 -J0.0\n",m_sPOVExe,buf,m_iPOVResX,m_iPOVResY);
		}
#else
		if (s == NULL)
		{
			if (m_iPOVThreads == 1)
			{
				sprintf(buf,"POV_Single/voro_box_%05d.pov",g_pVoroWrapper->m_iPOVFrameCounter);
				mfprintf(m_fPOVBoxScript[m_iPOVThreadPos],"%s voro_box_%05d.pov +w%d +h%d +fn +a0.3 -d -J0.0 +Oframe_%05d.png\n",m_sPOVExe,g_pVoroWrapper->m_iPOVFrameCounter,m_iPOVResX,m_iPOVResY,g_pVoroWrapper->m_iPOVFrameCounter);
			} else
			{
				sprintf(buf,"POV_Single/%d/voro_box_%05d.pov",m_iPOVThreadPos+1,g_pVoroWrapper->m_iPOVFrameCounter);
				mfprintf(m_fPOVBoxScript[m_iPOVThreadPos],"%s voro_box_%05d.pov +w%d +h%d +fn +a0.3 -d -J0.0 +Oframe_%05d.png\n",m_sPOVExe,g_pVoroWrapper->m_iPOVFrameCounter,m_iPOVResX,m_iPOVResY,g_pVoroWrapper->m_iPOVFrameCounter);
			}
		} else
		{
			strcpy(buf,s);
			mfprintf(m_fPOVBoxScript[m_iPOVThreadPos],"%s %s +w%d +h%d +fn +a0.3 -d -J0.0\n",m_sPOVExe,buf,m_iPOVResX,m_iPOVResY);
		}
#endif
	} else
	{
		strcpy(buf,s);
#ifdef TARGET_WINDOWS
		b = OpenFileWrite("render_single.bat",true);
		mfprintf(b,"%s /NR /EXIT /RENDER %s +w%d +h%d +fn +a0.3 -J0.0\n",m_sPOVExe,buf,m_iPOVResX,m_iPOVResY);
#else
		b = OpenFileWrite("render_single",true);
		mfprintf(b,"%s %s +w%d +h%d +fn +a0.3 -d -J0.0\n",m_sPOVExe,buf,m_iPOVResX,m_iPOVResY);
#endif
		fclose(b);
	}

	if (m_bWritePOVMovie)
	{
		fflush(m_fPOVBoxScript[m_iPOVThreadPos]);

		m_iPOVThreadPos++;
		if (m_iPOVThreadPos >= m_iPOVThreads)
			m_iPOVThreadPos = 0;
	}

	try { m_pContainer = new container_periodic_poly(g_fBoxX/1000.0,0,g_fBoxY/1000.0,0,0,g_fBoxZ/1000.0,m_iBlocksX,m_iBlocksY,m_iBlocksZ,g_iVoroMemory); } catch(...) { m_pContainer = NULL; }
	if (m_pContainer == NULL) NewException((double)sizeof(container_periodic_poly),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	for (z=0;z<g_iGesAtomCount;z++)
		m_pContainer->put(z,ts->m_vaCoords[z][0]/1000.0,ts->m_vaCoords[z][1]/1000.0,ts->m_vaCoords[z][2]/1000.0,0.05);

	c_loop_all_periodic vl(*m_pContainer);

//	mfprintf(m_fPOVScript,"\"%s\" /EXIT /RENDER \"%s\" +w%d +h%d +fn +a0.3\n",m_sPOVExe,s,m_iPOVResX,m_iPOVResY);
//	fflush(m_fPOVScript);

	angle = m_fPOVAngle / 180.0 * Pi;

	lb = m_fPOVCameraDist;

	cam[0] = 1.0 * lb * sin(angle) / m_fPOVZoom;
	cam[1] = 0.0 * lb / m_fPOVZoom;
	cam[2] = 1.0 * lb * cos(angle) / m_fPOVZoom;

	b = OpenFileWrite(buf,true);

	mfprintf(b,"// Written by TRAVIS\n");
	mfprintf(b,"// See http://www.travis-analyzer.de\n\n");
	mfprintf(b,"#version 3.6; // Please use version 3.7 anyway\n");
	mfprintf(b,"\n");


	mfprintf(b,"/**********************************************************************************************\n");
	mfprintf(b,"*********************** Variable Values *******************************************************\n");
	mfprintf(b,"**********************************************************************************************/\n\n");

	mfprintf(b,"#declare rot_angle        = %.6f; // In radians, not degrees\n",angle);
	mfprintf(b,"#declare zoom             = %.6f;\n",m_fPOVZoom);

	if (m_bWritePOVMovie)
		mfprintf(b,"#declare title_text       = \"%s\";\n",m_sPOVText);

	mfprintf(b,"#declare face_trans       = %.6f;\n",m_fPOVFaceTrans);
	mfprintf(b,"#declare edge_trans       = %.6f;\n",m_fPOVEdgeTrans);
	mfprintf(b,"#declare nb_trans         = %.6f;\n",m_fPOVNbTrans);
	mfprintf(b,"#declare ray_trans        = %.6f;\n",m_fPOVRayTrans);
	mfprintf(b,"#declare nb_highlight     = %.6f;\n",m_fPOVNbHighlightFac);
	mfprintf(b,"#declare facebleach_val   = %.6f;\n",m_fPOVFaceBleach);
	mfprintf(b,"#declare facebleach_r     = %.6f;\n",m_vPOVFaceBleach[0]);
	mfprintf(b,"#declare facebleach_g     = %.6f;\n",m_vPOVFaceBleach[1]);
	mfprintf(b,"#declare facebleach_b     = %.6f;\n",m_vPOVFaceBleach[2]);


	mfprintf(b,"\n/**********************************************************************************************\n");
	mfprintf(b,"**********************************************************************************************/\n\n");

	mfprintf(b,"#include \"settings_single.inc\"\n");

	if (!m_bIncludeWritten)
	{
		m_bIncludeWritten = true;
		if (m_bWritePOVMovie)
		{
#ifdef TARGET_WINDOWS
			a = OpenFileWrite("POV_Single\\settings_single.inc",true);
#else
			a = OpenFileWrite("POV_Single/settings_single.inc",true);
#endif
		} else
			a = OpenFileWrite("settings_single.inc",true);

		WritePOVHeader(a);
		fclose(a);
	}

	mfprintf(b,"\n/****** Below this line, no changes should be necessary **********/\n\n\n");

//	mfprintf(b,"\nunion {\n");

	if (vl.start()) 
	{
		do 
		{
			if (m_pContainer->compute_cell(c,vl))
			{
//				co++;

				ijk=vl.ijk;
				q=vl.q;
				pp=m_pContainer->p[ijk]+m_pContainer->ps*q;

				id = m_pContainer->id[ijk][q];

				va = (CVoroAtom*)m_oaVoroAtoms[m_pAssignAtoms[id]];
				vm = (CVoroMolecule*)m_oaVoroMolecules[m_pAssignMolecules[id]];

				ti = 0;

				if ((vm->m_iMolecule == mol) && (vm->m_iSingleMol == smol))
				{
					c.neighbors(nb);
					faces = c.number_of_faces();
					c.face_vertices(fv);

					mfprintf(b,"\n// Voronoi cell %d\n",m_pContainer->id[vl.ijk][vl.q]);
//					mfprintf(b,"union {\n");

					for (z=0;z<faces;z++)
					{
				//		mprintf("Face %d/%d:\n",z+1,faces);
						fc = fv[ti];
				//		mprintf("  %d Vertices\n",fc);

						if (m_pAssignMolecules[id] != m_pAssignMolecules[nb[z]])
						{
							tx = 0;
							ty = 0;
							tz = 0;
							for (z2=0;z2<fc;z2++)
							{
					//			mprintf("  - Vertex %d/%d; Id=%d, ti=%d (%.6f, %.6f, %.6f) -> (%.6f, %.6f, %.6f)\n",z2+1,fc,fv[ti+z2+1]+1,ti,c.pts[fv[ti+z2+1]*3]*0.5,c.pts[fv[ti+z2+1]*3+1]*0.5,c.pts[fv[ti+z2+1]*3+2]*0.5,*pp+c.pts[fv[ti+z2+1]*3]*0.5,pp[1]+c.pts[fv[ti+z2+1]*3+1]*0.5,pp[2]+c.pts[fv[ti+z2+1]*3+2]*0.5);

								vec1 = CxVector3(*pp+c.pts[fv[ti+z2+1]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+z2+1]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+z2+1]*3+2]*0.5-g_fBoxZ/2000.0);

								if (m_bPOVRot)
									vec1 = m_mPOVMat * vec1;

								if (m_bPOVVertices)
								{
									mfprintf(b,"#if (vertex_draw)\n");
									mfprintf(b,"  sphere { < %.6f, %.6f, %.6f >, vertex_r\n",vec1[0],vec1[1],vec1[2]);
									mfprintf(b,"    pigment { rgbft vertex_color } finish { reflection vertex_reflect specular vertex_specular ambient vertex_ambient diffuse vertex_diffuse } no_shadow }\n");
									mfprintf(b,"#end\n");
								}

								if (m_bPOVEdges)
								{
									n = false;

									if (z2 < fc-1)
									{
										if ((pow(c.pts[fv[ti+z2+1]*3]-c.pts[fv[ti+z2+2]*3],2)+pow(c.pts[fv[ti+z2+1]*3+1]-c.pts[fv[ti+z2+2]*3+1],2)+pow(c.pts[fv[ti+z2+1]*3+2]-c.pts[fv[ti+z2+2]*3+2],2)) > 0.0000000001)
										{
											vec2 = CxVector3(*pp+c.pts[fv[ti+z2+2]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+z2+2]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+z2+2]*3+2]*0.5-g_fBoxZ/2000.0);
											n = true;
										}
									} else
									{
										if ((pow(c.pts[fv[ti+z2+1]*3]-c.pts[fv[ti+1]*3],2)+pow(c.pts[fv[ti+z2+1]*3+1]-c.pts[fv[ti+1]*3+1],2)+pow(c.pts[fv[ti+z2+1]*3+2]-c.pts[fv[ti+1]*3+2],2)) > 0.0000000001)
										{
											vec2 = CxVector3(*pp+c.pts[fv[ti+1]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+1]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+1]*3+2]*0.5-g_fBoxZ/2000.0);
											n = true;
										}
									}

									if (n)
									{
										if (m_fPOVEdgeTrans != 1.0f)
										{
											if (m_bPOVRot)
												vec2 = m_mPOVMat * vec2;

											mfprintf(b,"#if (edge_draw)\n");
											mfprintf(b,"  cylinder { < %.6f, %.6f, %.6f >, < %.6f, %.6f, %.6f >, edge_r\n",vec1[0],vec1[1],vec1[2],vec2[0],vec2[1],vec2[2]);
											mfprintf(b,"    pigment { rgbft edge_color } finish { reflection edge_reflect specular edge_specular ambient edge_ambient diffuse edge_diffuse } no_shadow }\n");
											mfprintf(b,"#end\n");

											vec3 = cam - (vec1 + vec2) / 2.0;
											vec3.Normalize();

											vec2b = vec2-vec1;
											vec2b.Normalize();
											vec1b = CrossP(vec3,vec2b);

											vec1b.Normalize();

							//				if (halo)
											{
												mfprintf(b,"#if (edge_draw_halo1)\n");
												mfprintf(b,"  triangle {\n    < ");
												mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1) - (%.6f)*edge_l_halo1 - (%.6f)*edge_d_halo1, ",vec1[0],vec1b[0],vec2b[0],vec3[0]);
												mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1) - (%.6f)*edge_l_halo1 - (%.6f)*edge_d_halo1, ",vec1[1],vec1b[1],vec2b[1],vec3[1]);
												mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1) - (%.6f)*edge_l_halo1 - (%.6f)*edge_d_halo1 >,\n",vec1[2],vec1b[2],vec2b[2],vec3[2]);
												mfprintf(b,"    < ");
												mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1) + (%.6f)*edge_l_halo1 - (%.6f)*edge_d_halo1, ",vec2[0],vec1b[0],vec2b[0],vec3[0]);
												mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1) + (%.6f)*edge_l_halo1 - (%.6f)*edge_d_halo1, ",vec2[1],vec1b[1],vec2b[1],vec3[1]);
												mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1) + (%.6f)*edge_l_halo1 - (%.6f)*edge_d_halo1 >,\n",vec2[2],vec1b[2],vec2b[2],vec3[2]);
												mfprintf(b,"    < ");
												mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1) + (%.6f)*edge_l_halo1 - (%.6f)*edge_d_halo1, ",vec2[0],vec1b[0],vec2b[0],vec3[0]);
												mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1) + (%.6f)*edge_l_halo1 - (%.6f)*edge_d_halo1, ",vec2[1],vec1b[1],vec2b[1],vec3[1]);
												mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1) + (%.6f)*edge_l_halo1 - (%.6f)*edge_d_halo1 >\n",vec2[2],vec1b[2],vec2b[2],vec3[2]);
												mfprintf(b,"    pigment { rgbft edge_color_halo1 } finish { reflection edge_reflect_halo1 specular edge_specular_halo1 ambient edge_ambient_halo1 diffuse edge_diffuse_halo1 } no_shadow no_reflection no_radiosity }\n");

												mfprintf(b,"  triangle {\n    < ");
												mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1) - (%.6f)*edge_l_halo1 - (%.6f)*edge_d_halo1, ",vec1[0],vec1b[0],vec2b[0],vec3[0]);
												mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1) - (%.6f)*edge_l_halo1 - (%.6f)*edge_d_halo1, ",vec1[1],vec1b[1],vec2b[1],vec3[1]);
												mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1) - (%.6f)*edge_l_halo1 - (%.6f)*edge_d_halo1 >,\n",vec1[2],vec1b[2],vec2b[2],vec3[2]);
												mfprintf(b,"    < ");
												mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1) - (%.6f)*edge_l_halo1 - (%.6f)*edge_d_halo1, ",vec1[0],vec1b[0],vec2b[0],vec3[0]);
												mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1) - (%.6f)*edge_l_halo1 - (%.6f)*edge_d_halo1, ",vec1[1],vec1b[1],vec2b[1],vec3[1]);
												mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1) - (%.6f)*edge_l_halo1 - (%.6f)*edge_d_halo1 >,\n",vec1[2],vec1b[2],vec2b[2],vec3[2]);
												mfprintf(b,"    < ");
												mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1) + (%.6f)*edge_l_halo1 - (%.6f)*edge_d_halo1, ",vec2[0],vec1b[0],vec2b[0],vec3[0]);
												mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1) + (%.6f)*edge_l_halo1 - (%.6f)*edge_d_halo1, ",vec2[1],vec1b[1],vec2b[1],vec3[1]);
												mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1) + (%.6f)*edge_l_halo1 - (%.6f)*edge_d_halo1 >\n",vec2[2],vec1b[2],vec2b[2],vec3[2]);
												mfprintf(b,"    pigment { rgbft edge_color_halo1 } finish { reflection edge_reflect_halo1 specular edge_specular_halo1 ambient edge_ambient_halo1 diffuse edge_diffuse_halo1 } no_shadow no_reflection no_radiosity }\n");
												mfprintf(b,"#end\n");


												mfprintf(b,"#if (edge_draw_halo2)\n");
												mfprintf(b,"  triangle {\n    < ");
												mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2) - (%.6f)*edge_l_halo2 - (%.6f)*edge_d_halo2, ",vec1[0],vec1b[0],vec2b[0],vec3[0]);
												mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2) - (%.6f)*edge_l_halo2 - (%.6f)*edge_d_halo2, ",vec1[1],vec1b[1],vec2b[1],vec3[1]);
												mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2) - (%.6f)*edge_l_halo2 - (%.6f)*edge_d_halo2 >,\n",vec1[2],vec1b[2],vec2b[2],vec3[2]);
												mfprintf(b,"    < ");
												mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2) + (%.6f)*edge_l_halo2 - (%.6f)*edge_d_halo2, ",vec2[0],vec1b[0],vec2b[0],vec3[0]);
												mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2) + (%.6f)*edge_l_halo2 - (%.6f)*edge_d_halo2, ",vec2[1],vec1b[1],vec2b[1],vec3[1]);
												mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2) + (%.6f)*edge_l_halo2 - (%.6f)*edge_d_halo2 >,\n",vec2[2],vec1b[2],vec2b[2],vec3[2]);
												mfprintf(b,"    < ");
												mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2) + (%.6f)*edge_l_halo2 - (%.6f)*edge_d_halo2, ",vec2[0],vec1b[0],vec2b[0],vec3[0]);
												mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2) + (%.6f)*edge_l_halo2 - (%.6f)*edge_d_halo2, ",vec2[1],vec1b[1],vec2b[1],vec3[1]);
												mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2) + (%.6f)*edge_l_halo2 - (%.6f)*edge_d_halo2 >\n",vec2[2],vec1b[2],vec2b[2],vec3[2]);
												mfprintf(b,"    pigment { rgbft edge_color_halo2 } finish { reflection edge_reflect_halo2 specular edge_specular_halo2 ambient edge_ambient_halo2 diffuse edge_diffuse_halo2 } no_shadow no_reflection no_radiosity }\n");

												mfprintf(b,"  triangle {\n    < ");
												mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2) - (%.6f)*edge_l_halo2 - (%.6f)*edge_d_halo2, ",vec1[0],vec1b[0],vec2b[0],vec3[0]);
												mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2) - (%.6f)*edge_l_halo2 - (%.6f)*edge_d_halo2, ",vec1[1],vec1b[1],vec2b[1],vec3[1]);
												mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2) - (%.6f)*edge_l_halo2 - (%.6f)*edge_d_halo2 >,\n",vec1[2],vec1b[2],vec2b[2],vec3[2]);
												mfprintf(b,"    < ");
												mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2) - (%.6f)*edge_l_halo2 - (%.6f)*edge_d_halo2, ",vec1[0],vec1b[0],vec2b[0],vec3[0]);
												mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2) - (%.6f)*edge_l_halo2 - (%.6f)*edge_d_halo2, ",vec1[1],vec1b[1],vec2b[1],vec3[1]);
												mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2) - (%.6f)*edge_l_halo2 - (%.6f)*edge_d_halo2 >,\n",vec1[2],vec1b[2],vec2b[2],vec3[2]);
												mfprintf(b,"    < ");
												mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2) + (%.6f)*edge_l_halo2 - (%.6f)*edge_d_halo2, ",vec2[0],vec1b[0],vec2b[0],vec3[0]);
												mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2) + (%.6f)*edge_l_halo2 - (%.6f)*edge_d_halo2, ",vec2[1],vec1b[1],vec2b[1],vec3[1]);
												mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2) + (%.6f)*edge_l_halo2 - (%.6f)*edge_d_halo2 >\n",vec2[2],vec1b[2],vec2b[2],vec3[2]);
												mfprintf(b,"    pigment { rgbft edge_color_halo2 } finish { reflection edge_reflect_halo2 specular edge_specular_halo2 ambient edge_ambient_halo2 diffuse edge_diffuse_halo2 } no_shadow no_reflection no_radiosity }\n");
												mfprintf(b,"#end\n");

												
												mfprintf(b,"#if (edge_draw_halo3)\n");
												mfprintf(b,"  triangle {\n    < ");
												mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2 + edge_r_halo3) - (%.6f)*edge_l_halo3 - (%.6f)*edge_d_halo3, ",vec1[0],vec1b[0],vec2b[0],vec3[0]);
												mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2 + edge_r_halo3) - (%.6f)*edge_l_halo3 - (%.6f)*edge_d_halo3, ",vec1[1],vec1b[1],vec2b[1],vec3[1]);
												mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2 + edge_r_halo3) - (%.6f)*edge_l_halo3 - (%.6f)*edge_d_halo3 >,\n",vec1[2],vec1b[2],vec2b[2],vec3[2]);
												mfprintf(b,"    < ");
												mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2 + edge_r_halo3) + (%.6f)*edge_l_halo3 - (%.6f)*edge_d_halo3, ",vec2[0],vec1b[0],vec2b[0],vec3[0]);
												mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2 + edge_r_halo3) + (%.6f)*edge_l_halo3 - (%.6f)*edge_d_halo3, ",vec2[1],vec1b[1],vec2b[1],vec3[1]);
												mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2 + edge_r_halo3) + (%.6f)*edge_l_halo3 - (%.6f)*edge_d_halo3 >,\n",vec2[2],vec1b[2],vec2b[2],vec3[2]);
												mfprintf(b,"    < ");
												mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2 + edge_r_halo3) + (%.6f)*edge_l_halo3 - (%.6f)*edge_d_halo3, ",vec2[0],vec1b[0],vec2b[0],vec3[0]);
												mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2 + edge_r_halo3) + (%.6f)*edge_l_halo3 - (%.6f)*edge_d_halo3, ",vec2[1],vec1b[1],vec2b[1],vec3[1]);
												mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2 + edge_r_halo3) + (%.6f)*edge_l_halo3 - (%.6f)*edge_d_halo3 >\n",vec2[2],vec1b[2],vec2b[2],vec3[2]);
												mfprintf(b,"    pigment { rgbft edge_color_halo3 } finish { reflection edge_reflect_halo3 specular edge_specular_halo3 ambient edge_ambient_halo3 diffuse edge_diffuse_halo3 } no_shadow no_reflection no_radiosity }\n");

												mfprintf(b,"  triangle {\n    < ");
												mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2 + edge_r_halo3) - (%.6f)*edge_l_halo3 - (%.6f)*edge_d_halo3, ",vec1[0],vec1b[0],vec2b[0],vec3[0]);
												mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2 + edge_r_halo3) - (%.6f)*edge_l_halo3 - (%.6f)*edge_d_halo3, ",vec1[1],vec1b[1],vec2b[1],vec3[1]);
												mfprintf(b,"%.6f + (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2 + edge_r_halo3) - (%.6f)*edge_l_halo3 - (%.6f)*edge_d_halo3 >,\n",vec1[2],vec1b[2],vec2b[2],vec3[2]);
												mfprintf(b,"    < ");
												mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2 + edge_r_halo3) - (%.6f)*edge_l_halo3 - (%.6f)*edge_d_halo3, ",vec1[0],vec1b[0],vec2b[0],vec3[0]);
												mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2 + edge_r_halo3) - (%.6f)*edge_l_halo3 - (%.6f)*edge_d_halo3, ",vec1[1],vec1b[1],vec2b[1],vec3[1]);
												mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2 + edge_r_halo3) - (%.6f)*edge_l_halo3 - (%.6f)*edge_d_halo3 >,\n",vec1[2],vec1b[2],vec2b[2],vec3[2]);
												mfprintf(b,"    < ");
												mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2 + edge_r_halo3) + (%.6f)*edge_l_halo3 - (%.6f)*edge_d_halo3, ",vec2[0],vec1b[0],vec2b[0],vec3[0]);
												mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2 + edge_r_halo3) + (%.6f)*edge_l_halo3 - (%.6f)*edge_d_halo3, ",vec2[1],vec1b[1],vec2b[1],vec3[1]);
												mfprintf(b,"%.6f - (%.6f)*(edge_r + edge_r_halo1 + edge_r_halo2 + edge_r_halo3) + (%.6f)*edge_l_halo3 - (%.6f)*edge_d_halo3 >\n",vec2[2],vec1b[2],vec2b[2],vec3[2]);
												mfprintf(b,"    pigment { rgbft edge_color_halo3 } finish { reflection edge_reflect_halo3 specular edge_specular_halo3 ambient edge_ambient_halo3 diffuse edge_diffuse_halo3 } no_shadow no_reflection no_radiosity }\n");
												mfprintf(b,"#end\n");
											}
										}
									}
								}

								tx += vec1[0];
								ty += vec1[1];
								tz += vec1[2];
							}

							if (m_bPOVFaces && (m_fPOVFaceTrans != 1.0f))
							{
								tx /= fc;
								ty /= fc;
								tz /= fc;

								if (m_bPOVDrawNeighbors && (m_fPOVRayTrans != 1.0f))
								{
									vec1 = ts->m_vaCoords[nb[z]];
									vec1 /= 1000.0;
									vec1 -= CxVector3(g_fBoxX/2000.0,g_fBoxY/2000.0,g_fBoxZ/2000.0);

									el = ((CAtom*)g_oaAtoms[g_waAtomRealElement[nb[z]]])->m_pElement;

									if (m_bPOVRot)
										vec1 = m_mPOVMat * vec1;

									vec2[0] = tx;
									vec2[1] = ty;
									vec2[2] = tz;

									mfprintf(b,"#if (ray_draw)\n");
									mfprintf(b,"  cylinder { <%.6f, %.6f, %.6f>, <%.6f, %.6f, %.6f>, ray_r\n",vec1[0],vec1[1],vec1[2],vec2[0],vec2[1],vec2[2]);
									mfprintf(b,"    pigment { rgbft ray_color } finish { reflection ray_reflect specular ray_specular ambient ray_ambient diffuse ray_diffuse } no_shadow }\n");
									mfprintf(b,"#end\n");

									vec3 = cam - (vec1 + vec2) / 2.0;
									vec3.Normalize();

									vec2b = vec2-vec1;
									vec1b = CrossP(vec3,vec2b);

									vec1b.Normalize();

					//				if (shadow)
									{
										mfprintf(b,"#if (ray_draw_halo1)\n");
										mfprintf(b,"  triangle {\n    < ");
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1) - (%.6f)*ray_d_halo1, ",vec1[0],vec1b[0],vec3[0]);
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1) - (%.6f)*ray_d_halo1, ",vec1[1],vec1b[1],vec3[1]);
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1) - (%.6f)*ray_d_halo1 >,\n",vec1[2],vec1b[2],vec3[2]);
										mfprintf(b,"    < ");
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1) - (%.6f)*ray_d_halo1, ",vec2[0],vec1b[0],vec3[0]);
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1) - (%.6f)*ray_d_halo1, ",vec2[1],vec1b[1],vec3[1]);
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1) - (%.6f)*ray_d_halo1 >,\n",vec2[2],vec1b[2],vec3[2]);
										mfprintf(b,"    < ");
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1) - (%.6f)*ray_d_halo1, ",vec2[0],vec1b[0],vec3[0]);
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1) - (%.6f)*ray_d_halo1, ",vec2[1],vec1b[1],vec3[1]);
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1) - (%.6f)*ray_d_halo1 >\n",vec2[2],vec1b[2],vec3[2]);
										mfprintf(b,"    pigment { rgbft ray_color_halo1 } finish { reflection ray_reflect_halo1 specular ray_specular_halo1 ambient ray_ambient_halo1 diffuse ray_diffuse_halo1 } no_shadow no_reflection no_radiosity }\n");

										mfprintf(b,"  triangle {\n    < ");
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1) - (%.6f)*ray_d_halo1, ",vec1[0],vec1b[0],vec3[0]);
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1) - (%.6f)*ray_d_halo1, ",vec1[1],vec1b[1],vec3[1]);
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1) - (%.6f)*ray_d_halo1 >,\n",vec1[2],vec1b[2],vec3[2]);
										mfprintf(b,"    < ");
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1) - (%.6f)*ray_d_halo1, ",vec1[0],vec1b[0],vec3[0]);
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1) - (%.6f)*ray_d_halo1, ",vec1[1],vec1b[1],vec3[1]);
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1) - (%.6f)*ray_d_halo1 >,\n",vec1[2],vec1b[2],vec3[2]);
										mfprintf(b,"    < ");
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1) - (%.6f)*ray_d_halo1, ",vec2[0],vec1b[0],vec3[0]);
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1) - (%.6f)*ray_d_halo1, ",vec2[1],vec1b[1],vec3[1]);
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1) - (%.6f)*ray_d_halo1 >\n",vec2[2],vec1b[2],vec3[2]);
										mfprintf(b,"    pigment { rgbft ray_color_halo1 } finish { reflection ray_reflect_halo1 specular ray_specular_halo1 ambient ray_ambient_halo1 diffuse ray_diffuse_halo1 } no_shadow no_reflection no_radiosity }\n");
										mfprintf(b,"#end\n");
									}

					//				if (halo)
									{
										mfprintf(b,"#if (ray_draw_halo2)\n");
										mfprintf(b,"  triangle {\n    < ");
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2) - (%.6f)*ray_d_halo2, ",vec1[0],vec1b[0],vec3[0]);
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2) - (%.6f)*ray_d_halo2, ",vec1[1],vec1b[1],vec3[1]);
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2) - (%.6f)*ray_d_halo2 >,\n",vec1[2],vec1b[2],vec3[2]);
										mfprintf(b,"    < ");
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2) - (%.6f)*ray_d_halo2, ",vec2[0],vec1b[0],vec3[0]);
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2) - (%.6f)*ray_d_halo2, ",vec2[1],vec1b[1],vec3[1]);
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2) - (%.6f)*ray_d_halo2 >,\n",vec2[2],vec1b[2],vec3[2]);
										mfprintf(b,"    < ");
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2) - (%.6f)*ray_d_halo2, ",vec2[0],vec1b[0],vec3[0]);
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2) - (%.6f)*ray_d_halo2, ",vec2[1],vec1b[1],vec3[1]);
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2) - (%.6f)*ray_d_halo2 >\n",vec2[2],vec1b[2],vec3[2]);
										mfprintf(b,"    pigment { rgbft ray_color_halo2 } finish { reflection ray_reflect_halo2 specular ray_specular_halo2 ambient ray_ambient_halo2 diffuse ray_diffuse_halo2 } no_shadow no_reflection no_radiosity }\n");

										mfprintf(b,"  triangle {\n    < ");
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2) - (%.6f)*ray_d_halo2, ",vec1[0],vec1b[0],vec3[0]);
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2) - (%.6f)*ray_d_halo2, ",vec1[1],vec1b[1],vec3[1]);
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2) - (%.6f)*ray_d_halo2 >,\n",vec1[2],vec1b[2],vec3[2]);
										mfprintf(b,"    < ");
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2) - (%.6f)*ray_d_halo2, ",vec1[0],vec1b[0],vec3[0]);
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2) - (%.6f)*ray_d_halo2, ",vec1[1],vec1b[1],vec3[1]);
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2) - (%.6f)*ray_d_halo2 >,\n",vec1[2],vec1b[2],vec3[2]);
										mfprintf(b,"    < ");
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2) - (%.6f)*ray_d_halo2, ",vec2[0],vec1b[0],vec3[0]);
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2) - (%.6f)*ray_d_halo2, ",vec2[1],vec1b[1],vec3[1]);
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2) - (%.6f)*ray_d_halo2 >\n",vec2[2],vec1b[2],vec3[2]);
										mfprintf(b,"    pigment { rgbft ray_color_halo2 } finish { reflection ray_reflect_halo2 specular ray_specular_halo2 ambient ray_ambient_halo2 diffuse ray_diffuse_halo2 } no_shadow no_reflection no_radiosity }\n");
										mfprintf(b,"#end\n");

										
										mfprintf(b,"#if (ray_draw_halo3)\n");
										mfprintf(b,"  triangle {\n    < ");
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2 + ray_r_halo3) - (%.6f)*ray_d_halo3, ",vec1[0],vec1b[0],vec3[0]);
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2 + ray_r_halo3) - (%.6f)*ray_d_halo3, ",vec1[1],vec1b[1],vec3[1]);
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2 + ray_r_halo3) - (%.6f)*ray_d_halo3 >,\n",vec1[2],vec1b[2],vec3[2]);
										mfprintf(b,"    < ");
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2 + ray_r_halo3) - (%.6f)*ray_d_halo3, ",vec2[0],vec1b[0],vec3[0]);
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2 + ray_r_halo3) - (%.6f)*ray_d_halo3, ",vec2[1],vec1b[1],vec3[1]);
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2 + ray_r_halo3) - (%.6f)*ray_d_halo3 >,\n",vec2[2],vec1b[2],vec3[2]);
										mfprintf(b,"    < ");
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2 + ray_r_halo3) - (%.6f)*ray_d_halo3, ",vec2[0],vec1b[0],vec3[0]);
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2 + ray_r_halo3) - (%.6f)*ray_d_halo3, ",vec2[1],vec1b[1],vec3[1]);
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2 + ray_r_halo3) - (%.6f)*ray_d_halo3 >\n",vec2[2],vec1b[2],vec3[2]);
										mfprintf(b,"    pigment { rgbft ray_color_halo3 } finish { reflection ray_reflect_halo3 specular ray_specular_halo3 ambient ray_ambient_halo3 diffuse ray_diffuse_halo3 } no_shadow no_reflection no_radiosity }\n");

										mfprintf(b,"  triangle {\n    < ");
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2 + ray_r_halo3) - (%.6f)*ray_d_halo3, ",vec1[0],vec1b[0],vec3[0]);
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2 + ray_r_halo3) - (%.6f)*ray_d_halo3, ",vec1[1],vec1b[1],vec3[1]);
										mfprintf(b,"%.6f + (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2 + ray_r_halo3) - (%.6f)*ray_d_halo3 >,\n",vec1[2],vec1b[2],vec3[2]);
										mfprintf(b,"    < ");
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2 + ray_r_halo3) - (%.6f)*ray_d_halo3, ",vec1[0],vec1b[0],vec3[0]);
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2 + ray_r_halo3) - (%.6f)*ray_d_halo3, ",vec1[1],vec1b[1],vec3[1]);
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2 + ray_r_halo3) - (%.6f)*ray_d_halo3 >,\n",vec1[2],vec1b[2],vec3[2]);
										mfprintf(b,"    < ");
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2 + ray_r_halo3) - (%.6f)*ray_d_halo3, ",vec2[0],vec1b[0],vec3[0]);
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2 + ray_r_halo3) - (%.6f)*ray_d_halo3, ",vec2[1],vec1b[1],vec3[1]);
										mfprintf(b,"%.6f - (%.6f)*(ray_r + ray_r_halo1 + ray_r_halo2 + ray_r_halo3) - (%.6f)*ray_d_halo3 >\n",vec2[2],vec1b[2],vec3[2]);
										mfprintf(b,"    pigment { rgbft ray_color_halo3 } finish { reflection ray_reflect_halo3 specular ray_specular_halo3 ambient ray_ambient_halo3 diffuse ray_diffuse_halo3 } no_shadow no_reflection no_radiosity }\n");
										mfprintf(b,"#end\n");
									}

					//				if (shadow)
									{
										vec3 = vec2 - vec1;
										vec3.Normalize();

	//				if (m_bPOVHighlightNbAtoms)
	//					mfprintf(b,"  sphere { <%.6f, %.6f, %.6f>, element_%s_radius*nbatom_r*%f\n",vec1[0],vec1[1],vec1[2],el->m_sLabel,n?1.5:1.0);
	//						else mfprintf(b,"  sphere { <%.6f, %.6f, %.6f>, element_%s_radius*nbatom_r\n",vec1[0],vec1[1],vec1[2],el->m_sLabel);

										mfprintf(b,"#if (ray_draw_stub)\n");
										mfprintf(b,"  cylinder { <%.6f, %.6f, %.6f>,\n    < ",vec1[0],vec1[1],vec1[2]);
										if (m_bPOVHighlightNbAtoms)
										{
											mfprintf(b,"%.6f + (%.6f) * ( (element_%s_radius * nbatom_r * nbhighlight_fac) - (element_%s_radius * nbatom_r * nbhighlight_fac - 0.5 * sqrt(4*element_%s_radius*element_%s_radius*nbhighlight_fac*nbhighlight_fac*nbatom_r*nbatom_r - 4*(ray_r+ray_r_stub)*(ray_r+ray_r_stub) )) + ray_l_stub ), ",vec1[0],vec3[0],el->m_sLabel,el->m_sLabel,el->m_sLabel,el->m_sLabel);
											mfprintf(b,"%.6f + (%.6f) * ( (element_%s_radius * nbatom_r * nbhighlight_fac) - (element_%s_radius * nbatom_r * nbhighlight_fac - 0.5 * sqrt(4*element_%s_radius*element_%s_radius*nbhighlight_fac*nbhighlight_fac*nbatom_r*nbatom_r - 4*(ray_r+ray_r_stub)*(ray_r+ray_r_stub) )) + ray_l_stub ), ",vec1[1],vec3[1],el->m_sLabel,el->m_sLabel,el->m_sLabel,el->m_sLabel);
											mfprintf(b,"%.6f + (%.6f) * ( (element_%s_radius * nbatom_r * nbhighlight_fac) - (element_%s_radius * nbatom_r * nbhighlight_fac - 0.5 * sqrt(4*element_%s_radius*element_%s_radius*nbhighlight_fac*nbhighlight_fac*nbatom_r*nbatom_r - 4*(ray_r+ray_r_stub)*(ray_r+ray_r_stub) )) + ray_l_stub ) >,\n",vec1[2],vec3[2],el->m_sLabel,el->m_sLabel,el->m_sLabel,el->m_sLabel);
										} else
										{
											mfprintf(b,"%.6f + (%.6f) * ( (element_%s_radius * nbatom_r) - (element_%s_radius * nbatom_r - 0.5 * sqrt(4*element_%s_radius*element_%s_radius*nbatom_r*nbatom_r - 4*(ray_r+ray_r_stub)*(ray_r+ray_r_stub) )) + ray_l_stub ), ",vec1[0],vec3[0],el->m_sLabel,el->m_sLabel,el->m_sLabel,el->m_sLabel);
											mfprintf(b,"%.6f + (%.6f) * ( (element_%s_radius * nbatom_r) - (element_%s_radius * nbatom_r - 0.5 * sqrt(4*element_%s_radius*element_%s_radius*nbatom_r*nbatom_r - 4*(ray_r+ray_r_stub)*(ray_r+ray_r_stub) )) + ray_l_stub ), ",vec1[1],vec3[1],el->m_sLabel,el->m_sLabel,el->m_sLabel,el->m_sLabel);
											mfprintf(b,"%.6f + (%.6f) * ( (element_%s_radius * nbatom_r) - (element_%s_radius * nbatom_r - 0.5 * sqrt(4*element_%s_radius*element_%s_radius*nbatom_r*nbatom_r - 4*(ray_r+ray_r_stub)*(ray_r+ray_r_stub) )) + ray_l_stub ) >,\n",vec1[2],vec3[2],el->m_sLabel,el->m_sLabel,el->m_sLabel,el->m_sLabel);
										}
										mfprintf(b,"    ray_r+ray_r_stub\n");
										mfprintf(b,"    pigment { rgbft ray_color_stub } finish { reflection ray_reflect_stub specular ray_specular_stub ambient ray_ambient_stub diffuse ray_diffuse_stub } no_shadow no_radiosity }\n");

										mfprintf(b,"  cylinder { <%.6f, %.6f, %.6f>,\n    < ",vec2[0],vec2[1],vec2[2]);
										mfprintf(b,"%.6f - (%.6f) * ray_l_stub, ",vec2[0],vec3[0]);
										mfprintf(b,"%.6f - (%.6f) * ray_l_stub, ",vec2[1],vec3[1]);
										mfprintf(b,"%.6f - (%.6f) * ray_l_stub >,",vec2[2],vec3[2]);
										mfprintf(b,"    ray_r+ray_r_stub\n");
										mfprintf(b,"    pigment { rgbft ray_color_stub } finish { reflection ray_reflect_stub specular ray_specular_stub ambient ray_ambient_stub diffuse ray_diffuse_stub } no_shadow no_radiosity }\n");
										mfprintf(b,"#end\n");
									}
								}

								mfprintf(b,"#if (face_draw)\n");
								mfprintf(b,"\n  mesh { \n");

								for (z2=0;z2<fc;z2++)
								{
									vec1 = CxVector3(*pp+c.pts[fv[ti+z2+1]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+z2+1]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+z2+1]*3+2]*0.5-g_fBoxZ/2000.0);

									if (z2 < fc-1)
										vec2 = CxVector3(*pp+c.pts[fv[ti+z2+2]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+z2+2]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+z2+2]*3+2]*0.5-g_fBoxZ/2000.0);
											else vec2 = CxVector3(*pp+c.pts[fv[ti+1]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+1]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+1]*3+2]*0.5-g_fBoxZ/2000.0);

									if (m_bPOVRot)
									{
										vec1 = m_mPOVMat * vec1;
										vec2 = m_mPOVMat * vec2;
									}

									mfprintf(b,"    triangle { <%.6f, %.6f, %.6f>, <%.6f, %.6f, %.6f>, <%.6f, %.6f, %.6f> }\n",tx,ty,tz,vec1[0],vec1[1],vec1[2],vec2[0],vec2[1],vec2[2]);
								}

								if (m_bPOVDrawNeighbors)
								{
									nbatoms.Add(nb[z]);
									for (z2=0;z2<nbmol.GetSize();z2++)
									{
										if ((nbmol[z2] == ((CVoroMolecule*)m_oaVoroMolecules[m_pAssignMolecules[nb[z]]])->m_iMolecule) && (nbsm[z2] == ((CVoroMolecule*)m_oaVoroMolecules[m_pAssignMolecules[nb[z]]])->m_iSingleMol))
											goto _nbdone;
									}
									nbmol.Add(((CVoroMolecule*)m_oaVoroMolecules[m_pAssignMolecules[nb[z]]])->m_iMolecule);
									nbsm.Add(((CVoroMolecule*)m_oaVoroMolecules[m_pAssignMolecules[nb[z]]])->m_iSingleMol);
_nbdone:;
								}

								if (m_bPOVFaceColor)
								{
									if (m_bPOVFaceColorRef)
										POVFindColor(id,&cr,&cg,&cb,0);
											else POVFindColor(nb[z],&cr,&cg,&cb,0);

									mfprintf(b,"    texture { pigment { rgbft m_face_color(<%.3f, %.3f, %.3f>) } finish { reflection face_reflect specular face_specular ambient face_ambient diffuse face_diffuse } } no_shadow\n",cr,cg,cb);
								} else mfprintf(b,"    texture { pigment { rgbft m_face_color(<1,1,1>) } finish { reflection face_reflect specular face_specular ambient face_ambient diffuse face_diffuse } } no_shadow\n");

								mfprintf(b,"  }\n\n");
								mfprintf(b,"#end\n");
							}
						}

						ti += fv[ti]+1;
					}

					mfprintf(b,"\n\n");
//					mfprintf(b,"}\n\n");

					if (m_bPOVAtoms)
					{
						vec1 = CxVector3(*pp-g_fBoxX/2000.0,pp[1]-g_fBoxY/2000.0,pp[2]-g_fBoxZ/2000.0);
						el = ((CAtom*)g_oaAtoms[va->m_iRealAtomType])->m_pElement;

						if (m_bPOVRot)
							vec1 = m_mPOVMat * vec1;

						mfprintf(b,"#if (atom_draw)\n");
						mfprintf(b,"// Particle %d\nsphere { < %.6f, %.6f, %.6f >, element_%s_radius*atom_r\n",m_pContainer->id[vl.ijk][vl.q],vec1[0],vec1[1],vec1[2],el->m_sLabel);

						if (m_bPOVAtomGrey)
						{
							z2 = (int)((el->m_iColorR + el->m_iColorG + el->m_iColorB) / 4.0);
							mfprintf(b,"  pigment { rgb < %f, %f, %f > } finish { reflection atom_reflect specular atom_specular ambient atom_ambient diffuse atom_diffuse } no_shadow }\n",z2/255.0,z2/255.0,z2/255.0);
						} else mfprintf(b,"  pigment { rgbft element_%s_color } finish { reflection atom_reflect specular atom_specular ambient atom_ambient diffuse atom_diffuse } no_shadow }\n",el->m_sLabel);
						mfprintf(b,"#end\n");

						vec2 = cam - vec1;
						vec2.Normalize();

		//				if (shadow)
						{
							mfprintf(b,"#if (atom_draw_halo1)\n");
							mfprintf(b,"  disc { < %.6f - (%.6f) * atom_d_halo1, %.6f - (%.6f) * atom_d_halo1, %.6f - (%.6f) * atom_d_halo1 >,\n",vec1[0],vec2[0],vec1[1],vec2[1],vec1[2],vec2[2]);
							mfprintf(b,"    < %.6f, %.6f, %.6f >, (%f * atom_r) + atom_r_halo1\n",vec2[0],vec2[1],vec2[2],el->m_fRadius/1000.0);
							mfprintf(b,"    pigment { rgbft atom_color_halo1 } finish { reflection atom_reflect_halo1 specular atom_specular_halo1 ambient atom_ambient_halo1 diffuse atom_diffuse_halo1 } no_shadow no_reflection no_radiosity }\n");
							mfprintf(b,"#end\n");
						}

		//				if (halo)
						{
							mfprintf(b,"#if (atom_draw_halo2)\n");
							mfprintf(b,"  disc { < %.6f - (%.6f) * atom_d_halo2, %.6f - (%.6f) * atom_d_halo2, %.6f - (%.6f) * atom_d_halo2 >,\n",vec1[0],vec2[0],vec1[1],vec2[1],vec1[2],vec2[2]);
							mfprintf(b,"    < %.6f, %.6f, %.6f >, (%f * atom_r) + atom_r_halo1 + atom_r_halo2\n",vec2[0],vec2[1],vec2[2],el->m_fRadius/1000.0);
							mfprintf(b,"    pigment { rgbft atom_color_halo2 } finish { reflection atom_reflect_halo2 specular atom_specular_halo2 ambient atom_ambient_halo2 diffuse atom_diffuse_halo2 } no_shadow no_reflection no_radiosity }\n");
							mfprintf(b,"#end\n");

							mfprintf(b,"#if (atom_draw_halo3)\n");
							mfprintf(b,"  disc { < %.6f - (%.6f) * atom_d_halo3, %.6f - (%.6f) * atom_d_halo3, %.6f - (%.6f) * atom_d_halo3 >,\n",vec1[0],vec2[0],vec1[1],vec2[1],vec1[2],vec2[2]);
							mfprintf(b,"    < %.6f, %.6f, %.6f >, (%f * atom_r) + atom_r_halo1 + atom_r_halo2 + atom_r_halo3\n",vec2[0],vec2[1],vec2[2],el->m_fRadius/1000.0);
							mfprintf(b,"    pigment { rgbft atom_color_halo3 } finish { reflection atom_reflect_halo3 specular atom_specular_halo3 ambient atom_ambient_halo3 diffuse atom_diffuse_halo3 } no_shadow no_reflection no_radiosity }\n");
							mfprintf(b,"#end\n");
						}
					}
				}
			}
		} while (vl.inc());
	}

	for (z2=0;z2<sm->m_oaBonds.GetSize();z2++)
	{
		mb = (CMolBond*)sm->m_oaBonds[z2];
		o = mb->m_iAtomOffset[0];
		o2 = mb->m_iAtomOffset[1];

		el = ((CAtom*)g_oaAtoms[g_waAtomRealElement[o]])->m_pElement;
		el2 = ((CAtom*)g_oaAtoms[g_waAtomRealElement[o2]])->m_pElement;

		vec1 = ts->m_vaCoords[o];
		vec1 /= 1000.0;
		vec1 -= CxVector3(g_fBoxX/2000.0,g_fBoxY/2000.0,g_fBoxZ/2000.0);

		vec2 = ts->m_vaCoords[o2];
		vec2 /= 1000.0;
		vec2 -= CxVector3(g_fBoxX/2000.0,g_fBoxY/2000.0,g_fBoxZ/2000.0);

		if ((vec1-vec2).GetLength() > 0.3)
			continue;

		if (m_bPOVRot)
		{
			vec1 = m_mPOVMat * vec1;
			vec2 = m_mPOVMat * vec2;
		}

//		mprintf("Mol %d, SM %d, Bond %d, O1=%d, O2=%d, (%f|%f|%f), (%f|%f|%f).\n",mol,smol,z2,o,o2,vec1[0],vec1[1],vec1[2],vec2[0],vec2[1],vec2[2]);

		vec3 = (vec1/el->m_fRadius + vec2/el2->m_fRadius) / (1.0/el->m_fRadius+1.0/el2->m_fRadius);

		mfprintf(b,"#if (bond_draw)\n");
		mfprintf(b,"  cylinder { < %.6f, %.6f, %.6f >, < %.6f, %.6f, %.6f >, bond_r\n",vec1[0],vec1[1],vec1[2],vec3[0],vec3[1],vec3[2]);
		mfprintf(b,"    pigment { rgbft element_%s_color } finish { reflection bond_reflect specular bond_specular ambient bond_ambient diffuse bond_diffuse } no_shadow }\n",el->m_sLabel);

		mfprintf(b,"  cylinder { < %.6f, %.6f, %.6f >, < %.6f, %.6f, %.6f >, bond_r\n",vec3[0],vec3[1],vec3[2],vec2[0],vec2[1],vec2[2]);
		mfprintf(b,"    pigment { rgbft element_%s_color } finish { reflection bond_reflect specular bond_specular ambient bond_ambient diffuse bond_diffuse } no_shadow }\n",el2->m_sLabel);
		mfprintf(b,"#end\n");

		vec3 = cam - (vec1 + vec2) / 2.0;
		vec3.Normalize();

		vec2b = vec2-vec1;
		vec1b = CrossP(vec3,vec2b);

		vec1b.Normalize();

//		if (shadow)
		{
			mfprintf(b,"#if (bond_draw_halo1)\n");
			mfprintf(b,"  triangle {\n    < ");
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1) - (%.6f) * bond_d_halo1, ",vec1[0],vec1b[0],vec3[0]);
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1) - (%.6f) * bond_d_halo1, ",vec1[1],vec1b[1],vec3[1]);
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1) - (%.6f) * bond_d_halo1 >,\n",vec1[2],vec1b[2],vec3[2]);
			mfprintf(b,"    < ");
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1) - (%.6f) * bond_d_halo1, ",vec2[0],vec1b[0],vec3[0]);
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1) - (%.6f) * bond_d_halo1, ",vec2[1],vec1b[1],vec3[1]);
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1) - (%.6f) * bond_d_halo1 >,\n",vec2[2],vec1b[2],vec3[2]);
			mfprintf(b,"    < ");
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1) - (%.6f) * bond_d_halo1, ",vec2[0],vec1b[0],vec3[0]);
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1) - (%.6f) * bond_d_halo1, ",vec2[1],vec1b[1],vec3[1]);
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1) - (%.6f) * bond_d_halo1 >\n",vec2[2],vec1b[2],vec3[2]);
			mfprintf(b,"    pigment { rgbft bond_color_halo1 } finish { reflection bond_reflect_halo1 specular bond_specular_halo1 ambient bond_ambient_halo1 diffuse bond_diffuse_halo1 } no_shadow no_reflection no_radiosity }\n");

			mfprintf(b,"  triangle {\n    < ");
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1) - (%.6f) * bond_d_halo1, ",vec1[0],vec1b[0],vec3[0]);
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1) - (%.6f) * bond_d_halo1, ",vec1[1],vec1b[1],vec3[1]);
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1) - (%.6f) * bond_d_halo1 >,\n",vec1[2],vec1b[2],vec3[2]);
			mfprintf(b,"    < ");
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1) - (%.6f) * bond_d_halo1, ",vec1[0],vec1b[0],vec3[0]);
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1) - (%.6f) * bond_d_halo1, ",vec1[1],vec1b[1],vec3[1]);
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1) - (%.6f) * bond_d_halo1 >,\n",vec1[2],vec1b[2],vec3[2]);
			mfprintf(b,"    < ");
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1) - (%.6f) * bond_d_halo1, ",vec2[0],vec1b[0],vec3[0]);
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1) - (%.6f) * bond_d_halo1, ",vec2[1],vec1b[1],vec3[1]);
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1) - (%.6f) * bond_d_halo1 >\n",vec2[2],vec1b[2],vec3[2]);
			mfprintf(b,"    pigment { rgbft bond_color_halo1 } finish { reflection bond_reflect_halo1 specular bond_specular_halo1 ambient bond_ambient_halo1 diffuse bond_diffuse_halo1 } no_shadow no_reflection no_radiosity }\n");
			mfprintf(b,"#end\n");
		}

//		if (halo)
		{
			mfprintf(b,"#if (bond_draw_halo2)\n");
			mfprintf(b,"  triangle {\n    < ");
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2) - (%.6f) * bond_d_halo2, ",vec1[0],vec1b[0],vec3[0]);
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2) - (%.6f) * bond_d_halo2, ",vec1[1],vec1b[1],vec3[1]);
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2) - (%.6f) * bond_d_halo2 >,\n",vec1[2],vec1b[2],vec3[2]);
			mfprintf(b,"    < ");
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2) - (%.6f) * bond_d_halo2, ",vec2[0],vec1b[0],vec3[0]);
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2) - (%.6f) * bond_d_halo2, ",vec2[1],vec1b[1],vec3[1]);
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2) - (%.6f) * bond_d_halo2 >,\n",vec2[2],vec1b[2],vec3[2]);
			mfprintf(b,"    < ");
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2) - (%.6f) * bond_d_halo2, ",vec2[0],vec1b[0],vec3[0]);
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2) - (%.6f) * bond_d_halo2, ",vec2[1],vec1b[1],vec3[1]);
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2) - (%.6f) * bond_d_halo2 >\n",vec2[2],vec1b[2],vec3[2]);
			mfprintf(b,"    pigment { rgbft bond_color_halo2 } finish { reflection bond_reflect_halo2 specular bond_specular_halo2 ambient bond_ambient_halo2 diffuse bond_diffuse_halo2 } no_shadow no_reflection no_radiosity }\n");

			mfprintf(b,"  triangle {\n    < ");
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2) - (%.6f) * bond_d_halo2, ",vec1[0],vec1b[0],vec3[0]);
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2) - (%.6f) * bond_d_halo2, ",vec1[1],vec1b[1],vec3[1]);
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2) - (%.6f) * bond_d_halo2 >,\n",vec1[2],vec1b[2],vec3[2]);
			mfprintf(b,"    < ");
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2) - (%.6f) * bond_d_halo2, ",vec1[0],vec1b[0],vec3[0]);
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2) - (%.6f) * bond_d_halo2, ",vec1[1],vec1b[1],vec3[1]);
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2) - (%.6f) * bond_d_halo2 >,\n",vec1[2],vec1b[2],vec3[2]);
			mfprintf(b,"    < ");
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2) - (%.6f) * bond_d_halo2, ",vec2[0],vec1b[0],vec3[0]);
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2) - (%.6f) * bond_d_halo2, ",vec2[1],vec1b[1],vec3[1]);
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2) - (%.6f) * bond_d_halo2 >\n",vec2[2],vec1b[2],vec3[2]);
			mfprintf(b,"    pigment { rgbft bond_color_halo2 } finish { reflection bond_reflect_halo2 specular bond_specular_halo2 ambient bond_ambient_halo2 diffuse bond_diffuse_halo2 } no_shadow no_reflection no_radiosity }\n");
			mfprintf(b,"#end\n");

			
			mfprintf(b,"#if (bond_draw_halo3)\n");
			mfprintf(b,"  triangle {\n    < ");
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2 + bond_r_halo3) - (%.6f) * bond_d_halo3, ",vec1[0],vec1b[0],vec3[0]);
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2 + bond_r_halo3) - (%.6f) * bond_d_halo3, ",vec1[1],vec1b[1],vec3[1]);
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2 + bond_r_halo3) - (%.6f) * bond_d_halo3 >,\n",vec1[2],vec1b[2],vec3[2]);
			mfprintf(b,"    < ");
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2 + bond_r_halo3) - (%.6f) * bond_d_halo3, ",vec2[0],vec1b[0],vec3[0]);
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2 + bond_r_halo3) - (%.6f) * bond_d_halo3, ",vec2[1],vec1b[1],vec3[1]);
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2 + bond_r_halo3) - (%.6f) * bond_d_halo3 >,\n",vec2[2],vec1b[2],vec3[2]);
			mfprintf(b,"    < ");
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2 + bond_r_halo3) - (%.6f) * bond_d_halo3, ",vec2[0],vec1b[0],vec3[0]);
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2 + bond_r_halo3) - (%.6f) * bond_d_halo3, ",vec2[1],vec1b[1],vec3[1]);
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2 + bond_r_halo3) - (%.6f) * bond_d_halo3 >\n",vec2[2],vec1b[2],vec3[2]);
			mfprintf(b,"    pigment { rgbft bond_color_halo3 } finish { reflection bond_reflect_halo3 specular bond_specular_halo3 ambient bond_ambient_halo3 diffuse bond_diffuse_halo3 } no_shadow no_reflection no_radiosity }\n");

			mfprintf(b,"  triangle {\n    < ");
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2 + bond_r_halo3) - (%.6f) * bond_d_halo3, ",vec1[0],vec1b[0],vec3[0]);
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2 + bond_r_halo3) - (%.6f) * bond_d_halo3, ",vec1[1],vec1b[1],vec3[1]);
			mfprintf(b,"%.6f + (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2 + bond_r_halo3) - (%.6f) * bond_d_halo3 >,\n",vec1[2],vec1b[2],vec3[2]);
			mfprintf(b,"    < ");
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2 + bond_r_halo3) - (%.6f) * bond_d_halo3, ",vec1[0],vec1b[0],vec3[0]);
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2 + bond_r_halo3) - (%.6f) * bond_d_halo3, ",vec1[1],vec1b[1],vec3[1]);
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2 + bond_r_halo3) - (%.6f) * bond_d_halo3 >,\n",vec1[2],vec1b[2],vec3[2]);
			mfprintf(b,"    < ");
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2 + bond_r_halo3) - (%.6f) * bond_d_halo3, ",vec2[0],vec1b[0],vec3[0]);
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2 + bond_r_halo3) - (%.6f) * bond_d_halo3, ",vec2[1],vec1b[1],vec3[1]);
			mfprintf(b,"%.6f - (%.6f) * (bond_r + bond_r_halo1 + bond_r_halo2 + bond_r_halo3) - (%.6f) * bond_d_halo3 >\n",vec2[2],vec1b[2],vec3[2]);
			mfprintf(b,"    pigment { rgbft bond_color_halo3 } finish { reflection bond_reflect_halo3 specular bond_specular_halo3 ambient bond_ambient_halo3 diffuse bond_diffuse_halo3 } no_shadow no_reflection no_radiosity }\n");
			mfprintf(b,"#end\n");
		}

//		if (shadow)
		{
			vec3 = vec2 - vec1;
			vec3.Normalize();

			mfprintf(b,"#if (bond_draw_stub)\n");
			mfprintf(b,"  cylinder {\n    < %.6f, %.6f, %.6f >,\n",vec1[0],vec1[1],vec1[2]);
			mfprintf(b,"    < %.6f + (%.6f) * ( (%.6f)*atom_r - ( (%.6f)*atom_r - 0.5*sqrt(4*(%.6f)*(%.6f)*atom_r*atom_r - 4*(bond_r+bond_r_stub)*(bond_r+bond_r_stub) ) ) + bond_l_stub ),\n",vec1[0],vec3[0],el->m_fRadius/1000.0,el->m_fRadius/1000.0,el->m_fRadius/1000.0,el->m_fRadius/1000.0);
			mfprintf(b,"      %.6f + (%.6f) * ( (%.6f)*atom_r - ( (%.6f)*atom_r - 0.5*sqrt(4*(%.6f)*(%.6f)*atom_r*atom_r - 4*(bond_r+bond_r_stub)*(bond_r+bond_r_stub) ) ) + bond_l_stub ),\n",vec1[1],vec3[1],el->m_fRadius/1000.0,el->m_fRadius/1000.0,el->m_fRadius/1000.0,el->m_fRadius/1000.0);
			mfprintf(b,"      %.6f + (%.6f) * ( (%.6f)*atom_r - ( (%.6f)*atom_r - 0.5*sqrt(4*(%.6f)*(%.6f)*atom_r*atom_r - 4*(bond_r+bond_r_stub)*(bond_r+bond_r_stub) ) ) + bond_l_stub ) >\n,",vec1[2],vec3[2],el->m_fRadius/1000.0,el->m_fRadius/1000.0,el->m_fRadius/1000.0,el->m_fRadius/1000.0);
			mfprintf(b,"    bond_r + bond_r_stub\n");
			mfprintf(b,"    pigment { rgbft bond_color_stub } finish { reflection bond_reflect_stub specular bond_specular_stub ambient bond_ambient_stub diffuse bond_diffuse_stub } no_shadow no_radiosity }\n");

			mfprintf(b,"  cylinder {\n    < %.6f, %.6f, %.6f >,\n",vec2[0],vec2[1],vec2[2]);
			mfprintf(b,"    < %.6f - (%.6f) * ( (%.6f)*atom_r - ( (%.6f)*atom_r - 0.5*sqrt(4*(%.6f)*(%.6f)*atom_r*atom_r - 4*(bond_r+bond_r_stub)*(bond_r+bond_r_stub) ) ) + bond_l_stub ),\n",vec2[0],vec3[0],el2->m_fRadius/1000.0,el2->m_fRadius/1000.0,el2->m_fRadius/1000.0,el2->m_fRadius/1000.0);
			mfprintf(b,"      %.6f - (%.6f) * ( (%.6f)*atom_r - ( (%.6f)*atom_r - 0.5*sqrt(4*(%.6f)*(%.6f)*atom_r*atom_r - 4*(bond_r+bond_r_stub)*(bond_r+bond_r_stub) ) ) + bond_l_stub ),\n",vec2[1],vec3[1],el2->m_fRadius/1000.0,el2->m_fRadius/1000.0,el2->m_fRadius/1000.0,el2->m_fRadius/1000.0);
			mfprintf(b,"      %.6f - (%.6f) * ( (%.6f)*atom_r - ( (%.6f)*atom_r - 0.5*sqrt(4*(%.6f)*(%.6f)*atom_r*atom_r - 4*(bond_r+bond_r_stub)*(bond_r+bond_r_stub) ) ) + bond_l_stub ) >\n,",vec2[2],vec3[2],el2->m_fRadius/1000.0,el2->m_fRadius/1000.0,el2->m_fRadius/1000.0,el2->m_fRadius/1000.0);
			mfprintf(b,"    bond_r + bond_r_stub\n");
			mfprintf(b,"    pigment { rgbft bond_color_stub } finish { reflection bond_reflect_stub specular bond_specular_stub ambient bond_ambient_stub diffuse bond_diffuse_stub } no_shadow no_radiosity }\n");
			mfprintf(b,"#end\n");
		}
	}


	if (m_bPOVDrawNeighbors && (m_fPOVNbTrans != 1.0f))
	{
		for (z=0;z<nbmol.GetSize();z++)
		{
			m = (CMolecule*)g_oaMolecules[nbmol[z]];
			sm = (CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[nbsm[z]]];

			for (z2=0;z2<m->m_baAtomIndex.GetSize();z2++)
			{
				if (m->m_baAtomIndex[z2] == g_iVirtAtomType)
					continue;

				el = ((CAtom*)g_oaAtoms[m->m_baAtomIndex[z2]])->m_pElement;
				for (z3=0;z3<((CxIntArray*)sm->m_oaAtomOffset[z2])->GetSize();z3++)
				{
					o = ((CxIntArray*)sm->m_oaAtomOffset[z2])->GetAt(z3);
					vec1 = ts->m_vaCoords[o];
					vec1 /= 1000.0;

					vec1 -= CxVector3(g_fBoxX/2000.0,g_fBoxY/2000.0,g_fBoxZ/2000.0);

					if (m_bPOVRot)
						vec1 = m_mPOVMat * vec1;

					if (m_bPOVHighlightNbAtoms)
					{
						n = false;
						for (z4=0;z4<nbatoms.GetSize();z4++)
						{
							if (nbatoms[z4] == o)
							{
								n = true;
								break;
							}
						}
					}

					if (m_bPOVNbColorFace)
					{
						if (m_bPOVHighlightNbAtoms)
							POVFindColor(o,&cr,&cg,&cb,n?0:0.3);
								else POVFindColor(o,&cr,&cg,&cb,0);
					} else
					{
						cr = el->m_iColorR / 255.0;
						cg = el->m_iColorG / 255.0;
						cb = el->m_iColorB / 255.0;
					}

					mfprintf(b,"#if (nbatom_draw)\n");
					if (m_bPOVHighlightNbAtoms && n)
						mfprintf(b,"  sphere { <%.6f, %.6f, %.6f>, element_%s_radius*nbatom_r*nbhighlight_fac\n",vec1[0],vec1[1],vec1[2],el->m_sLabel);
							else mfprintf(b,"  sphere { <%.6f, %.6f, %.6f>, element_%s_radius*nbatom_r\n",vec1[0],vec1[1],vec1[2],el->m_sLabel);

					if (m_bPOVNbColorFace && m_bPOVHighlightNbAtoms)
						mfprintf(b,"    pigment { rgbft < %f, %f, %f, 0, %.6f > } finish { reflection nbatom_reflect specular nbatom_specular ambient nbatom_ambient diffuse nbatom_diffuse } no_shadow }\n",cr,cg,cb,m_fPOVNbTrans);
							else mfprintf(b,"    pigment { rgbft nb_%s_color } finish { reflection nbatom_reflect specular nbatom_specular ambient nbatom_ambient diffuse nbatom_diffuse } no_shadow }\n",el->m_sLabel);
					mfprintf(b,"#end\n");

					vec2 = cam - vec1;
					vec2.Normalize();

	//				if (shadow)
					{
						mfprintf(b,"#if (nbatom_draw_halo1)\n");
						mfprintf(b,"  disc { < %.6f - (%.6f) * nbatom_d_halo1, %.6f - (%.6f) * nbatom_d_halo1, %.6f - (%.6f) * nbatom_d_halo1 >,\n",vec1[0],vec2[0],vec1[1],vec2[1],vec1[2],vec2[2]);
						if (m_bPOVHighlightNbAtoms && n)
							mfprintf(b,"    < %.6f, %.6f, %.6f >, (element_%s_radius*nbatom_r*nbhighlight_fac) + nbatom_r_halo1\n",vec2[0],vec2[1],vec2[2],el->m_sLabel);
								else mfprintf(b,"    < %.6f, %.6f, %.6f >, (element_%s_radius*nbatom_r) + nbatom_r_halo1\n",vec2[0],vec2[1],vec2[2],el->m_sLabel);
						mfprintf(b,"    pigment { rgbft nbatom_color_halo1 } finish { reflection nbatom_reflect_halo1 specular nbatom_specular_halo1 ambient nbatom_ambient_halo1 diffuse nbatom_diffuse_halo1 } no_shadow no_reflection no_radiosity }\n");
						mfprintf(b,"#end\n");
					}

	//				if (halo)
					{
						mfprintf(b,"#if (nbatom_draw_halo2)\n");
						mfprintf(b,"  disc { < %.6f - (%.6f) * nbatom_d_halo2, %.6f - (%.6f) * nbatom_d_halo2, %.6f - (%.6f) * nbatom_d_halo2 >,\n",vec1[0],vec2[0],vec1[1],vec2[1],vec1[2],vec2[2]);
						if (m_bPOVHighlightNbAtoms && n)
							mfprintf(b,"    < %.6f, %.6f, %.6f >, (element_%s_radius*nbatom_r*nbhighlight_fac) + nbatom_r_halo1 + nbatom_r_halo2\n",vec2[0],vec2[1],vec2[2],el->m_sLabel);
								else mfprintf(b,"    < %.6f, %.6f, %.6f >, (element_%s_radius*nbatom_r) + nbatom_r_halo1 + nbatom_r_halo2\n",vec2[0],vec2[1],vec2[2],el->m_sLabel);
						mfprintf(b,"    pigment { rgbft nbatom_color_halo2 } finish { reflection nbatom_reflect_halo2 specular nbatom_specular_halo2 ambient nbatom_ambient_halo2 diffuse nbatom_diffuse_halo2 } no_shadow no_reflection no_radiosity }\n");
						mfprintf(b,"#end\n");

						mfprintf(b,"#if (nbatom_draw_halo3)\n");
						mfprintf(b,"  disc { < %.6f - (%.6f) * nbatom_d_halo3, %.6f - (%.6f) * nbatom_d_halo3, %.6f - (%.6f) * nbatom_d_halo3 >,\n",vec1[0],vec2[0],vec1[1],vec2[1],vec1[2],vec2[2]);
						if (m_bPOVHighlightNbAtoms && n)
							mfprintf(b,"    < %.6f, %.6f, %.6f >, (element_%s_radius*nbatom_r*nbhighlight_fac) + nbatom_r_halo1 + nbatom_r_halo2 + nbatom_r_halo3\n",vec2[0],vec2[1],vec2[2],el->m_sLabel);
								else mfprintf(b,"    < %.6f, %.6f, %.6f >, (element_%s_radius*nbatom_r) + nbatom_r_halo1 + nbatom_r_halo2 + nbatom_r_halo3\n",vec2[0],vec2[1],vec2[2],el->m_sLabel);
						mfprintf(b,"    pigment { rgbft nbatom_color_halo3 } finish { reflection nbatom_reflect_halo3 specular nbatom_specular_halo3 ambient nbatom_ambient_halo3 diffuse nbatom_diffuse_halo3 } no_shadow no_reflection no_radiosity }\n");
						mfprintf(b,"#end\n");
					}
				}
			}

			for (z2=0;z2<sm->m_oaBonds.GetSize();z2++)
			{
				mb = (CMolBond*)sm->m_oaBonds[z2];
				o = mb->m_iAtomOffset[0];
				o2 = mb->m_iAtomOffset[1];

				el = ((CAtom*)g_oaAtoms[g_waAtomRealElement[o]])->m_pElement;
				el2 = ((CAtom*)g_oaAtoms[g_waAtomRealElement[o2]])->m_pElement;

				vec1 = ts->m_vaCoords[o];
				vec1 /= 1000.0;
				vec1 -= CxVector3(g_fBoxX/2000.0,g_fBoxY/2000.0,g_fBoxZ/2000.0);

				vec2 = ts->m_vaCoords[o2];
				vec2 /= 1000.0;
				vec2 -= CxVector3(g_fBoxX/2000.0,g_fBoxY/2000.0,g_fBoxZ/2000.0);

				if ((vec1-vec2).GetLength() > 0.3)
					continue;

				if (m_bPOVRot)
				{
					vec1 = m_mPOVMat * vec1;
					vec2 = m_mPOVMat * vec2;
				}

				vec3 = (vec1/el->m_fRadius + vec2/el2->m_fRadius) / (1.0/el->m_fRadius+1.0/el2->m_fRadius);

				if (m_bPOVNbColorFace)
				{
					if (m_bPOVHighlightNbAtoms)
					{
						n = false;
						for (z4=0;z4<nbatoms.GetSize();z4++)
						{
							if (nbatoms[z4] == o)
							{
								n = true;
								break;
							}
						}
						POVFindColor(o,&cr,&cg,&cb,n?0:0.3);
					} else POVFindColor(o,&cr,&cg,&cb,0);
				} else
				{
					cr = el->m_iColorR / 255.0;
					cg = el->m_iColorG / 255.0;
					cb = el->m_iColorB / 255.0;
				}

				mfprintf(b,"#if (nbbond_draw)\n");
				mfprintf(b,"  cylinder { < %.6f, %.6f, %.6f >, < %.6f, %.6f, %.6f >, nbbond_r\n",vec1[0],vec1[1],vec1[2],vec3[0],vec3[1],vec3[2]);

				if (m_bPOVNbColorFace && m_bPOVHighlightNbAtoms)
					mfprintf(b,"    pigment { rgbft < %f, %f, %f, 0, %.6f > } finish { reflection nbbond_reflect specular nbbond_specular ambient nbbond_ambient diffuse nbbond_diffuse } no_shadow }\n",cr,cg,cb,m_fPOVNbTrans);
						else mfprintf(b,"    pigment { rgbft nb_%s_color } finish { reflection nbbond_reflect specular nbbond_specular ambient nbbond_ambient diffuse nbbond_diffuse } no_shadow }\n",el->m_sLabel);

				if (m_bPOVNbColorFace)
				{
					if (m_bPOVHighlightNbAtoms)
					{
						n = false;
						for (z4=0;z4<nbatoms.GetSize();z4++)
						{
							if (nbatoms[z4] == o2)
							{
								n = true;
								break;
							}
						}
						POVFindColor(o2,&cr,&cg,&cb,n?0:0.3);
					} else POVFindColor(o2,&cr,&cg,&cb,0);
				} else
				{
					cr = el2->m_iColorR / 255.0;
					cg = el2->m_iColorG / 255.0;
					cb = el2->m_iColorB / 255.0;
				}

				mfprintf(b,"  cylinder { < %.6f, %.6f, %.6f >, < %.6f, %.6f, %.6f >, nbbond_r\n",vec3[0],vec3[1],vec3[2],vec2[0],vec2[1],vec2[2]);
				if (m_bPOVNbColorFace && m_bPOVHighlightNbAtoms)
					mfprintf(b,"    pigment { rgbft < %f, %f, %f, 0, %.6f > } finish { reflection nbbond_reflect specular nbbond_specular ambient nbbond_ambient diffuse nbbond_diffuse } no_shadow }\n",cr,cg,cb,m_fPOVNbTrans);
						else mfprintf(b,"    pigment { rgbft nb_%s_color } finish { reflection nbbond_reflect specular nbbond_specular ambient nbbond_ambient diffuse nbbond_diffuse } no_shadow }\n",el->m_sLabel);
				mfprintf(b,"#end\n");

				vec3 = cam - (vec1 + vec2) / 2.0;
				vec3.Normalize();

				vec2b = vec2-vec1;
				vec1b = CrossP(vec3,vec2b);

				vec1b.Normalize();

//				if (shadow)
				{
					mfprintf(b,"#if (nbbond_draw_halo1)\n");
					mfprintf(b,"  triangle {\n    < ");
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1) - (%.6f) * nbbond_d_halo1, ",vec1[0],vec1b[0],vec3[0]);
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1) - (%.6f) * nbbond_d_halo1, ",vec1[1],vec1b[1],vec3[1]);
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1) - (%.6f) * nbbond_d_halo1 >,\n",vec1[2],vec1b[2],vec3[2]);
					mfprintf(b,"    < ");
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1) - (%.6f) * nbbond_d_halo1, ",vec2[0],vec1b[0],vec3[0]);
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1) - (%.6f) * nbbond_d_halo1, ",vec2[1],vec1b[1],vec3[1]);
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1) - (%.6f) * nbbond_d_halo1 >,\n",vec2[2],vec1b[2],vec3[2]);
					mfprintf(b,"    < ");
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1) - (%.6f) * nbbond_d_halo1, ",vec2[0],vec1b[0],vec3[0]);
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1) - (%.6f) * nbbond_d_halo1, ",vec2[1],vec1b[1],vec3[1]);
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1) - (%.6f) * nbbond_d_halo1 >\n",vec2[2],vec1b[2],vec3[2]);
					mfprintf(b,"    pigment { rgbft nbbond_color_halo1 } finish { reflection nbbond_reflect_halo1 specular nbbond_specular_halo1 ambient nbbond_ambient_halo1 diffuse nbbond_diffuse_halo1 } no_shadow no_reflection no_radiosity }\n");

					mfprintf(b,"  triangle {\n    < ");
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1) - (%.6f) * nbbond_d_halo1, ",vec1[0],vec1b[0],vec3[0]);
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1) - (%.6f) * nbbond_d_halo1, ",vec1[1],vec1b[1],vec3[1]);
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1) - (%.6f) * nbbond_d_halo1 >,\n",vec1[2],vec1b[2],vec3[2]);
					mfprintf(b,"    < ");
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1) - (%.6f) * nbbond_d_halo1, ",vec1[0],vec1b[0],vec3[0]);
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1) - (%.6f) * nbbond_d_halo1, ",vec1[1],vec1b[1],vec3[1]);
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1) - (%.6f) * nbbond_d_halo1 >,\n",vec1[2],vec1b[2],vec3[2]);
					mfprintf(b,"    < ");
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1) - (%.6f) * nbbond_d_halo1, ",vec2[0],vec1b[0],vec3[0]);
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1) - (%.6f) * nbbond_d_halo1, ",vec2[1],vec1b[1],vec3[1]);
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1) - (%.6f) * nbbond_d_halo1 >\n",vec2[2],vec1b[2],vec3[2]);
					mfprintf(b,"    pigment { rgbft nbbond_color_halo1 } finish { reflection nbbond_reflect_halo1 specular nbbond_specular_halo1 ambient nbbond_ambient_halo1 diffuse nbbond_diffuse_halo1 } no_shadow no_reflection no_radiosity }\n");
					mfprintf(b,"#end\n");
				}

//				if (halo)
				{
					mfprintf(b,"#if (nbbond_draw_halo2)\n");
					mfprintf(b,"  triangle {\n    < ");
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2) - (%.6f) * nbbond_d_halo2, ",vec1[0],vec1b[0],vec3[0]);
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2) - (%.6f) * nbbond_d_halo2, ",vec1[1],vec1b[1],vec3[1]);
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2) - (%.6f) * nbbond_d_halo2 >,\n",vec1[2],vec1b[2],vec3[2]);
					mfprintf(b,"    < ");
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2) - (%.6f) * nbbond_d_halo2, ",vec2[0],vec1b[0],vec3[0]);
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2) - (%.6f) * nbbond_d_halo2, ",vec2[1],vec1b[1],vec3[1]);
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2) - (%.6f) * nbbond_d_halo2 >,\n",vec2[2],vec1b[2],vec3[2]);
					mfprintf(b,"    < ");
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2) - (%.6f) * nbbond_d_halo2, ",vec2[0],vec1b[0],vec3[0]);
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2) - (%.6f) * nbbond_d_halo2, ",vec2[1],vec1b[1],vec3[1]);
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2) - (%.6f) * nbbond_d_halo2 >\n",vec2[2],vec1b[2],vec3[2]);
					mfprintf(b,"    pigment { rgbft nbbond_color_halo2 } finish { reflection nbbond_reflect_halo2 specular nbbond_specular_halo2 ambient nbbond_ambient_halo2 diffuse nbbond_diffuse_halo2 } no_shadow no_reflection no_radiosity }\n");

					mfprintf(b,"  triangle {\n    < ");
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2) - (%.6f) * nbbond_d_halo2, ",vec1[0],vec1b[0],vec3[0]);
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2) - (%.6f) * nbbond_d_halo2, ",vec1[1],vec1b[1],vec3[1]);
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2) - (%.6f) * nbbond_d_halo2 >,\n",vec1[2],vec1b[2],vec3[2]);
					mfprintf(b,"    < ");
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2) - (%.6f) * nbbond_d_halo2, ",vec1[0],vec1b[0],vec3[0]);
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2) - (%.6f) * nbbond_d_halo2, ",vec1[1],vec1b[1],vec3[1]);
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2) - (%.6f) * nbbond_d_halo2 >,\n",vec1[2],vec1b[2],vec3[2]);
					mfprintf(b,"    < ");
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2) - (%.6f) * nbbond_d_halo2, ",vec2[0],vec1b[0],vec3[0]);
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2) - (%.6f) * nbbond_d_halo2, ",vec2[1],vec1b[1],vec3[1]);
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2) - (%.6f) * nbbond_d_halo2 >\n",vec2[2],vec1b[2],vec3[2]);
					mfprintf(b,"    pigment { rgbft nbbond_color_halo2 } finish { reflection nbbond_reflect_halo2 specular nbbond_specular_halo2 ambient nbbond_ambient_halo2 diffuse nbbond_diffuse_halo2 } no_shadow no_reflection no_radiosity }\n");
					mfprintf(b,"#end\n");

					
					mfprintf(b,"#if (nbbond_draw_halo3)\n");
					mfprintf(b,"  triangle {\n    < ");
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2 + nbbond_r_halo3) - (%.6f) * nbbond_d_halo3, ",vec1[0],vec1b[0],vec3[0]);
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2 + nbbond_r_halo3) - (%.6f) * nbbond_d_halo3, ",vec1[1],vec1b[1],vec3[1]);
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2 + nbbond_r_halo3) - (%.6f) * nbbond_d_halo3 >,\n",vec1[2],vec1b[2],vec3[2]);
					mfprintf(b,"    < ");
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2 + nbbond_r_halo3) - (%.6f) * nbbond_d_halo3, ",vec2[0],vec1b[0],vec3[0]);
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2 + nbbond_r_halo3) - (%.6f) * nbbond_d_halo3, ",vec2[1],vec1b[1],vec3[1]);
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2 + nbbond_r_halo3) - (%.6f) * nbbond_d_halo3 >,\n",vec2[2],vec1b[2],vec3[2]);
					mfprintf(b,"    < ");
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2 + nbbond_r_halo3) - (%.6f) * nbbond_d_halo3, ",vec2[0],vec1b[0],vec3[0]);
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2 + nbbond_r_halo3) - (%.6f) * nbbond_d_halo3, ",vec2[1],vec1b[1],vec3[1]);
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2 + nbbond_r_halo3) - (%.6f) * nbbond_d_halo3 >\n",vec2[2],vec1b[2],vec3[2]);
					mfprintf(b,"    pigment { rgbft nbbond_color_halo3 } finish { reflection nbbond_reflect_halo3 specular nbbond_specular_halo3 ambient nbbond_ambient_halo3 diffuse nbbond_diffuse_halo3 } no_shadow no_reflection no_radiosity }\n");

					mfprintf(b,"  triangle {\n    < ");
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2 + nbbond_r_halo3) - (%.6f) * nbbond_d_halo3, ",vec1[0],vec1b[0],vec3[0]);
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2 + nbbond_r_halo3) - (%.6f) * nbbond_d_halo3, ",vec1[1],vec1b[1],vec3[1]);
					mfprintf(b,"%.6f + (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2 + nbbond_r_halo3) - (%.6f) * nbbond_d_halo3 >,\n",vec1[2],vec1b[2],vec3[2]);
					mfprintf(b,"    < ");
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2 + nbbond_r_halo3) - (%.6f) * nbbond_d_halo3, ",vec1[0],vec1b[0],vec3[0]);
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2 + nbbond_r_halo3) - (%.6f) * nbbond_d_halo3, ",vec1[1],vec1b[1],vec3[1]);
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2 + nbbond_r_halo3) - (%.6f) * nbbond_d_halo3 >,\n",vec1[2],vec1b[2],vec3[2]);
					mfprintf(b,"    < ");
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2 + nbbond_r_halo3) - (%.6f) * nbbond_d_halo3, ",vec2[0],vec1b[0],vec3[0]);
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2 + nbbond_r_halo3) - (%.6f) * nbbond_d_halo3, ",vec2[1],vec1b[1],vec3[1]);
					mfprintf(b,"%.6f - (%.6f) * (nbbond_r + nbbond_r_halo1 + nbbond_r_halo2 + nbbond_r_halo3) - (%.6f) * nbbond_d_halo3 >\n",vec2[2],vec1b[2],vec3[2]);
					mfprintf(b,"    pigment { rgbft nbbond_color_halo3 } finish { reflection nbbond_reflect_halo3 specular nbbond_specular_halo3 ambient nbbond_ambient_halo3 diffuse nbbond_diffuse_halo3 } no_shadow no_reflection no_radiosity }\n");
					mfprintf(b,"#end\n");
				}

				if (m_bPOVHighlightNbAtoms)
				{
					n = false;
					n2 = false;
					for (z4=0;z4<nbatoms.GetSize();z4++)
					{
						if (nbatoms[z4] == o)
							n = true;
						if (nbatoms[z4] == o2)
							n2 = true;
					}
				}

				vec3 = vec2 - vec1;
				vec3.Normalize();

				mfprintf(b,"#if (nbbond_draw_stub)\n");
				mfprintf(b,"  cylinder {\n    < %.6f, %.6f, %.6f >,\n",vec1[0],vec1[1],vec1[2]);

				if (m_bPOVHighlightNbAtoms && n)
				{
					mfprintf(b,"    < %.6f + (%.6f) * ( element_%s_radius*nbatom_r*nbhighlight_fac - ( element_%s_radius*nbatom_r*nbhighlight_fac - 0.5*sqrt(4*element_%s_radius*element_%s_radius*nbatom_r*nbatom_r*nbhighlight_fac*nbhighlight_fac - 4*(nbbond_r+nbbond_r_stub)*(nbbond_r+nbbond_r_stub) ) ) + nbbond_l_stub ),\n",vec1[0],vec3[0],el->m_sLabel,el->m_sLabel,el->m_sLabel,el->m_sLabel);
					mfprintf(b,"      %.6f + (%.6f) * ( element_%s_radius*nbatom_r*nbhighlight_fac - ( element_%s_radius*nbatom_r*nbhighlight_fac - 0.5*sqrt(4*element_%s_radius*element_%s_radius*nbatom_r*nbatom_r*nbhighlight_fac*nbhighlight_fac - 4*(nbbond_r+nbbond_r_stub)*(nbbond_r+nbbond_r_stub) ) ) + nbbond_l_stub ),\n",vec1[1],vec3[1],el->m_sLabel,el->m_sLabel,el->m_sLabel,el->m_sLabel);
					mfprintf(b,"      %.6f + (%.6f) * ( element_%s_radius*nbatom_r*nbhighlight_fac - ( element_%s_radius*nbatom_r*nbhighlight_fac - 0.5*sqrt(4*element_%s_radius*element_%s_radius*nbatom_r*nbatom_r*nbhighlight_fac*nbhighlight_fac - 4*(nbbond_r+nbbond_r_stub)*(nbbond_r+nbbond_r_stub) ) ) + nbbond_l_stub ) >\n,",vec1[2],vec3[2],el->m_sLabel,el->m_sLabel,el->m_sLabel,el->m_sLabel);
				} else
				{
					mfprintf(b,"    < %.6f + (%.6f) * ( element_%s_radius*nbatom_r - ( element_%s_radius*nbatom_r - 0.5*sqrt(4*element_%s_radius*element_%s_radius*nbatom_r*nbatom_r - 4*(nbbond_r+nbbond_r_stub)*(nbbond_r+nbbond_r_stub) ) ) + nbbond_l_stub ),\n",vec1[0],vec3[0],el->m_sLabel,el->m_sLabel,el->m_sLabel,el->m_sLabel);
					mfprintf(b,"      %.6f + (%.6f) * ( element_%s_radius*nbatom_r - ( element_%s_radius*nbatom_r - 0.5*sqrt(4*element_%s_radius*element_%s_radius*nbatom_r*nbatom_r - 4*(nbbond_r+nbbond_r_stub)*(nbbond_r+nbbond_r_stub) ) ) + nbbond_l_stub ),\n",vec1[1],vec3[1],el->m_sLabel,el->m_sLabel,el->m_sLabel,el->m_sLabel);
					mfprintf(b,"      %.6f + (%.6f) * ( element_%s_radius*nbatom_r - ( element_%s_radius*nbatom_r - 0.5*sqrt(4*element_%s_radius*element_%s_radius*nbatom_r*nbatom_r - 4*(nbbond_r+nbbond_r_stub)*(nbbond_r+nbbond_r_stub) ) ) + nbbond_l_stub ) >\n,",vec1[2],vec3[2],el->m_sLabel,el->m_sLabel,el->m_sLabel,el->m_sLabel);
				}

				mfprintf(b,"    nbbond_r + nbbond_r_stub\n");
				mfprintf(b,"    pigment { rgbft nbbond_color_stub } finish { reflection nbbond_reflect_stub specular nbbond_specular_stub ambient nbbond_ambient_stub diffuse nbbond_diffuse_stub } no_shadow no_radiosity }\n");

				mfprintf(b,"  cylinder {\n    < %.6f, %.6f, %.6f >,\n",vec2[0],vec2[1],vec2[2]);

				if (m_bPOVHighlightNbAtoms && n2)
				{
					mfprintf(b,"    < %.6f - (%.6f) * ( element_%s_radius*nbatom_r*nbhighlight_fac - ( element_%s_radius*nbatom_r*nbhighlight_fac - 0.5*sqrt(4*element_%s_radius*element_%s_radius*nbatom_r*nbatom_r*nbhighlight_fac*nbhighlight_fac - 4*(nbbond_r+nbbond_r_stub)*(nbbond_r+nbbond_r_stub) ) ) + nbbond_l_stub ),\n",vec2[0],vec3[0],el2->m_sLabel,el2->m_sLabel,el2->m_sLabel,el2->m_sLabel);
					mfprintf(b,"      %.6f - (%.6f) * ( element_%s_radius*nbatom_r*nbhighlight_fac - ( element_%s_radius*nbatom_r*nbhighlight_fac - 0.5*sqrt(4*element_%s_radius*element_%s_radius*nbatom_r*nbatom_r*nbhighlight_fac*nbhighlight_fac - 4*(nbbond_r+nbbond_r_stub)*(nbbond_r+nbbond_r_stub) ) ) + nbbond_l_stub ),\n",vec2[1],vec3[1],el2->m_sLabel,el2->m_sLabel,el2->m_sLabel,el2->m_sLabel);
					mfprintf(b,"      %.6f - (%.6f) * ( element_%s_radius*nbatom_r*nbhighlight_fac - ( element_%s_radius*nbatom_r*nbhighlight_fac - 0.5*sqrt(4*element_%s_radius*element_%s_radius*nbatom_r*nbatom_r*nbhighlight_fac*nbhighlight_fac - 4*(nbbond_r+nbbond_r_stub)*(nbbond_r+nbbond_r_stub) ) ) + nbbond_l_stub ) >\n,",vec2[2],vec3[2],el2->m_sLabel,el2->m_sLabel,el2->m_sLabel,el2->m_sLabel);
				} else
				{
					mfprintf(b,"    < %.6f - (%.6f) * ( element_%s_radius*nbatom_r - ( element_%s_radius*nbatom_r - 0.5*sqrt(4*element_%s_radius*element_%s_radius*nbatom_r*nbatom_r - 4*(nbbond_r+nbbond_r_stub)*(nbbond_r+nbbond_r_stub) ) ) + nbbond_l_stub ),\n",vec2[0],vec3[0],el2->m_sLabel,el2->m_sLabel,el2->m_sLabel,el2->m_sLabel);
					mfprintf(b,"      %.6f - (%.6f) * ( element_%s_radius*nbatom_r - ( element_%s_radius*nbatom_r - 0.5*sqrt(4*element_%s_radius*element_%s_radius*nbatom_r*nbatom_r - 4*(nbbond_r+nbbond_r_stub)*(nbbond_r+nbbond_r_stub) ) ) + nbbond_l_stub ),\n",vec2[1],vec3[1],el2->m_sLabel,el2->m_sLabel,el2->m_sLabel,el2->m_sLabel);
					mfprintf(b,"      %.6f - (%.6f) * ( element_%s_radius*nbatom_r - ( element_%s_radius*nbatom_r - 0.5*sqrt(4*element_%s_radius*element_%s_radius*nbatom_r*nbatom_r - 4*(nbbond_r+nbbond_r_stub)*(nbbond_r+nbbond_r_stub) ) ) + nbbond_l_stub ) >\n,",vec2[2],vec3[2],el2->m_sLabel,el2->m_sLabel,el2->m_sLabel,el2->m_sLabel);
				}

				mfprintf(b,"    nbbond_r + nbbond_r_stub\n");
				mfprintf(b,"    pigment { rgbft nbbond_color_stub } finish { reflection nbbond_reflect_stub specular nbbond_specular_stub ambient nbbond_ambient_stub diffuse nbbond_diffuse_stub } no_shadow no_radiosity }\n");
				mfprintf(b,"#end\n");
			}
		}
	}

//	mfprintf(b,"\n/*** Manual change of cell rotation only works if drawing of all halos is disabled! ****/\n");
//	mfprintf(b,"    rotate <%f, %f, %f>\n\n    no_shadow\n",m_vPOVRotPos[0],m_vPOVRotPos[1],m_vPOVRotPos[2]);
	mfprintf(b,"\n");
	fclose(b);

	delete m_pContainer;
}


/*void CVoroWrapper::WriteXYZCell(CTimeStep *ts, int mol, int smol)
{
	FILE *a, *b;
	int ijk, q, z, z2, z3, id, faces, co, fc, ti, ec;
	CVoroAtom *va;
	CVoroMolecule *vm;
	CMolecule *m;
	CSingleMolecule *sm;
	double *pp, tx, ty, tz;
	voronoicell_neighbor c;
	vector<int> nb;
	vector<int> fo;
	vector<double> fa;
	vector<int> fv;
	int *nbtemp;

	m = (CMolecule*)g_oaMolecules[mol];
	sm = (CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[smol]];

	try { nbtemp = new int[m_oaVoroAtoms.GetSize()]; } catch(...) { nbtemp = NULL; }
	if (nbtemp == NULL) NewException((double)m_oaVoroAtoms.GetSize()*sizeof(int),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	try { m_pContainer = new container_periodic_poly(g_fBoxX/1000.0,0,g_fBoxY/1000.0,0,0,g_fBoxZ/1000.0,m_iBlocksX,m_iBlocksY,m_iBlocksZ,g_iVoroMemory); } catch(...) { m_pContainer = NULL; }
	if (m_pContainer == NULL) NewException((double)sizeof(container_periodic_poly),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	for (z=0;z<g_iGesAtomCount;z++)
		m_pContainer->put(z,ts->m_vaCoords[z][0]/1000.0,ts->m_vaCoords[z][1]/1000.0,ts->m_vaCoords[z][2]/1000.0,0.05);

	c_loop_all_periodic vl(*m_pContainer);

	a = OpenFileWrite(s,true);

	ec = 0;
	for (z=0;z<m->m_baAtomIndex.GetSize()-1;z++)
		ec += ((CxIntArray*)sm->m_oaAtomOffset[z])->GetSize();

	mfprintf(a,"%d\n\n",ec+2000);

	for (z=0;z<m->m_baAtomIndex.GetSize()-1;z++)
		for (z2=0;z2<((CxIntArray*)sm->m_oaAtomOffset[z])->GetSize();z2++)
			mfprintf(a," %s  %g  %g  %g\n",((CAtom*)g_oaAtoms[m->m_baAtomIndex[z]])->m_sName,ts->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z])->GetAt(z2)][0]/100.0-g_fBoxX/200.0,ts->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z])->GetAt(z2)][1]/100.0-g_fBoxY/200.0,ts->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z])->GetAt(z2)][2]/100.0-g_fBoxZ/200.0);

	for (z=0;z<10;z++)
	{
		for (z2=0;z2<10;z2++)
		{
			for (z3=0;z3<10;z3++)
			{
				mfprintf(a,"B  %g  %g  %g\n",30.0+z*3.0,z2*3.0,z3*3.0);
				mfprintf(a,"B  %g  %g  %g\n",30.2+z*3.0,z2*3.0,z3*3.0);
			}
		}
	}

	mfprintf(a,"%d\n\n",m->m_iAtomGes+2000);
	for (z=0;z<m->m_baAtomIndex.GetSize()-1;z++)
		for (z2=0;z2<((CxIntArray*)sm->m_oaAtomOffset[z])->GetSize();z2++)
			mfprintf(a," %s  %g  %g  %g\n",((CAtom*)g_oaAtoms[m->m_baAtomIndex[z]])->m_sName,ts->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z])->GetAt(z2)][0]/100.0-g_fBoxX/200.0,ts->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z])->GetAt(z2)][1]/100.0-g_fBoxY/200.0,ts->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z])->GetAt(z2)][2]/100.0-g_fBoxZ/200.0);

	b = OpenFileWrite("voro.pov",true);


	mfprintf(b,"#version 3.6;\n");
	mfprintf(b,"\n");

	mfprintf(b,"#declare r_atom=1.0;\n");
	mfprintf(b,"#declare r_vertex=0;\n");
	mfprintf(b,"#declare r_edge=0.004;\n");
	mfprintf(b,"//#declare r_vertex=0.01;\n");
	mfprintf(b,"//#declare r_edge=0.01;\n");
	mfprintf(b,"#declare color_edge=<0,1,0,0,0>;\n");
	mfprintf(b,"#declare color_vertex=<0,1,0,0,0>;\n");
	mfprintf(b,"#declare color_face=<0.5,0.5,1,0,0.4>;\n");
	mfprintf(b,"//#declare color_face=<0,0,0,1,1>;\n");
	mfprintf(b,"#declare spec_atom=0.7;\n");
	mfprintf(b,"#declare spec_edge=0.3;\n");
	mfprintf(b,"#declare spec_vertex=0.3;\n");
	mfprintf(b,"#declare spec_face=0.3;\n");
	mfprintf(b,"#declare refl_atom=0;\n");
	mfprintf(b,"#declare ambient_vertex=0.2;\n");
	mfprintf(b,"#declare ambient_edge=0.2;\n");
	mfprintf(b,"#declare ambient_atom=0.4;\n");
	mfprintf(b,"#declare ambient_face=0.3;\n");
	mfprintf(b,"#declare diffuse_vertex=0.8;\n");
	mfprintf(b,"#declare diffuse_edge=0.8;\n");
	mfprintf(b,"#declare diffuse_atom=0.7;\n");
	mfprintf(b,"#declare diffuse_face=0.9;\n");


	mfprintf(b,"// Right-handed coordinate system in which the z-axis points upwards\n");
	mfprintf(b,"camera {\n");
	mfprintf(b,"	location <0,-20,0>\n");
	mfprintf(b,"	sky z\n");
	mfprintf(b,"	right -0.06*x*image_width/image_height\n");
	mfprintf(b,"	up 0.06*z\n");
	mfprintf(b,"	look_at <0, 0, 0>\n");
	mfprintf(b,"}\n");
	mfprintf(b,"\n");
	mfprintf(b,"// White background\n");
	mfprintf(b,"background{rgb 1}\n");
	mfprintf(b,"\n");
	mfprintf(b,"// Two lights with slightly different colors\n");
	mfprintf(b,"light_source{<-8,-20,30> color rgb <0.77,0.75,0.75>}\n");
	mfprintf(b,"light_source{<25,-12,12> color rgb <0.38,0.40,0.40>}\n\n");

	mfprintf(b,"\nunion {\n");

	ec = 0;
	if (vl.start()) 
	{
		do 
		{
			if (m_pContainer->compute_cell(c,vl))
			{
				co++;

				ijk=vl.ijk;
				q=vl.q;
				pp=m_pContainer->p[ijk]+m_pContainer->ps*q;

				id = m_pContainer->id[ijk][q];

				va = (CVoroAtom*)m_oaVoroAtoms[m_pAssignAtoms[id]];
				vm = (CVoroMolecule*)m_oaVoroMolecules[m_pAssignMolecules[id]];

				ti = 0;

				if ((vm->m_iMolecule == mol) && (vm->m_iSingleMol == smol))
				{
					c.neighbors(nb);
					faces = c.number_of_faces();
					c.face_vertices(fv);

					mfprintf(b,"// Voronoi cell %d\n",m_pContainer->id[vl.ijk][vl.q]);
					mfprintf(b,"union {\n");

					for (z=0;z<faces;z++)
					{
				//		mprintf("Face %d/%d:\n",z+1,faces);
						fc = fv[ti];
				//		mprintf("  %d Vertices\n",fc);

						if (m_pAssignMolecules[id] != m_pAssignMolecules[nb[z]])
						{
							tx = 0;
							ty = 0;
							tz = 0;
							for (z2=0;z2<fc;z2++)
							{
					//			mprintf("  - Vertex %d/%d; Id=%d, ti=%d (%g, %g, %g) -> (%g, %g, %g)\n",z2+1,fc,fv[ti+z2+1]+1,ti,c.pts[fv[ti+z2+1]*3]*0.5,c.pts[fv[ti+z2+1]*3+1]*0.5,c.pts[fv[ti+z2+1]*3+2]*0.5,*pp+c.pts[fv[ti+z2+1]*3]*0.5,pp[1]+c.pts[fv[ti+z2+1]*3+1]*0.5,pp[2]+c.pts[fv[ti+z2+1]*3+2]*0.5);
								mfprintf(b,"sphere{<%g,%g,%g>, r_vertex\n pigment{rgbft color_vertex } finish{specular spec_vertex ambient ambient_vertex diffuse diffuse_vertex } }\n",*pp+c.pts[fv[ti+z2+1]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+z2+1]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+z2+1]*3+2]*0.5-g_fBoxZ/2000.0);
								tx += *pp+c.pts[fv[ti+z2+1]*3]*0.5-g_fBoxX/2000.0;
								ty += pp[1]+c.pts[fv[ti+z2+1]*3+1]*0.5-g_fBoxY/2000.0;
								tz += pp[2]+c.pts[fv[ti+z2+1]*3+2]*0.5-g_fBoxZ/2000.0;
								if (z2 < fc-1)
								{
									if ((pow(c.pts[fv[ti+z2+1]*3]-c.pts[fv[ti+z2+2]*3],2)+pow(c.pts[fv[ti+z2+1]*3+1]-c.pts[fv[ti+z2+2]*3+1],2)+pow(c.pts[fv[ti+z2+1]*3+2]-c.pts[fv[ti+z2+2]*3+2],2)) > 0.0000000001)
									{
										mfprintf(a," B  %g  %g  %g\n",(*pp+c.pts[fv[ti+z2+1]*3]*0.5-g_fBoxX/2000.0)*10.0,(pp[1]+c.pts[fv[ti+z2+1]*3+1]*0.5-g_fBoxY/2000.0)*10.0,(pp[2]+c.pts[fv[ti+z2+1]*3+2]*0.5-g_fBoxZ/2000.0)*10.0);
										mfprintf(a," B  %g  %g  %g\n",(*pp+c.pts[fv[ti+z2+2]*3]*0.5-g_fBoxX/2000.0)*10.0,(pp[1]+c.pts[fv[ti+z2+2]*3+1]*0.5-g_fBoxY/2000.0)*10.0,(pp[2]+c.pts[fv[ti+z2+2]*3+2]*0.5-g_fBoxZ/2000.0)*10.0);
										ec++;
										if (ec == 1000)
										{
											eprintf("Out of edges.\n");
											return;
										}
										mfprintf(b,"cylinder{<%g,%g,%g>,<%g,%g,%g>, r_edge\n pigment{rgbft color_edge } finish{specular spec_edge ambient ambient_edge diffuse diffuse_edge } }\n",*pp+c.pts[fv[ti+z2+1]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+z2+1]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+z2+1]*3+2]*0.5-g_fBoxZ/2000.0,*pp+c.pts[fv[ti+z2+2]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+z2+2]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+z2+2]*3+2]*0.5-g_fBoxZ/2000.0);
									}
								} else
								{
									if ((pow(c.pts[fv[ti+z2+1]*3]-c.pts[fv[ti+1]*3],2)+pow(c.pts[fv[ti+z2+1]*3+1]-c.pts[fv[ti+1]*3+1],2)+pow(c.pts[fv[ti+z2+1]*3+2]-c.pts[fv[ti+1]*3+2],2)) > 0.0000000001)
									{
										mfprintf(a," B  %g  %g  %g\n",(*pp+c.pts[fv[ti+z2+1]*3]*0.5-g_fBoxX/2000.0)*10.0,(pp[1]+c.pts[fv[ti+z2+1]*3+1]*0.5-g_fBoxY/2000.0)*10.0,(pp[2]+c.pts[fv[ti+z2+1]*3+2]*0.5-g_fBoxZ/2000.0)*10.0);
										mfprintf(a," B  %g  %g  %g\n",(*pp+c.pts[fv[ti+1]*3]*0.5-g_fBoxX/2000.0)*10.0,(pp[1]+c.pts[fv[ti+1]*3+1]*0.5-g_fBoxY/2000.0)*10.0,(pp[2]+c.pts[fv[ti+1]*3+2]*0.5-g_fBoxZ/2000.0)*10.0);
										ec++;
										if (ec == 1000)
										{
											eprintf("Out of edges.\n");
											return;
										}
										mfprintf(b,"cylinder{<%g,%g,%g>,<%g,%g,%g>, r_edge\n pigment{rgbft color_edge } finish{specular spec_edge ambient ambient_edge diffuse diffuse_edge } }\n",*pp+c.pts[fv[ti+z2+1]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+z2+1]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+z2+1]*3+2]*0.5-g_fBoxZ/2000.0,*pp+c.pts[fv[ti+1]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+1]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+1]*3+2]*0.5-g_fBoxZ/2000.0);
									}
								}
								fflush(b);
							}
							tx /= fc;
							ty /= fc;
							tz /= fc;

							mfprintf(b,"mesh { \n");
							for (z2=0;z2<fc-1;z2++)
							{
								mfprintf(b," triangle { <%g, %g, %g>, <%g, %g, %g>, <%g, %g, %g> }\n",tx,ty,tz,*pp+c.pts[fv[ti+z2+1]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+z2+1]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+z2+1]*3+2]*0.5-g_fBoxZ/2000.0,*pp+c.pts[fv[ti+z2+2]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+z2+2]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+z2+2]*3+2]*0.5-g_fBoxZ/2000.0);
							}
							mfprintf(b," triangle { <%g, %g, %g>, <%g, %g, %g>, <%g, %g, %g> }\n",tx,ty,tz,*pp+c.pts[fv[ti+z2+1]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+z2+1]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+z2+1]*3+2]*0.5-g_fBoxZ/2000.0,*pp+c.pts[fv[ti+1]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+1]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+1]*3+2]*0.5-g_fBoxZ/2000.0);
							mfprintf(b,"\ntexture {\n");
							mfprintf(b,"pigment{rgbft color_face } finish{specular spec_face ambient ambient_face diffuse diffuse_face }\n");
							mfprintf(b,"  }\n}\n");
						}

						ti += fv[ti]+1;
					}

					mfprintf(b,"\n");
					mfprintf(b,"}\n\n");

					mfprintf(b,"// Particle %d\nsphere{<%g,%g,%g>, %g*r_atom\n",m_pContainer->id[vl.ijk][vl.q],*pp-g_fBoxX/2000.0,pp[1]-g_fBoxY/2000.0,pp[2]-g_fBoxZ/2000.0,((CAtom*)g_oaAtoms[va->m_iRealAtomType])->m_pElement->m_fRadius/1000.0);
					mfprintf(b,"pigment{rgb <%f,%f,%f>} finish{ reflection refl_atom specular spec_atom ambient ambient_atom diffuse diffuse_atom } }\n",((CAtom*)g_oaAtoms[va->m_iRealAtomType])->m_pElement->m_iColorR/255.0,((CAtom*)g_oaAtoms[va->m_iRealAtomType])->m_pElement->m_iColorG/255.0,((CAtom*)g_oaAtoms[va->m_iRealAtomType])->m_pElement->m_iColorB/255.0);
				}
			}
		} while (vl.inc());
	}

	mfprintf(b,"\n    rotate <0, 0, 50>\n  rotate <-30, 0, 0>\n}\n");

	fclose(b);

	mprintf("%d/1000 Edges used.\n",ec);

	for (z=ec;z<1000;z++)
		mfprintf(a," B  30 0 0\n He  30 0 0\n");

	fclose(a);

}*/

/*void CVoroWrapper::WriteXYZCell_Start(CTimeStep *ts, const char *s, int mol, int smol)
{
	FILE *a, *b;
	int ijk, q, z, z2, z3, id,  faces, co, fc, ti, ec;
	CVoroAtom *va;
	CVoroMolecule *vm;
	CMolecule *m;
	CSingleMolecule *sm;
	double *pp, tx, ty, tz;
	voronoicell_neighbor c;
	vector<int> nb;
	vector<int> fo;
	vector<double> fa;
	vector<int> fv;
	int *nbtemp;

	m = (CMolecule*)g_oaMolecules[mol];
	sm = (CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[smol]];

	try { nbtemp = new int[m_oaVoroAtoms.GetSize()]; } catch(...) { nbtemp = NULL; }
	if (nbtemp == NULL) NewException((double)m_oaVoroAtoms.GetSize()*sizeof(int),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	try { m_pContainer = new container_periodic_poly(g_fBoxX/1000.0,0,g_fBoxY/1000.0,0,0,g_fBoxZ/1000.0,m_iBlocksX,m_iBlocksY,m_iBlocksZ,g_iVoroMemory); } catch(...) { m_pContainer = NULL; }
	if (m_pContainer == NULL) NewException((double)sizeof(container_periodic_poly),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	for (z=0;z<g_iGesAtomCount;z++)
		m_pContainer->put(z,ts->m_vaCoords[z][0]/1000.0,ts->m_vaCoords[z][1]/1000.0,ts->m_vaCoords[z][2]/1000.0,0.05);

	c_loop_all_periodic vl(*m_pContainer);

	a = OpenFileWrite(s,true);

	ec = 0;
	for (z=0;z<m->m_baAtomIndex.GetSize()-1;z++)
		ec += ((CxIntArray*)sm->m_oaAtomOffset[z])->GetSize();

	mfprintf(a,"%d\n\n",ec+m_iMaxEdgesR3);

	for (z=0;z<m->m_baAtomIndex.GetSize()-1;z++)
		for (z2=0;z2<((CxIntArray*)sm->m_oaAtomOffset[z])->GetSize();z2++)
			mfprintf(a," %s  %g  %g  %g\n",((CAtom*)g_oaAtoms[m->m_baAtomIndex[z]])->m_sName,ts->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z])->GetAt(z2)][0]/100.0-g_fBoxX/200.0,ts->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z])->GetAt(z2)][1]/100.0-g_fBoxY/200.0,ts->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z])->GetAt(z2)][2]/100.0-g_fBoxZ/200.0);

	for (z=0;z<10;z++)
	{
		for (z2=0;z2<10;z2++)
		{
			for (z3=0;z3<10;z3++)
			{
				mfprintf(a,"B  %g  %g  %g\n",30.0+z*3.0,z2*3.0,z3*3.0);
				mfprintf(a,"B  %g  %g  %g\n",30.2+z*3.0,z2*3.0,z3*3.0);
			}
		}
	}

	mfprintf(a,"%d\n\n",m->m_iAtomGes+2000);
	for (z=0;z<m->m_baAtomIndex.GetSize()-1;z++)
		for (z2=0;z2<((CxIntArray*)sm->m_oaAtomOffset[z])->GetSize();z2++)
			mfprintf(a," %s  %g  %g  %g\n",((CAtom*)g_oaAtoms[m->m_baAtomIndex[z]])->m_sName,ts->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z])->GetAt(z2)][0]/100.0-g_fBoxX/200.0,ts->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z])->GetAt(z2)][1]/100.0-g_fBoxY/200.0,ts->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z])->GetAt(z2)][2]/100.0-g_fBoxZ/200.0);

	b = OpenFileWrite("voro.pov",true);


	mfprintf(b,"#version 3.6;\n");
	mfprintf(b,"\n");

	mfprintf(b,"#declare r_atom=1.0;\n");
	mfprintf(b,"#declare r_vertex=0;\n");
	mfprintf(b,"#declare r_edge=0.004;\n");
	mfprintf(b,"//#declare r_vertex=0.01;\n");
	mfprintf(b,"//#declare r_edge=0.01;\n");
	mfprintf(b,"#declare color_edge=<0,1,0,0,0>;\n");
	mfprintf(b,"#declare color_vertex=<0,1,0,0,0>;\n");
	mfprintf(b,"#declare color_face=<0.5,0.5,1,0,0.4>;\n");
	mfprintf(b,"//#declare color_face=<0,0,0,1,1>;\n");
	mfprintf(b,"#declare spec_atom=0.7;\n");
	mfprintf(b,"#declare spec_edge=0.3;\n");
	mfprintf(b,"#declare spec_vertex=0.3;\n");
	mfprintf(b,"#declare spec_face=0.3;\n");
	mfprintf(b,"#declare refl_atom=0;\n");
	mfprintf(b,"#declare ambient_vertex=0.2;\n");
	mfprintf(b,"#declare ambient_edge=0.2;\n");
	mfprintf(b,"#declare ambient_atom=0.4;\n");
	mfprintf(b,"#declare ambient_face=0.3;\n");
	mfprintf(b,"#declare diffuse_vertex=0.8;\n");
	mfprintf(b,"#declare diffuse_edge=0.8;\n");
	mfprintf(b,"#declare diffuse_atom=0.7;\n");
	mfprintf(b,"#declare diffuse_face=0.9;\n");


	mfprintf(b,"// Right-handed coordinate system in which the z-axis points upwards\n");
	mfprintf(b,"camera {\n");
	mfprintf(b,"	location <0,-20,0>\n");
	mfprintf(b,"	sky z\n");
	mfprintf(b,"	right -0.06*x*image_width/image_height\n");
	mfprintf(b,"	up 0.06*z\n");
	mfprintf(b,"	look_at <0, 0, 0>\n");
	mfprintf(b,"}\n");
	mfprintf(b,"\n");
	mfprintf(b,"// White background\n");
	mfprintf(b,"background{rgb 1}\n");
	mfprintf(b,"\n");
	mfprintf(b,"// Two lights with slightly different colors\n");
	mfprintf(b,"light_source{<-8,-20,30> color rgb <0.77,0.75,0.75>}\n");
	mfprintf(b,"light_source{<25,-12,12> color rgb <0.38,0.40,0.40>}\n\n");

	mfprintf(b,"\nunion {\n");

	ec = 0;
	if (vl.start()) 
	{
		do 
		{
			if (m_pContainer->compute_cell(c,vl))
			{
				co++;

				ijk=vl.ijk;
				q=vl.q;
				pp=m_pContainer->p[ijk]+m_pContainer->ps*q;

				id = m_pContainer->id[ijk][q];

				va = (CVoroAtom*)m_oaVoroAtoms[m_pAssignAtoms[id]];
				vm = (CVoroMolecule*)m_oaVoroMolecules[m_pAssignMolecules[id]];

				ti = 0;

				if ((vm->m_iMolecule == mol) && (vm->m_iSingleMol == smol))
				{
					c.neighbors(nb);
					faces = c.number_of_faces();
					c.face_vertices(fv);

					mfprintf(b,"// Voronoi cell %d\n",m_pContainer->id[vl.ijk][vl.q]);
					mfprintf(b,"union {\n");

					for (z=0;z<faces;z++)
					{
				//		mprintf("Face %d/%d:\n",z+1,faces);
						fc = fv[ti];
				//		mprintf("  %d Vertices\n",fc);

						if (m_pAssignMolecules[id] != m_pAssignMolecules[nb[z]])
						{
							tx = 0;
							ty = 0;
							tz = 0;
							for (z2=0;z2<fc;z2++)
							{
					//			mprintf("  - Vertex %d/%d; Id=%d, ti=%d (%g, %g, %g) -> (%g, %g, %g)\n",z2+1,fc,fv[ti+z2+1]+1,ti,c.pts[fv[ti+z2+1]*3]*0.5,c.pts[fv[ti+z2+1]*3+1]*0.5,c.pts[fv[ti+z2+1]*3+2]*0.5,*pp+c.pts[fv[ti+z2+1]*3]*0.5,pp[1]+c.pts[fv[ti+z2+1]*3+1]*0.5,pp[2]+c.pts[fv[ti+z2+1]*3+2]*0.5);
								mfprintf(b,"sphere{<%g,%g,%g>, r_vertex\n pigment{rgbft color_vertex } finish{specular spec_vertex ambient ambient_vertex diffuse diffuse_vertex } }\n",*pp+c.pts[fv[ti+z2+1]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+z2+1]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+z2+1]*3+2]*0.5-g_fBoxZ/2000.0);
								tx += *pp+c.pts[fv[ti+z2+1]*3]*0.5-g_fBoxX/2000.0;
								ty += pp[1]+c.pts[fv[ti+z2+1]*3+1]*0.5-g_fBoxY/2000.0;
								tz += pp[2]+c.pts[fv[ti+z2+1]*3+2]*0.5-g_fBoxZ/2000.0;
								if (z2 < fc-1)
								{
									if ((pow(c.pts[fv[ti+z2+1]*3]-c.pts[fv[ti+z2+2]*3],2)+pow(c.pts[fv[ti+z2+1]*3+1]-c.pts[fv[ti+z2+2]*3+1],2)+pow(c.pts[fv[ti+z2+1]*3+2]-c.pts[fv[ti+z2+2]*3+2],2)) > 0.0000000001)
									{
										mfprintf(a," B  %g  %g  %g\n",(*pp+c.pts[fv[ti+z2+1]*3]*0.5-g_fBoxX/2000.0)*10.0,(pp[1]+c.pts[fv[ti+z2+1]*3+1]*0.5-g_fBoxY/2000.0)*10.0,(pp[2]+c.pts[fv[ti+z2+1]*3+2]*0.5-g_fBoxZ/2000.0)*10.0);
										mfprintf(a," B  %g  %g  %g\n",(*pp+c.pts[fv[ti+z2+2]*3]*0.5-g_fBoxX/2000.0)*10.0,(pp[1]+c.pts[fv[ti+z2+2]*3+1]*0.5-g_fBoxY/2000.0)*10.0,(pp[2]+c.pts[fv[ti+z2+2]*3+2]*0.5-g_fBoxZ/2000.0)*10.0);
										ec++;
										if (ec == 1000)
										{
											eprintf("Out of edges.\n");
											return;
										}
										mfprintf(b,"cylinder{<%g,%g,%g>,<%g,%g,%g>, r_edge\n pigment{rgbft color_edge } finish{specular spec_edge ambient ambient_edge diffuse diffuse_edge } }\n",*pp+c.pts[fv[ti+z2+1]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+z2+1]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+z2+1]*3+2]*0.5-g_fBoxZ/2000.0,*pp+c.pts[fv[ti+z2+2]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+z2+2]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+z2+2]*3+2]*0.5-g_fBoxZ/2000.0);
									}
								} else
								{
									if ((pow(c.pts[fv[ti+z2+1]*3]-c.pts[fv[ti+1]*3],2)+pow(c.pts[fv[ti+z2+1]*3+1]-c.pts[fv[ti+1]*3+1],2)+pow(c.pts[fv[ti+z2+1]*3+2]-c.pts[fv[ti+1]*3+2],2)) > 0.0000000001)
									{
										mfprintf(a," B  %g  %g  %g\n",(*pp+c.pts[fv[ti+z2+1]*3]*0.5-g_fBoxX/2000.0)*10.0,(pp[1]+c.pts[fv[ti+z2+1]*3+1]*0.5-g_fBoxY/2000.0)*10.0,(pp[2]+c.pts[fv[ti+z2+1]*3+2]*0.5-g_fBoxZ/2000.0)*10.0);
										mfprintf(a," B  %g  %g  %g\n",(*pp+c.pts[fv[ti+1]*3]*0.5-g_fBoxX/2000.0)*10.0,(pp[1]+c.pts[fv[ti+1]*3+1]*0.5-g_fBoxY/2000.0)*10.0,(pp[2]+c.pts[fv[ti+1]*3+2]*0.5-g_fBoxZ/2000.0)*10.0);
										ec++;
										if (ec == 1000)
										{
											eprintf("Out of edges.\n");
											return;
										}
										mfprintf(b,"cylinder{<%g,%g,%g>,<%g,%g,%g>, r_edge\n pigment{rgbft color_edge } finish{specular spec_edge ambient ambient_edge diffuse diffuse_edge } }\n",*pp+c.pts[fv[ti+z2+1]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+z2+1]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+z2+1]*3+2]*0.5-g_fBoxZ/2000.0,*pp+c.pts[fv[ti+1]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+1]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+1]*3+2]*0.5-g_fBoxZ/2000.0);
									}
								}
								fflush(b);
							}
							tx /= fc;
							ty /= fc;
							tz /= fc;

							mfprintf(b,"mesh { \n");
							for (z2=0;z2<fc-1;z2++)
							{
								mfprintf(b," triangle { <%g, %g, %g>, <%g, %g, %g>, <%g, %g, %g> }\n",tx,ty,tz,*pp+c.pts[fv[ti+z2+1]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+z2+1]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+z2+1]*3+2]*0.5-g_fBoxZ/2000.0,*pp+c.pts[fv[ti+z2+2]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+z2+2]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+z2+2]*3+2]*0.5-g_fBoxZ/2000.0);
							}
							mfprintf(b," triangle { <%g, %g, %g>, <%g, %g, %g>, <%g, %g, %g> }\n",tx,ty,tz,*pp+c.pts[fv[ti+z2+1]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+z2+1]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+z2+1]*3+2]*0.5-g_fBoxZ/2000.0,*pp+c.pts[fv[ti+1]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+1]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+1]*3+2]*0.5-g_fBoxZ/2000.0);
							mfprintf(b,"\ntexture {\n");
							mfprintf(b,"pigment{rgbft color_face } finish{specular spec_face ambient ambient_face diffuse diffuse_face }\n");
							mfprintf(b,"  }\n}\n");
						}

						ti += fv[ti]+1;
					}

					mfprintf(b,"\n");
					mfprintf(b,"}\n\n");

					mfprintf(b,"// Particle %d\nsphere{<%g,%g,%g>, %g*r_atom\n",m_pContainer->id[vl.ijk][vl.q],*pp-g_fBoxX/2000.0,pp[1]-g_fBoxY/2000.0,pp[2]-g_fBoxZ/2000.0,((CAtom*)g_oaAtoms[va->m_iRealAtomType])->m_pElement->m_fRadius/1000.0);
					mfprintf(b,"pigment{rgb <%f,%f,%f>} finish{ reflection refl_atom specular spec_atom ambient ambient_atom diffuse diffuse_atom } }\n",((CAtom*)g_oaAtoms[va->m_iRealAtomType])->m_pElement->m_iColorR/255.0,((CAtom*)g_oaAtoms[va->m_iRealAtomType])->m_pElement->m_iColorG/255.0,((CAtom*)g_oaAtoms[va->m_iRealAtomType])->m_pElement->m_iColorB/255.0);
				}
			}
		} while (vl.inc());
	}

	mfprintf(b,"\n    rotate <0, 0, 50>\n  rotate <-30, 0, 0>\n}\n");

	fclose(b);

	mprintf("%d/1000 Edges used.\n",ec);

	for (z=ec;z<1000;z++)
		mfprintf(a," B  30 0 0\n He  30 0 0\n");

	fclose(a);

}*/

void CVoroWrapper::ParsePOVFaceColor()
{
	int z, z2, z3;
	CMolecule *m;
	CAtom *at;
	char buf[256];
	CxWordArray wa;
	float fr, fg, fb;

	if (m_faPOVFaceColor.GetSize() == 0)
	{
		for (z=0;z<g_oaMolecules.GetSize();z++)
		{
			if (m_bPOVFaceColorRef && (z != m_iPOVMol))
				continue;

			m = (CMolecule*)g_oaMolecules[z];
			for (z2=0;z2<m->m_waAtomCount.GetSize()-1;z2++)
			{
				at = (CAtom*)g_oaAtoms[m->m_baAtomIndex[z2]];
				for (z3=0;z3<m->m_waAtomCount[z2];z3++)
				{
					m_faPOVFaceColor.Add(at->m_pElement->m_iColorR/255.0f);
					m_faPOVFaceColor.Add(at->m_pElement->m_iColorG/255.0f);
					m_faPOVFaceColor.Add(at->m_pElement->m_iColorB/255.0f);
					m_waPOVFaceColorMol.Add(z);
					m_waPOVFaceColorElem.Add(z2);
					m_waPOVFaceColorAtom.Add(z3);
				}
			}
		}
	}

_again:
	mprintf(WHITE,"\n *** Color assignment:\n\n");
	for (z=0;z<m_waPOVFaceColorMol.GetSize();z++)
	{
		m = (CMolecule*)g_oaMolecules[m_waPOVFaceColorMol[z]];
		mprintf(YELLOW,"  #");
		mprintf(WHITE,"%3d",z+1);
		mprintf("%15s %2s%-2d  ( %.3f, %.3f, %.3f )\n",m->m_sName,((CAtom*)g_oaAtoms[m->m_baAtomIndex[m_waPOVFaceColorElem[z]]])->m_sName,m_waPOVFaceColorAtom[z]+1,m_faPOVFaceColor[z*3],m_faPOVFaceColor[z*3+1],m_faPOVFaceColor[z*3+2]);
		if ((z+1)%10 == 0)
			mprintf("\n");
	}
	mprintf("\n");

_iagain:
	AskString("    Which colors to change (e.g. 1-5,8): [done] ",buf,"");

	if (strlen(buf) == 0)
		return;

	if (!ParseIntList(buf,&wa))
	{
		eprintf("Wrong input.\n");
		goto _iagain;
	}

	fr = AskFloat("    Enter red component (0.0 - 1.0): [0.7] ",0.7f);
	fg = AskFloat("    Enter green component (0.0 - 1.0): [0.7] ",0.7f);
	fb = AskFloat("    Enter blue component (0.0 - 1.0): [0.7] ",0.7f);

	for (z=0;z<wa.GetSize();z++)
	{
		m_faPOVFaceColor[(wa[z]-1)*3] = fr;
		m_faPOVFaceColor[(wa[z]-1)*3+1] = fg;
		m_faPOVFaceColor[(wa[z]-1)*3+2] = fb;
	}

	mprintf("\n    Changed %d color values.\n",wa.GetSize());
	wa.RemoveAll();

	goto _again;
}


void CVoroWrapper::POVFindColor(int offset, float *r, float *g, float *b, float bleach)
{
	int z;
	float gr;

	for (z=0;z<m_waPOVFaceColorMol.GetSize();z++)
	{
		if ((g_waAtomMolIndex[offset] == m_waPOVFaceColorMol[z]) &&
			(g_waAtomElement[offset] == m_waPOVFaceColorElem[z]) &&
			(g_waAtomMolNumber[offset] == m_waPOVFaceColorAtom[z]))
		{
			*r = m_faPOVFaceColor[z*3];
			*g = m_faPOVFaceColor[z*3+1];
			*b = m_faPOVFaceColor[z*3+2];

//			gr = (*r + *g + *b) / 3.0;
			gr = 1.0f;

			*r = *r * (1.0-bleach) + gr*bleach;
			*g = *g * (1.0-bleach) + gr*bleach;
			*b = *b * (1.0-bleach) + gr*bleach;

			return;
		}
	}

	*r = 0;
	*g = 0;
	*b = 0;
}


void CVoroWrapper::ProcessSurfCover(CTimeStep *ts)
{
	int ijk, q, z, z3, id, faces;
//	CVoroAtom *va;
	CVoroMolecule *vm;
//	CMolecule *m;
//	CSingleMolecule *sm;
	voronoicell_neighbor c;
	vector<int> nb;
	vector<int> fo;
	vector<double> fa;
	CxVector3 vec;
	double atot;
	bool *tb;
	CxDoubleArray *tfa;

//	m = (CMolecule*)g_oaMolecules[m_iSurfCoverMol];
/*	sm = (CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[m_iSurfCoverSM]];

	vec = ts->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[g_iFixAtomType[0]])->GetAt(g_iFixAtom[0])] - CxVector3(g_fBoxX/2.0,g_fBoxY/2.0,g_fBoxZ/2.0);

	ts->CenterPos(vec);*/

	ts->FoldAtomsPositive();

	try { m_pContainer = new container_periodic_poly(g_fBoxX/1000.0,0,g_fBoxY/1000.0,0,0,g_fBoxZ/1000.0,m_iBlocksX,m_iBlocksY,m_iBlocksZ,g_iVoroMemory); } catch(...) { m_pContainer = NULL; }
	if (m_pContainer == NULL) NewException((double)sizeof(container_periodic_poly),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	for (z=0;z<g_iGesAtomCount;z++)
		m_pContainer->put(z,ts->m_vaCoords[z][0]/1000.0,ts->m_vaCoords[z][1]/1000.0,ts->m_vaCoords[z][2]/1000.0,0.05);

	c_loop_all_periodic vl(*m_pContainer);

	atot = 0;

	try { tb = new bool[m_waSurfCoverAtom.GetSize()]; } catch(...) { tb = NULL; }
	if (tb == NULL) NewException((double)m_waSurfCoverAtom.GetSize()*sizeof(bool),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	for (z=0;z<m_waSurfCoverAtom.GetSize();z++)
		tb[z] = false;

	if (vl.start()) 
	{
		do 
		{
			if (m_pContainer->compute_cell(c,vl))
			{
				ijk=vl.ijk;
				q=vl.q;

				id = m_pContainer->id[ijk][q];

//				va = (CVoroAtom*)m_oaVoroAtoms[m_pAssignAtoms[id]];
				vm = (CVoroMolecule*)m_oaVoroMolecules[m_pAssignMolecules[id]];

				if ((vm->m_iMolecule == m_iSurfCoverMol) && (/*vm->m_iSingleMol == m_iSurfCoverSM*/m_waSurfCoverSM.Contains(vm->m_iSingleMol)))
				{
					c.neighbors(nb);
					faces = c.number_of_faces();
					c.face_areas(fa);

					for (z=0;z<faces;z++)
					{
						if (m_pAssignMolecules[id] != m_pAssignMolecules[nb[z]])
						{
							for (z3=0;z3<m_waSurfCoverMol.GetSize();z3++)
							{
								if ((g_waAtomMolIndex[nb[z]] == m_waSurfCoverMol[z3]) &&
									(g_waAtomElement[nb[z]] == m_waSurfCoverElem[z3]) &&
									(g_waAtomMolNumber[nb[z]] == m_waSurfCoverAtom[z3]))
								{
									tfa = (CxDoubleArray*)m_oaSurfCoverData[z3];

									if (!tb[z3])
									{
										tb[z3] = true;
										tfa->Add(fa[z]);
									} else (*tfa)[tfa->GetSize()-1] += fa[z];

									atot += fa[z];

									goto _done;
								}
							}
							eprintf("Error.\n");
_done:;
						}
					}
				}
			}
		} while (vl.inc());
	}

	for (z=0;z<m_waSurfCoverAtom.GetSize();z++)
	{
		if (!tb[z])
			((CxDoubleArray*)m_oaSurfCoverData[z])->Add(0);

		if (atot > 0)
			((CxDoubleArray*)m_oaSurfCoverData[z])->GetAt(((CxDoubleArray*)m_oaSurfCoverData[z])->GetSize()-1) /= atot;
	}

	delete m_pContainer;
	delete[] tb;
}


void CVoroWrapper::FinishSurfCover(const char *s)
{
	FILE *a;
	int z, z2;
	CMolecule *m;
	double t;

	a = OpenFileWrite(s,true);

	mfprintf(a,"# Step;  ");

	for (z=0;z<m_waSurfCoverMol.GetSize();z++)
	{
		m = (CMolecule*)g_oaMolecules[m_waSurfCoverMol[z]];
		mfprintf(a,"%s %s%d",m->m_sName,((CAtom*)g_oaAtoms[m->m_baAtomIndex[m_waSurfCoverElem[z]]])->m_sName,m_waSurfCoverAtom[z]+1);
		if (z < m_waSurfCoverMol.GetSize()-1)
			mfprintf(a,";  ");
	}
	mfprintf(a,"\n");

	mfprintf(a,"# Averages;  ");
	for (z2=0;z2<m_waSurfCoverMol.GetSize();z2++)
	{
		t = 0;
		for (z=0;z<((CxDoubleArray*)m_oaSurfCoverData[z2])->GetSize();z++)
			t += ((CxDoubleArray*)m_oaSurfCoverData[z2])->GetAt(z);

		if (t > 0)
			t /= ((CxDoubleArray*)m_oaSurfCoverData[z2])->GetSize();

		mfprintf(a,"%f",t*100.0f);
		if (z2 < m_waSurfCoverMol.GetSize()-1)
			mfprintf(a,";  ");
	}
	mfprintf(a,"\n");

	for (z=0;z<((CxDoubleArray*)m_oaSurfCoverData[0])->GetSize();z++)
	{
		mfprintf(a,"%d;  ",z);
		for (z2=0;z2<m_waSurfCoverMol.GetSize();z2++)
		{
			mfprintf(a,"%f",((CxDoubleArray*)m_oaSurfCoverData[z2])->GetAt(z)*100.0f);
			if (z2 < m_waSurfCoverMol.GetSize()-1)
				mfprintf(a,";  ");
		}
		mfprintf(a,"\n");
	}

	fclose(a);
}


void CVoroWrapper::WritePOVHeader_WholeBox(FILE *b)
{
	double lb;
	int z;
	CElement *el;
	
	mfprintf(b,"/*****************************************************************************\n");
	mfprintf(b,"********************* General Parameter Values *******************************\n");
	mfprintf(b,"*****************************************************************************/\n\n");


	mfprintf(b,"/**** Atoms ****/\n");

	mfprintf(b,"#declare atom_draw          = true;\n");
	mfprintf(b,"#declare atom_r             = 0.65;\n");
	mfprintf(b,"#declare atom_specular      = 0.6;\n");
	mfprintf(b,"#declare atom_reflect       = 0;\n");
	mfprintf(b,"#declare atom_ambient       = 0.2;\n");
	mfprintf(b,"#declare atom_diffuse       = 0.8;\n");
	mfprintf(b,"//#declare atom_color         = < 1.0, 1.0, 1.0, 0, 0 >;\n");
	mfprintf(b,"//#declare atom_trans         = 0.7;\n");

	mfprintf(b,"\n");

	mfprintf(b,"#declare atom_draw_halo1      = true;\n");
	mfprintf(b,"#declare atom_r_halo1         = 0.01;\n");
	mfprintf(b,"#declare atom_d_halo1         = 0.0125;\n");
	mfprintf(b,"#declare atom_color_halo1     = < 0, 0, 0, 0, 1 >;\n");
	mfprintf(b,"#declare atom_specular_halo1  = 0;\n");
	mfprintf(b,"#declare atom_reflect_halo1   = 0;\n");
	mfprintf(b,"#declare atom_ambient_halo1   = 1.0;\n");
	mfprintf(b,"#declare atom_diffuse_halo1   = 0;\n");

	mfprintf(b,"\n");

	mfprintf(b,"/**** Voronoi Surface ****/\n");

	mfprintf(b,"#declare voro_specular      = 0.0;\n");
	mfprintf(b,"#declare voro_reflect       = 0;\n");
	mfprintf(b,"#declare voro_ambient       = 0.2;\n");
	mfprintf(b,"#declare voro_diffuse       = 1.0;\n");

	mfprintf(b,"\n");

	mfprintf(b,"/**** Other Variables ****/\n");

	lb = g_fBoxX;
	if (g_fBoxY > lb)
		lb = g_fBoxY;
	if (g_fBoxZ > lb)
		lb = g_fBoxZ;

	mfprintf(b,"#declare var_outertrans   = 1.0-var_outeropac;\n");
	mfprintf(b,"#declare cyl_r            = 0.015 * %.6f;\n",lb/2121.24);
	mfprintf(b,"#declare doclip           = true;\n\n");

	mfprintf(b,"#declare camera_dist = %f;\n",lb);
	mfprintf(b,"#declare camera_x = 10.0 * camera_dist / 1566.0 * sin(rot_angle) / var_zoom;\n");
	mfprintf(b,"#declare camera_y = 2.5 * camera_dist / 1566.0 / var_zoom;\n");
	mfprintf(b,"#declare camera_z = 10.0 * camera_dist /1566.0 * cos(rot_angle) / var_zoom;\n");

	mfprintf(b,"\ncamera {\n");
	mfprintf(b,"	location < camera_x, camera_y, camera_z >\n");
	mfprintf(b,"	sky y\n");
	mfprintf(b,"	right -0.3*x*image_width/image_height\n");
	mfprintf(b,"	up 0.3*y\n");
	mfprintf(b,"	look_at < 0, 0, 0 >\n");
	mfprintf(b,"}\n");
	mfprintf(b,"\n");
	mfprintf(b,"// Solid colored Background\n");
	mfprintf(b,"background { rgb < 1.0, 1.0, 1.0 > }\n");
	mfprintf(b,"\n");


	mfprintf(b,"/***** Color Gradient Background ****/\n");
	mfprintf(b,"sky_sphere {\n");
	mfprintf(b,"  pigment {\n");
	mfprintf(b,"    gradient y\n");
	mfprintf(b,"    color_map {\n");
	mfprintf(b,"      [ 0 color rgb < 0.9, 0.9, 0.9 > ]\n");
	mfprintf(b,"      [ 1 color rgb < 0.85, 0.75, 1.0 > ]\n");
	mfprintf(b,"    }\n");
	mfprintf(b,"    scale 1\n");
	mfprintf(b,"    translate -0.05\n");
	mfprintf(b,"  }\n");
	mfprintf(b,"}\n\n");

	mfprintf(b,"// Two light sources\n");
	mfprintf(b,"light_source { < 21.54*sin(rot_angle-0.7297), 20, 21.54*cos(rot_angle-0.7297) > color rgb 0.8 }\n");
	mfprintf(b,"light_source { < 32.02*sin(rot_angle+0.547), 12, 32.02*cos(rot_angle+0.547) > color rgb 0.5 }\n\n");

	if (m_bWritePOVMovie)
	{
		if (strlen(m_sPOVText) != 0)
		{
			mfprintf(b,"text {\n");
			mfprintf(b,"  ttf \"courbd.ttf\" title_text 0.01, 0\n");
			mfprintf(b,"  pigment { rgb < 0.2, 0.2, 0.2 > } \n");
			mfprintf(b,"  no_shadow\n");
			mfprintf(b,"  scale %.6f / var_zoom\n");
			mfprintf(b,"  translate < -2.65*camera_dist/2121.24, 1.90*camera_dist/2121.24, 0 > / var_zoom\n");
			mfprintf(b,"  rotate < 0, rot_angle, 0 >\n");
			mfprintf(b,"}\n\n");
		}
	}

	mfprintf(b,"// The Simulation Box\n");
	mfprintf(b,"union {\n");
	mfprintf(b,"  cylinder { < %f, %f, %f > * var_explode, < %f, %f, %f > * var_explode, cyl_r open }\n",-g_fBoxX/2000.0*m_fPOVExplode,-g_fBoxY/2000.0*m_fPOVExplode,-g_fBoxZ/2000.0*m_fPOVExplode, g_fBoxX/2000.0*m_fPOVExplode,-g_fBoxY/2000.0*m_fPOVExplode,-g_fBoxZ/2000.0*m_fPOVExplode);
	mfprintf(b,"  cylinder { < %f, %f, %f > * var_explode, < %f, %f, %f > * var_explode, cyl_r open }\n", g_fBoxX/2000.0*m_fPOVExplode,-g_fBoxY/2000.0*m_fPOVExplode,-g_fBoxZ/2000.0*m_fPOVExplode, g_fBoxX/2000.0*m_fPOVExplode, g_fBoxY/2000.0*m_fPOVExplode,-g_fBoxZ/2000.0*m_fPOVExplode);
	mfprintf(b,"  cylinder { < %f, %f, %f > * var_explode, < %f, %f, %f > * var_explode, cyl_r open }\n", g_fBoxX/2000.0*m_fPOVExplode, g_fBoxY/2000.0*m_fPOVExplode,-g_fBoxZ/2000.0*m_fPOVExplode,-g_fBoxX/2000.0*m_fPOVExplode, g_fBoxY/2000.0*m_fPOVExplode,-g_fBoxZ/2000.0*m_fPOVExplode);
	mfprintf(b,"  cylinder { < %f, %f, %f > * var_explode, < %f, %f, %f > * var_explode, cyl_r open }\n",-g_fBoxX/2000.0*m_fPOVExplode, g_fBoxY/2000.0*m_fPOVExplode,-g_fBoxZ/2000.0*m_fPOVExplode,-g_fBoxX/2000.0*m_fPOVExplode,-g_fBoxY/2000.0*m_fPOVExplode,-g_fBoxZ/2000.0*m_fPOVExplode);
	mfprintf(b,"  cylinder { < %f, %f, %f > * var_explode, < %f, %f, %f > * var_explode, cyl_r open }\n",-g_fBoxX/2000.0*m_fPOVExplode,-g_fBoxY/2000.0*m_fPOVExplode, g_fBoxZ/2000.0*m_fPOVExplode, g_fBoxX/2000.0*m_fPOVExplode,-g_fBoxY/2000.0*m_fPOVExplode, g_fBoxZ/2000.0*m_fPOVExplode);
	mfprintf(b,"  cylinder { < %f, %f, %f > * var_explode, < %f, %f, %f > * var_explode, cyl_r open }\n", g_fBoxX/2000.0*m_fPOVExplode,-g_fBoxY/2000.0*m_fPOVExplode, g_fBoxZ/2000.0*m_fPOVExplode, g_fBoxX/2000.0*m_fPOVExplode, g_fBoxY/2000.0*m_fPOVExplode, g_fBoxZ/2000.0*m_fPOVExplode);
	mfprintf(b,"  cylinder { < %f, %f, %f > * var_explode, < %f, %f, %f > * var_explode, cyl_r open }\n", g_fBoxX/2000.0*m_fPOVExplode, g_fBoxY/2000.0*m_fPOVExplode, g_fBoxZ/2000.0*m_fPOVExplode,-g_fBoxX/2000.0*m_fPOVExplode, g_fBoxY/2000.0*m_fPOVExplode, g_fBoxZ/2000.0*m_fPOVExplode);
	mfprintf(b,"  cylinder { < %f, %f, %f > * var_explode, < %f, %f, %f > * var_explode, cyl_r open }\n",-g_fBoxX/2000.0*m_fPOVExplode, g_fBoxY/2000.0*m_fPOVExplode, g_fBoxZ/2000.0*m_fPOVExplode,-g_fBoxX/2000.0*m_fPOVExplode,-g_fBoxY/2000.0*m_fPOVExplode, g_fBoxZ/2000.0*m_fPOVExplode);
	mfprintf(b,"  cylinder { < %f, %f, %f > * var_explode, < %f, %f, %f > * var_explode, cyl_r open }\n",-g_fBoxX/2000.0*m_fPOVExplode,-g_fBoxY/2000.0*m_fPOVExplode,-g_fBoxZ/2000.0*m_fPOVExplode,-g_fBoxX/2000.0*m_fPOVExplode,-g_fBoxY/2000.0*m_fPOVExplode, g_fBoxZ/2000.0*m_fPOVExplode);
	mfprintf(b,"  cylinder { < %f, %f, %f > * var_explode, < %f, %f, %f > * var_explode, cyl_r open }\n", g_fBoxX/2000.0*m_fPOVExplode,-g_fBoxY/2000.0*m_fPOVExplode,-g_fBoxZ/2000.0*m_fPOVExplode, g_fBoxX/2000.0*m_fPOVExplode,-g_fBoxY/2000.0*m_fPOVExplode, g_fBoxZ/2000.0*m_fPOVExplode);
	mfprintf(b,"  cylinder { < %f, %f, %f > * var_explode, < %f, %f, %f > * var_explode, cyl_r open }\n", g_fBoxX/2000.0*m_fPOVExplode, g_fBoxY/2000.0*m_fPOVExplode,-g_fBoxZ/2000.0*m_fPOVExplode, g_fBoxX/2000.0*m_fPOVExplode, g_fBoxY/2000.0*m_fPOVExplode, g_fBoxZ/2000.0*m_fPOVExplode);
	mfprintf(b,"  cylinder { < %f, %f, %f > * var_explode, < %f, %f, %f > * var_explode, cyl_r open }\n",-g_fBoxX/2000.0*m_fPOVExplode, g_fBoxY/2000.0*m_fPOVExplode,-g_fBoxZ/2000.0*m_fPOVExplode,-g_fBoxX/2000.0*m_fPOVExplode, g_fBoxY/2000.0*m_fPOVExplode, g_fBoxZ/2000.0*m_fPOVExplode);
	mfprintf(b,"  pigment { rgbft < 0, 0, 1, 0, 0 > }\n  finish { reflection 0 specular 0.7 ambient 0.2 diffuse 0.7 }\n  no_shadow\n}\n\n");


	if (m_bPOVMolecules)
	{
		mfprintf(b,"\n/**** Element Radii ****/\n");
		for (z=0;z<g_oaAtoms.GetSize();z++)
		{
			if (z == g_iVirtAtomType)
				continue;
			el = ((CAtom*)g_oaAtoms[z])->m_pElement;
			mfprintf(b,"#declare elem_%s_r  = %f * 0.8;\n",el->m_sLabel,el->m_fRadius/1000.0);
		}
		mfprintf(b,"\n");
	}

	mfprintf(b,"/*****************************************************************************\n");
	mfprintf(b,"*****************************************************************************/\n\n");
}


void CVoroWrapper::WritePOV_WholeBox(CTimeStep *ts, const char *s)
{
	FILE *a, *b;
	int ijk, q, z, z2, id, faces, /*co,*/ fc, ti, ti2, zx, zy, zz;
	double *pp, tx, ty, tz, angle;
	voronoicell_neighbor c;
	vector<int> nb;
	vector<int> fo;
	vector<double> fa;
	vector<int> fv;
	CxDVector3 vec1, vec2, vec3, vec1b, vec2b, vecA, vecB, vecC, vecD, cam, vec0, vat, vcen, color;
	CxDVector3 plane_normal;
	CxIntArray nbmol, nbsm;
	CxIntArray nbatoms;
	double minx, miny, minz, maxx, maxy, maxz, lb;
	char buf[256];
//	CElement *el;

	if (m_bWritePOVMovie)
		sprintf(m_sPOVText,"t=%6.2fps",g_iSteps/g_iStride*g_fTimestepLength/1000.0);

	if (m_bWritePOVMovie)
	{
#ifdef TARGET_WINDOWS
		if (s == NULL)
		{
			if (m_iPOVThreads == 1)
			{
				sprintf(buf,"POV_Box\\voro_box_%05d.pov",m_iPOVFrameCounter);
				mfprintf(m_fPOVBoxScript[m_iPOVThreadPos],"%s /NR /EXIT /RENDER voro_box_%05d.pov +w%d +h%d +fn +a0.3 -J0.0 +Oframe_%05d.png\n",m_sPOVExe,m_iPOVFrameCounter,m_iPOVResX,m_iPOVResY,m_iPOVFrameCounter);
			} else
			{
				sprintf(buf,"POV_Box\\%d\\voro_box_%05d.pov",m_iPOVThreadPos+1,m_iPOVFrameCounter);
				mfprintf(m_fPOVBoxScript[m_iPOVThreadPos],"%s /NR /EXIT /RENDER voro_box_%05d.pov +w%d +h%d +fn +a0.3 -J0.0 +Oframe_%05d.png\n",m_sPOVExe,m_iPOVFrameCounter,m_iPOVResX,m_iPOVResY,m_iPOVFrameCounter);
			}
		} else
		{
			strcpy(buf,s);
			mfprintf(m_fPOVBoxScript[m_iPOVThreadPos],"%s /NR /EXIT /RENDER %s +w%d +h%d +fn +a0.3 -J0.0\n",m_sPOVExe,buf,m_iPOVResX,m_iPOVResY);
		}
#else
		if (s == NULL)
		{
			if (m_iPOVThreads == 1)
			{
				sprintf(buf,"POV_Box/voro_box_%05d.pov",m_iPOVFrameCounter);
				mfprintf(m_fPOVBoxScript[m_iPOVThreadPos],"%s voro_box_%05d.pov +w%d +h%d +fn +a0.3 -d -J0.0 +Oframe_%05d.png\n",m_sPOVExe,m_iPOVFrameCounter,m_iPOVResX,m_iPOVResY,m_iPOVFrameCounter);
			} else
			{
				sprintf(buf,"POV_Box/%d/voro_box_%05d.pov",m_iPOVThreadPos+1,m_iPOVFrameCounter);
				mfprintf(m_fPOVBoxScript[m_iPOVThreadPos],"%s voro_box_%05d.pov +w%d +h%d +fn +a0.3 -d -J0.0 +Oframe_%05d.png\n",m_sPOVExe,m_iPOVFrameCounter,m_iPOVResX,m_iPOVResY,m_iPOVFrameCounter);
			}
		} else
		{
			strcpy(buf,s);
			mfprintf(m_fPOVBoxScript[m_iPOVThreadPos],"%s %s +w%d +h%d +fn +a0.3 -d -J0.0\n",m_sPOVExe,buf,m_iPOVResX,m_iPOVResY);
		}
#endif
	} else
	{
		strcpy(buf,s);
#ifdef TARGET_WINDOWS
		b = OpenFileWrite("render_box.bat",true);
		mfprintf(b,"%s /NR /EXIT /RENDER %s +w%d +h%d +fn +a0.3 -J0.0\n",m_sPOVExe,buf,m_iPOVResX,m_iPOVResY);
#else
		b = OpenFileWrite("render_box",true);
		mfprintf(b,"%s %s +w%d +h%d +fn +a0.3 -d -J0.0\n",m_sPOVExe,buf,m_iPOVResX,m_iPOVResY);
#endif
		fclose(b);
	}

	if (m_bWritePOVMovie)
	{
		fflush(m_fPOVBoxScript[m_iPOVThreadPos]);

		m_iPOVThreadPos++;
		if (m_iPOVThreadPos >= m_iPOVThreads)
			m_iPOVThreadPos = 0;
	}

	angle = m_fPOVAngle / 180.0 * Pi;

	lb = g_fBoxX;
	if (g_fBoxY > lb)
		lb = g_fBoxY;
	if (g_fBoxZ > lb)
		lb = g_fBoxZ;

	cam[0] = 10.0 * lb/1566.0 * sin(angle);
	cam[1] = 2.5 * lb/1566.0;
	cam[2] = 10.0 * lb/1566.0 * cos(angle);

	try { m_pContainer = new container_periodic_poly(g_fBoxX/1000.0,0,g_fBoxY/1000.0,0,0,g_fBoxZ/1000.0,m_iBlocksX,m_iBlocksY,m_iBlocksZ,g_iVoroMemory); } catch(...) { m_pContainer = NULL; }
	if (m_pContainer == NULL) NewException((double)sizeof(container_periodic_poly),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	for (z=0;z<g_iGesAtomCount;z++)
		m_pContainer->put(z,ts->m_vaCoords[z][0]/1000.0,ts->m_vaCoords[z][1]/1000.0,ts->m_vaCoords[z][2]/1000.0,0.05);

	c_loop_all_periodic vl(*m_pContainer);

	b = OpenFileWrite(buf,true);

	mfprintf(b,"// Written by TRAVIS\n");
	mfprintf(b,"// See http://www.travis-analyzer.de\n\n");
	mfprintf(b,"#version 3.6; // Please use version 3.7 anyway\n");
	mfprintf(b,"\n");

	mfprintf(b,"/**********************************************************************************************\n");
	mfprintf(b,"*********************** Variable Values *******************************************************\n");
	mfprintf(b,"**********************************************************************************************/\n\n");

	mfprintf(b,"#declare rot_angle        = %.6f; // In radians, not degrees\n",angle);
	mfprintf(b,"#declare var_zoom         = %.6f;\n",m_fPOVZoom);

	if (m_bWritePOVMovie)
		mfprintf(b,"#declare title_text       = \"%s\";\n",m_sPOVText);

/*	mfprintf(b,"#declare face_trans       = %.6f;\n",m_fPOVFaceTrans);
	mfprintf(b,"#declare edge_trans       = %.6f;\n",m_fPOVEdgeTrans);
	mfprintf(b,"#declare nb_trans         = %.6f;\n",m_fPOVNbTrans);
	mfprintf(b,"#declare ray_trans        = %.6f;\n",m_fPOVRayTrans);
	mfprintf(b,"#declare nb_highlight     = %.6f;\n",m_fPOVNbHighlightFac);
	mfprintf(b,"#declare facebleach_val   = %.6f;\n",m_fPOVFaceBleach);
	mfprintf(b,"#declare facebleach_r     = %.6f;\n",m_vPOVFaceBleach[0]);
	mfprintf(b,"#declare facebleach_g     = %.6f;\n",m_vPOVFaceBleach[1]);
	mfprintf(b,"#declare facebleach_b     = %.6f;\n",m_vPOVFaceBleach[2]);*/

	mfprintf(b,"#declare var_scale        = %.6f;\n",m_fPOVScale);
	mfprintf(b,"#declare var_clip         = %.6f;\n",m_fPOVClip);
	mfprintf(b,"#declare var_explode      = %.6f;\n",m_fPOVExplode);
	mfprintf(b,"#declare var_outeropac    = %.6f;\n",m_fPOVBoxOuterOpacity);


	mfprintf(b,"\n/**********************************************************************************************\n");
	mfprintf(b,"**********************************************************************************************/\n\n");

	mfprintf(b,"#include \"settings_box.inc\"\n");

	if (!m_bIncludeWritten)
	{
		m_bIncludeWritten = true;
		if (m_bWritePOVMovie)
		{
#ifdef TARGET_WINDOWS
			a = OpenFileWrite("POV_Box\\settings_box.inc",true);
#else
			a = OpenFileWrite("POV_Box/settings_box.inc",true);
#endif
		} else
			a = OpenFileWrite("settings_box.inc",true);

		WritePOVHeader_WholeBox(a);
		fclose(a);
	}

	mfprintf(b,"\n/****** Below this line, no changes should be necessary **********/\n\n\n");

	if (m_fPOVClip < 2.0f)
	{
		mfprintf(b,"#if (doclip)\n");
		mfprintf(b,"intersection\n{\n");
		mfprintf(b,"#end\n\n");
	}

	mfprintf(b,"  union\n  {\n\n");

//	mfprintf(b,"    box { < %.6f, %.6f, %.6f >, < %.6f, %.6f, %.6f > texture { pigment { rgbft < 0, 0, 0, 0, 0 > } } }\n\n",-g_fBoxX/1000.0,-g_fBoxY/1000.0,-g_fBoxZ/1000.0, g_fBoxX/1000.0, g_fBoxY/1000.0, g_fBoxZ/1000.0);

	if (vl.start()) 
	{
		do 
		{
			if (m_pContainer->compute_cell(c,vl))
			{
//				co++;

				ijk=vl.ijk;
				q=vl.q;
				pp=m_pContainer->p[ijk]+m_pContainer->ps*q;

				id = m_pContainer->id[ijk][q];

//				va = (CVoroAtom*)m_oaVoroAtoms[m_pAssignAtoms[id]];
//				vm = (CVoroMolecule*)m_oaVoroMolecules[m_pAssignMolecules[id]];

				c.neighbors(nb);
				faces = c.number_of_faces();
				c.face_vertices(fv);

				vat[0] = ts->m_vaCoords[id][0]/1000.0-g_fBoxX/2000.0;
				vat[1] = ts->m_vaCoords[id][1]/1000.0-g_fBoxY/2000.0;
				vat[2] = ts->m_vaCoords[id][2]/1000.0-g_fBoxZ/2000.0;

				minx = 1e30;
				miny = 1e30;
				minz = 1e30;
				maxx = -1e30;
				maxy = -1e30;
				maxz = -1e30;

				color[0] = ((m_iaPOVBoxAtomColors[id] & 0xFF0000) >> 16) / 255.0;
				color[1] = ((m_iaPOVBoxAtomColors[id] & 0xFF00) >> 8) / 255.0;
				color[2] = (m_iaPOVBoxAtomColors[id] & 0xFF) / 255.0;

		//		mprintf("%03d: %06X --> %f, %f, %f\n",id,m_iaPOVBoxAtomColors[id],color[0],color[1],color[2]);

				ti = 0;
				ti2 = 0;
				vcen = 0;
				for (z=0;z<faces;z++)
				{
					fc = fv[ti];

					for (z2=0;z2<fc;z2++)
					{
						vec1 = CxVector3(*pp+c.pts[fv[ti+z2+1]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+z2+1]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+z2+1]*3+2]*0.5-g_fBoxZ/2000.0);
						vcen += vec1;
						ti2++;

						if (vec1[0] > maxx)
							maxx = vec1[0];
						if (vec1[0] < minx)
							minx = vec1[0];
						if (vec1[1] > maxy)
							maxy = vec1[1];
						if (vec1[1] < miny)
							miny = vec1[1];
						if (vec1[2] > maxz)
							maxz = vec1[2];
						if (vec1[2] < minz)
							minz = vec1[2];

					}
					ti += fv[ti]+1;
				}

				vcen /= ti2;

				vec1 = m_fPOVAtomCenter*vat + (1.0-m_fPOVAtomCenter)*vcen;
				vcen = vec1;

				for (zx=-1;zx<=1;zx++)
				{
					if ((zx == -1) && (maxx < g_fBoxX/2000.0))
						continue;
					if ((zx == 1) && (minx > -g_fBoxX/2000.0))
						continue;

					for (zy=-1;zy<=1;zy++)
					{
						if ((zy == -1) && (maxy < g_fBoxY/2000.0))
							continue;
						if ((zy == 1) && (miny > -g_fBoxY/2000.0))
							continue;

						for (zz=-1;zz<=1;zz++)
						{
							if ((zz == -1) && (maxz < g_fBoxZ/2000.0))
								continue;
							if ((zz == 1) && (minz > -g_fBoxZ/2000.0))
								continue;

							if (((zx != 0) || (zy != 0) || (zz != 0)) && (m_fPOVBoxOuterOpacity == 0))
								continue;

							if (m_bPOVMolecules)
							{
								vec1 = vat + CxVector3(zx*g_fBoxX,zy*g_fBoxY,zz*g_fBoxZ)/1000.0;

								mfprintf(b,"    sphere { < %g, %g, %g >, elem_%s_r * atom_r\n",vec1[0],vec1[1],vec1[2],((CAtom*)g_oaAtoms[g_waAtomRealElement[id]])->m_pElement->m_sLabel);

								if ((zx == 0) && (zy == 0) && (zz == 0))
								{
									mfprintf(b,"      pigment { rgbft < %f, %f, %f, 0, 0 > }\n",color[0],color[1],color[2]);
									mfprintf(b,"      finish { reflection atom_reflect specular atom_specular ambient atom_ambient diffuse atom_diffuse } no_shadow }\n\n");
								} else
								{
									mfprintf(b,"      pigment { rgbft < %f, %f, %f, 0, var_outertrans > }\n",color[0],color[1],color[2]);
									mfprintf(b,"      finish { reflection atom_reflect specular atom_specular*var_outeropac ambient atom_ambient diffuse atom_diffuse } no_shadow }\n\n");
								}


								vec2 = cam - vec1;
								vec2.Normalize();

								mfprintf(b,"    disc { < %g - (%g) * atom_d_halo1, %g - (%g) * atom_d_halo1, %g - (%g) * atom_d_halo1 >,\n",vec1[0],vec2[0],vec1[1],vec2[1],vec1[2],vec2[2]);
								mfprintf(b,"      < %g, %g, %g >, (elem_%s_r * atom_r) + atom_r_halo1, elem_%s_r * atom_r\n",vec2[0],vec2[1],vec2[2],((CAtom*)g_oaAtoms[g_waAtomRealElement[id]])->m_pElement->m_sLabel,((CAtom*)g_oaAtoms[g_waAtomRealElement[id]])->m_pElement->m_sLabel);

								if ((zx == 0) && (zy == 0) && (zz == 0))
									mfprintf(b,"      pigment { rgbft atom_color_halo1*0 }");
										else mfprintf(b,"      pigment { rgbft atom_color_halo1*var_outertrans*var_outertrans }");

								mfprintf(b,"finish { reflection atom_reflect_halo1 specular atom_specular_halo1 ambient atom_ambient_halo1 diffuse atom_diffuse_halo1 } no_reflection no_shadow }\n\n");
							}

							if (m_faPOVBoxAtomVisible[id] < 0.001)
								continue;

							if (m_fPOVScale < 0.001)
								continue;

							mfprintf(b,"\n    // Voronoi cell %d (x=%d, y=%d, z=%d)\n\n",m_pContainer->id[vl.ijk][vl.q],zx,zy,zz);

							mfprintf(b,"    object { \n");
							mfprintf(b,"      mesh { \n");

							ti = 0;

							for (z=0;z<faces;z++)
							{
						//		mprintf("Face %d/%d:\n",z+1,faces);
								fc = fv[ti];
						//		mprintf("  %d Vertices\n",fc);

								tx = 0;
								ty = 0;
								tz = 0;
								for (z2=0;z2<fc;z2++)
								{
									vec1 = CxVector3(*pp+c.pts[fv[ti+z2+1]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+z2+1]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+z2+1]*3+2]*0.5-g_fBoxZ/2000.0);

									tx += vec1[0];
									ty += vec1[1];
									tz += vec1[2];
								}

								tx /= fc;
								ty /= fc;
								tz /= fc;

								vec0[0] = tx;
								vec0[1] = ty;
								vec0[2] = tz;

								vec0 -= vcen;

								for (z2=0;z2<fc;z2++)
								{
									vec1 = CxVector3(*pp+c.pts[fv[ti+z2+1]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+z2+1]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+z2+1]*3+2]*0.5-g_fBoxZ/2000.0);

									if (z2 < fc-1)
										vec2 = CxVector3(*pp+c.pts[fv[ti+z2+2]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+z2+2]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+z2+2]*3+2]*0.5-g_fBoxZ/2000.0);
											else vec2 = CxVector3(*pp+c.pts[fv[ti+1]*3]*0.5-g_fBoxX/2000.0,pp[1]+c.pts[fv[ti+1]*3+1]*0.5-g_fBoxY/2000.0,pp[2]+c.pts[fv[ti+1]*3+2]*0.5-g_fBoxZ/2000.0);

									vec1 -= vcen;
									vec2 -= vcen;

								/*	vec3 = CrossP(vec0-vec1,vec0-vec2);
									if (DotP(vec3,vat-vec0) < 0)
										vec3 = -1.0f * vec3;

									vec3.Normalize();

									vec3 *= sepwidth;*/

									mfprintf(b,"        triangle { < %.6f, %.6f, %.6f >, < %.6f, %.6f, %.6f >, < %.6f, %.6f, %.6f > }\n",vec0[0],vec0[1],vec0[2],vec1[0],vec1[1],vec1[2],vec2[0],vec2[1],vec2[2]);
								}

								ti += fv[ti]+1;
							}

				//			mfprintf(b,"        inside_vector < 0, 0, 0 >\n");
							mfprintf(b,"        inside_vector < %.6f, %.6f, %.6f >\n",vat[0]+zx*g_fBoxX/1000.0,vat[1]+zy*g_fBoxY/1000.0,vat[2]+zz*g_fBoxZ/1000.0);

							if ((zx == 0) && (zy == 0) && (zz == 0))
								mfprintf(b,"        texture { pigment { rgbft < %f, %f, %f, 0, %f > } finish { reflection voro_reflect specular voro_specular ambient voro_ambient diffuse voro_diffuse }  }\n",color[0],color[1],color[2],1.0-m_faPOVBoxAtomVisible[id]);
									else mfprintf(b,"        texture { pigment { rgbft < %f, %f, %f, 0, %f + var_outertrans - %f*var_outertrans > } finish { reflection voro_reflect specular voro_specular ambient voro_ambient diffuse voro_diffuse }  }\n",color[0],color[1],color[2],1.0-m_faPOVBoxAtomVisible[id],1.0-m_faPOVBoxAtomVisible[id]);

							mfprintf(b,"        scale var_scale\n");
							mfprintf(b,"        no_shadow\n");
							mfprintf(b,"      }\n\n");
							mfprintf(b,"      translate < %.6f, %.6f, %.6f > * var_explode\n",vcen[0]+zx*g_fBoxX/1000.0,vcen[1]+zy*g_fBoxY/1000.0,vcen[2]+zz*g_fBoxZ/1000.0);
							mfprintf(b,"      no_shadow\n");
							mfprintf(b,"    }\n\n");
						}
					}
				}
			}
		} while (vl.inc());
	}

	mfprintf(b,"  }\n\n");

	if (m_fPOVClip < 2.0)
	{
		mfprintf(b,"#if (doclip)\n");
		mfprintf(b,"  box { < %.6f*%.6f, %.6f*%.6f, %.6f*%.6f > * var_clip * var_explode, < %.6f*%.6f, %.6f*%.6f, %.6f*%.6f > * var_clip * var_explode }\n",g_fBoxX/2000.0,m_vPOVBoxClipLow[0],g_fBoxY/2000.0,m_vPOVBoxClipLow[1],g_fBoxZ/2000.0,m_vPOVBoxClipLow[2],g_fBoxX/2000.0,m_vPOVBoxClipHigh[0],g_fBoxY/2000.0,m_vPOVBoxClipHigh[1],g_fBoxZ/2000.0,m_vPOVBoxClipHigh[2]);
		mfprintf(b,"  cutaway_textures\n");
		mfprintf(b,"  no_shadow\n");
		mfprintf(b,"}\n");
		mfprintf(b,"#end\n\n");
	}

	fclose(b);

	delete m_pContainer;
}


CVoroChoreographyPart* CVoroWrapper::AddVoroChoreographyPart(float seconds)
{	
	CVoroChoreographyPart *p, *p2;
	int z;

	p = NULL;

	if (seconds > 0)
		p = new CVoroChoreographyPart();

	if (m_oaBoxChoreography.GetSize() == 0)
	{
		if (seconds > 0)
		{
			p->m_iFrameStart = 0;
			p->m_iFrameEnd = seconds*m_fPOV_FPS-1;
		}
	} else
	{
		if (seconds > 0)
		{
			p->m_iFrameStart = ((CVoroChoreographyPart*)m_oaBoxChoreography[m_oaBoxChoreography.GetSize()-1])->m_iFrameEnd+1;
			p->m_iFrameEnd = p->m_iFrameStart + seconds*m_fPOV_FPS-1;
		}

		p2 = (CVoroChoreographyPart*)m_oaBoxChoreography[m_oaBoxChoreography.GetSize()-1];

		for (z=0;z<3;z++)
		{
			if (p2->m_vClipLowEnd[z] > 1.0e15)
				p2->m_vClipLowEnd[z] = p2->m_vClipLowBegin[z];

			if (p2->m_vClipHighEnd[z] > 1.0e15)
				p2->m_vClipHighEnd[z] = p2->m_vClipHighBegin[z];
		}

		if (p2->m_fFaceTransEnd < 0)
			p2->m_fFaceTransEnd = p2->m_fFaceTransBegin;

		if (p2->m_fEdgeTransEnd < 0)
			p2->m_fEdgeTransEnd = p2->m_fEdgeTransBegin;

		if (p2->m_fNbTransEnd < 0)
			p2->m_fNbTransEnd = p2->m_fNbTransBegin;

		if (p2->m_fRayTransEnd < 0)
			p2->m_fRayTransEnd = p2->m_fRayTransBegin;

		if (p2->m_fNbHLFacEnd < 0)
			p2->m_fNbHLFacEnd = p2->m_fNbHLFacBegin;

		if (p2->m_fFaceBleachEnd < 0)
			p2->m_fFaceBleachEnd = p2->m_fFaceBleachBegin;

		if (p2->m_fAngleEnd < 0)
			p2->m_fAngleEnd = p2->m_fAngleBegin;

		if (p2->m_fScaleEnd < 0)
			p2->m_fScaleEnd = p2->m_fScaleBegin;

		if (p2->m_fExplodeEnd < 0)
			p2->m_fExplodeEnd = p2->m_fExplodeBegin;

		if (p2->m_fZoomEnd < 0)
			p2->m_fZoomEnd = p2->m_fZoomBegin;

		if (p2->m_fClipEnd < 0)
			p2->m_fClipEnd = p2->m_fClipBegin;

		if (p2->m_fOuterOpacEnd < 0)
			p2->m_fOuterOpacEnd = p2->m_fOuterOpacBegin;

		if (p2->m_fAtomCenterEnd < 0)
			p2->m_fAtomCenterEnd = p2->m_fAtomCenterBegin;

		if (p2->m_fMoleculesEnd < 0)
			p2->m_fMoleculesEnd = p2->m_fMoleculesBegin;

		for (z=0;z<g_iGesAtomCount;z++)
			if (p2->m_faAtomVisibleEnd[z] < 0)
				p2->m_faAtomVisibleEnd[z] = p2->m_faAtomVisibleBegin[z];


		p2->m_fStepEnd = p2->m_fStepBegin + (p2->m_iFrameEnd-p2->m_iFrameStart)*p2->m_fStepIncrement;

//		mprintf("Old StepEnd = Begin+ (FEnd-FStart)*Inc = %f + (%d - %d)*%f = %f\n",p2->m_fStepBegin,p2->m_iFrameEnd,p2->m_iFrameStart,p2->m_fStepIncrement,p2->m_fStepEnd);

		if (seconds > 0)
		{
			p->m_fStepBegin = p2->m_fStepEnd + p2->m_fStepIncrement;

//			mprintf("StepBegin = StepEnd + Inc = %f + %f = %f\n",p2->m_fStepEnd,p2->m_fStepIncrement,p->m_fStepBegin);

			for (z=0;z<3;z++)
			{
				p->m_vClipLowBegin[z] = p2->m_vClipLowEnd[z];
				p->m_vClipHighBegin[z] = p2->m_vClipHighEnd[z];
			}
			p->m_fFaceTransBegin = p2->m_fFaceTransEnd;
			p->m_fEdgeTransBegin = p2->m_fEdgeTransEnd;
			p->m_fNbTransBegin = p2->m_fNbTransEnd;
			p->m_fRayTransBegin = p2->m_fRayTransEnd;
			p->m_fFaceBleachBegin = p2->m_fFaceBleachEnd;
			p->m_fNbHLFacBegin = p2->m_fNbHLFacEnd;
			p->m_fAngleBegin = p2->m_fAngleEnd;
			p->m_fScaleBegin = p2->m_fScaleEnd;
			p->m_fAtomCenterBegin = p2->m_fAtomCenterEnd;
			p->m_fOuterOpacBegin = p2->m_fOuterOpacEnd;
			p->m_fClipBegin = p2->m_fClipEnd;
			p->m_fExplodeBegin = p2->m_fExplodeEnd;
			p->m_fAngleBegin = p2->m_fAngleEnd;
			p->m_fZoomBegin = p2->m_fZoomEnd;
			p->m_fMoleculesBegin = p2->m_fMoleculesEnd;

			p->m_fStepIncrement = p2->m_fStepIncrement;

//			mprintf("(A) Clip %f.\n",p->m_fClipBegin);

			for (z=0;z<g_iGesAtomCount;z++)
				p->m_faAtomVisibleBegin[z] = p2->m_faAtomVisibleEnd[z];
		}
	}
	
	if (seconds > 0)
	{
		m_oaBoxChoreography.Add(p);
		return p;
	} else return NULL;
}


CVoroChoreographyPart::CVoroChoreographyPart()
{
	int z;

	for (z=0;z<3;z++)
	{
		m_vClipLowBegin[z] = 1.0e20;
		m_vClipHighBegin[z] = 1.0e20;
	}
	m_fFaceTransBegin = -1;
	m_fEdgeTransBegin = -1;
	m_fNbTransBegin = -1;
	m_fRayTransBegin = -1;
	m_fNbHLFacBegin = -1;
	m_fFaceBleachBegin = -1;
	m_fStepBegin = -1;
	m_fZoomBegin = -1;
	m_fAngleBegin = -1;
	m_fScaleBegin = -1;
	m_fExplodeBegin = -1;
	m_fClipBegin = -1;
	m_fOuterOpacBegin = -1;
	m_fAtomCenterBegin = -1;
	m_fMoleculesBegin = -1;
	m_faAtomVisibleBegin.SetSize(g_iGesAtomCount);
	for (z=0;z<g_iGesAtomCount;z++)
		m_faAtomVisibleBegin[z] = -1;


	for (z=0;z<3;z++)
	{
		m_vClipLowEnd[z] = 1.0e20;
		m_vClipHighEnd[z] = 1.0e20;
	}
	m_fFaceTransEnd = -1;
	m_fEdgeTransEnd = -1;
	m_fNbTransEnd = -1;
	m_fRayTransEnd = -1;
	m_fNbHLFacEnd = -1;
	m_fFaceBleachEnd = -1;
	m_fStepEnd = -1;
	m_fZoomEnd = -1;
	m_fAngleEnd = -1;
	m_fScaleEnd = -1;
	m_fExplodeEnd = -1;
	m_fClipEnd = -1;
	m_fOuterOpacEnd = -1;
	m_fAtomCenterEnd = -1;
	m_fMoleculesEnd = -1;
	m_faAtomVisibleEnd.SetSize(g_iGesAtomCount);
	for (z=0;z<g_iGesAtomCount;z++)
		m_faAtomVisibleEnd[z] = -1;


	for (z=0;z<3;z++)
	{
		m_vClipLowIncrement[z] = 0;
		m_vClipHighIncrement[z] = 0;
	}
	m_fFaceTransIncrement = -1;
	m_fEdgeTransIncrement = -1;
	m_fNbTransIncrement = -1;
	m_fRayTransIncrement = -1;
	m_fNbHLFacIncrement = -1;
	m_fFaceBleachIncrement = -1;
	m_fMoleculesIncrement = 0;
	m_fZoomIncrement = 0;
	m_fStepIncrement = 0;
	m_fAngleIncrement = 0;
	m_fScaleIncrement = 0;
	m_fExplodeIncrement = 0;
	m_fClipIncrement = 0;
	m_fOuterOpacIncrement = 0;
	m_fAtomCenterIncrement = 0;
	m_faAtomVisibleIncrement.SetSize(g_iGesAtomCount);
	for (z=0;z<g_iGesAtomCount;z++)
		m_faAtomVisibleIncrement[z] = 0;
}


CVoroChoreographyPart::~CVoroChoreographyPart()
{
}


void CVoroChoreographyPart::UpdateIncrement(CVoroChoreographyPart *p)
{
	int z;

	for (z=0;z<3;z++)
	{
		m_vClipLowBegin[z] += p->m_vClipLowIncrement[z];
		m_vClipHighBegin[z] += p->m_vClipHighIncrement[z];
	}

	m_fFaceTransBegin += p->m_fFaceTransIncrement;
	m_fEdgeTransBegin += p->m_fEdgeTransIncrement;
	m_fRayTransBegin += p->m_fRayTransIncrement;
	m_fNbTransBegin += p->m_fNbTransIncrement;
	m_fNbHLFacBegin += p->m_fNbHLFacIncrement;
	m_fFaceBleachBegin += p->m_fFaceBleachIncrement;
	m_fAngleBegin += p->m_fAngleIncrement;
	m_fScaleBegin += p->m_fScaleIncrement;
	m_fExplodeBegin += p->m_fExplodeIncrement;
	m_fOuterOpacBegin += p->m_fOuterOpacIncrement;
	m_fClipBegin += p->m_fClipIncrement;
//	mprintf("(C) Clip %f --> %f.\n",m_fClipBegin-p->m_fClipIncrement,m_fClipBegin);
	m_fAtomCenterBegin += p->m_fAtomCenterIncrement;
	m_fMoleculesBegin += p->m_fMoleculesIncrement;
//	mprintf("Step %f + %f = %f\n",m_fStepBegin,p->m_fStepIncrement,m_fStepBegin+p->m_fStepIncrement);
	m_fStepBegin += p->m_fStepIncrement;
	m_fZoomBegin += p->m_fZoomIncrement;

	for (z=0;z<g_iGesAtomCount;z++)
		m_faAtomVisibleBegin[z] += p->m_faAtomVisibleIncrement[z];
}


void CVoroChoreographyPart::CalcIncrements()
{
	int z;

	for (z=0;z<3;z++)
	{
		m_vClipLowIncrement[z] += (m_vClipLowEnd[z]-m_vClipLowBegin[z])/(m_iFrameEnd-m_iFrameStart);
		m_vClipHighIncrement[z] += (m_vClipHighEnd[z]-m_vClipHighBegin[z])/(m_iFrameEnd-m_iFrameStart);
	}

	m_fFaceTransIncrement = (m_fFaceTransEnd-m_fFaceTransBegin)/(m_iFrameEnd-m_iFrameStart);
	m_fEdgeTransIncrement = (m_fEdgeTransEnd-m_fEdgeTransBegin)/(m_iFrameEnd-m_iFrameStart);
	m_fRayTransIncrement = (m_fRayTransEnd-m_fRayTransBegin)/(m_iFrameEnd-m_iFrameStart);
	m_fNbTransIncrement = (m_fNbTransEnd-m_fNbTransBegin)/(m_iFrameEnd-m_iFrameStart);
	m_fNbHLFacIncrement = (m_fNbHLFacEnd-m_fNbHLFacBegin)/(m_iFrameEnd-m_iFrameStart);
	m_fFaceBleachIncrement = (m_fFaceBleachEnd-m_fFaceBleachBegin)/(m_iFrameEnd-m_iFrameStart);
	m_fAngleIncrement = (m_fAngleEnd-m_fAngleBegin)/(m_iFrameEnd-m_iFrameStart);
	m_fScaleIncrement = (m_fScaleEnd-m_fScaleBegin)/(m_iFrameEnd-m_iFrameStart);
	m_fClipIncrement = (m_fClipEnd-m_fClipBegin)/(m_iFrameEnd-m_iFrameStart);
	m_fExplodeIncrement = (m_fExplodeEnd-m_fExplodeBegin)/(m_iFrameEnd-m_iFrameStart);
	m_fOuterOpacIncrement = (m_fOuterOpacEnd-m_fOuterOpacBegin)/(m_iFrameEnd-m_iFrameStart);
	m_fStepIncrement = (m_fStepEnd-m_fStepBegin)/(m_iFrameEnd-m_iFrameStart);
	m_fAtomCenterIncrement = (m_fAtomCenterEnd-m_fAtomCenterBegin)/(m_iFrameEnd-m_iFrameStart);
	m_fMoleculesIncrement = (m_fMoleculesEnd-m_fMoleculesBegin)/(m_iFrameEnd-m_iFrameStart);
	m_fZoomIncrement = (m_fZoomEnd-m_fZoomBegin)/(m_iFrameEnd-m_iFrameStart);

//	mprintf("(B) Clip %f --> %f (%d steps)\n",m_fClipBegin,m_fClipEnd,m_iFrameEnd-m_iFrameStart);

	for (z=0;z<g_iGesAtomCount;z++)
		m_faAtomVisibleIncrement[z] = (m_faAtomVisibleEnd[z]-m_faAtomVisibleBegin[z])/(m_iFrameEnd-m_iFrameStart);
}


void CVoroWrapper::CreateChoreography()
{
	CVoroChoreographyPart *p;
	int z;

	p = AddVoroChoreographyPart(2.0);

	for (z=0;z<g_iGesAtomCount;z++)
		p->m_faAtomVisibleBegin[z] = 1.0;

	p->m_fAngleBegin = 25.0;
	p->m_fAtomCenterBegin = 0.0;
	p->m_fClipBegin = 5.0;
	p->m_fExplodeBegin = 1.0;
	p->m_fZoomBegin = 1.0;
	p->m_fMoleculesBegin = 0.0;
	p->m_fOuterOpacBegin = 1.0;
	p->m_fScaleBegin = 0.95;
	p->m_fStepBegin = 0.0;
	p->m_vClipLowBegin[0] = -1.0;
	p->m_vClipLowBegin[1] = -1.0;
	p->m_vClipLowBegin[2] = -1.0;
	p->m_vClipHighBegin[0] = 1.0;
	p->m_vClipHighBegin[1] = 1.0;
	p->m_vClipHighBegin[2] = 1.0;

	p = AddVoroChoreographyPart(10.0);
	p->m_fAngleEnd = 115.0;

	p = AddVoroChoreographyPart(1.0);
	p->m_fClipEnd = 1.5;

	p = AddVoroChoreographyPart(10.0);
	p->m_fClipEnd = 1.0;

	p = AddVoroChoreographyPart(1.0);

	p = AddVoroChoreographyPart(4.0);
	p->m_fZoomEnd = 1.25;

	p = AddVoroChoreographyPart(1.0);

	p = AddVoroChoreographyPart(10.0);
	p->m_fAngleEnd = 205.0;

	p = AddVoroChoreographyPart(1.0);

	p = AddVoroChoreographyPart(10.0);
	p->m_fStepIncrement = 1.0;

	for (z=0;z<20;z++)
	{
		p = AddVoroChoreographyPart(0.3f);
		p->m_fStepIncrement = exp(z/19.0*log(40.0));
	}

	p = AddVoroChoreographyPart(30.0);
	p->m_fStepIncrement = 40.0;

	p = AddVoroChoreographyPart(1.0);


/*	p = AddVoroChoreographyPart(2.0);

	for (z=0;z<g_iGesAtomCount;z++)
		p->m_faAtomVisibleBegin[z] = 1.0;

	p->m_fAngleBegin = 25.0;
	p->m_fAtomCenterBegin = 1.0;
	p->m_fClipBegin = 5.0;
	p->m_fExplodeBegin = 1.0;
	p->m_fZoomBegin = 1.25;
	p->m_fMoleculesBegin = 1.0;
	p->m_fOuterOpacBegin = 0.0;
	p->m_fScaleBegin = 0.0;
	p->m_fStepBegin = 0.0;
	p->m_vClipLowBegin[0] = -1.0;
	p->m_vClipLowBegin[1] = -1.0;
	p->m_vClipLowBegin[2] = -1.0;
	p->m_vClipHighBegin[0] = 1.0;
	p->m_vClipHighBegin[1] = 1.0;
	p->m_vClipHighBegin[2] = 1.0;

	p = AddVoroChoreographyPart(15.0);
	p->m_fStepIncrement = 1.0;

	p = AddVoroChoreographyPart(1.0);

	p = AddVoroChoreographyPart(10.0);
	p->m_fAngleEnd = 115.0;

	p = AddVoroChoreographyPart(1.0);

	p = AddVoroChoreographyPart(4.0);
	p->m_fZoomEnd = 1.0;

	p = AddVoroChoreographyPart(1.0);

	p = AddVoroChoreographyPart(5.0);
	p->m_fOuterOpacEnd = 1.0;

	p = AddVoroChoreographyPart(1.0);

	p = AddVoroChoreographyPart(10.0);
	p->m_fScaleEnd = 0.95;
	p->m_fAtomCenterEnd = 0.0;

	p = AddVoroChoreographyPart(1.0);
	p->m_fMoleculesEnd = 0.0;

	p = AddVoroChoreographyPart(10.0);
	p->m_fAngleEnd = 205.0;

	p = AddVoroChoreographyPart(1.0);
	p->m_fClipEnd = 1.4;

	p = AddVoroChoreographyPart(10.0);
	p->m_fClipEnd = 1.0;

	p = AddVoroChoreographyPart(1.0);

	p = AddVoroChoreographyPart(4.0);
	p->m_fZoomEnd = 1.25;

	p = AddVoroChoreographyPart(1.0);

	p = AddVoroChoreographyPart(10.0);
	p->m_fAngleEnd = 295.0;

	p = AddVoroChoreographyPart(1.0);

	p = AddVoroChoreographyPart(30.0);
	p->m_fStepIncrement = 1.0;

	p = AddVoroChoreographyPart(1.0);

	p = AddVoroChoreographyPart(10.0);
	for (z=3;z<g_iGesAtomCount;z++)
		p->m_faAtomVisibleEnd[z] = 0.0;

	p = AddVoroChoreographyPart(1.0);

	p = AddVoroChoreographyPart(30.0);
	p->m_fStepIncrement = 1.0;

	p = AddVoroChoreographyPart(1.0);

	p = AddVoroChoreographyPart(10.0);
	for (z=3;z<g_iGesAtomCount;z++)
		p->m_faAtomVisibleEnd[z] = 1.0;

	p = AddVoroChoreographyPart(1.0);

	p = AddVoroChoreographyPart(15.0);
	p->m_vClipLowEnd[0] = 0.98;

	p = AddVoroChoreographyPart(1.0);

	p = AddVoroChoreographyPart(15.0);
	p->m_vClipLowEnd[0] = -1.0;

	p = AddVoroChoreographyPart(1.0);

	p = AddVoroChoreographyPart(20.0);
	p->m_fStepIncrement = 1.0;*/

	AddVoroChoreographyPart(0);

	for (z=0;z<m_oaBoxChoreography.GetSize();z++)
		((CVoroChoreographyPart*)m_oaBoxChoreography[z])->CalcIncrements();

	m_pPOVCurrentValues = new CVoroChoreographyPart();
	m_pPOVCurrentValues->CopyFrom((CVoroChoreographyPart*)m_oaBoxChoreography[0]);

	m_iPOVCurrentChoreographyPart = 0;
}


void CVoroWrapper::ParseChoreography_Low(CVoroChoreographyPart *p, bool initial)
{
	int i, z;
	char buf[256];
	CxIntArray tia;
	float tf;

	mprintf("    Current parameter values:\n\n");

	mprintf(WHITE,"       #   Name                             Value     Sensible Range\n");
	if (m_bWritePOVMovie && !initial)
		mprintf("      (1)  Step increment / frame           %5.1f     0 - *\n",p->m_fStepIncrement);
	mprintf("      (2)  Rotation Angle (degree)         %6.1f     *\n",p->m_fAngleBegin);
	mprintf("      (3)  Voronoi Edge Transparency        %5.3f     0 - 1\n",p->m_fEdgeTransBegin);
	mprintf("      (4)  Voronoi Face Transparency        %5.3f     0 - 1\n",p->m_fFaceTransBegin);
	mprintf("      (5)  Voronoi Neighbors Transparency   %5.3f     0 - 1\n",p->m_fNbTransBegin);
	mprintf("      (6)  Voronoi Ray Transparency         %5.3f     0 - 1\n",p->m_fRayTransBegin);
	mprintf("      (7)  Voronoi Face Bleaching           %5.3f     0 - 1\n",p->m_fFaceBleachBegin);
	mprintf("      (8)  Voronoi Neighbor Highlight       %5.3f     1 - 2\n",p->m_fNbHLFacBegin);
	mprintf("      (9)  Zoom value (relative)            %5.3f     0.5 - 2\n",p->m_fZoomBegin);
	mprintf("     (10)  Atom Center value                %5.3f     0 - 1\n",p->m_fAtomCenterBegin);
	mprintf("     (11)  Clip value                       %5.3f     1 - 5\n",p->m_fClipBegin);
	mprintf("     (12)  Explode value                    %5.2f     1 - 10\n",p->m_fExplodeBegin);
	mprintf("     (13)  DrawMolecules value              %5.3f     0 - 1 (integer)\n",p->m_fMoleculesBegin);
	mprintf("     (14)  OuterOpac value                  %5.3f     0 - 1\n",p->m_fOuterOpacBegin);
	mprintf("     (15)  Voronoi scale value              %5.3f     0 - 1\n",p->m_fScaleBegin);
	mprintf("     (16)  Clipping Box                     <%.2f,%.2f,%.2f> - <%.2f,%.2f %.2f>\n",p->m_vClipLowBegin[0],p->m_vClipLowBegin[1],p->m_vClipLowBegin[2],p->m_vClipHighBegin[0],p->m_vClipHighBegin[1],p->m_vClipHighBegin[2]);
	mprintf("     (17)  Atom Visibility                  ...\n");
	mprintf("\n");

	while (true)
	{
		i = AskUnsignedInteger("    Which parameter to modify? [done] ",0);
		if (i == 0)
			break;

		switch(i)
		{
			case 1:
				tf = AskFloat("    Enter new value: [0] ",0);
				p->m_fStepIncrement = tf;
				break;

			case 2:
				tf = AskFloat("    Enter new value: [%.0f] ",p->m_fAngleBegin,p->m_fAngleBegin);
				if (initial)
					p->m_fAngleBegin = tf;
						else p->m_fAngleEnd = tf;
				break;

			case 3:
				tf = AskFloat("    Enter new value: [%.3f] ",p->m_fEdgeTransBegin,p->m_fEdgeTransBegin);
				if (initial)
					p->m_fEdgeTransBegin = tf;
						else p->m_fEdgeTransEnd = tf;
				break;

			case 4:
				tf = AskFloat("    Enter new value: [%.3f] ",p->m_fFaceTransBegin,p->m_fFaceTransBegin);
				if (initial)
					p->m_fFaceTransBegin = tf;
						else p->m_fFaceTransEnd = tf;
				break;

			case 5:
				tf = AskFloat("    Enter new value: [%.3f] ",p->m_fNbTransBegin,p->m_fNbTransBegin);
				if (initial)
					p->m_fNbTransBegin = tf;
						else p->m_fNbTransEnd = tf;
				break;

			case 6:
				tf = AskFloat("    Enter new value: [%.3f] ",p->m_fRayTransBegin,p->m_fRayTransBegin);
				if (initial)
					p->m_fRayTransBegin = tf;
						else p->m_fRayTransEnd = tf;
				break;

			case 7:
				tf = AskFloat("    Enter new value: [%.3f] ",p->m_fFaceBleachBegin,p->m_fFaceBleachBegin);
				if (initial)
					p->m_fFaceBleachBegin = tf;
						else p->m_fFaceBleachEnd = tf;
				break;

			case 8:
				tf = AskFloat("    Enter new value: [%.3f] ",p->m_fNbHLFacBegin,p->m_fNbHLFacBegin);
				if (initial)
					p->m_fNbHLFacBegin = tf;
						else p->m_fNbHLFacEnd = tf;
				break;

			case 9:
				tf = AskFloat("    Enter new value: [%.3f] ",p->m_fZoomBegin,p->m_fZoomBegin);
				if (initial)
					p->m_fZoomBegin = tf;
						else p->m_fZoomEnd = tf;
				break;

			case 10:
				tf = AskFloat("    Enter new value: [%.3f] ",p->m_fAtomCenterBegin,p->m_fAtomCenterBegin);
				if (initial)
					p->m_fAtomCenterBegin = tf;
						else p->m_fAtomCenterEnd = tf;
				break;

			case 11:
				tf = AskFloat("    Enter new value: [%.3f] ",p->m_fClipBegin,p->m_fClipBegin);
				if (initial)
					p->m_fClipBegin = tf;
						else p->m_fClipEnd = tf;
				break;

			case 12:
				tf = AskFloat("    Enter new value: [%.3f] ",p->m_fExplodeBegin,p->m_fExplodeBegin);
				if (initial)
					p->m_fExplodeBegin = tf;
						else p->m_fExplodeEnd = tf;
				break;

			case 13:
				tf = AskFloat("    Enter new value: [%.0f] ",p->m_fMoleculesBegin,p->m_fMoleculesBegin);
				if (initial)
					p->m_fMoleculesBegin = tf;
						else p->m_fMoleculesEnd = tf;
				break;

			case 14:
				tf = AskFloat("    Enter new value: [%.3f] ",p->m_fOuterOpacBegin,p->m_fOuterOpacBegin);
				if (initial)
					p->m_fOuterOpacBegin = tf;
						else p->m_fOuterOpacEnd = tf;
				break;

			case 15:
				tf = AskFloat("    Enter new value: [%.3f] ",p->m_fScaleBegin,p->m_fScaleBegin);
				if (initial)
					p->m_fScaleBegin = tf;
						else p->m_fScaleEnd = tf;
				break;

			case 16:
				if (initial)
				{
					p->m_vClipLowBegin[0] = AskFloat("    Enter lower clipping box X value: [%.3f] ",p->m_vClipLowBegin[0],p->m_vClipLowBegin[0]);
					p->m_vClipLowBegin[1] = AskFloat("    Enter lower clipping box Y value: [%.3f] ",p->m_vClipLowBegin[1],p->m_vClipLowBegin[1]);
					p->m_vClipLowBegin[2] = AskFloat("    Enter lower clipping box Z value: [%.3f] ",p->m_vClipLowBegin[2],p->m_vClipLowBegin[2]);
					p->m_vClipHighBegin[0] = AskFloat("    Enter upper clipping box X value: [%.3f] ",p->m_vClipHighBegin[0],p->m_vClipHighBegin[0]);
					p->m_vClipHighBegin[1] = AskFloat("    Enter upper clipping box Y value: [%.3f] ",p->m_vClipHighBegin[1],p->m_vClipHighBegin[1]);
					p->m_vClipHighBegin[2] = AskFloat("    Enter upper clipping box Z value: [%.3f] ",p->m_vClipHighBegin[2],p->m_vClipHighBegin[2]);
				} else
				{
					p->m_vClipLowEnd[0] = AskFloat("    Enter lower clipping box X value: [%.3f] ",p->m_vClipLowBegin[0],p->m_vClipLowBegin[0]);
					p->m_vClipLowEnd[1] = AskFloat("    Enter lower clipping box Y value: [%.3f] ",p->m_vClipLowBegin[1],p->m_vClipLowBegin[1]);
					p->m_vClipLowEnd[2] = AskFloat("    Enter lower clipping box Z value: [%.3f] ",p->m_vClipLowBegin[2],p->m_vClipLowBegin[2]);
					p->m_vClipHighEnd[0] = AskFloat("    Enter upper clipping box X value: [%.3f] ",p->m_vClipHighBegin[0],p->m_vClipHighBegin[0]);
					p->m_vClipHighEnd[1] = AskFloat("    Enter upper clipping box Y value: [%.3f] ",p->m_vClipHighBegin[1],p->m_vClipHighBegin[1]);
					p->m_vClipHighEnd[2] = AskFloat("    Enter upper clipping box Z value: [%.3f] ",p->m_vClipHighBegin[2],p->m_vClipHighBegin[2]);
				}
				break;

			case 17:
_visagain:
				tia.RemoveAll();
				AskString("    Which atom's visibility values to modify (absolute offsets, e.g. \"1,3-5,8\"): [all] ",buf,"");
				if (strlen(buf) == 0)
				{
					for (z=0;z<g_iGesAtomCount;z++)
						tia.Add(z);
				} else
				{
					if (!ParseIntList(buf,&tia))
					{
						eprintf("    Wrong input.\n");
						goto _visagain;
					}
					for (z=0;z<tia.GetSize();z++)
					{
						if ((tia[z] < 1) || (tia[z] > g_iGesAtomCount))
						{
							eprintf("    Offset out of range: %d.\n",tia[z]);
							goto _visagain;
						}
					}
				}
				tf = AskFloat("    Enter opacity value (=1-transparency) for those atoms: [1.0] ",1.0);
				for (z=0;z<tia.GetSize();z++)
				{
					if (initial)
						p->m_faAtomVisibleBegin[tia[z]-1] = tf;
							else p->m_faAtomVisibleEnd[tia[z]-1] = tf;
				}
				break;

			default:
				eprintf("    Unknown parameter.\n");
				break;
		}
		mprintf("\n");
	}

	mprintf("\n");
}


void CVoroWrapper::ParseChoreography()
{
	CVoroChoreographyPart *p;
	float tf;
	int z, zb;

	if (m_bWritePOVMovie)
	{
		mprintf(WHITE,"\n>>> Choreography input for whole box >>>\n\n");

		mprintf("    The choreography consists of blocks. All parameters are linearly interpolated from\n");
		mprintf("    the beginning to the end of a block. The end parameters are defined by the beginning\n");
		mprintf("    parameters of the subsequent block.\n\n");

		mprintf(YELLOW,"*** Block 1 ***\n\n");

		tf = AskFloat("    Please enter duration of this block in seconds: [5.0] ",5.0);

		p = AddVoroChoreographyPart(tf);

		mprintf(YELLOW,"\n    Enter initial values of block 1:\n\n");
	} else
	{
		p = AddVoroChoreographyPart(1.0);
		mprintf("\n");
	}

	// Reasonable Defaults

	for (z=0;z<g_iGesAtomCount;z++)
		p->m_faAtomVisibleBegin[z] = 1.0;

	p->m_fAtomCenterBegin = 0.0;
	p->m_fClipBegin = 1.0;
	p->m_fExplodeBegin = 1.0;
	p->m_fMoleculesBegin = 0.0;
	p->m_fOuterOpacBegin = 1.0;
	p->m_fScaleBegin = 0.95;
	p->m_fStepBegin = 0.0;
	p->m_vClipLowBegin[0] = -1.0;
	p->m_vClipLowBegin[1] = -1.0;
	p->m_vClipLowBegin[2] = -1.0;
	p->m_vClipHighBegin[0] = 1.0;
	p->m_vClipHighBegin[1] = 1.0;
	p->m_vClipHighBegin[2] = 1.0;

	p->m_fAngleBegin = 25.0;
	p->m_fZoomBegin = 1.0;
	p->m_fEdgeTransBegin = 1.0;
	p->m_fFaceTransBegin = 1.0;
	p->m_fNbTransBegin = 1.0;
	p->m_fRayTransBegin = 1.0;
	p->m_fFaceBleachBegin = 1.0;
	p->m_fNbHLFacBegin = 1.0;
	p->m_fStepIncrement = 1.0;

	ParseChoreography_Low(p,true);

	if (!m_bWritePOVMovie)
	{
		ChoreoToGlobal(p);
		return;
	}

	mprintf(YELLOW,"\n    Enter final values of block 1:\n\n");

	ParseChoreography_Low(p,false);

	zb = 2;
	while (true)
	{
		mprintf(YELLOW,"*** Block %d ***\n\n",zb);

		tf = AskFloat("    Please enter duration of this block in seconds: [done] ",0.0);

		p = AddVoroChoreographyPart(tf);

		if (tf == 0)
			break;

		mprintf("\n");

		mprintf(YELLOW,"    Enter final values of block %d:\n\n",zb);

		ParseChoreography_Low(p,false);

		zb++;
	}

	for (z=0;z<m_oaBoxChoreography.GetSize();z++)
		((CVoroChoreographyPart*)m_oaBoxChoreography[z])->CalcIncrements();

	m_pPOVCurrentValues = new CVoroChoreographyPart();
	m_pPOVCurrentValues->CopyFrom((CVoroChoreographyPart*)m_oaBoxChoreography[0]);

	m_iPOVCurrentChoreographyPart = 0;

	mprintf(WHITE,"\n<<< End of Choreography input for whole box <<<\n\n");
}


void CVoroWrapper::ParseChoreography_Single_Low(CVoroChoreographyPart *p, bool initial)
{
	int i;
	float tf;

	mprintf("    Current parameter values:\n\n");

	mprintf(WHITE,"       #   Name                             Value     Sensible Range\n");
	if (m_bWritePOVMovie && !initial)
		mprintf("      (1)  Step increment / frame           %5.1f     0 - *\n",p->m_fStepIncrement);
	mprintf("      (2)  Rotation Angle (degree)         %6.1f     *\n",p->m_fAngleBegin);
	mprintf("      (3)  Voronoi Edge Transparency        %5.3f     0 - 1\n",p->m_fEdgeTransBegin);
	mprintf("      (4)  Voronoi Face Transparency        %5.3f     0 - 1\n",p->m_fFaceTransBegin);
	mprintf("      (5)  Voronoi Neighbors Transparency   %5.3f     0 - 1\n",p->m_fNbTransBegin);
	mprintf("      (6)  Voronoi Ray Transparency         %5.3f     0 - 1\n",p->m_fRayTransBegin);
	mprintf("      (7)  Voronoi Face Bleaching           %5.3f     0 - 1\n",p->m_fFaceBleachBegin);
	mprintf("      (8)  Voronoi Neighbor Highlight       %5.3f     1 - 2\n",p->m_fNbHLFacBegin);
	mprintf("      (9)  Zoom value (relative)            %5.2f     0.5 - 2\n",p->m_fZoomBegin);
	mprintf("\n");

	while (true)
	{
		i = AskUnsignedInteger("    Which parameter to modify? [done] ",0);
		if (i == 0)
			break;
		switch(i)
		{
			case 1:
				tf = AskFloat("    Enter new value: [0] ",0);
				p->m_fStepIncrement = tf;
				break;

			case 2:
				tf = AskFloat("    Enter new value: [%.0f] ",p->m_fAngleBegin,p->m_fAngleBegin);
				if (initial)
					p->m_fAngleBegin = tf;
						else p->m_fAngleEnd = tf;
				break;

			case 3:
				tf = AskFloat("    Enter new value: [%.3f] ",p->m_fEdgeTransBegin,p->m_fEdgeTransBegin);
				if (initial)
					p->m_fEdgeTransBegin = tf;
						else p->m_fEdgeTransEnd = tf;
				break;

			case 4:
				tf = AskFloat("    Enter new value: [%.3f] ",p->m_fFaceTransBegin,p->m_fFaceTransBegin);
				if (initial)
					p->m_fFaceTransBegin = tf;
						else p->m_fFaceTransEnd = tf;
				break;

			case 5:
				tf = AskFloat("    Enter new value: [%.3f] ",p->m_fNbTransBegin,p->m_fNbTransBegin);
				if (initial)
					p->m_fNbTransBegin = tf;
						else p->m_fNbTransEnd = tf;
				break;

			case 6:
				tf = AskFloat("    Enter new value: [%.3f] ",p->m_fRayTransBegin,p->m_fRayTransBegin);
				if (initial)
					p->m_fRayTransBegin = tf;
						else p->m_fRayTransEnd = tf;
				break;

			case 7:
				tf = AskFloat("    Enter new value: [%.3f] ",p->m_fFaceBleachBegin,p->m_fFaceBleachBegin);
				if (initial)
					p->m_fFaceBleachBegin = tf;
						else p->m_fFaceBleachEnd = tf;
				break;

			case 8:
				tf = AskFloat("    Enter new value: [%.3f] ",p->m_fNbHLFacBegin,p->m_fNbHLFacBegin);
				if (initial)
					p->m_fNbHLFacBegin = tf;
						else p->m_fNbHLFacEnd = tf;
				break;

			case 9:
				tf = AskFloat("    Enter new value: [%.3f] ",p->m_fZoomBegin,p->m_fZoomBegin);
				if (initial)
					p->m_fZoomBegin = tf;
						else p->m_fZoomEnd = tf;
				break;

			default:
				eprintf("    Unknown parameter.\n");
				break;
		}
		mprintf("\n");
	}

	mprintf("\n");
}


void CVoroWrapper::ParseChoreography_Single()
{
	CVoroChoreographyPart *p;
	float tf;
	int z, zb;

	if (m_bWritePOVMovie)
	{
		mprintf(WHITE,"\n>>> Choreography input for one single molecule >>>\n\n");

		mprintf("    The choreography consists of blocks. All parameters are linearly interpolated from\n");
		mprintf("    the beginning to the end of a block. The end parameters are defined by the beginning\n");
		mprintf("    parameters of the subsequent block.\n\n");

		mprintf(YELLOW,"*** Block 1 ***\n\n");

		tf = AskFloat("    Please enter duration of this block in seconds: [5.0] ",5.0);

		p = AddVoroChoreographyPart(tf);

		mprintf(YELLOW,"\n    Enter initial values of block 1:\n\n");
	} else
	{
		p = AddVoroChoreographyPart(1.0);
		mprintf("\n");
	}

	// Reasonable Defaults

	for (z=0;z<g_iGesAtomCount;z++)
		p->m_faAtomVisibleBegin[z] = 1.0;

	p->m_fAtomCenterBegin = 0.0;
	p->m_fClipBegin = 5.0;
	p->m_fExplodeBegin = 1.0;
	p->m_fMoleculesBegin = 0.0;
	p->m_fOuterOpacBegin = 1.0;
	p->m_fScaleBegin = 0.95;
	p->m_fStepBegin = 0.0;
	p->m_vClipLowBegin[0] = -1.0;
	p->m_vClipLowBegin[1] = -1.0;
	p->m_vClipLowBegin[2] = -1.0;
	p->m_vClipHighBegin[0] = 1.0;
	p->m_vClipHighBegin[1] = 1.0;
	p->m_vClipHighBegin[2] = 1.0;

	p->m_fAngleBegin = 0.0;
	p->m_fZoomBegin = 1.0;
	p->m_fEdgeTransBegin = 0.0;
	p->m_fFaceTransBegin = 0.5;
	p->m_fNbTransBegin = 0.0;
	p->m_fRayTransBegin = 0.0;
	p->m_fFaceBleachBegin = 0.0;
	p->m_fNbHLFacBegin = 1.0;
	p->m_fStepIncrement = 1.0;

	ParseChoreography_Single_Low(p,true);

	if (!m_bWritePOVMovie)
	{
		ChoreoToGlobal(p);
		return;
	}

	mprintf(YELLOW,"\n    Enter final values of block 1:\n\n");

	ParseChoreography_Single_Low(p,false);

	zb = 2;
	while (true)
	{
		mprintf(YELLOW,"*** Block %d ***\n\n",zb);

		tf = AskFloat("    Please enter duration of this block in seconds: [done] ",0.0);

		p = AddVoroChoreographyPart(tf);

		if (tf == 0)
			break;

		mprintf("\n");

		mprintf(YELLOW,"    Enter final values of block %d:\n\n",zb);

		ParseChoreography_Single_Low(p,false);

		zb++;
	}

	for (z=0;z<m_oaBoxChoreography.GetSize();z++)
		((CVoroChoreographyPart*)m_oaBoxChoreography[z])->CalcIncrements();

	m_pPOVCurrentValues = new CVoroChoreographyPart();
	m_pPOVCurrentValues->CopyFrom((CVoroChoreographyPart*)m_oaBoxChoreography[0]);

	m_iPOVCurrentChoreographyPart = 0;

	mprintf(WHITE,"\n<<< End of Choreography input for one single molecule <<<\n\n");
}


void CVoroWrapper::CreateChoreography_Single()
{
	CVoroChoreographyPart *p;
	int z;

	p = AddVoroChoreographyPart(15.0);

	for (z=0;z<g_iGesAtomCount;z++)
		p->m_faAtomVisibleBegin[z] = 1.0;

	p->m_fAtomCenterBegin = 0.0;
	p->m_fClipBegin = 5.0;
	p->m_fExplodeBegin = 1.0;
	p->m_fMoleculesBegin = 0.0;
	p->m_fOuterOpacBegin = 1.0;
	p->m_fScaleBegin = 0.95;
	p->m_fStepBegin = 0.0;
	p->m_vClipLowBegin[0] = -1.0;
	p->m_vClipLowBegin[1] = -1.0;
	p->m_vClipLowBegin[2] = -1.0;
	p->m_vClipHighBegin[0] = 1.0;
	p->m_vClipHighBegin[1] = 1.0;
	p->m_vClipHighBegin[2] = 1.0;

	p->m_fAngleBegin = 0.0;
	p->m_fZoomBegin = 1.625;
	p->m_fEdgeTransBegin = 1.0;
	p->m_fFaceTransBegin = 1.0;
	p->m_fNbTransBegin = 1.0;
	p->m_fRayTransBegin = 1.0;
	p->m_fFaceBleachBegin = 1.0;
	p->m_fNbHLFacBegin = 1.0;
	p->m_fStepIncrement = 1.0;

	p = AddVoroChoreographyPart(8.0);
	p->m_fStepIncrement = 1.0;
	p->m_fEdgeTransEnd = 0;

	p = AddVoroChoreographyPart(2.0);
	p->m_fStepIncrement = 1.0;

	p = AddVoroChoreographyPart(15.0);
	p->m_fStepIncrement = 1.0;
	p->m_fAngleEnd = 180.0;

	p = AddVoroChoreographyPart(2.0);
	p->m_fStepIncrement = 1.0;

	p = AddVoroChoreographyPart(5.0);
	p->m_fStepIncrement = 1.0;
	p->m_fFaceTransEnd = 0.25;

	p = AddVoroChoreographyPart(2.0);
	p->m_fStepIncrement = 1.0;

	p = AddVoroChoreographyPart(15.0);
	p->m_fStepIncrement = 1.0;
	p->m_fAngleEnd = 360.0;

	p = AddVoroChoreographyPart(2.0);
	p->m_fStepIncrement = 1.0;

	p = AddVoroChoreographyPart(5.0);
	p->m_fStepIncrement = 1.0;
	p->m_fZoomEnd = 0.93;

	p = AddVoroChoreographyPart(2.0);
	p->m_fStepIncrement = 1.0;

	p = AddVoroChoreographyPart(6.0);
	p->m_fStepIncrement = 1.0;
	p->m_fNbTransEnd = 0.0;

	p = AddVoroChoreographyPart(4.0);
	p->m_fStepIncrement = 1.0;

	p = AddVoroChoreographyPart(6.0);
	p->m_fStepIncrement = 1.0;
	p->m_fRayTransEnd = 0.0;
	p->m_fFaceBleachEnd = 0.0;
	p->m_fNbHLFacEnd = 1.3;

	p = AddVoroChoreographyPart(2.0);
	p->m_fStepIncrement = 1.0;

	p = AddVoroChoreographyPart(30.0);
	p->m_fStepIncrement = 1.0;
	p->m_fAngleEnd = 720.0;

	p = AddVoroChoreographyPart(3.0);
	p->m_fStepIncrement = 1.0;

	p = AddVoroChoreographyPart(6.0);
	p->m_fStepIncrement = 1.0;
	p->m_fNbTransEnd = 1.0;
	p->m_fRayTransEnd = 1.0;

	p = AddVoroChoreographyPart(3.0);
	p->m_fStepIncrement = 1.0;

	p = AddVoroChoreographyPart(5.0);
	p->m_fStepIncrement = 1.0;
	p->m_fZoomEnd = 1.625;

	p = AddVoroChoreographyPart(2.0);
	p->m_fStepIncrement = 1.0;

	p = AddVoroChoreographyPart(3.0);
	p->m_fStepIncrement = 1.0;

	for (z=0;z<20;z++)
	{
		p = AddVoroChoreographyPart(0.3f);
		p->m_fStepIncrement = exp(z/19.0*log(40.0));
	}

	p = AddVoroChoreographyPart(30.0);
	p->m_fStepIncrement = 40.0;
	p->m_fAngleEnd = 1080.0;

	p = AddVoroChoreographyPart(0);

	for (z=0;z<m_oaBoxChoreography.GetSize();z++)
		((CVoroChoreographyPart*)m_oaBoxChoreography[z])->CalcIncrements();

	m_pPOVCurrentValues = new CVoroChoreographyPart();
	m_pPOVCurrentValues->CopyFrom((CVoroChoreographyPart*)m_oaBoxChoreography[0]);

	m_iPOVCurrentChoreographyPart = 0;
}


void CVoroWrapper::TestChoreography()
{
	FILE *a;
	int step, frame;
	CVoroChoreographyPart *p;
	double sep;

//	mprintf("## %d\n",m_iPOVCurrentChoreographyPart);

	a = OpenFileWrite("choreo_test.csv",true);

//	mprintf("A\n");

	frame = 0;
	step = 0;
	sep = 0.01;

	mfprintf(a,"Frame;Sep;Step;fStep;Zoom;Angle;Scale;Explode;Clip;OuterOpac;Molecules;AtomCenter;ClipLowX;ClipLowY;ClipLowZ;ClipHighX;ClipHighY;ClipHighZ;EdgeTrans;FaceTrans;NbTrans;RayTrans;NbHLFac;FaceBleach\n");

//	mprintf("B\n");

	for (step=0;step<100000;step++)
	{
//		mprintf("Step %d\n",step);
		while (true)
		{
//			mprintf("  Frame %d (part %d)\n",frame,m_iPOVCurrentChoreographyPart);

			if (m_iPOVCurrentChoreographyPart >= m_oaBoxChoreography.GetSize())
				break;

			if (((int)m_pPOVCurrentValues->m_fStepBegin) > step)
				break;

			p = m_pPOVCurrentValues;
			mfprintf(a,"%d;  %f;  %d;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f\n",frame,sep,step,p->m_fStepBegin,p->m_fZoomBegin,p->m_fAngleBegin,p->m_fScaleBegin,p->m_fExplodeBegin,p->m_fClipBegin,p->m_fOuterOpacBegin,p->m_fMoleculesBegin,p->m_fAtomCenterBegin,p->m_vClipLowBegin[0],p->m_vClipLowBegin[1],p->m_vClipLowBegin[2],p->m_vClipHighBegin[0],p->m_vClipHighBegin[1],p->m_vClipHighBegin[2],p->m_fEdgeTransBegin,p->m_fFaceTransBegin,p->m_fNbTransBegin,p->m_fRayTransBegin,p->m_fNbHLFacBegin,p->m_fFaceBleachBegin);

			frame++;

			p = (CVoroChoreographyPart*)m_oaBoxChoreography[m_iPOVCurrentChoreographyPart];

			if (frame > p->m_iFrameEnd)
			{
//				mprintf("  # Going to part %d.\n",m_iPOVCurrentChoreographyPart+1);
				m_iPOVCurrentChoreographyPart++;
				if (m_iPOVCurrentChoreographyPart < m_oaBoxChoreography.GetSize())
					m_pPOVCurrentValues->CopyFrom((CVoroChoreographyPart*)m_oaBoxChoreography[m_iPOVCurrentChoreographyPart]);
				if (sep == 0.01)
					sep = 10000;
						else sep = 0.01;
				continue;
			}

			m_pPOVCurrentValues->UpdateIncrement(p);
		}

		if (m_iPOVCurrentChoreographyPart >= m_oaBoxChoreography.GetSize())
			break;
	}

	m_iPOVBoxTotalFrames = frame;
	m_iPOVBoxTotalSteps = step+1;

	fclose(a);
}


void CVoroChoreographyPart::CopyFrom(CVoroChoreographyPart *p)
{
	int z;

	m_iFrameStart = p->m_iFrameStart;
	m_iFrameEnd = p->m_iFrameEnd;

	m_fFaceTransBegin = p->m_fFaceTransBegin;
	m_fEdgeTransBegin = p->m_fEdgeTransBegin;
	m_fNbTransBegin = p->m_fNbTransBegin;
	m_fRayTransBegin = p->m_fRayTransBegin;
	m_fNbHLFacBegin = p->m_fNbHLFacBegin;
	m_fFaceBleachBegin = p->m_fFaceBleachBegin;
	m_fZoomBegin = p->m_fZoomBegin;
	m_fMoleculesBegin = p->m_fMoleculesBegin;
	m_fStepBegin = p->m_fStepBegin;
	m_fAngleBegin = p->m_fAngleBegin;
	m_fScaleBegin = p->m_fScaleBegin;
	m_fExplodeBegin = p->m_fExplodeBegin;
	m_fClipBegin = p->m_fClipBegin;
	m_fOuterOpacBegin = p->m_fOuterOpacBegin;
	m_fAtomCenterBegin = p->m_fAtomCenterBegin;

	m_fFaceTransEnd = p->m_fFaceTransEnd;
	m_fEdgeTransEnd = p->m_fEdgeTransEnd;
	m_fNbTransEnd = p->m_fNbTransEnd;
	m_fRayTransEnd = p->m_fRayTransEnd;
	m_fNbHLFacEnd = p->m_fNbHLFacEnd;
	m_fFaceBleachEnd = p->m_fFaceBleachEnd;
	m_fZoomEnd = p->m_fZoomEnd;
	m_fMoleculesEnd = p->m_fMoleculesEnd;
	m_fStepEnd = p->m_fStepEnd;
	m_fAngleEnd = p->m_fAngleEnd;
	m_fScaleEnd = p->m_fScaleEnd;
	m_fExplodeEnd = p->m_fExplodeEnd;
	m_fClipEnd = p->m_fClipEnd;
	m_fOuterOpacEnd = p->m_fOuterOpacEnd;
	m_fAtomCenterEnd = p->m_fAtomCenterEnd;

	m_fFaceTransIncrement = p->m_fFaceTransIncrement;
	m_fEdgeTransIncrement = p->m_fEdgeTransIncrement;
	m_fNbTransIncrement = p->m_fNbTransIncrement;
	m_fRayTransIncrement = p->m_fRayTransIncrement;
	m_fNbHLFacIncrement = p->m_fNbHLFacIncrement;
	m_fFaceBleachIncrement = p->m_fFaceBleachIncrement;
	m_fZoomIncrement = p->m_fZoomIncrement;
	m_fMoleculesIncrement = p->m_fMoleculesIncrement;
	m_fStepIncrement = p->m_fStepIncrement;
	m_fAngleIncrement = p->m_fAngleIncrement;
	m_fScaleIncrement = p->m_fScaleIncrement;
	m_fExplodeIncrement = p->m_fExplodeIncrement;
	m_fClipIncrement = p->m_fClipIncrement;
	m_fOuterOpacIncrement = p->m_fOuterOpacIncrement;
	m_fAtomCenterIncrement = p->m_fAtomCenterIncrement;

	m_faAtomVisibleBegin.SetSize(g_iGesAtomCount); 
	m_faAtomVisibleEnd.SetSize(g_iGesAtomCount);
	m_faAtomVisibleIncrement.SetSize(g_iGesAtomCount);

	for (z=0;z<g_iGesAtomCount;z++)
	{
		m_faAtomVisibleBegin[z] = p->m_faAtomVisibleBegin[z];
		m_faAtomVisibleEnd[z] = p->m_faAtomVisibleEnd[z];
		m_faAtomVisibleIncrement[z] = p->m_faAtomVisibleIncrement[z];
	}

	for (z=0;z<3;z++)
	{
		m_vClipLowBegin[z] = p->m_vClipLowBegin[z];
		m_vClipHighBegin[z] = p->m_vClipHighBegin[z];
		m_vClipLowEnd[z] = p->m_vClipLowEnd[z];
		m_vClipHighEnd[z] = p->m_vClipHighEnd[z];
		m_vClipLowIncrement[z] = p->m_vClipLowIncrement[z];
		m_vClipHighIncrement[z] = p->m_vClipHighIncrement[z];
	}
}


void CVoroWrapper::ChoreoToGlobal(CVoroChoreographyPart *p)
{
	int z;

	m_fPOVAngle = p->m_fAngleBegin;
	for (z=0;z<g_iGesAtomCount;z++)
		m_faPOVBoxAtomVisible[z] = p->m_faAtomVisibleBegin[z];
	m_fPOVAtomCenter = p->m_fAtomCenterBegin;
	m_fPOVClip = p->m_fClipBegin;
	m_fPOVExplode = p->m_fExplodeBegin;
	m_bPOVMolecules = (p->m_fMoleculesBegin > 0.9);
	m_fPOVBoxOuterOpacity = p->m_fOuterOpacBegin;
	m_fPOVScale = p->m_fScaleBegin;
	m_fPOVZoom = p->m_fZoomBegin;
	m_vPOVBoxClipLow = p->m_vClipLowBegin;
	m_vPOVBoxClipHigh = p->m_vClipHighBegin;
	m_fPOVFaceTrans = p->m_fFaceTransBegin;
	m_fPOVEdgeTrans = p->m_fEdgeTransBegin;
	m_fPOVNbTrans = p->m_fNbTransBegin;
	m_fPOVRayTrans = p->m_fRayTransBegin;
	m_fPOVNbHighlightFac = p->m_fNbHLFacBegin;
	m_fPOVFaceBleach = p->m_fFaceBleachBegin;
}


void CVoroWrapper::PerformChoreography_Box(CTimeStep *ts)
{
	CVoroChoreographyPart *p;

	if (m_bWritePOVMovie)
	{
		while (true)
		{
			if (m_iPOVCurrentChoreographyPart >= m_oaBoxChoreography.GetSize())
				break;

			if (((int)(m_pPOVCurrentValues->m_fStepBegin+1.5)) > (int)(g_iSteps/g_iStride))
				break;

			ChoreoToGlobal(m_pPOVCurrentValues);

			mfprintf(m_fPOVCT,"%d;  %d;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f\n",m_iPOVFrameCounter,g_iSteps/g_iStride,m_fPOVZoom,m_fPOVAngle,m_fPOVScale,m_fPOVExplode,m_fPOVClip,m_fPOVBoxOuterOpacity,m_bPOVMolecules?1.0:0,m_fPOVAtomCenter,m_vPOVBoxClipLow[0],m_vPOVBoxClipLow[1],m_vPOVBoxClipLow[2],m_vPOVBoxClipHigh[0],m_vPOVBoxClipHigh[1],m_vPOVBoxClipHigh[2],m_fPOVEdgeTrans,m_fPOVFaceTrans,m_fPOVNbTrans,m_fPOVRayTrans,m_fPOVNbHighlightFac,m_fPOVFaceBleach);
			fflush(m_fPOVCT);

			WritePOV_WholeBox(ts,NULL);

			m_iPOVFrameCounter++;

			p = (CVoroChoreographyPart*)m_oaBoxChoreography[m_iPOVCurrentChoreographyPart];

			if (m_iPOVFrameCounter > p->m_iFrameEnd)
			{
				m_iPOVCurrentChoreographyPart++;
				if (m_iPOVCurrentChoreographyPart < m_oaBoxChoreography.GetSize())
					m_pPOVCurrentValues->CopyFrom((CVoroChoreographyPart*)m_oaBoxChoreography[m_iPOVCurrentChoreographyPart]);
				continue;
			}

			m_pPOVCurrentValues->UpdateIncrement(p);
		}
	} else
	{
		WritePOV_WholeBox(ts,"frame_box.pov");
	}
}


void CVoroWrapper::PerformChoreography_Single(CTimeStep *ts)
{
	CVoroChoreographyPart *p;
//	int z;

	if (m_bWritePOVMovie)
	{
		while (true)
		{
			if (m_iPOVCurrentChoreographyPart >= m_oaBoxChoreography.GetSize())
				break;

			if (((int)(m_pPOVCurrentValues->m_fStepBegin+1.5)) > (int)(g_iSteps/g_iStride))
				break;

			ChoreoToGlobal(m_pPOVCurrentValues);

			mfprintf(m_fPOVCT,"%d;  %d;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f\n",m_iPOVFrameCounter,g_iSteps/g_iStride,m_fPOVZoom,m_fPOVAngle,m_fPOVScale,m_fPOVExplode,m_fPOVClip,m_fPOVBoxOuterOpacity,m_bPOVMolecules?1.0:0,m_fPOVAtomCenter,m_vPOVBoxClipLow[0],m_vPOVBoxClipLow[1],m_vPOVBoxClipLow[2],m_vPOVBoxClipHigh[0],m_vPOVBoxClipHigh[1],m_vPOVBoxClipHigh[2],m_fPOVEdgeTrans,m_fPOVFaceTrans,m_fPOVNbTrans,m_fPOVRayTrans,m_fPOVNbHighlightFac,m_fPOVFaceBleach);
			fflush(m_fPOVCT);

//			mprintf("Current Frame %d, Step %d\n",m_iPOVFrameCounter,(g_iSteps/g_iStride));
			WritePOV(ts,NULL);

			m_iPOVFrameCounter++;

			p = (CVoroChoreographyPart*)m_oaBoxChoreography[m_iPOVCurrentChoreographyPart];

			if (m_iPOVFrameCounter > p->m_iFrameEnd)
			{
//				mprintf("Block %d: Frame %d > End=%d.\n",m_iPOVCurrentChoreographyPart,m_iPOVFrameCounter,p->m_iFrameEnd);
				m_iPOVCurrentChoreographyPart++;
				if (m_iPOVCurrentChoreographyPart < m_oaBoxChoreography.GetSize())
				{
//					mprintf("Next: Block %d (FrameBeg %d, FrameEnd %d).\n",m_iPOVCurrentChoreographyPart,((CVoroChoreographyPart*)m_oaBoxChoreography[m_iPOVCurrentChoreographyPart])->m_iFrameStart,((CVoroChoreographyPart*)m_oaBoxChoreography[m_iPOVCurrentChoreographyPart])->m_iFrameEnd);
					m_pPOVCurrentValues->CopyFrom((CVoroChoreographyPart*)m_oaBoxChoreography[m_iPOVCurrentChoreographyPart]);
				}
				continue;
			}

			m_pPOVCurrentValues->UpdateIncrement(p);
		}
	} else
	{
		WritePOV(ts,"frame_single.pov");
	}
}


void CVoroWrapper::BuildVoroMetric(CTimeStep *ts)
{
	int ijk, q, z, id, faces;
	double dist, area;
	voronoicell_neighbor c;
	vector<int> nb;
	vector<double> fa;

	try { m_pContainer = new container_periodic_poly(g_fBoxX/1000.0,0,g_fBoxY/1000.0,0,0,g_fBoxZ/1000.0,m_iBlocksX,m_iBlocksY,m_iBlocksZ,g_iVoroMemory); } catch(...) { m_pContainer = NULL; }
	if (m_pContainer == NULL) NewException((double)sizeof(container_periodic_poly),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	for (z=0;z<g_iGesAtomCount;z++)
		m_pContainer->put(z,ts->m_vaCoords[z][0]/1000.0,ts->m_vaCoords[z][1]/1000.0,ts->m_vaCoords[z][2]/1000.0,0.05);

	c_loop_all_periodic vl(*m_pContainer);

	for (z=0;z<g_iGesAtomCount*g_iGesAtomCount;z++)
		m_faVoroMetric[z] = 0;

	if (vl.start()) 
	{
		do 
		{
			if (m_pContainer->compute_cell(c,vl))
			{
				ijk=vl.ijk;
				q=vl.q;
		//		pp=m_pContainer->p[ijk]+m_pContainer->ps*q;

				id = m_pContainer->id[ijk][q];

				faces = c.number_of_faces();
				c.neighbors(nb);
				c.face_areas(fa);

				for (z=0;z<faces;z++)
				{
					area = fa[z]*100.0;
					dist = FoldedLength(ts->m_vaCoords[id] - ts->m_vaCoords[nb[z]]);

		//			mprintf("%d -- %d: %.3f pm, %.3f pm^2\n",id,nb[z],dist,area);

					if (m_bVoroMetricCDF)
					{
						m_pVoroMetricCDF->AddToBin(dist,area);
						m_pVoroMetricCDF2->AddToBin(dist,1000.0/area);
					}

					m_faVoroMetric[id*g_iGesAtomCount+nb[z]] = 1000.0/area;
				}
			}
		} while (vl.inc());
	}

/*	for (z=0;z<g_iGesAtomCount;z++)
	{
		for (q=0;q<g_iGesAtomCount;q++)
		{
			if ((m_faVoroMetric[z*g_iGesAtomCount+q] != 0) || (m_faVoroMetric[q*g_iGesAtomCount+z] != 0))
			{
				mprintf("%f -- %f (%f%c)\n",m_faVoroMetric[z*g_iGesAtomCount+q],m_faVoroMetric[q*g_iGesAtomCount+z],fabs((m_faVoroMetric[z*g_iGesAtomCount+q]-m_faVoroMetric[q*g_iGesAtomCount+z])/m_faVoroMetric[q*g_iGesAtomCount+z]*100.0),'%');
			}
		}
	}*/

	delete m_pContainer;
}


