/*
 *
 *  Visual Voicemail Daemon
 *
 *  Copyright (C) 2024, Chris Talbot <chris@talbothome.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "vvm.h"
#include "resolve.h"
#include <arpa/inet.h>
#include <ares.h>
#include <sys/select.h>
#include <netdb.h>
#include <stdio.h>

static void
resolve_callback (void    *arg,
                  int      status,
                  int      timeouts,
                  struct   hostent *host)
{
  char **host_ip = (char **)arg;
  gchar ip[64];

  if (!host || status != ARES_SUCCESS)
    {
      DBG ("Failed to resolve host: %s\n", ares_strerror (status));
      return;
    }

  inet_ntop (host->h_addrtype, host->h_addr_list[0], ip, sizeof(ip));

  *host_ip = g_strdup (ip);

  DBG ("Found IP for '%s': %s\n", host->h_name, *host_ip);
}

static void
resolve_wait (ares_channel channel)
{
  int nfds, count;
  fd_set readers, writers;
  struct timeval tv, *tvp;

  while (1)
    {
      FD_ZERO (&readers);
      FD_ZERO (&writers);
      nfds = ares_fds (channel, &readers, &writers);
      if (nfds == 0)
        break;
      tvp = ares_timeout (channel, NULL, &tv);
      count = select (nfds, &readers, &writers, NULL, tvp);
      if (count == -1)
        {
          g_critical ("Error waiting for c-ares read/write descriptors");
          break;
        }
      ares_process (channel, &readers, &writers);
    }
}

char *
vvm_resolve_resolve_host (const char *host,
                          const char *mailbox_interface,
                          const char *resolvers_ipv4_csv,
                          const char *resolvers_ipv6_csv)
{
  ares_channel chan;
  int ares_return;
  gchar *host_ip = NULL;
  g_autofree gchar *nsall_csv = NULL;

  DBG ("%s", __func__);

  ares_return = ares_init (&chan);
  if (ares_return != ARES_SUCCESS)
    {
      g_warning ("Ares init failed: %s\n", ares_strerror (ares_return));
      goto ares_out;
    }

  /*
   * ares_set_local_dev () works without root
   * https://github.com/c-ares/c-ares/issues/405
   */
  DBG ("Binding resolver queries to interface %s", mailbox_interface);
  ares_set_local_dev (chan, mailbox_interface);

  if (resolvers_ipv6_csv && *resolvers_ipv6_csv &&
      resolvers_ipv4_csv && *resolvers_ipv4_csv)
    nsall_csv = g_strjoin (",", resolvers_ipv6_csv, resolvers_ipv4_csv, NULL);
  else if (resolvers_ipv6_csv && *resolvers_ipv6_csv)
    nsall_csv = g_strdup (resolvers_ipv6_csv);
  else if (resolvers_ipv4_csv && *resolvers_ipv4_csv)
    nsall_csv = g_strdup (resolvers_ipv4_csv);
  else {
      g_warning ("No active DNS Servers\n");
      goto ares_out;
    }

  DBG ("All Nameservers: %s", nsall_csv);

  ares_return = ares_set_servers_csv (chan, nsall_csv);
  if (ares_return != ARES_SUCCESS)
    g_warning ("Ares failed to set list of nameservers ('%s'), %s\n",
               nsall_csv, ares_strerror (ares_return));
  ares_gethostbyname (chan, host, AF_UNSPEC, resolve_callback,
                      (void *)&host_ip);
  resolve_wait (chan);
  if (host_ip == NULL)
    {
      g_warning ("Failed to resolve '%s'\n", host);
      goto ares_out;
    }

 ares_out:
  ares_destroy (chan);
  return host_ip;
}

int
vvm_resolve_ip_version(const char *src)
{
  char buf[INET6_ADDRSTRLEN];

  if (inet_pton(AF_INET, src, buf))
    return 4;
  else if (inet_pton(AF_INET6, src, buf))
    return 6;

  return -1;
}
