/* normalmode_copy.cc
 * This file belongs to Worker, a file manager for UN*X/X11.
 * Copyright (C) 2011-2012 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "normalmode.h"
#include "worker_locale.h"
#include "worker.h"
#include "copyopwin.hh"
#include "copyop.h"
#include "nmcacheentry.hh"
#include "nmcopyopdir.hh"
#include "nmfieldlistviewdnd.hh"
#include "datei.h"
#include "fileentry.hh"
#include "simplelist.hh"

void NormalMode::copy( struct NM_copyorder *copyorder )
{
    Lister *olister=NULL;
    NormalMode *nm2=NULL,*updatenm=NULL;
    ListerMode *lm=NULL;
    NM_specialsourceInt *ss1, *tss;
    const FileEntry *tfe;
    bool enter;
    NM_CopyOp_Dir *cod1,*tcod;
    unsigned long files,dirs,gf,gd;
    loff_t bytes;
    bool cancel=false;
    CopyOpWin *cowin;
    nm_copy_t ce1;
    bool nodatabase;
    int erg;
    char *textstr,*tstr,*buttonstr;
    std::list<NM_specialsourceInt*> *copylist;
    std::list<NM_specialsourceInt*>::iterator iti1, iti2;
    std::list<NM_specialsourceInt*> *copy_lvc_list;
    std::list<const FileEntry*>::iterator itfe1;
  
    if(copyorder==NULL) return;
    if(copyorder->destdir==NULL) return;
    if ( getCurrentDir() == NULL ) {
        // no current dir => no files to copy
        return;
    }

    if ( ce->dirOpened() == false ) return;

    copyorder->overwrite=copyorder->NM_OVERWRITE_NORMAL;  // force to start in this mode

    finishsearchmode();
  
    if(Datei::fileExistsExt(copyorder->destdir)!=Datei::D_FE_DIR) {
        // no dir as destination
        textstr=(char*)_allocsafe(strlen(catalog.getLocale(67))+strlen(copyorder->destdir)+1);
        sprintf(textstr,catalog.getLocale(67),copyorder->destdir);
        buttonstr = dupstring( catalog.getLocale(11) );
        request(catalog.getLocale(125),textstr,buttonstr);
        _freesafe(textstr);
        _freesafe( buttonstr );
        return;
    }
  
    disableDirWatch();
  
    // clear the reclist so the slace doesn't block us because he is still reading
    // from disk
    ft_rec_list_clear();
  
    if ( strcmp( getCurrentDir(), copyorder->destdir ) == 0 ) {
        copyorder->do_rename=true;
        updatenm=this;
    } else {
        olister=parentlister->getWorker()->getOtherLister(parentlister);
        if(olister!=NULL) {
            lm=olister->getActiveMode();
            if(lm!=NULL) {
                if(lm->isType("NormalMode")==true) {
                    nm2=(NormalMode*)lm;
                    if ( nm2->getCurrentDir() != NULL ) {
                        if ( strcmp( nm2->getCurrentDir(), copyorder->destdir ) == 0 ) {
                            updatenm=nm2;
                        }
                    }
                }
            }
        }
    }
    buf=_allocsafe(BUFSIZE);

    if ( updatenm != this ) {
        updatenm->disableDirWatch();
    }
  
    if(copyorder->copymode==copyorder->NM_COPYMODE_NODATABASE) nodatabase=true;
    else nodatabase=false;
  
    // no sense for follow_symlinks when moving
    if(copyorder->move==true) copyorder->follow_symlinks=false;

    /* now presupposition checked
       next search for the files to copy
       therefore collect entries to copy in a list together with the LVC (when on root-level)
       then for each entry do recursive call to copy it
       after success deactivate LVC */

    /* first: create List contains NM_specialsource
       when source==NM_SPECIAL then the given list but check for existing of the FEs
       else source==NM_ONLYACTIVE => take the activefe (from ce) in this list
       else take all selected entries (except the "..")
    */
    cowin=copyorder->cowin;
    if(cowin!=NULL) {
        cowin->open(nodatabase);
        cowin->setmessage(catalog.getLocale(122),0);
        if ( cowin->redraw() != 0 ) cancel = true;
    }
    copylist = new std::list<NM_specialsourceInt*>;
    switch(copyorder->source) {
        case NM_copyorder::NM_SPECIAL:
            buildSpecialSourceIntList( *( copyorder->sources ),
                                       *copylist );
            break;
        case NM_copyorder::NM_ONLYACTIVE:
            getSelFiles( copylist, NM_GETFILES_ONLYACTIVE, false );
            break;
        default:  // all selected entries
            getSelFiles( copylist, NM_GETFILES_SELORACT, false );
            break;
    }
    if ( cowin != NULL ) {
        if ( cowin->redraw() != 0 ) cancel = true;
    }

    // create the NM_CopyOp_Dir for each dir in copylist
    files=dirs=gf=gd=0;
    bytes = 0;
  
    if(nodatabase==false) {
        for ( iti1 = copylist->begin(); iti1 != copylist->end(); iti1++ ) {
            if ( cancel == true ) break;
            ss1 = *iti1;
            enter=false;
            if ( ss1->entry()->isDir() == true ) {
                // fe is a dir, check if it is a link and take it only when follow_symlinks==true
                // entry is a dir so it cannot be a corrupt link so no need to check
                if ( ss1->entry()->isLink == false ) enter = true;
                else if(copyorder->follow_symlinks==true) enter=true;
            }
            if(enter==true) {
                // fe is a dir so creating corresponding entry
                cod1 = new NM_CopyOp_Dir( ss1->entry() );
                if ( cod1->user_abort == false ) {
                    // recursive call
                    if(cod1->createSubDirs(copyorder,&gf,&gd)!=0) cancel=true;
                } else cancel = true;
                // add the values from this subdir to this dir
                files+=cod1->files;
                dirs+=cod1->dirs;
                bytes+=cod1->bytes;

                ss1->cod=cod1;
                // this is a dir so inc the counter
                dirs++;
                gd++;
                if(cowin!=NULL) {
                    cowin->set_files_to_copy(gf);
                    cowin->set_dirs_to_copy(gd);
                    if(cowin->redraw()!=0) cancel=true;
                }
            } else {
                // is not dir (mostly a file but can also be links ...)
                files++;
                gf++;
                if ( ( ss1->entry()->isLink == true ) &&
                     ( copyorder->follow_symlinks == true ) &&
                     ( ss1->entry()->isCorrupt == false ) ) {
                    bytes += ss1->entry()->dsize();
                } else {
                    bytes += ss1->entry()->size();
                }
                ss1->cod=NULL;
            }
        }
    }

    if(cowin!=NULL) {
        cowin->set_files_to_copy(files);
        cowin->set_dirs_to_copy(dirs);
        cowin->set_bytes_to_copy(bytes);
        cowin->starttimer();
    }

    // this is just the value from the last update, it might not be correct
    if ( parentlister->getWorker()->PS_readSpace( copyorder->destdir ) == 0 ) {
        loff_t free_blocks = parentlister->getWorker()->PS_getFreeSpace();
        loff_t block_size = parentlister->getWorker()->PS_getBlocksize();

        if ( block_size > 0 && ( ( (loff_t)bytes ) / block_size ) + 1 >= free_blocks ) {
            // files will possibly not fit on target, ask to continue or cancel

            std::string buttons = catalog.getLocale( 629 );
            buttons += "|";
            buttons += catalog.getLocale( 8 );

            std::string freespaceh = parentlister->getWorker()->PS_getFreeSpaceH();

            std::string text = AGUIXUtils::formatStringToString( catalog.getLocale( 992 ),
                                                                 copyorder->destdir,
                                                                 freespaceh.c_str() );

            erg = request( cowin, catalog.getLocale( 123 ), text.c_str(), buttons.c_str() );

            if ( erg != 0 ) {
                cancel = true;
            }
        }
    }

    // now database ready, can start copy-process

    copy_deletefe_list = new std::list<const FileEntry*>;
    copy_lvc_list = new std::list<NM_specialsourceInt*>;

    for ( iti1 = copylist->begin(); ( iti1 != copylist->end() ) && ( cancel == false ); iti1++ ) {
        ss1 = *iti1;
        if(nodatabase==false) {
            tcod=ss1->cod;
        } else {
            enter=false;
            tcod=NULL;
            if ( ss1->entry()->isDir() == true ) {
                // entry is a dir so it cannot be a corrupt link so no need to check
                if ( ss1->entry()->isLink == false ) enter = true;
                else if(copyorder->follow_symlinks==true) enter=true;
            }
            if ( enter == true ) {
                tcod = new NM_CopyOp_Dir( ss1->entry() );
                if ( tcod->user_abort == true ) {
                    delete tcod;
                    tcod = NULL;
                    cancel = true;
                }
            }
        }
        if ( cancel == true ) break;
        if ( ss1->row >= 0 ) {
            lv->setVisMark( ss1->row, true );
            lv->showRow( ss1->row, false );

            // this is some optimization when copying a lot of small files
            // limit redrawing in this case
            if ( tcod == NULL ) {
                loff_t size = 0;

                if ( ss1->entry()->isLink && ! ss1->entry()->isCorrupt ) {
                    size = ss1->entry()->dsize();
                } else {
                    size = ss1->entry()->size();
                }

                if ( size > 128 * 1024 ) {
                    lv->redraw();
                } else {
                    lv->timeLimitedRedraw();
                }
            } else {
                lv->redraw();
            }
        }
        if(tcod!=NULL) {
            ce1 = copydir( ss1->entry(), tcod, copyorder, true, copyorder->destdir );
            if((ce1==NM_COPY_OK)||(ce1==NM_COPY_NEED_DELETE)) {
                // success
            } else if(ce1==NM_COPY_CANCEL) {
                cancel=true;
            }
            if(copyorder->move==true) {
                if((ce1==NM_COPY_OK)&&(tcod->error_counter==0)) {
                    if ( ss1->row >= 0 ) lv->showRow( ss1->row, false );
                    removeEntry( ss1->entry(), ss1->row );
                    fixSpecialSourceList( copylist, ss1->row );
                } else if(ce1==NM_COPY_NEED_DELETE) {
                    if(ss1->row>=0) {
                        tss = new NM_specialsourceInt( ss1->entry() );
                        tss->row=ss1->row;
                        copy_lvc_list->push_back(tss);
                    }
                }
            } else {
                if((tcod->error_counter==0)&&(cancel==false)) {
                    // only deselect when dir totally copied
                    if(ss1->row>=0) {
                        lv->setSelect( ss1->row, false );
                        lv->showRow( ss1->row, false );
                    }
                    ss1->entry()->select=false;
                    ce->recalcStats();
                    showCacheState();
                }
            }
            if(nodatabase==true) delete tcod;
        } else {
            ce1 = copyfile( ss1->entry(), copyorder, true, copyorder->destdir );
            if((ce1==NM_COPY_OK)||(ce1==NM_COPY_NEED_DELETE)) {
                // success
                if(copyorder->move==true) {
                    if(ce1==NM_COPY_OK) {
                        if ( ss1->row >= 0 ) lv->showRow( ss1->row, false );
                        removeEntry( ss1->entry(), ss1->row );
                        fixSpecialSourceList( copylist, ss1->row );
                    } else {
                        if(ss1->row>=0) {
                            tss = new NM_specialsourceInt( ss1->entry() );
                            tss->row=ss1->row;
                            copy_lvc_list->push_back(tss);
                        }
                    }
                } else {
                    if(ss1->row>=0) {
                        lv->setSelect(ss1->row,false);
                        lv->showRow( ss1->row, false );
                    }
                    ss1->entry()->select = false;
                    ce->recalcStats();
                    showCacheState();
                }
            } else if(ce1==NM_COPY_CANCEL) {
                cancel=true;
            }
        }
        if ( ss1->row >= 0 ) {
            // this can and will changed the wrong row when the entry
            // was moved successfully
            // but for this case this doesn't matter because
            // the disabled visual mark is the default
            lv->setVisMark( ss1->row, false );
        }
        if(cowin!=NULL) {
            if(cowin->redraw()!=0) cancel=true;
        }
    }
    lv->redraw();
  
    // now delete remaining files/dirs when moving
    if(copyorder->move==true) {
        bool delcancel; // additional cancel so we don't need to
        // clear cancel for the special case when move
        // has to copy&delete and was aborted somewhere
        // I ask the user what to do with the correctly copied
        // files
        std::string str1;
  
        if(cowin!=NULL) cowin->setmessage("",1);

        itfe1 = copy_deletefe_list->begin();

        if ( itfe1 != copy_deletefe_list->end() )
            tfe = *itfe1;
        else
            tfe = NULL;
        iti2 = copy_lvc_list->begin();
        if ( iti2 != copy_lvc_list->end() )
            tss = *iti2;
        else
            tss = NULL;
    
        delcancel = false;
        if ( ( cancel == true ) && ( tfe != NULL ) ) {
            // copy was canceled
            // ask the user what to do with the correctly moved files
            // (exactly what to do with the sources!!)
            str1 = catalog.getLocale( 11 );
            str1 += "|";
            str1 += catalog.getLocale( 8 );
            erg = request( cowin, catalog.getLocale( 123 ), catalog.getLocale( 528 ), str1.c_str() );
            if ( erg != 0 ) {
                delcancel = true;
            }
        }
    
        while ( ( tfe != NULL ) && ( delcancel == false ) ) {
            if(tfe->isDir()==true) {
                if(cowin!=NULL) {
                    tstr=(char*)_allocsafe(strlen(catalog.getLocale(141))+strlen(tfe->fullname)+1);
                    sprintf(tstr,catalog.getLocale(141),tfe->fullname);
                    cowin->setmessage(tstr,0);
                    _freesafe(tstr);
                }
            }
            // remove tfe
            if ( ( tfe->isDir() == true ) && ( tfe->isLink == false ) ) erg = worker_rmdir( tfe->fullname );
            else erg = worker_unlink( tfe->fullname );
            if(erg==0) {
                // success
                if(tss!=NULL) {
                    if ( tfe == tss->entry() ) {
                        lv->showRow(tss->row);
                        removeEntry( tss->entry(), tss->row );
                        fixSpecialSourceList( copy_lvc_list, tss->row );
                        iti2++;
                        if ( iti2 != copy_lvc_list->end() )
                            tss = *iti2;
                        else
                            tss = NULL;
                    }
                }
            } else {
                textstr=(char*)_allocsafe(strlen(catalog.getLocale(291))+strlen(tfe->fullname)+1);
                sprintf(textstr,catalog.getLocale(291),tfe->fullname);
                buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                            strlen(catalog.getLocale(8))+1);
                sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                        catalog.getLocale(8));
                erg = request( cowin, catalog.getLocale( 347 ), textstr, buttonstr );
                _freesafe(buttonstr);
                _freesafe(textstr);
                if ( erg == 1 ) delcancel = true;
            }
            if ( itfe1 != copy_deletefe_list->end() ) itfe1++;
            if ( itfe1 != copy_deletefe_list->end() )
                tfe = *itfe1;
            else
                tfe = NULL;
        }
        if ( delcancel == true ) cancel = true; // as I write this cancel is not used
        // but perhaps later I will use it and
        // so set it also to true in this case
    }
  
    if(cowin!=NULL) cowin->close();
  
    for ( iti1 = copylist->begin(); iti1 != copylist->end(); iti1++ ) {
        ss1 = *iti1;
        if(ss1->cod!=NULL) delete ss1->cod;
        delete ss1;
    }
    delete copylist;
    delete copy_deletefe_list;

    for ( iti1 = copy_lvc_list->begin(); iti1 != copy_lvc_list->end(); iti1++ ) {
        tss = *iti1;
        delete tss;
    } 

    delete copy_lvc_list;
    _freesafe(buf);
    if(copyorder->move==true) {
        ce->restartcheck();
        rebuildView();
    }
    if(updatenm!=NULL) updatenm->update(false);
    lv->showActive();
    aguix->Flush();

    enableDirWatch();
    if ( updatenm != this ) {
        updatenm->enableDirWatch();
    }
}

NormalMode::nm_newname_t NormalMode::getNewName( const FileEntry *oldfe, struct NM_copyorder *copyorder, const char *dest,
                                                 char **newname_return, bool requestDir, bool acceptrename )
{
    if(requestDir==true)
        return getNewName4Dir(oldfe,copyorder,dest,newname_return,acceptrename);
    else
        return getNewName4File(oldfe,copyorder,dest,newname_return,acceptrename);
}

NormalMode::nm_newname_t NormalMode::getNewName4Dir( const FileEntry *oldfe, struct NM_copyorder *copyorder, const char *dest,
                                                     char **newname_return, bool acceptrename )
{
    char *oldname=oldfe->name;
    char *newname=dupstring(oldname);
    bool nameok=false,skip=false,cancel=false;
    char *return_str=NULL,*buttonstr,*textstr,*newdest;
    int erg;
    Datei::d_fe_t se,lse;
    int round=1;
    nm_newname_t returnvalue=NM_NEWNAME_CANCEL;
    bool trydel,do_rename,strongdirs;
    char *title;
  
    title = (char*)_allocsafe( strlen( catalog.getLocale( 500 ) ) + strlen( oldfe->fullname ) + 1 );
    sprintf( title, catalog.getLocale( 500 ), oldfe->fullname );
  
    do_rename=false;
    if(copyorder->do_rename==false) do_rename=false;
    else if(acceptrename==true) do_rename=true;
  
    strongdirs=false;

    do {
        if(!((round==1)&&(do_rename==false))) {
            buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                        strlen(catalog.getLocale(225))+1+
                                        strlen(catalog.getLocale(8))+1);
            sprintf(buttonstr,"%s|%s|%s",catalog.getLocale(11),
                    catalog.getLocale(225),
                    catalog.getLocale(8));
            textstr=(char*)_allocsafe(strlen(catalog.getLocale(148))+strlen(oldname)+1);
            sprintf(textstr,catalog.getLocale(148),oldname);
            erg = string_request( copyorder->cowin,
                                  catalog.getLocale( 149 ),
                                  textstr,
                                  newname,
                                  buttonstr,
                                  &return_str,
                                  Requester::REQUEST_SELECTALL );
            _freesafe(buttonstr);
            _freesafe(textstr);
            if(return_str!=NULL) {
                _freesafe(newname);
                newname=return_str;
            }
            if(erg==1) {
                skip=true;
                returnvalue=NM_NEWNAME_SKIP;
                break;
            } else if(erg==2) {
                cancel=true;
                returnvalue=NM_NEWNAME_CANCEL;
                break;
            }
        }
    
        if ( strlen( newname ) < 1 ) {
            // empty name->request
            round++;  // not needed since empty names should only come from user input
            // but perhaps there is (will be) a OS which allows empty names
            continue;
        }
    
        newdest=(char*)_allocsafe(strlen(dest)+1+strlen(newname)+1);
        strcpy(newdest,dest);
        if(strlen(dest)>1) strcat(newdest,"/");
        strcat(newdest,newname);
        se=Datei::fileExistsExt(newdest);
        lse=Datei::lfileExistsExt(newdest);

        switch(lse) {
            case Datei::D_FE_DIR:
                if(strongdirs==true) {
                    textstr = (char*)_allocsafe( strlen( catalog.getLocale( 190 ) ) + strlen( newdest ) + 1 );
                    sprintf( textstr, catalog.getLocale( 190 ), newdest );

                    buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(230))+1+
                                                strlen(catalog.getLocale(273))+1+
                                                strlen(catalog.getLocale(225))+1+
                                                strlen(catalog.getLocale(8))+1);
                    sprintf(buttonstr,"%s|%s|%s|%s",catalog.getLocale(230),
                            catalog.getLocale(273),
                            catalog.getLocale(225),
                            catalog.getLocale(8));

                    erg = request( copyorder->cowin, title/*catalog.getLocale(123)*/, textstr, buttonstr );
                    _freesafe(buttonstr);
                    _freesafe(textstr);
                    if(erg==1) {
                        nameok=true;
                        returnvalue=NM_NEWNAME_USE;
                    } else if(erg==2) {
                        skip=true;
                        returnvalue=NM_NEWNAME_SKIP;
                    } else if(erg==3) {
                        cancel=true;
                        returnvalue=NM_NEWNAME_CANCEL;
                    }
                } else {
                    // just use the dir
                    nameok=true;
                    returnvalue=NM_NEWNAME_USE;
                }
                break;
            case Datei::D_FE_LINK:
            case Datei::D_FE_FILE:
                if((lse==Datei::D_FE_LINK)&&(se==Datei::D_FE_DIR)) {
                    // Symlink with dir as destination 
                    if(strongdirs==true) {
                        //            textstr="There is already a symlink called %s, which points to a dir|You can use this link or delete it to|create a real directory";
                        //            buttonstr="Enter new name|Use this dir|Delete link|Skip|Cancel";

                        textstr = (char*)_allocsafe( strlen( catalog.getLocale( 275 ) ) + strlen( newdest ) + 1 );
                        sprintf( textstr, catalog.getLocale( 275 ), newdest );
                        buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(230))+1+
                                                    strlen(catalog.getLocale(273))+1+
                                                    strlen(catalog.getLocale(274))+1+
                                                    strlen(catalog.getLocale(225))+1+
                                                    strlen(catalog.getLocale(8))+1);
                        sprintf(buttonstr,"%s|%s|%s|%s|%s",catalog.getLocale(230),
                                catalog.getLocale(273),
                                catalog.getLocale(274),
                                catalog.getLocale(225),
                                catalog.getLocale(8));

                        erg = request( copyorder->cowin, title/*catalog.getLocale(123)*/, textstr, buttonstr );
                        _freesafe(buttonstr);
                        _freesafe(textstr);
                        switch(erg) {
                            case 1:
                                nameok=true;
                                returnvalue=NM_NEWNAME_USE;
                                break;
                            case 2:
                                // try to remove the link
                                // if success set nameok to true
                                // in case of failure show a request and repeat the whole
                                if ( worker_unlink( newdest ) == 0 ) {
                                    nameok=true;
                                    returnvalue=NM_NEWNAME_OK;
                                } else {
                                    //                  textstr="Failed to remove this file|Please enter new name!";
                                    //                  buttonstr="Ok|Skip|Cancel";

                                    textstr = dupstring( catalog.getLocale(276) );
                                    buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                                                strlen(catalog.getLocale(225))+1+
                                                                strlen(catalog.getLocale(8))+1);
                                    sprintf(buttonstr,"%s|%s|%s",catalog.getLocale(11),
                                            catalog.getLocale(225),
                                            catalog.getLocale(8));

                                    erg = request( copyorder->cowin, catalog.getLocale( 347 ), textstr, buttonstr );
                                    _freesafe(buttonstr);
                                    _freesafe( textstr );
                                    if(erg==1) {
                                        skip=true;
                                        returnvalue=NM_NEWNAME_SKIP;
                                    } else if(erg==2) {
                                        cancel=true;
                                        returnvalue=NM_NEWNAME_CANCEL;
                                    }
                                }
                                break;
                            case 3:
                                skip=true;
                                returnvalue=NM_NEWNAME_SKIP;
                                break;
                            case 4:
                                cancel=true;
                                returnvalue=NM_NEWNAME_CANCEL;
                                break;
                        }
                    } else {
                        // just use the link to the dir
                        nameok=true;
                        returnvalue=NM_NEWNAME_USE;
                    }
                } else {
                    trydel=false;
                    if((do_rename==false)&&(copyorder->overwrite==copyorder->NM_OVERWRITE_ALWAYS)) {
                        trydel=true;
                    } else if((do_rename==false)&&(copyorder->overwrite==copyorder->NM_OVERWRITE_NEVER)) {
                        // skip
                        skip=true;
                        returnvalue=NM_NEWNAME_SKIP;
                    } else {
                        if(lse==Datei::D_FE_LINK) {
                            //              textstr="There is already a link named %s";
                            //              buttonstr="Enter new name|Delete link|Skip|Cancel";

                            textstr = (char*)_allocsafe( strlen( catalog.getLocale( 278 ) ) + strlen( newdest ) + 1 );
                            sprintf( textstr, catalog.getLocale( 278 ), newdest );
                            buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(230))+1+
                                                        strlen(catalog.getLocale(274))+1+
                                                        strlen(catalog.getLocale(225))+1+
                                                        strlen(catalog.getLocale(8))+1);
                            sprintf(buttonstr,"%s|%s|%s|%s",catalog.getLocale(230),
                                    catalog.getLocale(274),
                                    catalog.getLocale(225),
                                    catalog.getLocale(8));
                        } else {
                            //              textstr="There is already a file named %s";
                            //              buttonstr="Enter new name|Delete file|Skip|Cancel";

                            textstr = (char*)_allocsafe( strlen( catalog.getLocale( 279 ) ) + strlen( newdest ) + 1 );
                            sprintf( textstr, catalog.getLocale( 279 ), newdest );
                            buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(230))+1+
                                                        strlen(catalog.getLocale(277))+1+
                                                        strlen(catalog.getLocale(225))+1+
                                                        strlen(catalog.getLocale(8))+1);
                            sprintf(buttonstr,"%s|%s|%s|%s",catalog.getLocale(230),
                                    catalog.getLocale(277),
                                    catalog.getLocale(225),
                                    catalog.getLocale(8));
                        }
                        erg = request( copyorder->cowin, title/*catalog.getLocale(123)*/, textstr, buttonstr );
                        _freesafe(buttonstr);
                        _freesafe(textstr);
                        switch(erg) {
                            case 1:
                                trydel=true;
                                break;
                            case 2:
                                skip=true;
                                returnvalue=NM_NEWNAME_SKIP;
                                break;
                            case 3:
                                cancel=true;
                                returnvalue=NM_NEWNAME_CANCEL;
                                break;
                        }
                    }
                    if(trydel==true) {
                        if ( worker_unlink( newdest ) == 0 ) {
                            nameok=true;
                            returnvalue=NM_NEWNAME_OK;
                        } else {
                            //              textstr="Failed to remove this file|Please enter new name!";
                            //              buttonstr="Ok|Skip|Cancel";

                            textstr = dupstring( catalog.getLocale(276) );
                            buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                                        strlen(catalog.getLocale(225))+1+
                                                        strlen(catalog.getLocale(8))+1);
                            sprintf(buttonstr,"%s|%s|%s",catalog.getLocale(11),
                                    catalog.getLocale(225),
                                    catalog.getLocale(8));

                            erg = request( copyorder->cowin, catalog.getLocale( 347 ), textstr, buttonstr );
                            _freesafe(buttonstr);
                            _freesafe( textstr );
                            if(erg==1) {
                                skip=true;
                                returnvalue=NM_NEWNAME_SKIP;
                            } else if(erg==2) {
                                cancel=true;
                                returnvalue=NM_NEWNAME_CANCEL;
                            }
                        }
                    }
                }
                break;
            default:
                nameok=true;
                returnvalue=NM_NEWNAME_OK;
                break;
        }
        _freesafe(newdest);
        round++;
    } while((nameok==false)&&(skip==false)&&(cancel==false));
    if((skip==true)||(cancel==true)) {
        _freesafe(newname);
        *newname_return=NULL;
    } else {
        *newname_return=newname;
    }
    _freesafe( title );
    return returnvalue;
}

NormalMode::nm_newname_t NormalMode::getNewName4File( const FileEntry *oldfe, struct NM_copyorder *copyorder, const char *dest,
                                                      char **newname_return, bool acceptrename )
{
    char *oldname=oldfe->name;
    char *newname=dupstring(oldname);
    bool nameok=false,skip=false,cancel=false;
    char *return_str=NULL,*buttonstr,*textstr,*newdest;
    int erg;
    Datei::d_fe_t lse;
    int round=1;
    nm_newname_t returnvalue=NM_NEWNAME_CANCEL;
    bool do_rename,trydel;
    char *extrastr,*tstr,*newtime,*oldtime;
    int newsizelen,oldsizelen,maxsizelen;
    char *title;
    time_t tvar;
    std::string newsize, oldsize;
  
    title = (char*)_allocsafe( strlen( catalog.getLocale( 116 ) ) + strlen( oldfe->fullname ) + 1 );
    sprintf( title, catalog.getLocale( 116 ), oldfe->fullname );
  
    do_rename=false;
    if(copyorder->do_rename==false) do_rename=false;
    else if(acceptrename==true) do_rename=true;

    do {
        if(!((round==1)&&(do_rename==false))) {
            //      buttonstr="Ok|Skip|Cancel";
            //      textstr="Enter new name for ...";
            buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                        strlen(catalog.getLocale(225))+1+
                                        strlen(catalog.getLocale(8))+1);
            sprintf(buttonstr,"%s|%s|%s",catalog.getLocale(11),
                    catalog.getLocale(225),
                    catalog.getLocale(8));
            textstr=(char*)_allocsafe(strlen(catalog.getLocale(148))+strlen(oldname)+1);
            sprintf(textstr,catalog.getLocale(148),oldname);
            erg = string_request( copyorder->cowin,
                                  catalog.getLocale( 149 ),
                                  textstr,
                                  newname,
                                  buttonstr,
                                  &return_str,
                                  Requester::REQUEST_SELECTALL );
            _freesafe(buttonstr);
            _freesafe(textstr);
            if(return_str!=NULL) {
                _freesafe(newname);
                newname=return_str;
            }
            if(erg==1) {
                skip=true;
                returnvalue=NM_NEWNAME_SKIP;
                break;
            } else if(erg==2) {
                cancel=true;
                returnvalue=NM_NEWNAME_CANCEL;
                break;
            }
        }
    
        if ( strlen( newname ) < 1 ) {
            // empty name->request
            round++;  // not needed since empty names should only come from user input
            // but perhaps there is (will be) a OS which allows empty names
            continue;
        }
    
        newdest=(char*)_allocsafe(strlen(dest)+1+strlen(newname)+1);
        strcpy(newdest,dest);
        if(strlen(dest)>1) strcat(newdest,"/");
        strcat(newdest,newname);
        lse=Datei::lfileExistsExt(newdest);

        switch(lse) {
            case Datei::D_FE_DIR:
                //        textstr="there is already a dir called %s";
                //        buttonstr="Enter new name|Skip|Cancel";

                textstr = (char*)_allocsafe( strlen( catalog.getLocale( 190 ) ) + strlen( newdest ) + 1 );
                sprintf( textstr, catalog.getLocale( 190 ), newdest );

                buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(230))+1+
                                            strlen(catalog.getLocale(225))+1+
                                            strlen(catalog.getLocale(8))+1);
                sprintf(buttonstr,"%s|%s|%s",catalog.getLocale(230),
                        catalog.getLocale(225),
                        catalog.getLocale(8));

                erg = request( copyorder->cowin, title/*catalog.getLocale(123)*/, textstr, buttonstr );
                _freesafe(buttonstr);
                _freesafe(textstr);
                if(erg==1) {
                    skip=true;
                    returnvalue=NM_NEWNAME_SKIP;
                } else if(erg==2) {
                    cancel=true;
                    returnvalue=NM_NEWNAME_CANCEL;
                }
                break;
            case Datei::D_FE_LINK:
                trydel=false;
                if((do_rename==false)&&(copyorder->overwrite==copyorder->NM_OVERWRITE_ALWAYS)) {
                    trydel=true;
                } else if((do_rename==false)&&(copyorder->overwrite==copyorder->NM_OVERWRITE_NEVER)) {
                    // skip
                    skip=true;
                    returnvalue=NM_NEWNAME_SKIP;
                } else {
                    //          textstr="there is already a link named %s";
                    //          buttonstr="Enter new name|Delete link|Skip|Cancel";

                    textstr = (char*)_allocsafe( strlen( catalog.getLocale( 278 ) ) + strlen( newdest ) + 1 );
                    sprintf( textstr, catalog.getLocale( 278 ), newdest );
                    buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(230))+1+
                                                strlen(catalog.getLocale(274))+1+
                                                strlen(catalog.getLocale(225))+1+
                                                strlen(catalog.getLocale(8))+1);
                    sprintf(buttonstr,"%s|%s|%s|%s",catalog.getLocale(230),
                            catalog.getLocale(274),
                            catalog.getLocale(225),
                            catalog.getLocale(8));

                    erg = request( copyorder->cowin, title/*catalog.getLocale(123)*/, textstr, buttonstr );
                    _freesafe(buttonstr);
                    _freesafe(textstr);
                    switch(erg) {
                        case 1:
                            trydel=true;
                            break;
                        case 2:
                            skip=true;
                            returnvalue=NM_NEWNAME_SKIP;
                            break;
                        case 3:
                            cancel=true;
                            returnvalue=NM_NEWNAME_CANCEL;
                            break;
                    }
                }
                if(trydel==true) {
                    if ( worker_unlink( newdest ) == 0 ) {
                        nameok=true;
                        returnvalue=NM_NEWNAME_OK;
                    } else {
                        //            textstr="Failed to remove this file|Please enter new name!";
                        //            buttonstr="Ok|Skip|Cancel";

                        textstr = dupstring( catalog.getLocale(276) );
                        buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                                    strlen(catalog.getLocale(225))+1+
                                                    strlen(catalog.getLocale(8))+1);
                        sprintf(buttonstr,"%s|%s|%s",catalog.getLocale(11),
                                catalog.getLocale(225),
                                catalog.getLocale(8));

                        erg = request( copyorder->cowin, catalog.getLocale( 347 ), textstr, buttonstr );
                        _freesafe(buttonstr);
                        _freesafe( textstr );
                        if(erg==1) {
                            skip=true;
                            returnvalue=NM_NEWNAME_SKIP;
                        } else if(erg==2) {
                            cancel=true;
                            returnvalue=NM_NEWNAME_CANCEL;
                        }
                    }
                }
                break;
            case Datei::D_FE_FILE:
                if((do_rename==false)&&(copyorder->overwrite==copyorder->NM_OVERWRITE_ALWAYS)) {
                    nameok=true;
                    returnvalue=NM_NEWNAME_OVERWRITE;
                } else if((do_rename==false)&&(copyorder->overwrite==copyorder->NM_OVERWRITE_NEVER)) {
                    skip=true;
                    returnvalue=NM_NEWNAME_SKIP;
                } else {
                    FileEntry *newfe=new FileEntry();
                    if(newfe->name!=NULL) _freesafe(newfe->name);
                    if(newfe->fullname!=NULL) _freesafe(newfe->fullname);
                    newfe->name=dupstring(newname);
                    newfe->fullname=dupstring(newdest);
                    newfe->readInfos();
          
                    MakeLong2NiceStr( newfe->size(), newsize );
                    MakeLong2NiceStr( oldfe->size(), oldsize );
                    oldsizelen = oldsize.length();
                    newsizelen = newsize.length();
                    maxsizelen = ( newsizelen > oldsizelen ) ? newsizelen : oldsizelen;
                    if ( maxsizelen > newsizelen )
                        newsize.insert( newsize.begin(), maxsizelen - newsizelen, ' ' );
                    if ( maxsizelen > oldsizelen )
                        oldsize.insert( oldsize.begin(), maxsizelen - oldsizelen, ' ' );

                    tvar = newfe->lastmod();
                    newtime = dupstring( ctime( &( tvar ) ) );
                    tvar = oldfe->lastmod();
                    oldtime = dupstring( ctime( &( tvar ) ) );
                    newtime[strlen(newtime)-1]='\0';  // remove return
                    oldtime[strlen(oldtime)-1]='\0';  // remove return
          
                    extrastr=(char*)_allocsafe(strlen(catalog.getLocale(137))+newsize.length()+oldsize.length()+
                                               strlen(newtime)+strlen(oldtime)+1);
                    sprintf(extrastr,catalog.getLocale(137),newsize.c_str(),oldsize.c_str(),newtime,oldtime);
          
                    _freesafe(newtime);
                    _freesafe(oldtime);
                    delete newfe;
                    //          textstr="There is already a file named %s";
                    //          buttonstr="Enter new name|Overwrite|Overwrite all|Overwrite none|Skip|Cancel";

                    textstr = (char*)_allocsafe( strlen( catalog.getLocale( 279 ) ) + strlen( newdest ) + 1 );
                    sprintf( textstr, catalog.getLocale( 279 ), newdest );

                    tstr=catstring(textstr,"|");
                    _freesafe(textstr);
                    textstr=catstring(tstr,extrastr);
                    _freesafe(tstr);
                    _freesafe(extrastr);

                    buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(230))+1+
                                                strlen(catalog.getLocale(280))+1+
                                                strlen(catalog.getLocale(135))+1+
                                                strlen(catalog.getLocale(136))+1+
                                                strlen(catalog.getLocale(225))+1+
                                                strlen(catalog.getLocale(8))+1);
                    sprintf(buttonstr,"%s|%s|%s|%s|%s|%s",catalog.getLocale(230),
                            catalog.getLocale(280),
                            catalog.getLocale(135),
                            catalog.getLocale(136),
                            catalog.getLocale(225),
                            catalog.getLocale(8));

                    erg = request( copyorder->cowin, title/*catalog.getLocale(123)*/, textstr, buttonstr );
                    _freesafe(buttonstr);
                    _freesafe(textstr);
                    switch(erg) {
                        case 1:
                            nameok=true;
                            returnvalue=NM_NEWNAME_OVERWRITE;
                            break;
                        case 2:
                            nameok=true;
                            returnvalue=NM_NEWNAME_OVERWRITE;
                            copyorder->overwrite=copyorder->NM_OVERWRITE_ALWAYS;
                            break;
                        case 3:
                            skip=true;
                            returnvalue=NM_NEWNAME_SKIP;
                            copyorder->overwrite=copyorder->NM_OVERWRITE_NEVER;
                            break;
                        case 4:
                            skip=true;
                            returnvalue=NM_NEWNAME_SKIP;
                            break;
                        case 5:
                            cancel=true;
                            returnvalue=NM_NEWNAME_CANCEL;
                            break;
                    }
                }
                break;
            default:
                nameok=true;
                returnvalue=NM_NEWNAME_OK;
                break;
        }
        _freesafe(newdest);
        round++;
    } while((nameok==false)&&(skip==false)&&(cancel==false));
    if((skip==true)||(cancel==true)) {
        _freesafe(newname);
        *newname_return=NULL;
    } else {
        *newname_return=newname;
    }
    _freesafe( title );
    return returnvalue;
}

NormalMode::nm_copy_t
NormalMode::copydir( const FileEntry *fe, NM_CopyOp_Dir *cod, struct NM_copyorder *copyorder, bool acceptrename, const char *destdir )
{
    NM_CopyOp_Dir *cod1=NULL,*tcod;
    bool nameok,createdir;
    nm_newname_t e1;
    char *destsubdir;
    Datei::d_fe_t de1;
    char *newname=NULL;
    int id1;
    FileEntry *subfe;
    bool isdir,skip,cancel;
    char *buttonstr,*textstr;
    int erg;
    nm_copy_t ce1;
    CopyOpWin *cowin=copyorder->cowin;
    bool nodatabase;
    bool docopy,failedmove=false;
    struct utimbuf utb;
    bool corrupt_entries_skipped = false;

    if(copyorder->copymode==copyorder->NM_COPYMODE_NODATABASE) nodatabase=true;
    else nodatabase=false;

    skip = cancel = false;
    if ( cowin != NULL ) {
        // call newfile without destination so the user
        // can see the filename in the copyopwin when he will
        // be asked for the new name
        cowin->newfile( fe->name, NULL );
        if ( cowin->redraw() != 0 ) cancel = true;
    }

    if ( cancel == false ) {
        e1=getNewName(fe,copyorder,destdir,&newname,true,acceptrename);
        if((e1==NM_NEWNAME_OK)||(e1==NM_NEWNAME_USE)) {
            // build new name
            destsubdir=(char*)_allocsafe(strlen(destdir)+1+strlen(newname)+1);
            strcpy(destsubdir,destdir);
            if(strlen(destsubdir)>1) strcat(destsubdir,"/");
            strcat(destsubdir,newname);
      
            nameok=false;
            createdir=true;
            if(e1==NM_NEWNAME_USE) {
                // there is already such dir (or a symlink to a dir) -> use it
                nameok=true;
                createdir=false;
            } else if(e1==NM_NEWNAME_OK) {
                // check for fileexists (should not)
                // if nethertheless then skip this dir because the rename-func wasn't able to remove it
                de1=Datei::lfileExistsExt(destsubdir);
                if(de1==Datei::D_FE_NOFILE) {
                    // everything is ok
                    nameok=true;
                } else if(de1==Datei::D_FE_DIR) {
                    // dest is a dir
                    // should not happend but ok
                    nameok=true;
                    createdir=false;
                    e1=NM_NEWNAME_USE; // for moving it's important to set this, so it doesn't try to
                    // move
                }
            }
            if(nameok==true) {
                docopy=true;
                failedmove=false;
                if(copyorder->move==true) {
                    if(e1==NM_NEWNAME_USE) {
                        // because of using the destination dir
                        // we need to do as failed rename
                        failedmove=true;
                    } else {
                        if ( worker_rename( fe->fullname, destsubdir ) == 0 ) {
                            // success
                            docopy=false;
                        } else {
                            // failure
                            failedmove=true;
                        }
                    }
                }
        
                if(docopy==true) {
                    if(cowin!=NULL) cowin->newfile(fe->name,destsubdir);
                    if(createdir==true) {
                        if ( worker_mkdir( destsubdir, 0700 ) != 0 ) {
                            // failed to create dir -> skip
                            //            textstr="Failed to create dir|I will skip this dir!";
                            //            buttonstr="Ok|Cancel";
              
                            if(cowin!=NULL) cowin->stoptimer();
                            textstr=(char*)_allocsafe(strlen(catalog.getLocale(126))+strlen(destsubdir)+1);
                            sprintf(textstr,catalog.getLocale(126),destsubdir);
                            buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                                        strlen(catalog.getLocale(8))+1);
                            sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                                    catalog.getLocale(8));
                            erg = request( copyorder->cowin, catalog.getLocale( 347 ), textstr, buttonstr );
                            _freesafe(buttonstr);
                            _freesafe(textstr);
                            if(erg==1) cancel=true;
                            nameok=false;
                            if(cowin!=NULL) cowin->conttimer();
                        }
                    }
                    if(nameok==true) {
                        // first copy all subdirs
                        // then copy all files 
                        if(nodatabase==false) {
                            id1=cod->subdirs->initEnum();
                            cod1=(NM_CopyOp_Dir*)cod->subdirs->getFirstElement(id1);
                            while(cod1!=NULL) {
                                ce1=copydir(cod1->fileentry,cod1,copyorder,false,destsubdir);
                                if((ce1==NM_COPY_OK)||(ce1==NM_COPY_NEED_DELETE)) {
                                    // success
                                    cod->error_counter+=cod1->error_counter;
                                } else {
                                    cod->error_counter+=cod1->error_counter+1;
                                    if(ce1==NM_COPY_CANCEL) {
                                        cancel=true;
                                        break;
                                    }
                                }
                                cod1=(NM_CopyOp_Dir*)cod->subdirs->getNextElement(id1);
                            }
                            cod->subdirs->closeEnum(id1);
                        }
            
                        // next only if read this dir correctly
                        if((cod->ok==true)&&(cancel==false)) {
                            for ( Verzeichnis::verz_it subfe_it1 = cod->verz->begin();
                                  subfe_it1 != cod->verz->end() && cancel == false;
                                  subfe_it1++ ) {
                                subfe = *subfe_it1;

                                if ( subfe->isCorrupt == true && subfe->isLink == false ) {
                                    if ( cowin != NULL ) cowin->stoptimer();
                    
                                    std::string text = AGUIXUtils::formatStringToString( catalog.getLocale( 909 ),
                                                                                         subfe->fullname );
                                    std::string button = std::string( catalog.getLocale( 225 ) ) + "|" + catalog.getLocale( 8 );
                                    erg = request( copyorder->cowin, catalog.getLocale( 347 ), text.c_str(), button.c_str() );
                                    if ( erg == 0 ) {
                                    } else {
                                        cancel = true;
                                    }
                    
                                    if ( cowin != NULL ) cowin->conttimer();
                    
                                    corrupt_entries_skipped = true;
                                    continue;
                                }

                                if(strcmp(subfe->name,"..")!=0) {
                                    isdir=false;
                                    if(subfe->isDir()==true) {
                                        if(subfe->isLink==false) isdir=true;
                                        else if(copyorder->follow_symlinks==true) isdir=true;
                                    }
                                    if(isdir==false) {
                                        ce1=copyfile(subfe,copyorder,false,destsubdir);
                                        if((ce1==NM_COPY_OK)||(ce1==NM_COPY_NEED_DELETE)) {
                                            // success
                                        } else {
                                            cod->error_counter++;
                                            if(ce1==NM_COPY_CANCEL) {
                                                cancel=true;
                                                break;
                                            }
                                        }
                                    } else {
                                        if(nodatabase==true) {
                                            // no database so copy the dir
                                            tcod=new NM_CopyOp_Dir(subfe);
                                            if ( tcod->user_abort == false ) {
                                                ce1=copydir(subfe,tcod,copyorder,false,destsubdir);
                                                if(ce1==NM_COPY_OK) {
                                                    // success
                                                    cod->error_counter+=tcod->error_counter;
                                                } else {
                                                    cod->error_counter+=tcod->error_counter+1;
                                                    if(ce1==NM_COPY_CANCEL) {
                                                        cancel=true;
                                                    }
                                                }
                                            } else cancel = true;
                                            delete tcod;
                                            if(cancel==true) break;
                                        }
                                    }
                                }
                            }
                        }
                        // finally change the permissions of the dir
                        if((createdir==true)&&(copyorder->preserve_attr==true)) {
                            // fe is a dir so if it is a link it is not corrupt
                            worker_chown( destsubdir, ( fe->isLink == true ) ? fe->duserid() : fe->userid(),
                                          ( fe->isLink == true ) ? fe->dgroupid() : fe->groupid() );
                            worker_chmod( destsubdir, ( fe->isLink == true ) ? fe->dmode() : fe->mode() );
                            utb.actime = ( fe->isLink == true ) ? fe->dlastaccess() : fe->lastaccess();
                            utb.modtime = ( fe->isLink == true ) ? fe->dlastmod() : fe->lastmod();
                            worker_utime(destsubdir,&utb);
                        }
                    } else skip=true;
                }
            } else skip=true;
            _freesafe(destsubdir);
        } else {
            switch(e1) {
                case NM_NEWNAME_CANCEL:
                    cancel=true;
                default:
                    skip=true;
                    break;
            }
        }
    }
    if((skip==true)&&(cowin!=NULL)) {
        // dir skiped so dec CopyOpWin counter by the files/dirs skiped
        cowin->dec_file_counter(cod->files);
        cowin->dec_dir_counter(cod->dirs);
        cowin->dec_byte_counter(cod->bytes);
    }

    if ( corrupt_entries_skipped == true ) skip = true;

    if(cowin!=NULL) {
        cowin->dir_finished();
        if(cowin->redraw()!=0) cancel=true;
    }
    if(newname!=NULL) _freesafe(newname);
    if(cancel==true) return NM_COPY_CANCEL;
    else if(skip==true) {
        cod->error_counter++;
        return NM_COPY_SKIP;
    }
    if(copyorder->move==true) {
        if((cod->error_counter==0)&&(failedmove==true)) {
            // put fe in list

            // Man koennte das FE auch also Kopie benutzen
            // man muesste nur den Vergleich beim spaeteren Loeschen anpassen
            copy_deletefe_list->push_back( fe );
            return NM_COPY_NEED_DELETE;
        }
    }
    return NM_COPY_OK;
}

NormalMode::nm_copy_t
NormalMode::copyfile( const FileEntry *fe, struct NM_copyorder *copyorder, bool acceptrename, const char *destdir )
{
    bool nameok;
    nm_newname_t e1;
    char *newfullname;
    Datei::d_fe_t de1;
    char *newname=NULL;
    bool useregcopy, skip, cancel;
    int erg;
    CopyOpWin *cowin=copyorder->cowin;
    bool docopy,failedmove=false,opok=false;
    struct utimbuf utb;
    const char *myerrstr;
    loff_t bytesToCopy;
    mode_t modeCheck;
    bool useLinkDest;
    std::string text, button;
  
    if ( ( fe->isLink == true ) && ( fe->isCorrupt == false ) )
        bytesToCopy = fe->dsize();
    else
        bytesToCopy = fe->size();

    skip = cancel = false;
    if ( cowin != NULL ) {
        // call newfile without destination so the user
        // can see the filename in the copyopwin when he will
        // be asked for the new name
        cowin->newfile( fe->name, NULL );
        if ( cowin->redraw() != 0 ) cancel = true;
    }

    if ( fe->isLink == false && fe->isCorrupt == true ) {
        if ( cowin != NULL ) cowin->stoptimer();

        text = AGUIXUtils::formatStringToString( catalog.getLocale( 909 ),
                                                 fe->fullname );
        button = std::string( catalog.getLocale( 225 ) ) + "|" + catalog.getLocale( 8 );
        erg = request( copyorder->cowin, catalog.getLocale( 347 ), text.c_str(), button.c_str() );
        if ( erg == 0 ) {
            skip = true;
        } else {
            cancel = true;
        }

        if ( cowin != NULL ) cowin->conttimer();
    }

    if ( cancel == false && skip == false ) {
        e1=getNewName(fe,copyorder,destdir,&newname,false,acceptrename);
    
        if((e1==NM_NEWNAME_OK)||(e1==NM_NEWNAME_OVERWRITE)) {
            // build new name
            newfullname=(char*)_allocsafe(strlen(destdir)+1+strlen(newname)+1);
            strcpy(newfullname,destdir);
            if(strlen(newfullname)>1) strcat(newfullname,"/");
            strcat(newfullname,newname);
      
            nameok=false;
            skip=false;
            if(e1==NM_NEWNAME_OVERWRITE) {
                nameok=true;
            } else if(e1==NM_NEWNAME_OK) {
                // check for fileexists (should not)
                de1=Datei::lfileExistsExt(newfullname);
                if(de1==Datei::D_FE_NOFILE) {
                    // everything is ok
                    nameok=true;
                } else if(de1==Datei::D_FE_FILE) {
                    // dest is a dir
                    // should not happend but ok try to overwrite this
                    nameok=true;
                }
            }
            if(nameok==true) {
                if ( fe->isSameFile( newfullname, copyorder->follow_symlinks ) == false ) {
                    // wenn move true, dann rename probieren
                    //        klappt rename nicht, muss flag gesetzt werden und normal weitergemacht
                    //        werden und bei korrekten kopieren FE in liste gepackt werden und NEED_DELETE
                    //        zurueckgeliefert werden
                    //      also: 1.flag docopy am Anfang auf true setzen
                    //            2.bei move==true rename probieren
                    //            2.1.bei Erfolg docopy auf false, Rueckgabewert ist automatisch OK
                    //            2.2.bei Misslingen failedmove auf true setzen
                    //            3.bei docopy==true normal kopieren
                    //            3.1.bei Misslingen kommt halt skip zurueck
                    //            3.2.Bei Erfolg und failedmove==true muss FE in liste gepackt werden
                    //                und NEED_DELETE zurueckgeliefert werden
                    docopy=true;
                    failedmove=false;
          
                    if ( cowin != NULL ) cowin->newfile( fe->name, newfullname );
          
                    if(copyorder->move==true) {
                        if ( worker_rename( fe->fullname, newfullname ) == 0 ) {
                            // success
                            docopy=false;
                        } else {
                            // failure
                            failedmove=true;
                        }
                    }
          
                    if(docopy==true) {
                        // first try to open the file
                        // in case of failure give a requester with choose "Delete"
                        // then try to delete it and retry to open
                        // if all fails, skip
                        useregcopy = false;
                        if ( S_ISREG( fe->mode() ) ) useregcopy = true;
                        else if ( ( fe->isLink == true ) &&
                                  ( copyorder->follow_symlinks == true ) &&
                                  ( fe->isCorrupt == false ) &&
                                  ( S_ISREG( fe->dmode() ) ) ) useregcopy = true;
            
                        if ( useregcopy == true ) {
                            reg_copy_erg_t rce = copyfile_reg( fe->fullname,
                                                               newfullname,
                                                               destdir,
                                                               newname,
                                                               ( fe->isLink == true ) ? fe->dmode() : fe->mode(),
                                                               copyorder->cowin,
                                                               bytesToCopy );
                            if ( rce == REG_COPY_OK ) {
                                opok = true;
                            } else if ( rce == REG_COPY_SKIP ) {
                                skip = true;
                            } else {
                                // everything else (including ERROR) is cancel
                                cancel = true;
                            }
                        } else {
                            // no regular copy
                            if ( e1 == NM_NEWNAME_OVERWRITE ) {
                                // there is already a file but because we cannot overwrite
                                // with special file I will delete it
                                // remember the user already choose to overwrite in getNewName!
                                if ( worker_remove( newfullname ) != 0 ) {
                                    // remove failed => cannot continue
                  
                                    if ( cowin != NULL ) cowin->stoptimer();
                                    myerrstr = strerror( errno );
                                    text = AGUIXUtils::formatStringToString( catalog.getLocale( 516 ), fe->fullname, newfullname, myerrstr );
                                    button = std::string( catalog.getLocale( 225 ) ) + "|" + catalog.getLocale( 8 );
                                    erg = request( copyorder->cowin, catalog.getLocale( 347 ), text.c_str(), button.c_str() );
                                    if ( erg == 0 ) {
                                        skip = true;
                                    } else {
                                        cancel = true;
                                    }
                                    if ( cowin != NULL ) cowin->conttimer();
                                }
                            }
                            //TODO: mc will skip corrupt links when follow_symlinks is used
                            //      I could do this too if I set modeCheck = 0 so it will got to
                            //      the else part
                            if ( ( fe->isLink == true ) &&
                                 ( copyorder->follow_symlinks == true ) &&
                                 ( fe->isCorrupt == false ) ) {
                                useLinkDest = true;
                                modeCheck = fe->dmode();
                            } else {
                                useLinkDest = false;
                                modeCheck = fe->mode();
                            }
              
                            if ( ( skip == false ) && ( cancel == false ) ) {
                                if ( ( S_ISCHR( modeCheck ) ) ||
#ifdef S_ISSOCK
                                     ( S_ISSOCK( modeCheck ) ) ||
#endif
                                     ( S_ISBLK( modeCheck ) ) ) {
                                    worker_struct_stat statbuf;
                  
                                    if ( useLinkDest == true ) {
                                        erg = worker_stat( fe->fullname, &statbuf );
                                    } else {
                                        erg = worker_lstat( fe->fullname, &statbuf );
                                    }
                                    if ( erg == 0 ) {
                                        // mknod
                                        if ( worker_mknod( newfullname, statbuf.st_mode, statbuf.st_rdev ) != 0 ) {
                                            if ( cowin != NULL ) cowin->stoptimer();
                                            myerrstr = strerror( errno );
                                            text = AGUIXUtils::formatStringToString( catalog.getLocale( 517 ), fe->fullname, newfullname, myerrstr );
                                            button = std::string( catalog.getLocale( 225 ) ) + "|" + catalog.getLocale( 8 );
                                            erg = request( copyorder->cowin, catalog.getLocale( 347 ), text.c_str(), button.c_str() );
                                            if ( erg == 0 ) {
                                                skip = true;
                                            } else {
                                                cancel = true;
                                            }
                                            if ( cowin != NULL ) cowin->conttimer();
                                        } else opok = true;
                                    } else {
                                        if ( cowin != NULL ) cowin->stoptimer();
                                        myerrstr = strerror( errno );
                                        text = AGUIXUtils::formatStringToString( catalog.getLocale( 518 ), fe->fullname, newfullname, myerrstr );
                                        button = std::string( catalog.getLocale( 225 ) ) + "|" + catalog.getLocale( 8 );
                                        erg = request( copyorder->cowin, catalog.getLocale( 347 ), text.c_str(), button.c_str() );
                                        if ( erg == 0 ) {
                                            skip = true;
                                        } else {
                                            cancel = true;
                                        }
                                        if ( cowin != NULL ) cowin->conttimer();
                                    }
                                } else if ( S_ISFIFO( modeCheck ) ) {
                                    // mkfifo
                                    if ( mkfifo( newfullname, modeCheck ) != 0 ) {
                                        if ( cowin != NULL ) cowin->stoptimer();
                                        myerrstr = strerror( errno );
                                        text = AGUIXUtils::formatStringToString( catalog.getLocale( 517 ), fe->fullname, newfullname, myerrstr );
                                        button = std::string( catalog.getLocale( 225 ) ) + "|" + catalog.getLocale( 8 );
                                        erg = request( copyorder->cowin, catalog.getLocale( 347 ), text.c_str(), button.c_str() );
                                        if ( erg == 0 ) {
                                            skip = true;
                                        } else {
                                            cancel = true;
                                        }
                                        if ( cowin != NULL ) cowin->conttimer();
                                    } else opok = true;
                                } else if ( S_ISLNK( modeCheck ) ) {
                                    // it's a link so create a new symlink which points to the same destination
                                    char *linkdest = fe->getDestination();
                                    if ( linkdest != NULL ) {
                                        if ( worker_symlink( linkdest, newfullname ) != 0 ) {
                                            if ( cowin != NULL ) cowin->stoptimer();
                                            myerrstr = strerror( errno );
                                            text = AGUIXUtils::formatStringToString( catalog.getLocale( 517 ), fe->fullname, newfullname, myerrstr );
                                            button = std::string( catalog.getLocale( 225 ) ) + "|" + catalog.getLocale( 8 );
                                            erg = request( copyorder->cowin, catalog.getLocale( 347 ), text.c_str(), button.c_str() );
                                            if ( erg == 0 ) {
                                                skip = true;
                                            } else {
                                                cancel = true;
                                            }
                                            if ( cowin != NULL ) cowin->conttimer();
                                        } else opok = true;
                                        _freesafe( linkdest );
                                    } else {
                                        if ( cowin != NULL ) cowin->stoptimer();
                                        myerrstr = strerror( errno );
                                        text = AGUIXUtils::formatStringToString( catalog.getLocale( 519 ), fe->fullname, newfullname, myerrstr );
                                        button = std::string( catalog.getLocale( 225 ) ) + "|" + catalog.getLocale( 8 );
                                        erg = request( copyorder->cowin, catalog.getLocale( 347 ), text.c_str(), button.c_str() );
                                        if ( erg == 0 ) {
                                            skip = true;
                                        } else {
                                            cancel = true;
                                        }
                                        if ( cowin != NULL ) cowin->conttimer();
                                    }
                                } else {
                                    if ( cowin != NULL ) cowin->stoptimer();
                                    myerrstr = strerror( errno );
                                    text = AGUIXUtils::formatStringToString( catalog.getLocale( 520 ), fe->fullname );
                                    button = std::string( catalog.getLocale( 225 ) ) + "|" + catalog.getLocale( 8 );
                                    erg = request( copyorder->cowin, catalog.getLocale( 347 ), text.c_str(), button.c_str() );
                                    if ( erg == 0 ) {
                                        skip = true;
                                    } else {
                                        cancel = true;
                                    }
                                    if ( cowin != NULL ) cowin->conttimer();
                                }
                            }
                            //TODO: muss hier nicht eher add_curbytes_copied aufgerufen werden?
                            //            if(cowin!=NULL) cowin->set_bytes_to_copy_curfile(fe->size);
                            if ( cowin != NULL ) cowin->add_curbytes_copied( fe->size() );
                        }
                    }
                } else {
                    text = AGUIXUtils::formatStringToString( catalog.getLocale( 113 ), fe->fullname );
                    button = catalog.getLocale( 11 );
          
                    request( copyorder->cowin, catalog.getLocale( 347 ), text.c_str(), button.c_str() );
                }
            } else skip=true;
      
            if ( opok == true ) {
                // try to change the mode because open modify it with umask
                // it doesn't matter if this fails 
                if(copyorder->preserve_attr==true) {
                    //TODO:currently deactivated
                    //     chown is necessary because link can have an owner
                    //     but chmod could be optional
                    // only apply new mode and time if it's not a link
                    // doing this would not do anything wrong
                    // but chown/chmod/utime would update the access/change time
                    // of the destination file although we don't have to touch it
                    //if ( ( fe->isLink == false ) || ( copyorder->follow_symlinks == true ) ) {
                    tryApplyOwnerPerm( newfullname,
                                       ( ( fe->isLink == true ) && ( fe->isCorrupt == false ) ) ? fe->duserid() : fe->userid(),
                                       ( ( fe->isLink == true ) && ( fe->isCorrupt == false ) ) ? fe->dgroupid() : fe->groupid(),
                                       ( ( fe->isLink == true ) && ( fe->isCorrupt == false ) ) ? fe->dmode() : fe->mode(),
                                       copyorder );
                    utb.actime = ( ( fe->isLink == true ) && ( fe->isCorrupt == false ) ) ? fe->dlastaccess() : fe->lastaccess();
                    utb.modtime = ( ( fe->isLink == true ) && ( fe->isCorrupt == false ) ) ? fe->dlastmod() : fe->lastmod();
                    worker_utime(newfullname,&utb);
                    //}
                }
            }
      
            _freesafe(newfullname);
        } else {
            switch(e1) {
                case NM_NEWNAME_CANCEL:
                    cancel=true;
                default:
                    skip=true;
                    break;
            }
        }
    }

    if(cowin!=NULL) {
        if ( skip == true ) {
            // the CopyOpWin need to substract bytesToCopy from global bytes sum
            // and already copied bytes from copyied bytes sum
            // if we called newfile() (which resets this counter)
            cowin->file_skipped( bytesToCopy, true );
        } else {
            cowin->file_finished();
        }
        if(cowin->redraw()!=0) cancel=true;
    }
    if(newname!=NULL) _freesafe(newname);
    if(cancel==true) return NM_COPY_CANCEL;
    else if(skip==true) return NM_COPY_SKIP;
    if(copyorder->move==true) {
        if((opok==true)&&(failedmove==true)) {
            // remember: failedmove is true when OS function rename failed
            //           in this case we have to copy the file and delete the source
            //           so add this entry only when copy is complete (opok) AND
            //           rename failed (failedmove)
            // put fe in list
            copy_deletefe_list->push_back( fe );
            return NM_COPY_NEED_DELETE;
        }
    }
    return NM_COPY_OK;
}

NormalMode::reg_copy_erg_t NormalMode::copyfile_reg( const char *sourcename,
						     const char *destname,
						     const char *destdirname,
						     const char *destbasename,
						     mode_t create_mode,
						     CopyOpWin *cowin,
						     loff_t bytesToCopy )
{
    int fdw, fdr;
    std::string text, button;
    ssize_t readbytes;
    reg_copy_erg_t return_value = REG_COPY_ERROR;
    int erg;
    const char *myerrstr;

    // firstly open source file
    fdr = worker_open( sourcename, O_RDONLY, 0 );
    if ( fdr == -1 ) {
        // can't open inputfile
        if ( cowin != NULL ) cowin->stoptimer();
    
        text = AGUIXUtils::formatStringToString( catalog.getLocale( 281 ), sourcename );
        button = std::string( catalog.getLocale( 225 ) ) + "|" + catalog.getLocale( 8 );
        erg = request( cowin, catalog.getLocale( 347 ), text.c_str(), button.c_str() );
    
        if ( erg == 1 ) return_value = REG_COPY_CANCEL;
        else return_value = REG_COPY_SKIP;
    
        if ( cowin != NULL ) cowin->conttimer();
        return return_value;
    }

    if ( cowin != NULL ) {
        cowin->set_bytes_to_copy_curfile( bytesToCopy );
    }
  
    // secondly try to open destination file
    // two runs as I first try to open an existing file and only remove it if it doesn't
    // work. This is because we can overwrite a file but not create it
    for ( int i = 0; i < 2; i++ ) {
        // fe is a regular file or a working symlink to a regular file so we can use dmode() without checking
        // for corrupt link
        fdw = worker_open( destname, O_CREAT | O_TRUNC | O_WRONLY, create_mode );

        if ( fdw != -1 ) break;

        // open failed
        if ( cowin != NULL ) cowin->stoptimer();
        if ( errno == ENOSPC ) {
            text = AGUIXUtils::formatStringToString( catalog.getLocale( 545 ), destbasename, destdirname );
            button = std::string( catalog.getLocale( 225 ) ) + "|" + catalog.getLocale( 41 ) + "|" + catalog.getLocale( 8 );
            erg = request( cowin, catalog.getLocale( 347 ), text.c_str(), button.c_str() );

            if ( erg == 2 ) {
                return_value = REG_COPY_CANCEL;
            } else if ( erg == 1 ) {
                // just try again
                i--;
            } else return_value = REG_COPY_SKIP;
        } else if ( errno == EACCES ) {
            if ( Datei::lfileExistsExt( destname ) != Datei::D_FE_NOFILE ) {
                /// error
                // give request
                // textstr="Can't overwrite the file|Shall I try to delete it?";
                // buttonstr="Delete file|Skip|Cancel";
	
                text = AGUIXUtils::formatStringToString( catalog.getLocale( 282 ), destname );
                button = std::string( catalog.getLocale( 277 ) ) +
                    "|" +
                    catalog.getLocale( 225 ) + 
                    "|" +
                    catalog.getLocale( 8 );
                erg = request( cowin, catalog.getLocale( 123 ), text.c_str(), button.c_str() );

                if ( erg == 1 ) return_value = REG_COPY_SKIP;
                else if ( erg == 2 ) return_value = REG_COPY_CANCEL;
                else {
                    if ( worker_remove( destname ) != 0 ) {
                        // textstr="Failed to remove this file|I will skip this file!";
                        // buttonstr="Ok|Cancel";
	    
                        text = AGUIXUtils::formatStringToString( catalog.getLocale( 138 ), destname );
                        button = std::string( catalog.getLocale( 11 ) ) + "|" + catalog.getLocale( 8 );
                        erg = request( cowin, catalog.getLocale( 347 ), text.c_str(), button.c_str() );

                        if ( erg == 1 ) return_value = REG_COPY_CANCEL;
                        else return_value = REG_COPY_SKIP;
                    }
                }
            } else {
                // "Can't open dest file"
                text = AGUIXUtils::formatStringToString( catalog.getLocale( 198 ), destname );
                button = std::string( catalog.getLocale( 225 ) ) + "|" + catalog.getLocale( 8 );
                erg = request( cowin, catalog.getLocale( 347 ), text.c_str(), button.c_str() );

                if ( erg == 1 ) return_value = REG_COPY_CANCEL;
                else return_value = REG_COPY_SKIP;
            }
        } else {
            text = AGUIXUtils::formatStringToString( catalog.getLocale( 198 ), destname );
            button = std::string( catalog.getLocale( 225 ) ) + "|" + catalog.getLocale( 8 );
            erg = request( cowin, catalog.getLocale( 347 ), text.c_str(), button.c_str() );

            if ( erg == 1 ) return_value = REG_COPY_CANCEL;
            else return_value = REG_COPY_SKIP;
        }
        if ( cowin != NULL ) cowin->conttimer();

        if ( ( return_value == REG_COPY_SKIP ) || ( return_value == REG_COPY_CANCEL ) ) break;
    }

    if ( fdw == -1 ) {
        worker_close( fdr );
        return return_value;
    }

    // now copy from fdr to fdw
    do {
        readbytes = worker_read( fdr, buf, BUFSIZE );
        if ( readbytes < 0 ) {
            // error while reading
      
            if ( cowin != NULL ) cowin->stoptimer();
      
            myerrstr = strerror( errno );
            text = AGUIXUtils::formatStringToString( catalog.getLocale( 525 ), sourcename, myerrstr );
            button = std::string( catalog.getLocale( 225 ) ) + "|" + catalog.getLocale( 8 );
            erg = request( cowin, catalog.getLocale( 347 ), text.c_str(), button.c_str() );

            if ( cowin != NULL ) cowin->conttimer();

            if ( erg == 0 ) return_value = REG_COPY_SKIP;
            else return_value = REG_COPY_CANCEL;
            break;
        } else if ( readbytes == 0 ) {
            // end of file
            break;
        }

        ssize_t bytes_written = 0;

        for ( ; bytes_written < readbytes; ) {
            const ssize_t bytes_left_to_write = readbytes - bytes_written;
            ssize_t writebytes;

            writebytes = worker_write( fdw, ((const char *)buf) + bytes_written, bytes_left_to_write );

            if ( writebytes > 0 ) {
                bytes_written += writebytes;
            }

            if ( writebytes > 0 && cowin != NULL ) {
                cowin->add_curbytes_copied( writebytes );

                //TODO if I use dynamic buffer I should skip some updates
                if ( cowin->redraw() != 0 )
                    return_value = REG_COPY_CANCEL;
            }

            if ( writebytes != bytes_left_to_write ) {
                // something went wrong
                // let the user choose to cancel or skip this file
                // Ask to delete the incomplete destfile
                // Attention: This method also moves files so be sure to
                //   NOT delete the source!
                if( ( ( writebytes >= 0 ) && ( writebytes < bytes_left_to_write ) ) ||
                    ( errno == ENOSPC ) ) {
                    // ENOSPC isn't always reported so assume ENOSPC
                    text = AGUIXUtils::formatStringToString( catalog.getLocale( 545 ), destbasename, destdirname );
                    button = std::string( catalog.getLocale( 225 ) ) + "|" + catalog.getLocale( 41 ) + "|" + catalog.getLocale( 8 );
                    erg = request( cowin, catalog.getLocale( 347 ), text.c_str(), button.c_str() );
	
                    if ( erg == 0 ) {
                        return_value = REG_COPY_SKIP;
                    } else if ( erg == 1 ) {
                        // try again
                    } else {
                        return_value = REG_COPY_CANCEL;
                    }
                } else {
                    if ( writebytes < 0 ) {
                        myerrstr = strerror( errno );
                        text = AGUIXUtils::formatStringToString( catalog.getLocale( 526 ), destname, myerrstr );
                    } else {
                        text = AGUIXUtils::formatStringToString( catalog.getLocale( 359 ), destname );
                    }
                    button = std::string( catalog.getLocale( 225 ) ) + "|" + catalog.getLocale( 8 );
                    erg = request( cowin, catalog.getLocale( 347 ), text.c_str(), button.c_str() );
	
                    if ( erg == 0 ) return_value = REG_COPY_SKIP;
                    else return_value = REG_COPY_CANCEL;
                }
            }

            if ( return_value == REG_COPY_CANCEL || return_value == REG_COPY_SKIP ) break;
        }
    } while ( ( return_value != REG_COPY_CANCEL ) && ( return_value != REG_COPY_SKIP ) );

    cowin->update_file_status();
    if ( cowin->redraw() != 0 ) return_value = REG_COPY_CANCEL;

    worker_close( fdr );
    if ( worker_close( fdw ) != 0 ) {
        // oops, close error
        myerrstr = strerror( errno );
        text = AGUIXUtils::formatStringToString( catalog.getLocale( 526 ), destname, myerrstr );
        button = std::string( catalog.getLocale( 225 ) ) + "|" + catalog.getLocale( 8 );
        erg = request( cowin, catalog.getLocale( 347 ), text.c_str(), button.c_str() );
    
        if ( erg == 0 ) return_value = REG_COPY_SKIP;
        else return_value = REG_COPY_CANCEL;
    }
  
    if ( ( return_value == REG_COPY_CANCEL ) || ( return_value == REG_COPY_SKIP ) ) {
        // error while copying!
        // ask to remove the incomplete destination file
    
        if ( cowin != NULL ) cowin->stoptimer();
    
        text = AGUIXUtils::formatStringToString( catalog.getLocale( 393 ), destname );
        button = std::string( catalog.getLocale( 394 ) ) + "|" + catalog.getLocale( 395 );
        erg = request( cowin, catalog.getLocale( 123 ), text.c_str(), button.c_str() );
        if ( erg == 1 ) {
            if ( worker_unlink( destname ) != 0 ) {
                // textstr="Failed to remove the destination file|This file is probably incomplete!";
                text = AGUIXUtils::formatStringToString( catalog.getLocale( 139 ), destname );
                button = catalog.getLocale( 11 );
                erg = request( cowin, catalog.getLocale( 347 ), text.c_str(), button.c_str() );
            }
        }
        if ( cowin != NULL ) cowin->conttimer();
    } else {
        // copy complete
        return_value = REG_COPY_OK;
    }
    return return_value;
}
