/*
 *  This file is part of X-File Manager XFM
 *  ----------------------------------------------------------------------
  FmFwCb.c

  (c) Simon Marlow 1990-92
  (c) Albert Graef 1994

  modified 7-1997 by strauman@sun6hft.ee.tu-berlin.de to add
  different enhancements (see README-1.4).

  modified 2004,2005,2006,2007 by Bernhard R. Link (see Changelog)

  Callback routines for widgets in a file window
 *  ----------------------------------------------------------------------
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
#include <xfmconfig.h>

#include <assert.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Xaw3d/Toggle.h>

#include "FileList.h"
#include "TextFileList.h"

#include <string.h>
#include <stdio.h>

#include "global.h"

#include "Fm.h"

#ifdef ENHANCE_SELECTION
#include "FmSelection.h"
#endif
/*-----------------------------------------------------------------------------
  This function is also used in FmFwActions when a directory is pulled onto
  the root window. In this case, w will be zero and we use this to popup
  the new window by the cursor.
-----------------------------------------------------------------------------*/
void fileOpenCb(Widget w, FileWindowRec *fw, UNUSED(XtPointer call_data))
{
  int i;

  assert( fw != NULL );

  for (i=0; i<fw->n_files; i++) {
    if (fw->files[i]->selected && S_ISDIR(fw->files[i]->stats.st_mode)) {
      char *pathname;

      pathname = dirConcat(fw->directory, fw->files[i]->name);
      newFileWindow(pathname,resources.default_display_type,
		    w ? False : True);
      XTFREE(pathname);
    }
  }
}

#ifdef ENHANCE_MENU
/*---------------------------------------------------------------------------*/


void fileCloneCb(UNUSED(Widget w), FileWindowRec *fw, UNUSED(XtPointer call_data))
{

 newFileWindow(fw->directory,fw->display_type,True);
 /* new file window was created at the beginning of the list */
 file_windows[0].showInode=fw->showInode;
 file_windows[0].showType=fw->showType;
 file_windows[0].showOwner=fw->showOwner;
 file_windows[0].showPermissions=fw->showPermissions;
 file_windows[0].showGroup=fw->showGroup;
 file_windows[0].showLinks=fw->showLinks;
 file_windows[0].showLength=fw->showLength;
 file_windows[0].showDate=fw->showDate;
 
 if (fw->display_type==Text) updateTxtOpts(&file_windows[0]);
}
#endif

/*---------------------------------------------------------------------------*/

void fileEditCb(UNUSED(Widget w), FileWindowRec *fw, UNUSED(XtPointer call_data))
{
  int i;

  assert( fw != NULL );

  for (i=0; i<fw->n_files; i++)
    if (fw->files[i]->selected && !S_ISDIR(fw->files[i]->stats.st_mode))
	doEdit(fw->directory,fw->files[i]->name);
}

/*---------------------------------------------------------------------------*/

void fileViewCb(UNUSED(Widget w), FileWindowRec *fw, UNUSED(XtPointer call_data))
{
  int i;

  assert( fw != NULL );

  for (i=0; i<fw->n_files; i++)
    if (fw->files[i]->selected && !S_ISDIR(fw->files[i]->stats.st_mode))
	doView(fw->directory,fw->files[i]->name);
}

/*---------------------------------------------------------------------------*/

void xtermCb(UNUSED(Widget w), FileWindowRec *fw, UNUSED(XtPointer call_data))
{
  assert( fw != NULL );
  doXterm(fw->directory);
}

/*---------------------------------------------------------------------------*/

void fileTreeCb(UNUSED(Widget w), FileWindowRec *fw, UNUSED(XtPointer call_data))
{
#ifdef ENHANCE_SCROLL
  Boolean keep_position=(fw->display_type==Tree);
#endif
  fw->display_type = Tree;
#ifdef ENHANCE_SCROLL
  updateFileDisplay(fw,keep_position);
#else
  updateFileDisplay(fw);
#endif
}

/*---------------------------------------------------------------------------*/

void fileIconsCb(UNUSED(Widget w), FileWindowRec *fw, UNUSED(XtPointer call_data))
{
  DisplayType t = fw->display_type;

#ifdef ENHANCE_SCROLL
  Boolean keep_position= (t==Icons);
#endif

  fw->display_type = Icons;
  if (t == Text)
    reDisplayFileWindow(fw);
  else
#ifdef ENHANCE_SCROLL
    updateFileDisplay(fw,keep_position);
#else
    updateFileDisplay(fw);
#endif
}

/*---------------------------------------------------------------------------*/

void fileTextCb(UNUSED(Widget w), FileWindowRec *fw, UNUSED(XtPointer call_data))
{
  DisplayType t = fw->display_type;

#ifdef ENHANCE_SCROLL
  Boolean keep_position= (t==Text);
#endif

  fw->display_type = Text;
  if (t == Icons)
    reDisplayFileWindow(fw);
  else
#ifdef ENHANCE_SCROLL
    updateFileDisplay(fw,keep_position);
#else
    updateFileDisplay(fw);
#endif
}

/*---------------------------------------------------------------------------*/

void fileSelectAllCb(UNUSED(Widget w), FileWindowRec *fw, UNUSED(XtPointer call_data))
{
  int i;
  Pixel pix;
  int onewid=(fw->icon_box && XtIsSubclass(fw->icon_box,fileListWidgetClass));
  
  fw->n_selections = 0;
  fw->n_bytes_selected = 0;
  for (i=0; i < fw->n_files; i++) {
    if ( (onewid || fw->files[i]->icon.toggle) &&
	strcmp(fw->files[i]->name, ".") &&
	strcmp(fw->files[i]->name, "..")) {
      fw->files[i]->selected = True;
      fw->n_selections++;
      fw->n_bytes_selected += fw->files[i]->stats.st_size;
    }
    else
      fw->files[i]->selected = False;
    if ( !onewid && fw->files[i]->icon.toggle) {
      XtVaGetValues(fw->files[i]->icon.toggle,
		    fw->files[i]->selected?XtNforeground:XtNbackground, &pix,
		    NULL);
      XtVaSetValues(fw->files[i]->icon.toggle, XtNborder, (XtArgVal) pix,
		    NULL);
    }
  }
  updateStatus(fw);
  /* normally the mayority of files is not selected, so we just force
   * a redraw of everything.
   */

  if (onewid)
    XClearArea(XtDisplay(fw->icon_box),XtWindow(fw->icon_box),0,0,0,0,True);
#ifdef ENHANCE_SELECTION
  FmOwnSelection(fw,CurrentTime);
#endif
}

/*---------------------------------------------------------------------------*/

void fileDeselectCb(UNUSED(Widget w), FileWindowRec *fw, UNUSED(XtPointer call_data))
{
  int i;
  Pixel pix;
  int onewid=(fw->icon_box && XtIsSubclass(fw->icon_box,fileListWidgetClass));
  
  for (i=0; i < fw->n_files; i++)
    if (fw->files[i]->selected && (onewid || fw->files[i]->icon.toggle)) {
      fw->files[i]->selected = False;
      if (!onewid) {
        XtVaGetValues(fw->files[i]->icon.toggle, XtNbackground, &pix, NULL);
        XtVaSetValues(fw->files[i]->icon.toggle, XtNborder, (XtArgVal) pix,
		    NULL);
      } else FileListRefreshItem((FileListWidget)fw->icon_box,i);
    }
  fw->n_selections = 0;
  fw->n_bytes_selected = 0;
  updateStatus(fw);
#ifdef ENHANCE_SELECTION
  FmDisownSelection(fw);
#endif
}

#ifdef ENHANCE_SELECTION
/*---------------------------------------------------------------------------*/

void selectionOwnCb(UNUSED(Widget w), FileWindowRec *fw, UNUSED(XtPointer call_data))
{
 FmOwnSelection(fw,CurrentTime);
}
#endif

/*---------------------------------------------------------------------------*/

void fileSortNameCb(UNUSED(Widget w), FileWindowRec *fw, UNUSED(XtPointer call_data))
{
  fw->sort_type = SortByName;
  reSortFileDisplay(fw);
}

/*---------------------------------------------------------------------------*/

void fileSortSizeCb(UNUSED(Widget w), FileWindowRec *fw, UNUSED(XtPointer call_data))
{
  fw->sort_type = SortBySize;
  reSortFileDisplay(fw);
}

/*---------------------------------------------------------------------------*/

void fileSortMTimeCb(UNUSED(Widget w), FileWindowRec *fw, UNUSED(XtPointer call_data))
{
  fw->sort_type = SortByMTime;
  reSortFileDisplay(fw);
}

/*---------------------------------------------------------------------------*/

void fileCloseCb(UNUSED(Widget w), FileWindowRec *fw, UNUSED(XtPointer call_data))
{
  if (fw == file_windows && fw->next == NULL && !resources.appmgr) {
    if (!resources.confirm_quit || fwConfirm(fw, "", "Exit file manager?", "", NULL))
      quit(0);
    else
      return;
  }

  XtDestroyWidget(fw->shell);
}

/*---------------------------------------------------------------------------*/

void fileHomeCb(UNUSED(Widget w), FileWindowRec *fw, UNUSED(XtPointer call_data))
{
#ifdef ENHANCE_SCROLL
  Boolean keep_position=True;
#endif
  freeze = True;
  if (chdir(user.home))
    sysError("Can't open folder:");
  else if (!getwd(fw->directory))
    sysError("System error:");
#ifdef ENHANCE_SCROLL
  else keep_position=(!strcmp(fw->directory,user.home));

  updateFileDisplay(fw,keep_position);
#else
  updateFileDisplay(fw);
#endif
  freeze = False;
}

/*---------------------------------------------------------------------------*/

void fileUpCb(UNUSED(Widget w), FileWindowRec *fw, UNUSED(XtPointer call_data))
{
#ifdef ENHANCE_SCROLL
  Boolean keep_position=True;
#endif
  freeze = True;
  if (chdir(fw->directory) || chdir(".."))
    sysError("Can't open folder:");
  else if (!getwd(fw->directory))
    sysError("System error:");
#ifdef ENHANCE_SCROLL
  else keep_position=False;

  updateFileDisplay(fw,keep_position);
#else
  updateFileDisplay(fw);
#endif
  freeze = False;
}

/*---------------------------------------------------------------------------*/

void mainArrowCb(Widget w, FileWindowRec *fw, UNUSED(XtPointer call_data))
{
  int i;

  freeze = True;
  for (i=0; i<fw->n_files; i++)
    if (fw->files[i]->icon.arrow == w) {
      if (chdir(fw->directory) || chdir(fw->files[i]->name))
	sysError("Can't open folder:");
      else if (!getwd(fw->directory))
	sysError("System error:");
      break;
    }
#ifdef ENHANCE_SCROLL
  updateFileDisplay(fw,False);
#else
  updateFileDisplay(fw);
#endif
  freeze = False;
}


/*---------------------------------------------------------------------------*/

void fileShowDirsCb(UNUSED(Widget w), FileWindowRec *fw, UNUSED(XtPointer call_data))
{
  fw->show_dirs = !fw->show_dirs;
#ifdef ENHANCE_SCROLL
  updateFileDisplay(fw,True);
#else
  updateFileDisplay(fw);
#endif
}

/*---------------------------------------------------------------------------*/

void fileDirsFirstCb(UNUSED(Widget w), FileWindowRec *fw, UNUSED(XtPointer call_data))
{
  fw->dirs_first = !fw->dirs_first;
  reSortFileDisplay(fw);
}

/*---------------------------------------------------------------------------*/

void fileShowHiddenCb(UNUSED(Widget w), FileWindowRec *fw, UNUSED(XtPointer call_data))
{
  fw->show_hidden = !fw->show_hidden;
#ifdef ENHANCE_SCROLL
  updateFileDisplay(fw,True);
#else
  updateFileDisplay(fw);
#endif
}

/*---------------------------------------------------------------------------*/

void timeoutCb(UNUSED(XtPointer data), UNUSED(XtIntervalId *id))
{
  if (!freeze) intUpdate();
  XtAppAddTimeOut(app_context, resources.update_interval, timeoutCb, NULL);
}

/*---------------------------------------------------------------------------*/

void showTxtOptsCb(Widget w, FileWindowRec *fw, UNUSED(XtPointer cad))
{
 int i;
 Boolean state,*fw_state=0;
 Widget tw=fw->icon_box;
 String resource=0;

 i=0;
 while(i<8 && w!=fw->option_items[i])i++;
 if (i>=8) return; /* not found */

 switch(i) {
	case 0: resource=XtNshowInode; fw_state=&fw->showInode; break;
	case 1: resource=XtNshowType;  fw_state=&fw->showType; break;
	case 2: resource=XtNshowPermissions; fw_state=&fw->showPermissions; break;
	case 3: resource=XtNshowLinks; fw_state=&fw->showLinks; break;
	case 4: resource=XtNshowOwner; fw_state=&fw->showOwner; break;
	case 5: resource=XtNshowGroup; fw_state=&fw->showGroup; break;
	case 6: resource=XtNshowLength;fw_state=&fw->showLength; break;
	case 7: resource=XtNshowDate;  fw_state=&fw->showDate; break;
 }

 XtVaGetValues(tw,resource,&state,NULL);
 state=!state;
 if (state) tick(w);
 else	    noTick(w);
 XtVaSetValues(tw,resource,state,NULL);

 /* update the state in the FileWindowRec as well */
 *fw_state=state;
}
/*---------------------------------------------------------------------------*/

void fileQuitCb(UNUSED(Widget w), FileWindowRec *fw,
		UNUSED(XtPointer call_data))
{
  if (resources.confirm_quit && !fwConfirm(fw, "", "Exit file manager?", "", NULL))
    return;

  quit(0);
}

/*---------------------------------------------------------------------------*/

void fileAboutCb(UNUSED(Widget w), FileWindowRec *fw,
             UNUSED(XtPointer call_data))
{
  aboutPopup(fw->shell);
}
