/*******************************************************************************
 * Copyright (C) 2004-2007 Intel Corp. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *   - Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 * 
 *   - Redistributions in binary form must reproduce the above copyright notice,
 *     this list of conditions and the following disclaimer in the documentation
 *     and/or other materials provided with the distribution.
 * 
 *   - Neither the name of Intel Corp. nor the names of its
 *     contributors may be used to endorse or promote products derived from this
 *     software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL Intel Corp. OR THE CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *******************************************************************************/

//----------------------------------------------------------------------------
//
//  File:       PTHICommand.cpp
//
//----------------------------------------------------------------------------

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <stdio.h>
#include <stdlib.h>
#include "ZTCLocalAgent.h"
#include "HECILinux.h"
#include "resource.h"
#include "version.h"
#include "PTHICommand.h"
#include "Utils.h"
#include "HECI_if.h"

// Define GUID used to connect to the PTHI client (via the HECI device)
// {12F80028-B4B7-4B2D-ACA8-46E0FF65814C}
DEFINE_GUID(HECI_PTHI_GUID,0x12F80028,0xB4B7,0x4b2d,0xAC,0xA8,0x46,0xE0,0xFF,0x65,0x81,0x4c);

PTHICommand::PTHICommand( bool verbose): PTHIClient(HECI_PTHI_GUID,verbose)
{
}

PTHICommand::~PTHICommand(void)
{
}

/*
* Confirms the correctness of the response message header
* and the response message size
* Arguments:
*	command	- appropriate Host interface command
*	response_header	- pointer to the response message header
*	response_size	- value that holds the actual size of the
*	                     response message
*	expected_size	- value that holds the expected size of the
*	                     response message
* Return values:
*	AMT_STATUS_SUCCESS - on success
*	PTSDK_STATUS_INTERNAL_ERROR - on failure
*/
UINT32 PTHICommand::_verifyResponseHeader(const UINT32 command,
										  const PTHI_MESSAGE_HEADER *response_header,
										  UINT32 response_size)
{
	UINT32 status = AMT_LOCAL_AGENT_STATUS_SUCCESS;

	if ( response_size != (response_header->Length + sizeof(PTHI_MESSAGE_HEADER)))
	{
		status = AMT_STATUS_INTERNAL_ERROR;
	}
	else if(	response_header->Command.cmd.val != command )
	{
		status = AMT_STATUS_INTERNAL_ERROR;
	}
	else if(response_header->Reserved != 0 )
	{
		status = AMT_STATUS_INTERNAL_ERROR;
	}

	return status;
}

/*
* Confirms the correctness of the GetCodeVersions response message
* Arguments:
*	response - pointer to the response message
* Return values:
*	AMT_STATUS_SUCCESS - on success
*	PTSDK_STATUS_INTERNAL_ERROR - on failure
*/
UINT32 PTHICommand::_verifyCodeVersions(const CFG_GET_CODE_VERSIONS_RESPONSE *response)
{
	UINT32 status = AMT_STATUS_SUCCESS;
	UINT32 codeVerLen;
	UINT32 ptVerTypeCount;
	UINT32 len = 0;
	UINT32 i;

	do
	{
		codeVerLen = response->Header.Length - sizeof(AMT_STATUS);
		ptVerTypeCount = codeVerLen - sizeof(response->CodeVersions.BiosVersion)- sizeof(response->CodeVersions.VersionsCount);
		if ( response->CodeVersions.VersionsCount != (ptVerTypeCount/sizeof( AMT_VERSION_TYPE )))
		{
			status = AMT_STATUS_INTERNAL_ERROR;
			break;
		}

		for ( i = 0; i < (response->CodeVersions.VersionsCount); i ++ )
		{
			len = response->CodeVersions.Versions[i].Description.Length;

			if(len > UNICODE_STRING_LEN)
			{
				status = AMT_STATUS_INTERNAL_ERROR;
				break;
			}

			len = response->CodeVersions.Versions[i].Version.Length;
			if ( response->CodeVersions.Versions[i].Version.String[len] != '\0' ||
				(len != strlen((PCHAR)(response->CodeVersions.Versions[i].Version.String))) )
			{
				status = AMT_STATUS_INTERNAL_ERROR;
				break;
			}
		}
	}while ( 0 );

	return status;
}
/*
* GetVersions response message PTHI command
* Arguments:
*	response - pointer to the CODE_VERSIONS struct
* Return values:
*	AMT_LOCAL_AGENT_STATUS_SUCCESS - on success
*	AMT_STATUS_INTERNAL_ERROR - on failure
*/
UINT32 PTHICommand::GetCodeVersions(CODE_VERSIONS *CodeVersions)
{
	AMT_STATUS status;
	UCHAR command[sizeof(GET_CODE_VERSION_HEADER)];
	UINT8 *readBuffer;
	UINT32 command_size = sizeof(GET_CODE_VERSION_HEADER);
	UINT32 inBuffSize;
	UINT32 outBuffSize=0;
	CFG_GET_CODE_VERSIONS_RESPONSE *tmp_response;
	PTHI_MESSAGE_HEADER *resp_header;
	int	   bytesWritten = 0;

	if ( CodeVersions == NULL )
	{
		return PTSDK_STATUS_INVALID_PARAM;
	}

	memcpy(command, &GET_CODE_VERSION_HEADER, sizeof(GET_CODE_VERSION_HEADER));

	bytesWritten = PTHIClient.SendMessage(command,command_size);
	if((UINT32) bytesWritten != command_size)
	{
		return AMT_STATUS_INTERNAL_ERROR;
	}
	inBuffSize = PTHIClient.GetBufferSize();
	readBuffer = (UINT8*) malloc (sizeof(UINT8) * inBuffSize);
	if(NULL == readBuffer)
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	memset(readBuffer,0,inBuffSize);

	outBuffSize = PTHIClient.ReceiveMessage(readBuffer,inBuffSize);
	do
	{
		if( 0 == outBuffSize)
		{
			status = AMT_STATUS_INTERNAL_ERROR;
			break;
		}

		tmp_response = (CFG_GET_CODE_VERSIONS_RESPONSE*)readBuffer;

		status = tmp_response->Status;
		if( status != AMT_STATUS_SUCCESS )
		{
			break;
		}

		resp_header = &(tmp_response->Header);

		status = _verifyResponseHeader(CODE_VERSIONS_RESPONSE,
			resp_header,outBuffSize);
		if ( status != AMT_LOCAL_AGENT_STATUS_SUCCESS )
		{
			break;
		}

		status = _verifyCodeVersions(tmp_response);
		if ( status != AMT_LOCAL_AGENT_STATUS_SUCCESS )
		{
			break;
		}

		memcpy( CodeVersions,&(tmp_response->CodeVersions),sizeof(CODE_VERSIONS)) ;
	}while(0);

	if ( readBuffer != NULL )
	{
		free(readBuffer);
		readBuffer = NULL;
	}
	return status;
}
/*
* DisplayCodeVersions print code version
* Arguments:
*	response - pointer to the CODE_VERSIONS struct
* Return values:
*	NONE
*/
VOID PTHICommand::DisplayCodeVersions(const CODE_VERSIONS *versions)
{
	UINT32 i,j;
	UINT16 len;

	// BIOS version is free format null terminated string,
	// max length of BIOS version string is 64 byte
	DisplayMessage(BIOS_VERSION);
	fprintf(stdout,"\t\t %s\n\n",versions->BiosVersion);
	DisplayMessage(AMT_CODE_VERSION);
	fprintf(stdout,"\n");
	for ( i = 0; i < versions->VersionsCount; i ++ )
	{
		fprintf(stdout,"\t");
		len = versions->Versions[i].Description.Length;
		for ( j = 0; j < len; j ++ )
		{
			fprintf(stdout,"%c",versions->Versions[i].Description.String[j]);
		}
		fprintf(stdout,":");
		for ( j = 0 ; j < (UINT32)(24 - len); j ++ )
		{
			fprintf(stdout," ");
		}

		fprintf(stdout,"%s\n", versions->Versions[i].Version.String);
	}
	fprintf(stdout,"\n");
}
/*
* Calls to GetProvisioningMode Host interface command
* Arguments:
*	legacy - pointer to the pre-allocated structure
*       which will hold the result
* Return values:
*	AMT_STATUS_SUCCESS - on success
*	appropriate error value defined in StatusCodeDefinitions.h - on failure
*/
AMT_STATUS PTHICommand::GetProvisioningMode(AMT_BOOLEAN *legacy, CFG_PROVISIONING_MODE *mode)
{
	UCHAR command[sizeof(GET_PROVISIONING_MODE_HEADER)];
	UINT8 *readBuffer;
	UINT32 inBuffSize =0;
	UINT32 outBuffSize=0;
	int	   bytesWritten = 0;
	ULONG command_size = sizeof(GET_PROVISIONING_MODE_HEADER);
	AMT_STATUS status;

	if ( legacy == NULL )
	{
		return PTSDK_STATUS_INVALID_PARAM;
	}

	memcpy(command, &GET_PROVISIONING_MODE_HEADER, sizeof(GET_PROVISIONING_MODE_HEADER));

	bytesWritten = PTHIClient.SendMessage(command,command_size);
	if((UINT32)bytesWritten != command_size)
	{
		return AMT_STATUS_INTERNAL_ERROR;
	}
	inBuffSize = PTHIClient.GetBufferSize();
	readBuffer = (UINT8*) malloc (sizeof(UINT8) * inBuffSize);
	if(NULL == readBuffer)
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	memset(readBuffer,0,inBuffSize);

	outBuffSize = PTHIClient.ReceiveMessage(readBuffer,inBuffSize);

	do
	{
		if( 0 == outBuffSize)
		{
			status = AMT_STATUS_INTERNAL_ERROR;
			break;
		}
		CFG_GET_PROVISIONING_MODE_RESPONSE *tmp_response =
			(CFG_GET_PROVISIONING_MODE_RESPONSE*)readBuffer;

		status = tmp_response->Status;

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}

		PTHI_MESSAGE_HEADER *resp_header =
			&(tmp_response->Header);

		status = _verifyResponseHeader(PROVISIONING_MODE_RESPONSE,resp_header,outBuffSize);

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}

		status = _verifyProvisioningMode(tmp_response);

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}

		*legacy = tmp_response->LegacyMode;
		*mode = tmp_response->ProvisioningMode;

	}while ( 0 );

	if ( readBuffer != NULL )
	{
		free(readBuffer);
		readBuffer = NULL;
	}
	return status;
}

/*
* Confirms the correctness of the GetProvisioningMode response message
* Arguments:
*	response - pointer to the response message
* Return values:
*	AMT_STATUS_SUCCESS - on success
*	PTSDK_STATUS_INTERNAL_ERROR - on failure
*/
AMT_STATUS PTHICommand::_verifyProvisioningMode(const CFG_GET_PROVISIONING_MODE_RESPONSE *response)
{
	ULONG ByteCount;

	ByteCount = response->Header.Length;

	if (ByteCount !=
		(sizeof(CFG_GET_PROVISIONING_MODE_RESPONSE) - sizeof(PTHI_MESSAGE_HEADER)))
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	return AMT_STATUS_SUCCESS;
}
/*
* DisplayAMTMode print legacy mode
* Arguments:
*	*legacy - pointer to the AMT_BOOLEAN enum
* Return values:
*	NONE
*/
void PTHICommand::DisplayAMTMode(const AMT_BOOLEAN *legacy)
{
	DisplayMessage(AMT_MODE);
	fprintf(stdout,"\t\t\t");
	if (*legacy)
	{
		DisplayMessage(AMT_MODE_1);
		fprintf(stdout,"\n");
	}
	else
	{
		DisplayMessage(AMT_MODE_2);
		fprintf(stdout,"\n");
	}
}
/*
* Calls to GetProvisioningState Host interface command
* Arguments:
*	state - pointer to the pre-allocated structure
*	   which will hold the result
* Return values:
*	PT_STATUS_SUCCESS - on success
*	appropriate error value defined in StatusCodeDefinitions.h - on failure
*/
AMT_STATUS PTHICommand::GetProvisioningState(AMT_PROVISIONING_STATE *state)
{
	AMT_STATUS status;
	UCHAR command[sizeof(GET_PROVISIONING_STATE_HEADER)];
	UINT8 *readBuffer;
	UINT32 command_size = sizeof(GET_PROVISIONING_STATE_HEADER);
	UINT32 inBuffSize;
	UINT32 outBuffSize=0;
	int	   bytesWritten = 0;

	if ( state == NULL )
	{
		return PTSDK_STATUS_INVALID_PARAM;
	}

	memcpy(command, &GET_PROVISIONING_STATE_HEADER, sizeof(GET_PROVISIONING_STATE_HEADER));

	bytesWritten = PTHIClient.SendMessage(command,command_size);
	if((UINT32)bytesWritten != command_size)
	{
		return AMT_STATUS_INTERNAL_ERROR;
	}
	inBuffSize = PTHIClient.GetBufferSize();
	readBuffer = (UINT8*) malloc (sizeof(UINT8) * inBuffSize);
	if(NULL == readBuffer)
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	memset(readBuffer,0,inBuffSize);

	outBuffSize = PTHIClient.ReceiveMessage(readBuffer,inBuffSize);
	do
	{
		if( 0 == outBuffSize)
		{
			status = AMT_STATUS_INTERNAL_ERROR;
			break;
		}

		CFG_GET_PROVISIONING_STATE_RESPONSE *tmp_response =
			(CFG_GET_PROVISIONING_STATE_RESPONSE*)readBuffer;

		status = tmp_response->Status;

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}

		PTHI_MESSAGE_HEADER *resp_header =
			&(tmp_response->Header);

		status = _verifyResponseHeader(PROVISIONING_STATE_RESPONSE,
			resp_header,outBuffSize);

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}

		status = _verifyProvisioningState(tmp_response);

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}

		*state = tmp_response->ProvisioningState;

	}while ( 0 );

	if ( readBuffer != NULL )
	{
		free(readBuffer);
	}
	return status;
}
/*
* Confirms the correctness of the GetProvisioningState response message
* Arguments:
*	response - pointer to the response message
* Return values:
*	PT_STATUS_SUCCESS - on success
*	PTSDK_STATUS_INTERNAL_ERROR - on failure
*/
AMT_STATUS PTHICommand::_verifyProvisioningState(const CFG_GET_PROVISIONING_STATE_RESPONSE *response)
{
	ULONG ByteCount;

	ByteCount = response->Header.Length;

	if (ByteCount !=
		(sizeof(CFG_GET_PROVISIONING_STATE_RESPONSE) - sizeof(PTHI_MESSAGE_HEADER)))
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	return AMT_STATUS_SUCCESS;
}
/*
* DisplayProvisioningState print provisioning state
* Arguments:
*	*state - pointer to the AMT_PROVISIONING_STATE enum
* Return values:
*	NONE
*/
VOID PTHICommand::DisplayProvisioningState(const AMT_PROVISIONING_STATE *state)
{
	DisplayMessage(PROVISIONING_STATE);
	fprintf(stdout,"\t");
	switch (*state)
	{
	case PROVISIONING_STATE_PRE:
		DisplayMessage(STATE_PRE);
		break;
	case PROVISIONING_STATE_IN:
		DisplayMessage(STATE_IN);
		break;
	case PROVISIONING_STATE_POST:
		DisplayMessage(STATE_POST);
		break;
	default:
		DisplayMessage(UNKNOWN);
		break;
	}
	fprintf(stdout,"\n");
}
/*
* Calls to GenerateRngKey Host interface command
* Arguments:
*	None
* Return values:
*	AMT_STATUS_SUCCESS - or AMT_STATUS_IN_PROGRESS on success
*	appropriate error value defined in StatusCodeDefinitions.h - on failure
*/
AMT_STATUS PTHICommand::GenerateRngKey()
{
	AMT_STATUS status;
	UCHAR command[sizeof(GENERATE_RNG_SEED_HEADER)];
	UINT8 *readBuffer;
	UINT32 command_size = sizeof(GENERATE_RNG_SEED_HEADER);
	UINT32 inBuffSize;
	UINT32 outBuffSize=0;
	int	   bytesWritten = 0;

	memcpy(command, &GENERATE_RNG_SEED_HEADER, sizeof(GENERATE_RNG_SEED_HEADER));

	bytesWritten = PTHIClient.SendMessage(command,command_size);
	if((UINT32)bytesWritten != command_size)
	{
		return AMT_STATUS_INTERNAL_ERROR;
	}
	inBuffSize = PTHIClient.GetBufferSize();
	readBuffer = (UINT8*) malloc (sizeof(UINT8) * inBuffSize);
	if(NULL == readBuffer)
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	memset(readBuffer,0,inBuffSize);

	outBuffSize = PTHIClient.ReceiveMessage(readBuffer,inBuffSize);
	do
	{
		if( 0 == outBuffSize)
		{
			status = AMT_STATUS_INTERNAL_ERROR;
			break;
		}

		CFG_GENERATE_RNG_SEED_RESPONSE *tmp_response =
			(CFG_GENERATE_RNG_SEED_RESPONSE*)readBuffer;

		status = tmp_response->Status;

		if( status != AMT_STATUS_SUCCESS && status!= AMT_STATUS_RNG_GENERATION_IN_PROGRESS )
		{
			break;
		}

		PTHI_MESSAGE_HEADER *resp_header =
			&(tmp_response->Header);

		status = _verifyResponseHeader(GENERATE_RNG_SEED_RESPONSE,
			resp_header,outBuffSize);

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}
		status = _verifyGenerateRngKey(tmp_response);

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}
		status = tmp_response->Status;

	}while ( 0 );

	if ( readBuffer != NULL )
	{
		free(readBuffer);
	}
	return status;
}
/*
* Confirms the correctness of the GenerateRngKey response message
* Arguments:
*	response - pointer to the response message
* Return values:
*	PT_STATUS_SUCCESS - on success
*	PTSDK_STATUS_INTERNAL_ERROR - on failure
*/
AMT_STATUS PTHICommand::_verifyGenerateRngKey(const CFG_GENERATE_RNG_SEED_RESPONSE *response)
{
	ULONG ByteCount;

	ByteCount = response->Header.Length;

	if (ByteCount !=
		(sizeof(CFG_GENERATE_RNG_SEED_RESPONSE) - sizeof(PTHI_MESSAGE_HEADER)))
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	return AMT_STATUS_SUCCESS;
}

/*
* Calls to GetRngSeedStatus Host interface command
* Arguments:
*	state - pointer to the pre-allocated structure
*	   which will hold the result
* Return values:
*	PT_STATUS_SUCCESS - on success
*	appropriate error value defined in StatusCodeDefinitions.h - on failure
*/
AMT_STATUS PTHICommand::GetRngSeedStatus(AMT_RNG_STATUS *rngStatus)
{
	AMT_STATUS status;
	UCHAR command[sizeof(GET_RNG_SEED_STATUS_HEADER)];
	UINT8 *readBuffer;
	UINT32 command_size = sizeof(GET_RNG_SEED_STATUS_HEADER);
	UINT32 inBuffSize;
	UINT32 outBuffSize=0;
	int	   bytesWritten = 0;

	if ( rngStatus == NULL )
	{
		return PTSDK_STATUS_INVALID_PARAM;
	}

	memcpy(command, &GET_RNG_SEED_STATUS_HEADER, sizeof(GET_RNG_SEED_STATUS_HEADER));

	bytesWritten = PTHIClient.SendMessage(command,command_size);
	if((UINT32)bytesWritten != command_size)
	{
		return AMT_STATUS_INTERNAL_ERROR;
	}
	inBuffSize = PTHIClient.GetBufferSize();
	readBuffer = (UINT8*) malloc (sizeof(UINT8) * inBuffSize);
	if(NULL == readBuffer)
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	memset(readBuffer,0,inBuffSize);

	outBuffSize = PTHIClient.ReceiveMessage(readBuffer,inBuffSize);
	do
	{
		if( 0 == outBuffSize)
		{
			status = AMT_STATUS_INTERNAL_ERROR;
			break;
		}

		CFG_GET_RNG_SEED_STATUS_RESPONSE *tmp_response =
			(CFG_GET_RNG_SEED_STATUS_RESPONSE*)readBuffer;

		status = tmp_response->Status;

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}

		PTHI_MESSAGE_HEADER *resp_header =
			&(tmp_response->Header);

		status = _verifyResponseHeader(GET_RNG_SEED_STATUS_RESPONSE,
			resp_header,outBuffSize);

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}

		status = _verifyGetRngSeedStatus(tmp_response);

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}
		status = tmp_response->Status;
		*rngStatus = tmp_response->RngStatus;

	}while ( 0 );

	if ( readBuffer != NULL )
	{
		free(readBuffer);
	}
	return status;
}
/*
* Confirms the correctness of the GetRngSeedStatus response message
* Arguments:
*	response - pointer to the response message
* Return values:
*	PT_STATUS_SUCCESS - on success
*	PTSDK_STATUS_INTERNAL_ERROR - on failure
*/
AMT_STATUS PTHICommand::_verifyGetRngSeedStatus(const CFG_GET_RNG_SEED_STATUS_RESPONSE *response)
{
	ULONG ByteCount;

	ByteCount = response->Header.Length;

	if (ByteCount !=
		(sizeof(CFG_GET_RNG_SEED_STATUS_RESPONSE) - sizeof(PTHI_MESSAGE_HEADER)))
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	return AMT_STATUS_SUCCESS;
}

/*
* DisplayRngSeedStatus print RNG seed status
* Arguments:
*	*rngStatus - pointer to the AMT_RNG_STATUS enum
* Return values:
*	NONE
*/
VOID PTHICommand::DisplayRngSeedStatus(const AMT_RNG_STATUS *rngStatus)
{
	DisplayMessage(WORD_RNG_SEED_STATUS);
	fprintf(stdout,"\t");
	switch (*rngStatus)
	{
	case RNG_STATUS_EXIST:
		DisplayMessage(WORD_EXIST);
		break;
	case RNG_STATUS_IN_PROGRESS:
		DisplayMessage(WORD_IN_PROGRESS);
		break;
	case RNG_STATUS_NOT_EXIST:
		DisplayMessage(WORD_NOT_EXIST);
		break;
	default:
		DisplayMessage(UNKNOWN);
		break;
	}
	fprintf(stdout,"\n");
}
/*
* Calls to ZeroTouchEnabled Host interface command
* Arguments:
*	zeroTouchEnabled - pointer to the pre-allocated structure
*	   which will hold the result
* Return values:
*	AMT_STATUS_SUCCESS - on success
*	appropriate error value defined in StatusCodeDefinitions.h - on failure
*/
AMT_STATUS PTHICommand::GetZeroTouchEnabled(AMT_BOOLEAN *zeroTouchEnabled)
{
	AMT_STATUS status;
	UCHAR command[sizeof(GET_ZERO_TOUCH_ENABLED_HEADER)];
	UINT8 *readBuffer;
	UINT32 command_size = sizeof(GET_ZERO_TOUCH_ENABLED_HEADER);
	UINT32 inBuffSize =0;
	UINT32 outBuffSize=0;
	int	   bytesWritten = 0;

	if ( zeroTouchEnabled == NULL )
	{
		return PTSDK_STATUS_INVALID_PARAM;
	}

	memcpy(command, &GET_ZERO_TOUCH_ENABLED_HEADER, sizeof(GET_ZERO_TOUCH_ENABLED_HEADER));

	bytesWritten = PTHIClient.SendMessage(command,command_size);
	if((UINT32)bytesWritten != command_size)
	{
		return AMT_STATUS_INTERNAL_ERROR;
	}
	inBuffSize = PTHIClient.GetBufferSize();
	readBuffer = (UINT8*) malloc (sizeof(UINT8) * inBuffSize);
	if(NULL == readBuffer)
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	memset(readBuffer,0,inBuffSize);

	outBuffSize = PTHIClient.ReceiveMessage(readBuffer,inBuffSize);

	do
	{
		if( 0 == outBuffSize)
		{
			status = AMT_STATUS_INTERNAL_ERROR;
			break;
		}
		CFG_GET_ZERO_TOUCH_ENABLED_RESPONSE *tmp_response =
			(CFG_GET_ZERO_TOUCH_ENABLED_RESPONSE*)readBuffer;

		status = tmp_response->Status;

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}

		PTHI_MESSAGE_HEADER *resp_header =
			&(tmp_response->Header);

		status = _verifyResponseHeader(GET_ZERO_TOUCH_ENABLED_RESPONSE,resp_header,outBuffSize);

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}

		status = _verifyGetZeroTouchEnabled(tmp_response);

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}

		*zeroTouchEnabled = tmp_response->ZeroTouchEnabled;

	}while ( 0 );

	if ( readBuffer != NULL )
	{
		free(readBuffer);
		readBuffer = NULL;
	}
	return status;
}

/*
* Confirms the correctness of the GetZeroTouchEnabled response message
* Arguments:
*	response - pointer to the response message
* Return values:
*	AMT_STATUS_SUCCESS - on success
*	PTSDK_STATUS_INTERNAL_ERROR - on failure
*/
AMT_STATUS PTHICommand::_verifyGetZeroTouchEnabled(const CFG_GET_ZERO_TOUCH_ENABLED_RESPONSE *response)
{
	ULONG ByteCount;

	ByteCount = response->Header.Length;

	if (ByteCount !=
		(sizeof(CFG_GET_ZERO_TOUCH_ENABLED_RESPONSE) - sizeof(PTHI_MESSAGE_HEADER)))
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	return AMT_STATUS_SUCCESS;
}

/*
* DisplayZTCEnabled print if ZTC enabled
* Arguments:
*	*ztcEnabled - pointer to the AMT_BOOLEAN enum
* Return values:
*	NONE
*/
VOID PTHICommand::DisplayZTCEnabled(const AMT_BOOLEAN *ztcEnabled)
{
	DisplayMessage(WORD_ZTC);
	fprintf(stdout,"\t\t\t");
	if (*ztcEnabled)
	{
		DisplayMessage(WORD_ENABLED);
		fprintf(stdout,"\n");
	}
	else
	{
		DisplayMessage(WORD_DISABLED);
		fprintf(stdout,"\n");
	}
}

/*
* Calls to GetProvisioningTlsMode Host interface command
* Arguments:
*	state - pointer to the pre-allocated structure
*	   which will hold the result
* Return values:
*	PT_STATUS_SUCCESS - on success
*	appropriate error value defined in StatusCodeDefinitions.h - on failure
*/
AMT_STATUS PTHICommand::GetProvisioningTlsMode(AMT_PROVISIONING_TLS_MODE *provisioningTlsMode)
{
	AMT_STATUS status;
	UCHAR command[sizeof(GET_PROVISIONING_TLS_MODE_HEADER)];
	UINT8 *readBuffer;
	UINT32 command_size = sizeof(GET_PROVISIONING_TLS_MODE_HEADER);
	UINT32 inBuffSize;
	UINT32 outBuffSize=0;
	int	   bytesWritten = 0;

	if ( provisioningTlsMode == NULL )
	{
		return PTSDK_STATUS_INVALID_PARAM;
	}

	memcpy(command, &GET_PROVISIONING_TLS_MODE_HEADER, sizeof(GET_PROVISIONING_TLS_MODE_HEADER));

	bytesWritten = PTHIClient.SendMessage(command,command_size);
	if((UINT32)bytesWritten != command_size)
	{
		return AMT_STATUS_INTERNAL_ERROR;
	}
	inBuffSize = PTHIClient.GetBufferSize();
	readBuffer = (UINT8*) malloc (sizeof(UINT8) * inBuffSize);
	if(NULL == readBuffer)
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	memset(readBuffer,0,inBuffSize);

	outBuffSize = PTHIClient.ReceiveMessage(readBuffer,inBuffSize);
	do
	{
		if( 0 == outBuffSize)
		{
			status = AMT_STATUS_INTERNAL_ERROR;
			break;
		}

		CFG_GET_PROVISIONING_TLS_MODE_RESPONSE *tmp_response =
			(CFG_GET_PROVISIONING_TLS_MODE_RESPONSE*)readBuffer;

		status = tmp_response->Status;

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}

		PTHI_MESSAGE_HEADER *resp_header =
			&(tmp_response->Header);

		status = _verifyResponseHeader(GET_PROVISIONING_TLS_MODE_RESPONSE,
			resp_header,outBuffSize);

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}

		status = _verifyGetProvisioningTlsMode(tmp_response);

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}
		status = tmp_response->Status;
		*provisioningTlsMode = tmp_response->ProvisioningTlsMode;

	}while ( 0 );

	if ( readBuffer != NULL )
	{
		free(readBuffer);
	}
	return status;
}
/*
* Confirms the correctness of the GetProvisioningTlsMode response message
* Arguments:
*	response - pointer to the response message
* Return values:
*	PT_STATUS_SUCCESS - on success
*	PTSDK_STATUS_INTERNAL_ERROR - on failure
*/
AMT_STATUS PTHICommand::_verifyGetProvisioningTlsMode(const CFG_GET_PROVISIONING_TLS_MODE_RESPONSE *response)
{
	ULONG ByteCount;

	ByteCount = response->Header.Length;

	if (ByteCount !=
		(sizeof(CFG_GET_PROVISIONING_TLS_MODE_RESPONSE) - sizeof(PTHI_MESSAGE_HEADER)))
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	return AMT_STATUS_SUCCESS;
}
/*
* DisplayProvisioningTlsMode print provisioning TLS mode
* Arguments:
*	*provisioningTlsMode - pointer to the AMT_PROVISIONING_TLS_MODE enum
* Return values:
*	NONE
*/
VOID PTHICommand::DisplayProvisioningTlsMode(const AMT_PROVISIONING_TLS_MODE *provisioningTlsMode)
{
	DisplayMessage(WORD_PROVISIONING_TLS_MODE);
	fprintf(stdout,"\t");
	switch (*provisioningTlsMode)
	{
	case PKI:
		DisplayMessage(WORD_PKI);
		break;
	case PSK:
		DisplayMessage(WORD_PSK);
		break;
	case NOT_READY:
		DisplayMessage(WORD_NOT_READY);
		break;
	default:
		DisplayMessage(UNKNOWN);
		break;
	}
	fprintf(stdout,"\n");
}

/*
* Calls to StartConfiguration Host interface command
* Arguments:
*	None
* Return values:
*	AMT_STATUS_SUCCESS - or AMT_STATUS_CERTIFICATE_NOT_READY on success
*	appropriate error value defined in StatusCodeDefinitions.h - on failure
*/
AMT_STATUS PTHICommand::StartConfiguration()
{
	AMT_STATUS status;
	UCHAR command[sizeof(START_CONFIGURATION_HEADER)];
	UINT8 *readBuffer;
	UINT32 command_size = sizeof(START_CONFIGURATION_HEADER);
	UINT32 inBuffSize;
	UINT32 outBuffSize=0;
	int	   bytesWritten = 0;

	memcpy(command, &START_CONFIGURATION_HEADER, sizeof(START_CONFIGURATION_HEADER));

	bytesWritten = PTHIClient.SendMessage(command,command_size);
	if((UINT32)bytesWritten != command_size)
	{
		return AMT_STATUS_INTERNAL_ERROR;
	}
	inBuffSize = PTHIClient.GetBufferSize();
	readBuffer = (UINT8*) malloc (sizeof(UINT8) * inBuffSize);
	if(NULL == readBuffer)
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	memset(readBuffer,0,inBuffSize);

	outBuffSize = PTHIClient.ReceiveMessage(readBuffer,inBuffSize);
	do
	{
		if( 0 == outBuffSize)
		{
			status = AMT_STATUS_INTERNAL_ERROR;
			break;
		}

		CFG_START_CONFIGURATION_RESPONSE *tmp_response =
			(CFG_START_CONFIGURATION_RESPONSE*)readBuffer;

		status = tmp_response->Status;

		if ( status != AMT_STATUS_SUCCESS ||status!= AMT_STATUS_CERTIFICATE_NOT_READY)
		{
			break;
		}

		PTHI_MESSAGE_HEADER *resp_header =
			&(tmp_response->Header);

		status = _verifyResponseHeader(START_CONFIGURATION_RESPONSE,
			resp_header,outBuffSize);

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}
		status = _verifyStartConfiguration(tmp_response);

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}
		status = tmp_response->Status;

	}while ( 0 );

	if ( readBuffer != NULL )
	{
		free(readBuffer);
	}
	return status;
}
/*
* Confirms the correctness of the StartConfiguration response message
* Arguments:
*	response - pointer to the response message
* Return values:
*	PT_STATUS_SUCCESS - on success
*	PTSDK_STATUS_INTERNAL_ERROR - on failure
*/
AMT_STATUS PTHICommand::_verifyStartConfiguration(const CFG_START_CONFIGURATION_RESPONSE *response)
{
	ULONG ByteCount;

	ByteCount = response->Header.Length;

	if (ByteCount !=
		(sizeof(CFG_START_CONFIGURATION_RESPONSE) - sizeof(PTHI_MESSAGE_HEADER)))
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	return AMT_STATUS_SUCCESS;
}

/*
* Calls to SetProvisioningServerOTP Host interface command
* Arguments:
*	passwordOTP AMT_ANSI_STRING structure of OTP password
* Return values:
*	AMT_STATUS_SUCCESS - on success
*	appropriate error value defined in StatusCodeDefinitions.h - on failure
*/
AMT_STATUS PTHICommand::SetProvisioningServerOTP(AMT_ANSI_STRING passwordOTP)
{
	if ( NULL == passwordOTP.Buffer )
	{
		return PTSDK_STATUS_INVALID_PARAM;
	}

	AMT_STATUS status;
	UINT32 msgLength = sizeof(passwordOTP.Length)  + (passwordOTP.Length*sizeof(CHAR));
	PTHI_MESSAGE_HEADER SET_PROVISIONING_SERVER_OTP_HEADER =
	{
		{AMT_MAJOR_VERSION,AMT_MINOR_VERSION},0,{SET_PROVISIONING_SERVER_OTP_REQUEST},msgLength
	};

	UINT32 command_size = sizeof(START_CONFIGURATION_HEADER) + msgLength;
	UCHAR *command;
	command = (UCHAR*) malloc(command_size);
	if(NULL == command)
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	memcpy(command, &SET_PROVISIONING_SERVER_OTP_HEADER, sizeof(SET_PROVISIONING_SERVER_OTP_HEADER));
	memcpy(command + sizeof(SET_PROVISIONING_SERVER_OTP_HEADER),&(passwordOTP.Length), sizeof(passwordOTP.Length));
	memcpy(
		command + sizeof(SET_PROVISIONING_SERVER_OTP_HEADER) + sizeof(passwordOTP.Length),
		passwordOTP.Buffer, passwordOTP.Length);

	UINT8 *readBuffer;
	UINT32 inBuffSize;
	UINT32 outBuffSize=0;
	int	   bytesWritten = 0;

	bytesWritten = PTHIClient.SendMessage(command,command_size);
	if ( command != NULL )
	{
		free(command);
	}
	if((UINT32)bytesWritten != command_size)
	{
		return AMT_STATUS_INTERNAL_ERROR;
	}
	inBuffSize = PTHIClient.GetBufferSize();
	readBuffer = (UINT8*) malloc (sizeof(UINT8) * inBuffSize);
	if(NULL == readBuffer)
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	memset(readBuffer,0,inBuffSize);

	outBuffSize = PTHIClient.ReceiveMessage(readBuffer,inBuffSize);
	do
	{
		if( 0 == outBuffSize)
		{
			status = AMT_STATUS_INTERNAL_ERROR;
			break;
		}

		CFG_SET_PROVISIONING_SERVER_OTP_RESPONSE *tmp_response =
			(CFG_SET_PROVISIONING_SERVER_OTP_RESPONSE*)readBuffer;

		status = tmp_response->Status;

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}

		PTHI_MESSAGE_HEADER *resp_header =
			&(tmp_response->Header);

		status = _verifyResponseHeader(SET_PROVISIONING_SERVER_OTP_RESPONSE,
			resp_header,outBuffSize);

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}
		status = _verifySetProvisioningServerOTP(tmp_response);

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}
		status = tmp_response->Status;

	}while ( 0 );

	if ( readBuffer != NULL )
	{
		free(readBuffer);
	}
	return status;
}
/*
* Confirms the correctness of the SetProvisioningServerOTP response message
* Arguments:
*	response - pointer to the response message
* Return values:
*	PT_STATUS_SUCCESS - on success
*	PTSDK_STATUS_INTERNAL_ERROR - on failure
*/
AMT_STATUS PTHICommand::_verifySetProvisioningServerOTP(const CFG_SET_PROVISIONING_SERVER_OTP_RESPONSE *response)
{
	ULONG ByteCount;

	ByteCount = response->Header.Length;

	if (ByteCount !=
		(sizeof(CFG_SET_PROVISIONING_SERVER_OTP_RESPONSE) - sizeof(PTHI_MESSAGE_HEADER)))
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	return AMT_STATUS_SUCCESS;
}
/*
* Calls to SetDnsSuffix Host interface command
* Arguments:
*	dnsSuffix AMT_ANSI_STRING structure of DNS suffix
* Return values:
*	AMT_STATUS_SUCCESS - on success
*	appropriate error value defined in StatusCodeDefinitions.h - on failure
*/
AMT_STATUS PTHICommand::SetDnsSuffix(AMT_ANSI_STRING dnsSuffix)
{
	if ( NULL == dnsSuffix.Buffer )
	{
		return PTSDK_STATUS_INVALID_PARAM;
	}

	AMT_STATUS status;
	UINT32 msgLength = sizeof(dnsSuffix.Length)  + (dnsSuffix.Length*sizeof(CHAR));
	PTHI_MESSAGE_HEADER SET_DNS_SUFFIX_HEADER =
	{
		{AMT_MAJOR_VERSION,AMT_MINOR_VERSION},0,{SET_DNS_SUFFIX_REQUEST},msgLength
	};

	UINT32 command_size = sizeof(SET_DNS_SUFFIX_HEADER) + msgLength;
	UCHAR *command;
	command = (UCHAR*) malloc(command_size);
	if(NULL == command)
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	memcpy(command, &SET_DNS_SUFFIX_HEADER, sizeof(SET_DNS_SUFFIX_HEADER));
	memcpy(command + sizeof(SET_DNS_SUFFIX_HEADER),&(dnsSuffix.Length), sizeof(dnsSuffix.Length));
	memcpy(command + sizeof(SET_DNS_SUFFIX_HEADER) + sizeof(dnsSuffix.Length),dnsSuffix.Buffer, dnsSuffix.Length);

	UINT8 *readBuffer;
	UINT32 inBuffSize;
	UINT32 outBuffSize=0;
	int	   bytesWritten = 0;

	bytesWritten = PTHIClient.SendMessage(command,command_size);
	if ( command != NULL )
	{
		free(command);
	}
	if((UINT32)bytesWritten != command_size)
	{
		return AMT_STATUS_INTERNAL_ERROR;
	}
	inBuffSize = PTHIClient.GetBufferSize();
	readBuffer = (UINT8*) malloc (sizeof(UINT8) * inBuffSize);
	if(NULL == readBuffer)
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	memset(readBuffer,0,inBuffSize);

	outBuffSize = PTHIClient.ReceiveMessage(readBuffer,inBuffSize);
	do
	{
		if( 0 == outBuffSize)
		{
			status = AMT_STATUS_INTERNAL_ERROR;
			break;
		}

		CFG_SET_DNS_SUFFIX_RESPONSE *tmp_response =
			(CFG_SET_DNS_SUFFIX_RESPONSE*)readBuffer;

		status = tmp_response->Status;

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}

		PTHI_MESSAGE_HEADER *resp_header =
			&(tmp_response->Header);

		status = _verifyResponseHeader(SET_DNS_SUFFIX_RESPONSE,
			resp_header,outBuffSize);

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}
		status = _verifySetDnsSuffix(tmp_response);

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}
		status = tmp_response->Status;

	}while ( 0 );

	if ( readBuffer != NULL )
	{
		free(readBuffer);
	}
	return status;
}
/*
* Confirms the correctness of the SetDnsSuffix response message
* Arguments:
*	response - pointer to the response message
* Return values:
*	PT_STATUS_SUCCESS - on success
*	PTSDK_STATUS_INTERNAL_ERROR - on failure
*/
AMT_STATUS PTHICommand::_verifySetDnsSuffix(const CFG_SET_DNS_SUFFIX_RESPONSE *response)
{
	ULONG ByteCount;

	ByteCount = response->Header.Length;

	if (ByteCount !=
		(sizeof(CFG_SET_DNS_SUFFIX_RESPONSE) - sizeof(PTHI_MESSAGE_HEADER)))
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
		//return AMT_STATUS_SUCCESS;
	}
	return AMT_STATUS_SUCCESS;
}

/*
* Calls to EnumerateHashHandles Host interface command
* Arguments:
*	hashHandles - pointer to the pre-allocated structure
*	   which will hold the result
* Return values:
*	AMT_STATUS_SUCCESS - on success
*	appropriate error value defined in StatusCodeDefinitions.h - on failure
*/
AMT_STATUS PTHICommand::EnumerateHashHandles(AMT_HASH_HANDLES *hashHandles)
{
	AMT_STATUS status;
	UCHAR command[sizeof(ENUMERATE_HASH_HANDLES_HEADER)];
	UINT8 *readBuffer;
	UINT32 command_size = sizeof(ENUMERATE_HASH_HANDLES_HEADER);
	UINT32 inBuffSize;
	UINT32 outBuffSize=0;
	int	   bytesWritten = 0;

	if (NULL == hashHandles)
	{
		return PTSDK_STATUS_INVALID_PARAM;
	}

	memcpy(command, &ENUMERATE_HASH_HANDLES_HEADER, sizeof(ENUMERATE_HASH_HANDLES_HEADER));

	bytesWritten = PTHIClient.SendMessage(command,command_size);
	if((UINT32)bytesWritten != command_size)
	{
		return AMT_STATUS_INTERNAL_ERROR;
	}
	inBuffSize = PTHIClient.GetBufferSize();
	readBuffer = (UINT8*) malloc (sizeof(UINT8) * inBuffSize);
	if(NULL == readBuffer)
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	memset(readBuffer,0,inBuffSize);

	outBuffSize = PTHIClient.ReceiveMessage(readBuffer,inBuffSize);
	do
	{
		if( 0 == outBuffSize)
		{
			status = AMT_STATUS_INTERNAL_ERROR;
			break;
		}

		CFG_GET_HASH_HANDLES_RESPONSE *tmp_response =
			(CFG_GET_HASH_HANDLES_RESPONSE*)readBuffer;

		status = tmp_response->Status;

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}

		PTHI_MESSAGE_HEADER *resp_header =
			&(tmp_response->Header);

		status = _verifyResponseHeader(ENUMERATE_HASH_HANDLES_RESPONSE,
			resp_header,outBuffSize);

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}

		status = _verifyHashHandles(tmp_response);

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}

		memset(hashHandles->Handles, 0, sizeof(UINT32) * CERT_HASH_MAX_NUMBER);

		hashHandles->Length = tmp_response->HashHandles.Length;

		if (CERT_HASH_MAX_NUMBER < hashHandles->Length) {
			status = PTSDK_STATUS_INTERNAL_ERROR;
			break;
		}

		memcpy(hashHandles->Handles, tmp_response->HashHandles.Handles, sizeof(UINT32) * hashHandles->Length);
	} while ( 0 );

	if ( readBuffer != NULL )
	{
		free(readBuffer);
	}
	return status;
}

/*
* Confirms the correctness of the EnumerateHashHandles response message
* Arguments:
*	response - pointer to the response message
* Return values:
*	PT_STATUS_SUCCESS - on success
*	PTSDK_STATUS_INTERNAL_ERROR - on failure
*/
AMT_STATUS PTHICommand::_verifyHashHandles(const CFG_GET_HASH_HANDLES_RESPONSE *response)
{
	ULONG ByteCount;

	ByteCount = response->Header.Length;

	if (ByteCount !=
		sizeof(AMT_STATUS) + sizeof(response->HashHandles.Length)  +(sizeof(UINT32) * response->HashHandles.Length))
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	return AMT_STATUS_SUCCESS;
}


/*
* Calls to GetCertificateHashEntry Host interface command
* Arguments:
*	passwordOTP AMT_ANSI_STRING structure of DNS suffix
* Return values:
*	AMT_STATUS_SUCCESS - on success
*	appropriate error value defined in StatusCodeDefinitions.h - on failure
*/
AMT_STATUS PTHICommand::GetCertificateHashEntry(UINT32 hashHandle, CERTHASH_ENTRY *hashEntry)
{
	if ( NULL == hashEntry )
	{
		return PTSDK_STATUS_INVALID_PARAM;
	}

	AMT_STATUS status;
	UINT32 msgLength = sizeof(hashHandle);
	PTHI_MESSAGE_HEADER GET_CERTHASH_ENTRY_HEADER =
	{
		{AMT_MAJOR_VERSION,AMT_MINOR_VERSION},0,{GET_CERTHASH_ENTRY_REQUEST},msgLength
	};

	UINT32 command_size = sizeof(GET_CERTHASH_ENTRY_HEADER) + msgLength;
	UCHAR *command;
	command = (UCHAR*) malloc(command_size);
	if(NULL == command)
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	memcpy(command, &GET_CERTHASH_ENTRY_HEADER, sizeof(GET_CERTHASH_ENTRY_HEADER));
	memcpy(command + sizeof(GET_CERTHASH_ENTRY_HEADER),&(hashHandle), sizeof(hashHandle));
	UINT8 *readBuffer;
	UINT32 inBuffSize;
	UINT32 outBuffSize=0;
	int	   bytesWritten = 0;

	bytesWritten = PTHIClient.SendMessage(command,command_size);
	if ( command != NULL )
	{
		free(command);
	}
	if((UINT32)bytesWritten != command_size)
	{
		return AMT_STATUS_INTERNAL_ERROR;
	}
	inBuffSize = PTHIClient.GetBufferSize();
	readBuffer = (UINT8*) malloc (sizeof(UINT8) * inBuffSize);
	if(NULL == readBuffer)
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	memset(readBuffer,0,inBuffSize);

	outBuffSize = PTHIClient.ReceiveMessage(readBuffer,inBuffSize);
	do
	{
		if( 0 == outBuffSize)
		{
			status = AMT_STATUS_INTERNAL_ERROR;
			break;
		}

		CFG_GET_CERTHASH_ENTRY_RESPONSE *tmp_response =
			(CFG_GET_CERTHASH_ENTRY_RESPONSE*)readBuffer;

		status = tmp_response->Status;

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}

		PTHI_MESSAGE_HEADER *resp_header =
			&(tmp_response->Header);

		status = _verifyResponseHeader(GET_CERTHASH_ENTRY_RESPONSE,
			resp_header,outBuffSize);

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}
		status = _verifyGetCertificateHashEntry(tmp_response);

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}
		status = tmp_response->Status;
		hashEntry->IsActive = tmp_response->Hash.IsActive;
		hashEntry->IsDefault = tmp_response->Hash.IsDefault;
		hashEntry->Name.Length = tmp_response->Hash.Name.Length;
		hashEntry->HashAlgorithm = tmp_response->Hash.HashAlgorithm;
		memcpy(hashEntry->CertificateHash,tmp_response->Hash.CertificateHash,sizeof(tmp_response->Hash.CertificateHash));
		hashEntry->Name.Buffer =(CHAR*)malloc(hashEntry->Name.Length*sizeof(CHAR));
		if(NULL == hashEntry->Name.Buffer)
		{
			status = PTSDK_STATUS_INTERNAL_ERROR;
			break;
		}
		memcpy(hashEntry->Name.Buffer,&(tmp_response->Hash.Name.Buffer) ,hashEntry->Name.Length*sizeof(CHAR));


	}while ( 0 );

	if ( readBuffer != NULL )
	{
		free(readBuffer);
	}
	return status;
}
/*
* Confirms the correctness of the GetCertificateHashEntry response message
* Arguments:
*	response - pointer to the response message
* Return values:
*	PT_STATUS_SUCCESS - on success
*	PTSDK_STATUS_INTERNAL_ERROR - on failure
*/
AMT_STATUS PTHICommand::_verifyGetCertificateHashEntry(const CFG_GET_CERTHASH_ENTRY_RESPONSE *response)
{
	ULONG ByteCount;

	ByteCount = response->Header.Length;

	if (ByteCount !=
		(sizeof(CFG_GET_CERTHASH_ENTRY_RESPONSE) - sizeof(PTHI_MESSAGE_HEADER)
		- sizeof(CHAR*)+ response->Hash.Name.Length))
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	return AMT_STATUS_SUCCESS;
}

/*
* DisplayHashEntry print Certificate Hash Entry
* Arguments:
*	hashEntry -   the CERTHASH_ENTRY struct
* Return values:
*	NONE
*/
VOID PTHICommand::DisplayHashEntry(CERTHASH_ENTRY hashEntry)
{
	char *printString;
	DisplayMessage(HASH_ENTRY);
	fprintf(stdout,"\n");
	DisplayMessage(FRIENDLY_NAME);
	printString =(char*) malloc((hashEntry.Name.Length+1)*sizeof(CHAR));
	if (NULL != printString)
	{
		memcpy(printString,hashEntry.Name.Buffer,hashEntry.Name.Length*sizeof(CHAR));
		printString[hashEntry.Name.Length*sizeof(CHAR)] ='\0';
		fprintf(stdout, "%s", printString);
		free(printString);
	}
	fprintf(stdout,"\n");
	fprintf(stdout, "Default = ");
	fprintf(stdout, "%s\n", hashEntry.IsDefault ? "true" : "false");
	fprintf(stdout, "Active = ");
	fprintf(stdout, "%s\n", hashEntry.IsActive ? "true" : "false");
	fprintf(stdout, "Hash Algorithm = ");
	int len;
	int rows;
	switch(hashEntry.HashAlgorithm)
	{
	case CERT_HASH_ALGORITHM_MD5:
		fprintf(stdout, "MD5");
		len = 16;
		rows = 2;
		break;
	case CERT_HASH_ALGORITHM_SHA1:
		fprintf(stdout, "SHA1");
		len = 20;
		rows = 4;
		break;
	default:
		fprintf(stdout, "UNKNOWN");
		len = 0;
		break;
	}
	fprintf(stdout,"\n");
	//break the print into rows
	DisplayMessage(CERT_HASH);
	fprintf(stdout,"\n");

	for(int j= 0; j < rows;j++)
	{
		for(int i = 0; i < len/rows; i++)
		{
			fprintf(stdout, " %02X", hashEntry.CertificateHash[j*(len/rows)+i]);
		}
		fprintf(stdout,"\n");
	}
}
/*
* Calls to GetDnsSuffix Host interface command
* Arguments:
*	dnsSuffix - pointer to the pre-allocated structure
*	   which will hold the result
* Return values:
*	PT_STATUS_SUCCESS - on success
*	appropriate error value defined in StatusCodeDefinitions.h - on failure
*/
AMT_STATUS PTHICommand::GetDnsSuffix(AMT_ANSI_STRING *dnsSuffix)
{
	AMT_STATUS status;
	UCHAR command[sizeof(GET_PKI_FQDN_SUFFIX_HEADER)];
	UINT8 *readBuffer;
	UINT32 command_size = sizeof(GET_PKI_FQDN_SUFFIX_HEADER);
	UINT32 inBuffSize;
	UINT32 outBuffSize=0;
	int	   bytesWritten = 0;

	if ( dnsSuffix == NULL )
	{
		return PTSDK_STATUS_INVALID_PARAM;
	}

	memcpy(command, &GET_PKI_FQDN_SUFFIX_HEADER, sizeof(GET_PKI_FQDN_SUFFIX_HEADER));

	bytesWritten = PTHIClient.SendMessage(command,command_size);
	if((UINT32)bytesWritten != command_size)
	{
		return AMT_STATUS_INTERNAL_ERROR;
	}
	inBuffSize = PTHIClient.GetBufferSize();
	readBuffer = (UINT8*) malloc (sizeof(UINT8) * inBuffSize);
	if(NULL == readBuffer)
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	memset(readBuffer,0,inBuffSize);

	outBuffSize = PTHIClient.ReceiveMessage(readBuffer,inBuffSize);
	do
	{
		if( 0 == outBuffSize)
		{
			status = AMT_STATUS_INTERNAL_ERROR;
			break;
		}

		CFG_GET_PKI_FQDN_SUFFIX_RESPONSE *tmp_response =
			(CFG_GET_PKI_FQDN_SUFFIX_RESPONSE*)readBuffer;

		status = tmp_response->Status;

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}

		PTHI_MESSAGE_HEADER *resp_header =
			&(tmp_response->Header);

		status = _verifyResponseHeader(GET_PKI_FQDN_SUFFIX_RESPONSE,
			resp_header,outBuffSize);

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}

		status = _verifyGetDnsSuffix(tmp_response);

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}
		status = tmp_response->Status;
		dnsSuffix->Length = tmp_response->Suffix.Length;
		dnsSuffix->Buffer =(CHAR*)malloc(dnsSuffix->Length * sizeof(CHAR));
		if(NULL == dnsSuffix->Buffer)
		{
			status = PTSDK_STATUS_INTERNAL_ERROR;
			break;
		}
		memcpy(dnsSuffix->Buffer,&(tmp_response->Suffix.Buffer),dnsSuffix->Length * sizeof(CHAR));
	}while ( 0 );

	if ( readBuffer != NULL )
	{
		free(readBuffer);
	}
	return status;
}
/*
* Confirms the correctness of the GetDnsSuffix response message
* Arguments:
*	response - pointer to the response message
* Return values:
*	PT_STATUS_SUCCESS - on success
*	PTSDK_STATUS_INTERNAL_ERROR - on failure
*/
AMT_STATUS PTHICommand :: _verifyGetDnsSuffix(const CFG_GET_PKI_FQDN_SUFFIX_RESPONSE *response)
{
	ULONG ByteCount;

	ByteCount = response->Header.Length;

	if (ByteCount  !=
		sizeof(AMT_STATUS) + sizeof(response->Suffix.Length)  + response->Suffix.Length*sizeof(CHAR))
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	return AMT_STATUS_SUCCESS;
}
/*
* Calls to GetAMTSetupAuditRecord Host interface command
* Arguments:
*	auditRecord - pointer to the pre-allocated structure
*	   which will hold the result
* Return values:
*	PT_STATUS_SUCCESS - on success
*	appropriate error value defined in StatusCodeDefinitions.h - on failure
*/
AMT_STATUS PTHICommand::GetAMTSetupAuditRecord(AMT_PROV_AUDIT_RECORD *auditRecord)
{
	AMT_STATUS status;
	UCHAR command[sizeof(GET_AUDIT_RECORD_HEADER)];
	UINT8 *readBuffer;
	UINT32 command_size = sizeof(GET_AUDIT_RECORD_HEADER);
	UINT32 inBuffSize;
	UINT32 outBuffSize=0;
	int	   bytesWritten = 0;

	if ( auditRecord == NULL )
	{
		return PTSDK_STATUS_INVALID_PARAM;
	}

	memcpy(command, &GET_AUDIT_RECORD_HEADER, sizeof(GET_AUDIT_RECORD_HEADER));

	bytesWritten = PTHIClient.SendMessage(command,command_size);
	if((UINT32)bytesWritten != command_size)
	{
		return AMT_STATUS_INTERNAL_ERROR;
	}
	inBuffSize = PTHIClient.GetBufferSize();
	readBuffer = (UINT8*) malloc (sizeof(UINT8) * inBuffSize);
	if(NULL == readBuffer)
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	memset(readBuffer,0,inBuffSize);

	outBuffSize = PTHIClient.ReceiveMessage(readBuffer,inBuffSize);
	do
	{
		if( 0 == outBuffSize)
		{
			status = AMT_STATUS_INTERNAL_ERROR;
			break;
		}

		CFG_GET_AUDIT_RECORD_RESPONSE *tmp_response =
			(CFG_GET_AUDIT_RECORD_RESPONSE*)readBuffer;

		status = tmp_response->Status;

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}

		PTHI_MESSAGE_HEADER *resp_header =
			&(tmp_response->Header);

		status = _verifyResponseHeader(GET_AUDIT_RECORD_RESPONSE,
			resp_header,outBuffSize);

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}

		status = _verifyGetAMTSetupAuditRecord(tmp_response);

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}
		status = tmp_response->Status;
		auditRecord->ProvServerFQDN.Length = tmp_response->AuditRecord.ProvServerFQDN.Length;
		auditRecord->ProvServerFQDN.Buffer =(CHAR*)malloc(auditRecord->ProvServerFQDN.Length * sizeof(CHAR));
		if(NULL == auditRecord->ProvServerFQDN.Buffer)
		{
			status = PTSDK_STATUS_INTERNAL_ERROR;
			break;
		}
		memcpy(auditRecord,&tmp_response->AuditRecord,sizeof(AMT_PROV_AUDIT_RECORD) - sizeof(AMT_ANSI_STRING));
		memcpy(auditRecord->ProvServerFQDN.Buffer,&(tmp_response->AuditRecord.ProvServerFQDN.Buffer),auditRecord->ProvServerFQDN.Length * sizeof(CHAR));

	}while ( 0 );

	if ( readBuffer != NULL )
	{
		free(readBuffer);
	}
	return status;
}
/*
* Confirms the correctness of the GetAMTSetupAuditRecord response message
* Arguments:
*	response - pointer to the response message
* Return values:
*	PT_STATUS_SUCCESS - on success
*	PTSDK_STATUS_INTERNAL_ERROR - on failure
*/
AMT_STATUS PTHICommand :: _verifyGetAMTSetupAuditRecord(const CFG_GET_AUDIT_RECORD_RESPONSE *response)
{
	ULONG ByteCount;

	ByteCount = response->Header.Length;

	if (ByteCount  !=
		(sizeof(CFG_GET_AUDIT_RECORD_RESPONSE) - sizeof(PTHI_MESSAGE_HEADER) - sizeof(AMT_ANSI_STRING) + sizeof(UINT16) + response->AuditRecord.ProvServerFQDN.Length * sizeof(CHAR)))
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	return AMT_STATUS_SUCCESS;
}



/*
* Calls to GetPID Host interface command
* Arguments:
*	pid - pointer to the pre-allocated structure
*       which will hold the result
* Return values:
*	PT_STATUS_SUCCESS - on success
*	appropriate error value defined in StatusCodeDefinitions.h - on failure
*/
AMT_STATUS PTHICommand::GetPID(UINT8 *pid)
{
	AMT_STATUS status;
	UCHAR command[sizeof(GET_PID_HEADER)];
	UINT8 *readBuffer;
	UINT32 command_size = sizeof(GET_PID_HEADER);
	UINT32 inBuffSize;
	UINT32 outBuffSize=0;
	int	   bytesWritten = 0;

	if ( pid == NULL )
	{
		return PTSDK_STATUS_INVALID_PARAM;
	}

	memcpy(command, &GET_PID_HEADER, sizeof(GET_PID_HEADER));

	bytesWritten = PTHIClient.SendMessage(command,command_size);
	if((UINT32)bytesWritten != command_size)
	{
		return AMT_STATUS_INTERNAL_ERROR;
	}
	inBuffSize = PTHIClient.GetBufferSize();
	readBuffer = (UINT8*) malloc (sizeof(UINT8) * inBuffSize);
	if(NULL == readBuffer)
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	memset(readBuffer,0,inBuffSize);

	outBuffSize = PTHIClient.ReceiveMessage(readBuffer,inBuffSize);
	do
	{
		if( 0 == outBuffSize)
		{
			status = AMT_STATUS_INTERNAL_ERROR;
			break;
		}

		CFG_GET_PID_RESPONSE *tmp_response =
			(CFG_GET_PID_RESPONSE*)readBuffer;

		status = tmp_response->Status;

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}

		PTHI_MESSAGE_HEADER *resp_header =
			&(tmp_response->Header);

		status = _verifyResponseHeader(GET_PID_RESPONSE,
			resp_header,outBuffSize);

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}

		status = _verifyGetPID(tmp_response);

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}
		status = tmp_response->Status;
		memcpy(pid,tmp_response->PID,sizeof(tmp_response->PID));
	}while ( 0 );

	if ( readBuffer != NULL )
	{
		free(readBuffer);
	}
	return status;
}
/*
* Confirms the correctness of the GetPID response message
* Arguments:
*	response - pointer to the response message
* Return values:
*	PT_STATUS_SUCCESS - on success
*	PTSDK_STATUS_INTERNAL_ERROR - on failure
*/
AMT_STATUS PTHICommand :: _verifyGetPID(const CFG_GET_PID_RESPONSE *response)
{
	ULONG ByteCount;

	ByteCount = response->Header.Length;

	if (ByteCount  !=
		(sizeof(CFG_GET_PID_RESPONSE) - sizeof(PTHI_MESSAGE_HEADER)))
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	return AMT_STATUS_SUCCESS;
}



/*
* Calls to SetConfigurationServerFQDN Host interface command
* Arguments:
*	confServerFQDN AMT_ANSI_STRING structure of DNS suffix
* Return values:
*	AMT_STATUS_SUCCESS - on success
*	appropriate error value defined in StatusCodeDefinitions.h - on failure
*/
AMT_STATUS PTHICommand::SetConfServerFQDN(AMT_ANSI_STRING confServerFQDN)
{
	if ( NULL == confServerFQDN.Buffer )
	{
		return PTSDK_STATUS_INVALID_PARAM;
	}

	AMT_STATUS status;
	UINT32 msgLength = sizeof(confServerFQDN.Length)
		+ (confServerFQDN.Length*sizeof(CHAR));
	PTHI_MESSAGE_HEADER SET_CONF_SERVER_FQDN_HEADER =
	{
		{AMT_MAJOR_VERSION,AMT_MINOR_VERSION},0,{SET_CONF_SERVER_FQDN_REQUEST},msgLength
	};

	UINT32 command_size = sizeof(SET_CONF_SERVER_FQDN_HEADER) + msgLength;
	UCHAR *command;
	command = (UCHAR*) malloc(command_size);
	if(NULL == command)
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	memcpy(command, &SET_CONF_SERVER_FQDN_HEADER, sizeof(SET_CONF_SERVER_FQDN_HEADER));
	memcpy(command + sizeof(SET_CONF_SERVER_FQDN_HEADER), &(confServerFQDN.Length),
		sizeof(confServerFQDN.Length));
	memcpy(command + sizeof(SET_CONF_SERVER_FQDN_HEADER) +
		sizeof(confServerFQDN.Length), confServerFQDN.Buffer, confServerFQDN.Length);

	UINT8 *readBuffer;
	UINT32 inBuffSize;
	UINT32 outBuffSize=0;
	int	   bytesWritten = 0;

	bytesWritten = PTHIClient.SendMessage(command,command_size);
	if ( command != NULL )
	{
		free(command);
	}
	if((UINT32)bytesWritten != command_size)
	{
		return AMT_STATUS_INTERNAL_ERROR;
	}
	inBuffSize = PTHIClient.GetBufferSize();
	readBuffer = (UINT8*) malloc (sizeof(UINT8) * inBuffSize);
	if(NULL == readBuffer)
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	memset(readBuffer,0,inBuffSize);

	outBuffSize = PTHIClient.ReceiveMessage(readBuffer,inBuffSize);
	do
	{
		if( 0 == outBuffSize)
		{
			status = AMT_STATUS_INTERNAL_ERROR;
			break;
		}

		CFG_SET_CONF_SERVER_FQDN_RESPONSE *tmp_response =
			(CFG_SET_CONF_SERVER_FQDN_RESPONSE*)readBuffer;

		status = tmp_response->Status;

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}

		PTHI_MESSAGE_HEADER *resp_header =
			&(tmp_response->Header);

		status = _verifyResponseHeader(SET_CONF_SERVER_FQDN_RESPONSE,
			resp_header,outBuffSize);

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}
		status = _verifySetConfServerFQDN(tmp_response);

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}
		status = tmp_response->Status;

	}while ( 0 );

	if ( readBuffer != NULL )
	{
		free(readBuffer);
	}
	return status;
}
/*
* Confirms the correctness of the SetConfServerFQDN response message
* Arguments:
*	response - pointer to the response message
* Return values:
*	PT_STATUS_SUCCESS - on success
*	PTSDK_STATUS_INTERNAL_ERROR - on failure
*/
AMT_STATUS PTHICommand::_verifySetConfServerFQDN(const CFG_SET_CONF_SERVER_FQDN_RESPONSE *response)
{
	ULONG ByteCount;

	ByteCount = response->Header.Length;

	if (ByteCount !=
		(sizeof(CFG_SET_CONF_SERVER_FQDN_RESPONSE) - sizeof(PTHI_MESSAGE_HEADER)))
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	return AMT_STATUS_SUCCESS;
}


/*
* Calls to GetConfServerFQDN Host interface command
* Arguments:
*	confServerFQDN - pointer to the pre-allocated structure
*       which will hold the result
* Return values:
*	PT_STATUS_SUCCESS - on success
*	appropriate error value defined in StatusCodeDefinitions.h - on failure
*/
AMT_STATUS PTHICommand::GetConfServerFQDN(AMT_ANSI_STRING *confServerFQDN)
{
	AMT_STATUS status;
	UCHAR command[sizeof(GET_CONF_SERVER_FQDN_HEADER)];
	UINT8 *readBuffer;
	UINT32 command_size = sizeof(GET_CONF_SERVER_FQDN_HEADER);
	UINT32 inBuffSize;
	UINT32 outBuffSize=0;
	int	   bytesWritten = 0;

	if ( confServerFQDN == NULL )
	{
		return PTSDK_STATUS_INVALID_PARAM;
	}

	memcpy(command, &GET_CONF_SERVER_FQDN_HEADER, sizeof(GET_CONF_SERVER_FQDN_HEADER));

	bytesWritten = PTHIClient.SendMessage(command,command_size);
	if((UINT32)bytesWritten != command_size)
	{
		return AMT_STATUS_INTERNAL_ERROR;
	}
	inBuffSize = PTHIClient.GetBufferSize();
	readBuffer = (UINT8*) malloc (sizeof(UINT8) * inBuffSize);
	if(NULL == readBuffer)
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	memset(readBuffer,0,inBuffSize);

	outBuffSize = PTHIClient.ReceiveMessage(readBuffer,inBuffSize);
	do
	{
		if( 0 == outBuffSize)
		{
			status = AMT_STATUS_INTERNAL_ERROR;
			break;
		}

		CFG_GET_CONF_SERVER_FQDN_RESPONSE *tmp_response =
			(CFG_GET_CONF_SERVER_FQDN_RESPONSE*)readBuffer;

		status = tmp_response->Status;

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}

		PTHI_MESSAGE_HEADER *resp_header =
			&(tmp_response->Header);

		status = _verifyResponseHeader(GET_CONF_SERVER_FQDN_RESPONSE,
			resp_header,outBuffSize);

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}

		status = _verifyGetConfServerFQDN(tmp_response);

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}
		status = tmp_response->Status;
		confServerFQDN->Length = tmp_response->FQDN.Length;
		confServerFQDN->Buffer =(CHAR*)malloc(confServerFQDN->Length * sizeof(CHAR));
		if(NULL == confServerFQDN->Buffer)
		{
			status = PTSDK_STATUS_INTERNAL_ERROR;
			break;
		}
		memcpy(confServerFQDN->Buffer,&(tmp_response->FQDN.Buffer),confServerFQDN->Length * sizeof(CHAR));
	}while ( 0 );

	if ( readBuffer != NULL )
	{
		free(readBuffer);
	}
	return status;
}
/*
* Confirms the correctness of the GetConfServerFQDN response message
* Arguments:
*	response - pointer to the response message
* Return values:
*	PT_STATUS_SUCCESS - on success
*	PTSDK_STATUS_INTERNAL_ERROR - on failure
*/
AMT_STATUS PTHICommand :: _verifyGetConfServerFQDN(const CFG_GET_CONF_SERVER_FQDN_RESPONSE *response)
{
	ULONG ByteCount;

	ByteCount = response->Header.Length;

	if (ByteCount  !=
		sizeof(AMT_STATUS) + sizeof(response->FQDN.Length)  + response->FQDN.Length*sizeof(CHAR))
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	return AMT_STATUS_SUCCESS;
}
/*
* Calls to SetCertificateHashState Host interface command
* Arguments:
*	handle Reference handle to the certificate entry
*  isActive Indicates whether this certificate active for provisioning.
*  FALSE  not active; TRUE  active.
* Return values:
*	AMT_STATUS_SUCCESS - on success
*	appropriate error value defined in StatusCodeDefinitions.h - on failure
*/
AMT_STATUS PTHICommand::SetCertificateHashState(UINT32 handle , AMT_BOOLEAN isActive )
{
	AMT_STATUS status;
	UCHAR *command;
	UINT8 *readBuffer;
	UINT32 command_size = sizeof(CFG_SET_CERTHASH_STATE_REQUEST);
	UINT32 inBuffSize;
	UINT32 outBuffSize=0;
	int	   bytesWritten = 0;

	command = (UCHAR*) malloc(command_size);
	if(NULL == command)
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	memcpy(command, &SET_CERTHASH_STATE_HEADER, sizeof(SET_CERTHASH_STATE_HEADER));
	memcpy(command + sizeof(SET_CERTHASH_STATE_HEADER), &(handle),
		sizeof(handle));
	memcpy(command + sizeof(SET_CERTHASH_STATE_HEADER) +
		sizeof(handle), &(isActive), sizeof(isActive));

	bytesWritten = PTHIClient.SendMessage(command,command_size);
	if ( command != NULL )
	{
		free(command);
	}
	if((UINT32)bytesWritten != command_size)
	{
		return AMT_STATUS_INTERNAL_ERROR;
	}
	inBuffSize = PTHIClient.GetBufferSize();
	readBuffer = (UINT8*) malloc (sizeof(UINT8) * inBuffSize);
	if(NULL == readBuffer)
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	memset(readBuffer,0,inBuffSize);

	outBuffSize = PTHIClient.ReceiveMessage(readBuffer,inBuffSize);
	do
	{
		if( 0 == outBuffSize)
		{
			status = AMT_STATUS_INTERNAL_ERROR;
			break;
		}

		CFG_SET_CERTHASH_STATE_RESPONSE *tmp_response =
			(CFG_SET_CERTHASH_STATE_RESPONSE*)readBuffer;

		status = tmp_response->Status;

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}

		PTHI_MESSAGE_HEADER *resp_header =
			&(tmp_response->Header);

		status = _verifyResponseHeader(SET_CERTHASH_STATE_RESPONSE,
			resp_header,outBuffSize);

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}
		status = _verifySetCertificateHashState(tmp_response);

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}
		status = tmp_response->Status;

	}while ( 0 );

	if ( readBuffer != NULL )
	{
		free(readBuffer);
	}
	return status;
}
/*
* Confirms the correctness of the SetCertificateHashState response message
* Arguments:
*	response - pointer to the response message
* Return values:
*	PT_STATUS_SUCCESS - on success
*	PTSDK_STATUS_INTERNAL_ERROR - on failure
*/
AMT_STATUS PTHICommand::_verifySetCertificateHashState(const CFG_SET_CERTHASH_STATE_RESPONSE *response)
{
	ULONG ByteCount;

	ByteCount = response->Header.Length;

	if (ByteCount !=
		(sizeof(CFG_SET_CERTHASH_STATE_RESPONSE) - sizeof(PTHI_MESSAGE_HEADER)))
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	return AMT_STATUS_SUCCESS;
}


/*
* Calls to SetConfigurationServerFQDN Host interface command
* Arguments:
*	isDefault Is this a default certificate or not. FALSE  not a default; TRUE  default.
*  isDefault Indicates whether this certificate active for provisioning. FALSE  not active; TRUE  active.
*  certificateHash Hashed certificate.
*  name ASCII char set, contains certificate friendly name which is up to 32 bytes.
*  handle The new certificate handle in AMT data base.(OUT)
*
* Return values:
*	AMT_STATUS_SUCCESS - on success
*	appropriate error value defined in StatusCodeDefinitions.h - on failure
*/
AMT_STATUS PTHICommand::AddCertificateHashEntry( AMT_BOOLEAN	isDefault,
												AMT_BOOLEAN	isActive,
												UINT8	certificateHash[20],
												AMT_ANSI_STRING	name ,
												UINT32 *handle)
{
	if ( NULL == name.Buffer )
	{
		return PTSDK_STATUS_INVALID_PARAM;
	}

	AMT_STATUS status;
	UINT32 msgLength =sizeof(isDefault) + sizeof(isActive) + sizeof(UINT8)*20 +
		sizeof(name.Length)  + (name.Length*sizeof(CHAR));
	PTHI_MESSAGE_HEADER ADD_CERTHASH_ENTRY_HEADER =
	{
		{AMT_MAJOR_VERSION,AMT_MINOR_VERSION},0,{ADD_CERTHASH_ENTRY_REQUEST},msgLength
	};

	UINT32 command_size = sizeof(ADD_CERTHASH_ENTRY_HEADER) + msgLength;
	UCHAR *command;
	UINT8 *readBuffer;
	UINT32 inBuffSize;
	UINT32 outBuffSize=0;
	int	   bytesWritten = 0;

	command = (UCHAR*) malloc(command_size);
	if(NULL == command)
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	memcpy(command, &ADD_CERTHASH_ENTRY_HEADER, sizeof(ADD_CERTHASH_ENTRY_HEADER));
	memcpy(command + sizeof(ADD_CERTHASH_ENTRY_HEADER),&(isDefault), sizeof(isDefault));
	memcpy(command + sizeof(ADD_CERTHASH_ENTRY_HEADER) + sizeof(isDefault), &(isActive), sizeof(isActive));
	memcpy(command + sizeof(ADD_CERTHASH_ENTRY_HEADER)
		+ sizeof(isDefault) + sizeof(isActive), certificateHash, sizeof(UINT8)* 20);
	memcpy(command + sizeof(ADD_CERTHASH_ENTRY_HEADER) + sizeof(isDefault) + sizeof(isActive)
		+ sizeof(UINT8)* 20,&(name.Length), sizeof(name.Length));
	memcpy(command + sizeof(ADD_CERTHASH_ENTRY_HEADER) + sizeof(isDefault) + sizeof(isActive)
		+ sizeof(UINT8)* 20 + sizeof(name.Length), name.Buffer, name.Length);

	bytesWritten = PTHIClient.SendMessage(command,command_size);
	if ( command != NULL )
	{
		free(command);
	}
	if((UINT32)bytesWritten != command_size)
	{
		return AMT_STATUS_INTERNAL_ERROR;
	}
	inBuffSize = PTHIClient.GetBufferSize();
	readBuffer = (UINT8*) malloc (sizeof(UINT8) * inBuffSize);
	if(NULL == readBuffer)
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	memset(readBuffer,0,inBuffSize);

	outBuffSize = PTHIClient.ReceiveMessage(readBuffer,inBuffSize);
	do
	{
		if( 0 == outBuffSize)
		{
			status = AMT_STATUS_INTERNAL_ERROR;
			break;
		}

		CFG_ADD_CERTHASH_ENTRY_RESPONSE *tmp_response =
			(CFG_ADD_CERTHASH_ENTRY_RESPONSE*)readBuffer;

		status = tmp_response->Status;

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}

		PTHI_MESSAGE_HEADER *resp_header =
			&(tmp_response->Header);

		status = _verifyResponseHeader(ADD_CERTHASH_ENTRY_RESPONSE,
			resp_header,outBuffSize);

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}
		status = _verifyAddCertificateHashEntry(tmp_response);

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}
		status = tmp_response->Status;
		memcpy(handle,&(tmp_response->Handle),sizeof(tmp_response->Handle));

	}while ( 0 );

	if ( readBuffer != NULL )
	{
		free(readBuffer);
	}
	return status;
}
/*
* Confirms the correctness of the AddCertificateHashEntry response message
* Arguments:
*	response - pointer to the response message
* Return values:
*	PT_STATUS_SUCCESS - on success
*	PTSDK_STATUS_INTERNAL_ERROR - on failure
*/
AMT_STATUS PTHICommand::_verifyAddCertificateHashEntry(const CFG_ADD_CERTHASH_ENTRY_RESPONSE *response)
{
	ULONG ByteCount;

	ByteCount = response->Header.Length;

	if (ByteCount !=  (sizeof(CFG_ADD_CERTHASH_ENTRY_RESPONSE) - sizeof(PTHI_MESSAGE_HEADER)))
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	return AMT_STATUS_SUCCESS;
}

/*
* Calls to DeleteCustomizedCertificateHashEntry Host interface command
* Arguments:
*	handle Reference handle to the certificate entry
* Return values:
*	AMT_STATUS_SUCCESS - on success
*	appropriate error value defined in StatusCodeDefinitions.h - on failure
*/
AMT_STATUS PTHICommand::DeleteCustomizedCertificateHashEntry(UINT32 handle)
{
	AMT_STATUS status;
	UCHAR *command;
	UINT8 *readBuffer;
	UINT32 command_size =  sizeof(CFG_DELETE_CERTHASH_ENTRY_REQUEST);
	UINT32 inBuffSize;
	UINT32 outBuffSize=0;
	int	   bytesWritten = 0;

	command = (UCHAR*) malloc(command_size);
	if(NULL == command)
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	memcpy(command, &DELETE_CERTHASH_ENTRY_HEADER, sizeof(DELETE_CERTHASH_ENTRY_HEADER));
	memcpy(command + sizeof(DELETE_CERTHASH_ENTRY_HEADER), &(handle),
		sizeof(handle));

	bytesWritten = PTHIClient.SendMessage(command,command_size);
	if ( command != NULL )
	{
		free(command);
	}
	if((UINT32)bytesWritten != command_size)
	{
		return AMT_STATUS_INTERNAL_ERROR;
	}
	inBuffSize = PTHIClient.GetBufferSize();
	readBuffer = (UINT8*) malloc (sizeof(UINT8) * inBuffSize);
	if(NULL == readBuffer)
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	memset(readBuffer,0,inBuffSize);

	outBuffSize = PTHIClient.ReceiveMessage(readBuffer,inBuffSize);
	do
	{
		if( 0 == outBuffSize)
		{
			status = AMT_STATUS_INTERNAL_ERROR;
			break;
		}

		CFG_DELETE_CERTHASH_ENTRY_RESPONSE *tmp_response =
			(CFG_DELETE_CERTHASH_ENTRY_RESPONSE*)readBuffer;

		status = tmp_response->Status;

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}

		PTHI_MESSAGE_HEADER *resp_header =
			&(tmp_response->Header);

		status = _verifyResponseHeader(DELETE_CERTHASH_ENTRY_RESPONSE,
			resp_header,outBuffSize);

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}
		status = _verifyDeleteCertificateHashEntry(tmp_response);

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}
		status = tmp_response->Status;

	}while ( 0 );

	if ( readBuffer != NULL )
	{
		free(readBuffer);
	}
	return status;
}
/*
* Confirms the correctness of the DeleteCustomizedCertificateHashEntry response message
* Arguments:
*	response - pointer to the response message
* Return values:
*	PT_STATUS_SUCCESS - on success
*	PTSDK_STATUS_INTERNAL_ERROR - on failure
*/
AMT_STATUS PTHICommand::_verifyDeleteCertificateHashEntry(const CFG_DELETE_CERTHASH_ENTRY_RESPONSE *response)
{
	ULONG ByteCount;

	ByteCount = response->Header.Length;

	if (ByteCount !=
		(sizeof(CFG_DELETE_CERTHASH_ENTRY_RESPONSE) - sizeof(PTHI_MESSAGE_HEADER)))
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	return AMT_STATUS_SUCCESS;
}
/*
* Calls to SetZeroTouchEnabled Host interface command
* Arguments:
* mode False  ZTC disabled, True- ZTC enabled.
* Return values:
*	AMT_STATUS_SUCCESS - on success
*	appropriate error value defined in StatusCodeDefinitions.h - on failure
*/
AMT_STATUS PTHICommand::SetZeroTouchEnabled( AMT_BOOLEAN mode )
{
	AMT_STATUS status;
	UCHAR *command;
	UINT8 *readBuffer;
	UINT32 command_size = sizeof(CFG_SET_ZTC_MODE_REQUEST);
	UINT32 inBuffSize;
	UINT32 outBuffSize=0;
	int	   bytesWritten = 0;

	command = (UCHAR*) malloc(command_size);
	if(NULL == command)
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	memcpy(command, &SET_ZTC_MODE_HEADER, sizeof(SET_ZTC_MODE_HEADER));
	memcpy(command + sizeof(SET_ZTC_MODE_HEADER), &(mode),
		sizeof(mode));

	bytesWritten = PTHIClient.SendMessage(command,command_size);
	if ( command != NULL )
	{
		free(command);
	}
	if((UINT32)bytesWritten != command_size)
	{
		return AMT_STATUS_INTERNAL_ERROR;
	}
	inBuffSize = PTHIClient.GetBufferSize();
	readBuffer = (UINT8*) malloc (sizeof(UINT8) * inBuffSize);
	if(NULL == readBuffer)
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	memset(readBuffer,0,inBuffSize);

	outBuffSize = PTHIClient.ReceiveMessage(readBuffer,inBuffSize);
	do
	{
		if( 0 == outBuffSize)
		{
			status = AMT_STATUS_INTERNAL_ERROR;
			break;
		}

		CFG_SET_ZTC_MODE_RESPONSE *tmp_response =
			(CFG_SET_ZTC_MODE_RESPONSE*)readBuffer;

		status = tmp_response->Status;

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}

		PTHI_MESSAGE_HEADER *resp_header =
			&(tmp_response->Header);

		status = _verifyResponseHeader(SET_ZTC_MODE_RESPONSE,
			resp_header,outBuffSize);

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}
		status = _verifySetZeroTouchEnabled(tmp_response);

		if ( status != AMT_STATUS_SUCCESS )
		{
			break;
		}
		status = tmp_response->Status;

	}while ( 0 );

	if ( readBuffer != NULL )
	{
		free(readBuffer);
	}
	return status;
}
/*
* Confirms the correctness of the SetZeroTouchEnabled response message
* Arguments:
*	response - pointer to the response message
* Return values:
*	PT_STATUS_SUCCESS - on success
*	PTSDK_STATUS_INTERNAL_ERROR - on failure
*/
AMT_STATUS PTHICommand::_verifySetZeroTouchEnabled(const CFG_SET_ZTC_MODE_RESPONSE *response)
{
	ULONG ByteCount;

	ByteCount = response->Header.Length;

	if (ByteCount !=
		(sizeof(CFG_SET_ZTC_MODE_RESPONSE) - sizeof(PTHI_MESSAGE_HEADER)))
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	return AMT_STATUS_SUCCESS;
}
