/*
 * this is a modified source of vtblc, which uses libvtblc
 * A. Gebhardt <albrecht.gebhardt@uni-klu.ac.at>
 */

/*
 *      Copyright (C) 1997 Claus-Justus Heine

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *
 *     This program is a small utility to manipulate certain fields of
 *     the volume table of a floppy tape cartridge.  To be used with
 *     the QIC-40/80/3010/3020 floppy-tape driver "ftape" for Linux.
 */

/* char src[] = "$Source: /usr/local/cvsroot/libvtblc/vtblc-prog.c,v $";
 * char rev[] = "$Revision: 1.2 $";
 * char dat[] = "$Date: 1998/03/16 05:41:10 $";
 */
#include <sys/types.h>
#include <sys/ioctl.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#include <getopt.h>
#include <ctype.h>
#include <stdarg.h>
#include <time.h>

#include "vtblc.h"

static char *tape_dev = FTAPE_RAW_DEV;
static char *progname = NULL;

static vtbl volumes[MAX_VOLUMES];
static u_int8_t vtbl_buffer[FT_SEGMENT_SIZE];
static int num_volumes;
static int verbose;

static char errstr[256];
static char drivetype[256];

static const char *short_options = "f:Vdvlh?pa:t::m:#:";
static const struct option long_options[] =
{
	{"file",               1, 0, 'f'}, /* full path to device */
	{"version",            0, 0, 'V'}, /* version information */
	{"debug",              0, 0, 'd'}, /* switch on debugging, unused */
	{"verbose",            0, 0, 'v'}, /* verbosity level, unused */ 
	{"long",               0, 0, 'l'}, /* print entire label string */ 
	{"usage",              0, 0, 'h'},
	{"help",               0, 0, 'h'},
	{"print",              0, 0, 'p'}, /* pretty print the vtbl */
	{"append",             1, 0, 'a'}, /* append an entry at the end */
	{"truncate",           2, 0, 't'}, /* truncate to given size */
	{"modify",             1, 0, 'm'}, /* modify given vtbl */
	{"vtbl-entry",         1, 0, '#'}, /* vtbl to print or modify */ 
	{0, 0, 0, 0}
};

const char *vtbl_str_tags[] =                                   
{
	"label",
	"date",
	"start",
	"end",
	NULL
};


static int vtbl_no      = -1;
static char *label      = NULL;
static char *datestr    = NULL;
static int setdate      = 0;
static int start_seg    = -1;
static int end_seg      = -1;
static int vtbl_print   = 0;
static int vtbl_long    = 0;
static int vtbl_maxsize = -1;
static int debug        = 0;


#define VTBLC_HELP_TEXT														  \
VTBLC_VERSION "\n"															  \
"\n"																		  \
"Usage: %s [OPTIONS]\n"														  \
"Manipulate the volume table of a floppy tape cartridge, for use with\n"	  \
FTAPE_VERSION", "ZFTAPE_VERSION"\n"											  \
"\n"																		  \
"Mandatory or optional arguments to long options are mandatory or optional\n" \
"for short options too. Unique abbreviations for long options are "			  \
"accepted.\n"																  \
"\n"																		  \
"  -f, --file=FILE       Tape device to use. Default is "					  \
"\""FTAPE_RAW_DEV"\".\n"													  \
"  -h, --help            Print this help.\n"								  \
"  -?                    Same as \'-h\'.\n"									  \
"      --usage           Same as \'-h\'.\n"									  \
"  -V, --version         Print version information.\n"						  \
"  -d, --debug           Unused yet.\n"										  \
"  -v, --verbose         Unused yet.\n"										  \
"  -#, --vtbl-entry=NR   Specify the volume number for \"--print\" and\n"	  \
"                        \"--modify\".\n"									  \
"  -p, --print           Print the volume table entry given by\n"			  \
"                        \"--vtbl-entry\" or the entire table if\n"			  \
"                        \"--vtbl-entry\" has been omitted.\n"				  \
"                        If no other action is specified then \"--print\"\n"  \
"                        is the default.\n"									  \
"  -l, --long            Print entire label strings, requires \"--print\""    \
"  -t, --truncate[=SIZE] Truncate the volume table to SIZE entries by\n"	  \
"                        filling the remainder with zero bytes. If SIZE\n"	  \
"                        is omitted then the last entry is deleted.\n"		  \
"  -a, --append          Append an entry to the end of the volume table.\n"	  \
"  -m, --modify          Modify the volume table entry given by\n"			  \
"                        \"--vtbl-entry\". If \"--vtbl-entry\" has been\n"	  \
"                        omitted then the last entry is modified.\n"		  \
"\n"																		  \
"\"--apend\" and \"--modify\" understand the following sub-options:\n"		  \
"\n"																		  \
"      label=LABEL       The description for this volume table entry.\n"	  \
"                        LABEL will be truncated to 44 bytes.\n"			  \
"      date[=DATE]       The time stamp for this entry. DATE is parsed\n"     \
"                        by strptime(3) using the \"%%T %%D\" format (same\n" \
"                        as \"%%H:%%M:%%S %%m/%%d/%%y\").\n"	              \
"                        If DATE is omitted the current local time is used\n" \
"                        instead.\n"				                       	  \
"      start=SEG_ID      The starting segment for this volume table entry.\n" \
"      end=SEG_ID        The final segment for this volume table entry.\n"	  \
"\n"																		  \
"WARNING:\n"																  \
"\"start=SEG_ID\" and \"end=SEG_ID\" "										  \
"should be used with extreme prejudice.\n"									  \
"They are mainly intended for testing purposes to quickly create some\n"	  \
"dummy volumes to test the file mark support of ftape.\n"


static void usage(FILE *file, int exit_val, char *message, ...)
{
	va_list ap;
	if (message) {
		va_start(ap, message);
		vfprintf(file, message, ap);
		va_end(ap);
	}
	fprintf(file, VTBLC_HELP_TEXT, progname);
	exit(exit_val);
}


int main(int argc, char *argv[])
{
	char *subopts, *value;
	int c;
	int tapefd;
	int first_seg, last_seg;
	progname = argv[0];

	while (1) {
		int option_index = 0;
		
		c = getopt_long(argc, argv, short_options, long_options,&option_index);

		if (c == -1) {
            break;
		}

		switch(c) {
		case 0:
			break;
		case '#':
			vtbl_no = strtol(optarg, NULL, 0);
			break;
		case 'l':
			vtbl_long = 1;
			break;
		case 'a':
		case 'm':
			subopts = optarg;
			while (subopts && *subopts != '\0') {
				vtbl_tag = getsubopt(&subopts, vtbl_str_tags, &value);
				if (value == NULL && vtbl_tag != vtbl_date && vtbl_tag != -1) {
					usage(stderr, 1, "The option \"%s\" needs a value!\n",
						  vtbl_str_tags[vtbl_tag]);
				}
				switch(vtbl_tag) {
				case vtbl_label:
					label = value;
					break;
				case vtbl_date:
					datestr = value;
					setdate = 1;
					break;
				case vtbl_start:
					start_seg = strtol(value, NULL, 0);
					break;
				case vtbl_end:
					end_seg = strtol(value, NULL, 0);
					break;
				default:
					usage(stderr, 1, "Unknown option: %s\n", value);
					break;
				}
			}
			if (c == 'a') {
				op_mode = append;
			} else {
				op_mode = modify;
			}
			break;
		case 'p':
			vtbl_print = 1;
			break;
		case 't':
			if (optarg) vtbl_maxsize = strtol(optarg, NULL, 0);
			op_mode = trunc;
			break;
		case 'f':
			tape_dev = optarg;
			break;
		case 'V':
			printf("vtblc " VTBLC_VERSION "\n");
			exit(0);
			break;
		case 'd':
			debug ++;
			break;
		case 'v':
			verbose++;
			set_vtbl_cntl(verbose);
			break;
		case 'h':
		case '?':
			usage(stdout, 0, NULL);
			break;
		default:
			fprintf (stderr, "?? getopt returned character code 0%o ??\n", c);
			exit(1);
			break;
		}
	}
	if (op_mode == none) {
		vtbl_print ++;
	}
	if ((tapefd = raw_tape_open(tape_dev, op_mode)) == -1) {
		get_vtbl_cntl(errstr, drivetype);
		printf("%s\n", errstr);
		exit(1);
	}
	if ((num_volumes = read_vtbl(tapefd, volumes, vtbl_buffer,
								 &first_seg, &last_seg)) == -1 ) {
		(void)raw_tape_close(tapefd);
		get_vtbl_cntl(errstr, drivetype);
		printf("%s\n", errstr);
		exit(1);
	}
	if (op_mode == append) {
		if (++num_volumes > MAX_VOLUMES) {
			fprintf(stderr, "Too many volumes: %d (%d max)\n", 
					num_volumes, MAX_VOLUMES);
			(void)raw_tape_close(tapefd);
			get_vtbl_cntl(errstr, drivetype);
			printf("%s\n", errstr);
			exit(1);
		}
		if (end_seg == -1 || start_seg == -1) {
			fprintf(stderr, "Missing start- or ending segment\n");
		}
		op_mode = modify;
		vtbl_no = -1;
	}
	if (op_mode == modify) {
		if (setdate && set_date(datestr, volumes, num_volumes, vtbl_no)) {
			(void)raw_tape_close(tapefd);
			get_vtbl_cntl(errstr, drivetype);
			printf("%s\n", errstr);
			exit(1);
		}
		if (label && set_label(label, volumes, num_volumes, vtbl_no)) {
			(void)raw_tape_close(tapefd);
			get_vtbl_cntl(errstr, drivetype);
			printf("%s\n", errstr);
			exit(1);
		}
		if (start_seg != -1 &&
			set_start(start_seg, volumes, num_volumes, vtbl_no)) {
			(void)raw_tape_close(tapefd);
			get_vtbl_cntl(errstr, drivetype);
			printf("%s\n", errstr);
			exit(1);
		}
		if (end_seg != -1 &&
			set_end(end_seg, volumes, num_volumes, vtbl_no)) {
			(void)raw_tape_close(tapefd);
			get_vtbl_cntl(errstr, drivetype);
			printf("%s\n", errstr);
			exit(1);
		}
		if (set_id(volumes, num_volumes, vtbl_no)) {
			(void)raw_tape_close(tapefd);
			get_vtbl_cntl(errstr, drivetype);
			printf("%s\n", errstr);
			exit(1);
		}
	} else if (op_mode == trunc) {
		if (vtbl_maxsize == -1) {
			vtbl_maxsize = num_volumes - 1;
		}
		if (vtbl_maxsize < 0 || vtbl_maxsize > num_volumes) {
			fprintf(stderr, "Volume number too big or negative: %d\n", 
					vtbl_no);
			(void)raw_tape_close(tapefd);
			get_vtbl_cntl(errstr, drivetype);
			printf("%s\n", errstr);
			exit(1);
		}
		num_volumes = vtbl_maxsize;
	}
	if (vtbl_print) {
		if (vtbl_long) {
			print_long_vtbl(volumes, num_volumes, first_seg, last_seg);
		}
		else {
			print_vtbl(volumes, num_volumes, first_seg, last_seg);
		}
	}
	if (write_vtbl(tapefd, volumes, vtbl_buffer, num_volumes, first_seg,
				   op_mode == trunc)) {
		(void)raw_tape_close(tapefd);
		get_vtbl_cntl(errstr, drivetype);
		printf("%s\n", errstr);
		exit(1);
	}
	if (raw_tape_close(tapefd)) {
		get_vtbl_cntl(errstr, drivetype);
		printf("%s\n", errstr);
		exit(1);
	} else {
		exit(0);
	}
	exit(0);
}

/*
 * Local variables:
 *  version-control: t
 *  kept-new-versions: 5
 *  c-basic-offset: 4
 *  tab-width: 4
 * End:
 */
